<?php

namespace Upsellar\Engine\Blocks;

use Upsellar\Engine\Blocks\UpsellarCartBlockRenderer;
use Upsellar\Engine\Traditional\UpsellarTraditionalCartBlockRenderer;
use Upsellar\Engine\Blocks\UpsellarRelatedProductsBlockRenderer;

defined('ABSPATH') || exit;

class UpsellarBlockBuilder {

    /**
     * UpsellarBlockBuilder constructor.
     * 
     * Register all the blocks and their renderers.
     * 
     * @since 1.0.0
     */
    public function __construct() {

        // Handle AJAX requests from JavaScript
        add_action('wp_ajax_fetch_upsellar_blocks', array($this, 'upsellar_handle_fetch_upsellar_blocks'));
        add_action('wp_ajax_nopriv_fetch_upsellar_blocks', array($this,'upsellar_handle_fetch_upsellar_blocks'));

    }

    /**
     * Handle AJAX request to fetch blocks.
     * 
     * @return void
     * @since 1.0.0
     */
    public function upsellar_handle_fetch_upsellar_blocks(){
        check_ajax_referer('upsellar_track_product_interaction_nonce', 'nonce');
        
        if (isset($_POST['product_ids'])) {
            // Ensure product_ids is always treated as an array
            // $product_ids = (array) wp_unslash( $_POST['product_ids'] );
            // $product_ids = array_map('intval', $product_ids);
            $product_ids = array_map('intval', (array) wp_unslash($_POST['product_ids']));
            $product_ids = array_combine($product_ids, $product_ids);

        } else {
            $product_ids = array();
        }

        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 0;
        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : '';

        // Validate the inputs
        $product_ids = array_filter($product_ids, 'wc_get_product'); // Filter out invalid product IDs

        // print_r(array_keys($product_ids));
        if (empty($product_ids)) {
            wp_die('No valid product IDs.');
        }
        if ($limit <= 0) {
            wp_die('Invalid limit specified');
        }
        if (empty($type)) {
            wp_die('Type is required');
        }
        
        $response = null;
        switch($type){
            case 'related-products':
                $response = $this->upsellar_render_related_products_block($product_ids, $limit);
                break;
            case 'cart-cross-sells':
                $response = $this->upsellar_render_cart_block($product_ids, $limit);
                break;
            case 'traditional-cart-cross-sells':
                $response = $this->upsellar_render_traditional_cart_block($product_ids, $limit);
                break;
            default:
                wp_die();
        }
   
        // Set the Content-Type to text/html
        header('Content-Type: text/html; charset=utf-8');

        // Escape the output before sending it back
        echo wp_kses_post($response);
    
        wp_die();
    }

    /**
     * Render the related products block.
     * 
     * @param array $product_ids The product IDs to get related products for.
     * @param int $limit The maximum number of products to display.
     * @return string The HTML to display.
     * @since 1.0.0
     */
    private function upsellar_render_related_products_block($product_ids, $limit){
        $relatedProductsBlockRenderer = new UpsellarRelatedProductsBlockRenderer();
        $response = $relatedProductsBlockRenderer->upsellar_render_block($product_ids, $limit);
        return $response;
    }

    /**
     * Render the Gutenberg cart block.
     * 
     * @param array $product_ids The product IDs to get related products for.
     * @param int $limit The maximum number of products to display.
     * @return string The HTML to display.
     * @since 1.0.0
     */
    private function upsellar_render_cart_block($product_ids, $limit){
        $cartBlockRenderer = new UpsellarCartBlockRenderer();
        $response = $cartBlockRenderer->upsellar_render_block($product_ids, $limit);
        return $response;
    }

    /**
     * Render the traditional cart block.
     * 
     * @param array $product_ids The product IDs to get related products for.
     * @param int $limit The maximum number of products to display.
     * @return string The HTML to display.
     * @since 1.0.0
     */
    private function upsellar_render_traditional_cart_block($product_ids, $limit){
        $cartBlockRenderer = new UpsellarTraditionalCartBlockRenderer();
        $response = $cartBlockRenderer->upsellar_render_block($product_ids, $limit);
        return $response;
    }


}