<?php 

namespace Upsellar\Admin;
use WP_Error;

defined('ABSPATH') || exit;

class UpsellarSettingsManager {

    // Constants for option names
    const API_KEY_OPTION = 'upsellar_api_key';
    const PRODUCT_PAGE_DISPLAY = 'upsellar_product_page_display';
    const PRODUCT_PAGE_COLUMNS = 'upsellar_product_page_columns';
    const PRODUCT_PAGE_ROWS = 'upsellar_product_page_rows';
    const CROSS_SELLS_DISPLAY = 'upsellar_cart_cross_sells_display';
    const CROSS_SELLS_COLUMNS = 'upsellar_cart_cross_sells_columns';
    const CROSS_SELLS_ROWS = 'upsellar_cart_cross_sells_rows';

    /**
     * Register settings with default value.
     * 
     * @return void
     * @since 1.0.0
     */
    public function upsellar_register_settings() {
        // Register settings
        register_setting(
            'upsellar_settings', 
            self::API_KEY_OPTION,
            [
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
            ]
        );

        $this->upsellar_register_setting_with_default(self::PRODUCT_PAGE_DISPLAY, true, [$this, 'upsellar_validate_display'], 'integer');
        $this->upsellar_register_setting_with_default(self::PRODUCT_PAGE_COLUMNS, 5, [$this, 'upsellar_validate_columns'], 'integer');
        $this->upsellar_register_setting_with_default(self::PRODUCT_PAGE_ROWS, 1, [$this, 'upsellar_validate_rows'], 'integer');
        $this->upsellar_register_setting_with_default(self::CROSS_SELLS_DISPLAY, true, [$this, 'upsellar_validate_display'], 'integer');
        $this->upsellar_register_setting_with_default(self::CROSS_SELLS_COLUMNS, 3, [$this, 'upsellar_validate_columns'], 'integer');
        $this->upsellar_register_setting_with_default(self::CROSS_SELLS_ROWS, 1, [$this, 'upsellar_validate_rows'], 'integer');
    }

    /**
     * Register settings with default value.
     * 
     * @param string $name The name of the setting.
     * @param mixed $default The default value of the setting.
     * @param callable $callback The validation callback.
     * @return void
     * @since 1.0.0
     */
    private function upsellar_register_setting_with_default($name, $default, $callback = '', $callback_type='string') {
        register_setting(
            'upsellar_appearance_options', 
            $name,
            [
                'type' => $callback_type,
                'sanitize_callback' => $callback,
            ]
        );

        if (get_option($name) === false) {
            update_option($name, $default);
        }
    }

    /**
     * Validate the display setting.
     * 
     * @param mixed $input The input value.
     * @return mixed The validated input.
     * @since 1.0.0
     */
    public function upsellar_validate_display($input) {
        // should return true or false
        return ($input == true || $input == false) ? $input : true; // Default to true if invalid
    }

    /**
     * Validate the column settings.
     * 
     * @param mixed $input The input value.
     * @return mixed The validated input.
     * @since 1.0.0
     */
    public function upsellar_validate_columns($input) {
        return ($input >= 1 && $input <= 5) ? intval($input) : 5; // Default to 5 if invalid
    }
    
    /**
     * Validate the rows setting.
     * 
     * @param mixed $input The input value.
     * @return mixed The validated input.
     * @since 1.0.0
     */
    public function upsellar_validate_rows($input) {
        return ($input >= 1 && $input <= 3) ? intval($input) : 1; // Default to 1 if invalid
    }

    /**
     * Get the API key for use with the Upsellar server.
     * 
     * @return string The API key, or an empty string if not set.
     * @since 1.0.0
     */
    public function upsellar_get_api_key() {
        return get_option(self::API_KEY_OPTION);
    }

    /**
     * Test the API key to see if it works.
     * 
     * @param string $apiKey The API key to test.
     * @return string 'success' or 'failed' if the API key is valid or not, respectively.
     * @since 1.0.0
     */
    public function upsellar_test_api_key($apiKey) {
        $url = 'https://upsellar.com/api/test';

        return $this->upsellar_make_api_request($url, $apiKey, true) === "success" ? 'success' : 'failed';
    }

    /**
     * Test the key to see if it works.
     * 
     * @param string $option The option name.
     * @param string $old_value The old option value.
     * @param string $value The new option value.
     * @return void
     * @since 1.0.0
     */
    public function upsellar_test_api_key_on_update($option, $old_value, $value) {
        if ($option === self::API_KEY_OPTION) {
            $testResult = $this->upsellar_test_api_key($value);
            set_transient('upsellar_api_key_test_result', $testResult, 60);
        }
    }

    /**
     * Get the dashboard data from Upsellar API.
     * 
     * @param string $apiKey The API key to use for the request.
     * @return array The dashboard data from the Upsellar API, or an error message if something went wrong.
     * @since 1.0.0
     */
    public function upsellar_get_dashboard_data($apiKey) {
        $url = 'https://upsellar.com/api/v1/get-dashboard-stats';
        return $this->upsellar_make_api_request($url, $apiKey, true);
    }
    
    /**
     * Make an API request.
     * 
     * See README.TXT for the API documentation.
     * 
     * @param string $url The API endpoint to make the request to.
     * @param string $apiKey The API key to use for the request.
     * @param bool $returnJson Whether to return the JSON response or not. Defaults to false.
     * @return array|string The response from the API request. If $returnJson is true, this will be a JSON string; otherwise it will be an associative array.
     * @since 1.0.0
     */
    private function upsellar_make_api_request($url, $apiKey, $returnJson = false) {
        // Append the API key to the URL as a query parameter if needed
        $url = add_query_arg('api_key', $apiKey, $url);

        $args = [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Accept' => 'application/json',
            ],
        ];

        $response = wp_remote_post($url, $args);

        if (is_wp_error($response)) {
            // Handle error accordingly
            return new WP_Error('api_request_error', $response->get_error_message());
        }

        $httpCode = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($httpCode === 200) {
            return $returnJson ? json_decode($body, true) : $body;
        } else {
            // You can include more sophisticated error handling here if needed
            return new WP_Error('api_request_error', "API Request failed with response code $httpCode and message: $body");
        }

    }
}    