<?php 

namespace Upsellar\Admin;

use Upsellar\Admin\UpsellarSettingsManager;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class UpsellarProductsController {

    /**
     * Initialize the class.
     * 
     * @since 1.0.0
     */
    public function __construct() {

        // Hook into WooCommerce actions
        add_action('woocommerce_new_product', array($this, 'upsellar_handle_product_update'));
        add_action('woocommerce_update_product', array($this, 'upsellar_handle_product_update'));
        add_action('wp_trash_post', array($this, 'upsellar_handle_product_trash'), 99 );
        add_action('delete_post', array($this, 'upsellar_handle_product_delete'), 99 );
        add_action('untrashed_post', array($this, 'upsellar_handle_product_update'), 99 );
    }

    /**
     * Update Upsellar's server with product data.
     * 
     * @param int $product_id The product id.
     * @return void
     * @since 1.0.0
     */
    public function upsellar_handle_product_update($product_id) {

        if (!$this->upsellar_is_woocommerce_product($product_id)) {
            return;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            return;
        }

        $product_data = $this->upsellar_prepare_product_data($product);


        $this->upsellar_send_to_server('update', $product_data);
    }

    /**
     * Delete product from Upsellar's server.
     * 
     * @param int $product_id The product id.
     * @return void
     * @since 1.0.0
     */
    public function upsellar_handle_product_trash($product_id) {

        if (!$this->upsellar_is_woocommerce_product($product_id)) {
            return;
        }

        $this->upsellar_send_to_server('trash', array('id' => $product_id));
    }

    /**
     * Delete product from Upsellar's server.
     * 
     * @param int $product_id The product id.
     * @return void
     * @since 1.0.0
     */
    public function upsellar_handle_product_delete($product_id) {

        if (!$this->upsellar_is_woocommerce_product($product_id)) {
            return;
        }

        $this->upsellar_send_to_server('delete', array('id' => $product_id));
    }

    /**
     * Prepare the product data to be sent to the Upsellar server.
     * 
     * @param WC_Product $product The product object.
     * @return array The prepared product data.
     * @since 1.0.0
     */
    private function upsellar_prepare_product_data($product) {
        // Prepare product data array
        $product_data = array(
            'id' => $product->get_id(),
            'name' => $product->get_name(),
        );
    
        // Get product categories
        $categories = wp_get_post_terms($product->get_id(), 'product_cat');
        // $category_names = array_map(function($term) {
        //     return $term->name;
        // }, $categories);
    
        $category_ids = array_map(function($term) {
            return $term->term_id;
        }, $categories);
    
        // Add categories to product data
        //$product_data['categories'] = $category_names;
        $product_data['category_ids'] = $category_ids;
    
        return $product_data;
    }
    
    /**
     * Send the product data to the Upsellar server.
     * 
     * @param string $action The action to be performed (create or update).
     * @param array $data The product data to be sent.
     * @return void
     * @since 1.0.0
     */
    private function upsellar_send_to_server($action, $data) {
        
        $api_url = 'https://upsellar.com/api/v1/change-product';

        $data['action'] = $action;

        $response = $this->upsellar_make_api_request($api_url, true, $data);

    }

    /**
     * Make an API request to a given URL.
     * 
     * See README.TXT for the API documentation.
     * 
     * @param string $url The URL to make a request to.
     * @param bool $returnJson Whether to return the response in JSON format. Defaults to false.
     * @param array $postData The data to be sent with the request. Defaults to an empty array.
     * @return mixed The response from the server. If $returnJson is true, this will be a string in JSON format; otherwise it's an object or array depending on how cURL handles the response.
     * @since 1.0.0
     */
    private function upsellar_make_api_request($url, $returnJson = false, $postData = []) {
        $settingsManager = new UpsellarSettingsManager();
        $apiKey = $settingsManager->upsellar_get_api_key(); 

        // Prepare the request body as JSON if $postData is provided
        $body = wp_json_encode($postData);

        // Set up the request headers
        $headers = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ];

        // Set up the arguments for wp_remote_post
        $args = [
            'method'    => 'POST',
            'headers'   => $headers,
            'body'      => $body,
            'timeout'   => 45,
            'sslverify' => true,
        ];

        // Make the request
        $response = wp_remote_post($url, $args);

        // Check for WP_Error
        if (is_wp_error($response)) {
            // Optional: Handle error accordingly
            return false;
        }

        $httpCode = wp_remote_retrieve_response_code($response);
        $responseBody = wp_remote_retrieve_body($response);
    
        if ($httpCode === 200) {
            return $returnJson ? json_decode($responseBody, true) : $responseBody;
        } else {
            // Optional: Add error logging here
            return false;
        }
    }

    /**
     * Check if a product is WooCommerce product.
     * 
     * @param int $product_id The product ID.
     * @return bool True if the product is WooCommerce product, false otherwise.
     * @since 1.0.0
     */
    private function upsellar_is_woocommerce_product($product_id) {
        return get_post_type($product_id) === 'product';
    }
    
}
