<?php 
namespace Upsellar\Admin;

use Upsellar\Interaction\UpsellarInteractionTrackingController;
use Upsellar\Interaction\UpsellarWoocommerceInteractionController;
use Upsellar\Engine\UpsellarEngineController;
use Upsellar\Admin\UpsellarProductsController;

defined('ABSPATH') || exit;

class UpsellarPluginInitializer {

    private $main_file;
    private $configLoader;

    /**
     * UpsellarPluginInitializer constructor.
     * 
     * @param $main_file string Main plugin file path.
     * @param $configLoader UpsellarConfigLoader object.
     * @return void
     * @since  1.0.0
     */
    public function __construct($main_file) {
        $this->main_file = $main_file;
        $this->configLoader = new UpsellarConfigLoader(defined('UPSELLAR_ENVIRONMENT') ? UPSELLAR_ENVIRONMENT : 'prod');

        if ($this->upsellar_get_config('require_ssl', true)) {
            $httpsEnforcer = new UpsellarHttpsEnforcer($this->main_file);
            add_action('admin_init', [$httpsEnforcer, 'upsellar_enforce_https']);
            register_activation_hook($this->main_file, [$httpsEnforcer, 'upsellar_check_on_activation']);
        }

        add_action('admin_menu', [$this, 'upsellar_add_admin_menu']);
        add_action('admin_init', [$this, 'upsellar_register_settings']);

        // the hooks for enqueuing scripts and styles
        $assetManager = new UpsellarAssetManager();
        add_action('admin_enqueue_scripts', [$assetManager, 'upsellar_admin_enqueue_scripts']);
        add_action('admin_enqueue_scripts', [$assetManager, 'upsellar_admin_enqueue_styles']);
        
        
        // Retrieve the API key using UpsellarSettingsManager
        $settingsManager = new UpsellarSettingsManager();
        $apiKey = $settingsManager->upsellar_get_api_key();

        // Only proceed with certain actions and initializations if API key is present
        if (!empty($apiKey)) {
            //add_action( 'enqueue_block_editor_assets', [$assetManager, 'blockEditorEnqueueScripts']);

            // Hooks for public-facing pages
            add_action('wp_enqueue_scripts', [$assetManager, 'upsellar_enqueue_scripts']);
            // add_action('wp_enqueue_scripts', [$assetManager, 'upsellar_enqueue_styles']); // Temporarily disabling styles for now since we're not using them yet
            
            $this->upsellar_initialize_interaction_tracking();
            $this->upsellar_initialize_engine();
            $this->upsellar_initialize_products_controller();
        }
    }

    /**
     * Adds the menu item to the admin panel.
     * 
     * @return void
     * @since  1.0.0
     */
    public function upsellar_add_admin_menu() {
        (new UpsellarAdminPage())->upsellar_add_admin_menu();
    }

    /**
     * Registers the settings for this plugin.
     * 
     * @return void
     * @since  1.0.0
     */
    public function upsellar_register_settings() {
        (new UpsellarSettingsManager())->upsellar_register_settings();
    }

    /**
     * Initializes the interaction tracking.
     * 
     * @return void
     * @since  1.0.0
     */
    private function upsellar_initialize_interaction_tracking() {
        $interactionTracker = new \Upsellar\Interaction\UpsellarInteractionTrackingController();
        $woocommerceTracker = new \Upsellar\Interaction\UpsellarWoocommerceInteractionController($interactionTracker);
    }
    
    /**
     * Initializes the engine.
     * 
     * @return void
     * @since 1.0.0
     */
    private function upsellar_initialize_engine() {
        new UpsellarEngineController();
    }
    
    /**
     * Initializes the products controller.
     * 
     * @return void
     * @since  1.0.0
     */
    private function upsellar_initialize_products_controller() {
        new UpsellarProductsController();
    }

    /**
     * Gets a config value.
     * 
     * @param string $key The key of the config value.
     * @param mixed  $default The default value to use if the key is not set.
     * @return mixed The config value or the default if not set.
     * @since  1.0.0
     */
    public function upsellar_get_config($key, $default = null) {
        return $this->configLoader->upsellar_get_config_value($key, $default);
    }
}

