<?php

namespace Upsellar\Admin;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class UpsellarAdminPage {
    private $settingsManager;

    /**
     * Initialize the class and set its properties.
     * 
     * @since 1.0.0
     */
    public function __construct() {
        add_action('admin_menu', [$this, 'upsellar_add_admin_menu']);

        $this->settingsManager = new UpsellarSettingsManager();

    }

    /**
     * Adds the admin menu and submenus.
     * 
     * @return void
     * @since 1.0.0
     */
    public function upsellar_add_admin_menu() {
        // Main menu page
        add_menu_page(
            'Upsellar Dashboard',
            'Upsellar',
            'manage_options',
            'upsellar-dashboard',
            [$this, 'upsellar_display_dashboard'],
            'dashicons-chart-line'
        );

        // Submenu page for settings
        add_submenu_page(
            'upsellar-dashboard',
            'Upsellar Settings',
            'Dashboard',
            'manage_options',
            'upsellar-dashboard',
            [$this, 'upsellar_display_dashboard']
        );

        // Submenu page for settings
        add_submenu_page(
            'upsellar-dashboard',
            'Upsellar Settings',
            'Appearance',
            'manage_options',
            'upsellar-appearance',
            [$this, 'upsellar_display_appearance']
        );
        
        // Submenu page for settings
        add_submenu_page(
            'upsellar-dashboard',
            'Upsellar Settings',
            'Settings',
            'manage_options',
            'upsellar-settings',
            [$this, 'upsellar_display_settings']
        );
    }

    /**
     * Display the settings page.
     * 
     * @return void
     * @since 1.0.0
     */
    public function upsellar_display_dashboard() {
        $this->upsellar_check_permissions();

        // Retrieve the current API key
        $apiKey = $this->settingsManager->upsellar_get_api_key();

        // Get the dashboard data from the API
        $dashboardData = $this->settingsManager->upsellar_get_dashboard_data($apiKey);
    
        // Check if the response is an error
        if (is_wp_error($dashboardData)) {
            // Store the error message in a transient
            set_transient('upsellar_dashboard_error', "Could not retrieve dashboard data. Make sure you're API key is correct in the Upsellar settings tab.", 60);
        } else {
            // Process data for charts
            $ctrLabels = array_keys($dashboardData['ctrData']);
            $ctrDataValues = array_values($dashboardData['ctrData']);
    
            $recommendationLogsLabels = array_keys($dashboardData['recommendationLogsData']);
            $recommendationLogsDataValues = array_map('intval', array_values($dashboardData['recommendationLogsData']));    
            
            // Prepare data for Chart.js
            $chartData = array(
                'ctrLabels' => $ctrLabels,
                'ctrData' => array(
                    array(
                        'data' => $ctrDataValues,
                        'tension' => 0.4,
                        'label' => 'CTR',
                        'borderColor' => '#ef4444',
                        'fill' => false,
                    ),
                ),
                'recommendationLogsLabels' => $recommendationLogsLabels,
                'recommendationLogsData' => array(
                    array(
                        'data' => $recommendationLogsDataValues,
                        'tension' => 0.4,
                        'label' => 'Recommendation Logs',
                        'borderColor' => '#0ea5e9',
                        'fill' => false,
                    ),
                ),
            );

            set_transient('upsellar_dashboard_data', $dashboardData, 60);

            wp_localize_script('upsellar-scripts-admin', 'upsellarChartData', $chartData);

        }

        include_once plugin_dir_path(__FILE__) . 'views/admin-dashboard.php';
    }

    /**
     * Display the appearance page of the plugin admin area.
     * 
     * This function checks if the user has the necessary permissions to view and manage the plugin options.
     * If the check passes, it includes the 'admin-appearance.php' file from the views directory which is responsible for displaying the appearance settings in the admin area.
     *
     * @return void This function does not return a value.
     * @since 1.0.0
     */
    public function upsellar_display_appearance() {
        $this->upsellar_check_permissions();
        include_once plugin_dir_path(__FILE__) . 'views/admin-appearance.php';
    }

    /**
     * Display the settings page of the plugin admin area.
     * 
     * This function checks if the user has the necessary permissions to view and manage the plugin options.
     * If the check passes, it includes the 'admin-settings.php' file from the views directory which is responsible for displaying the settings in the admin area.
     * 
     * See README.TXT for the API documentation.
     * 
     * @return void This function does not return a value.
     * @since 1.0.0
     */
    public function upsellar_display_settings() {
        $this->upsellar_check_permissions();
    
        // Handle import request
        if (isset($_POST['upsellar_import_sales_data']) && $_POST['upsellar_import_sales_data'] == '1') {
            // Check for nonce security
            if (check_admin_referer('upsellar_import_nonce', 'upsellar_import_nonce_field')) {
                // Only proceed if current user has the capability (e.g., 'manage_options')
                if (current_user_can('manage_options')) {
                    $controller = new \Upsellar\Interaction\UpsellarWoocommerceInteractionController();
                    $response = $controller->upsellar_import_existing_sales();
    
                    // Assuming the response is a WP_Error or WP_HTTP_Response object
                    if (!is_wp_error($response)) {
                        // Set a transient for showing a success message
                        set_transient('upsellar_import_success', true, 10);
                    } else {
                        // Handle errors, maybe set a different transient for error message
                    }
    
                    // Redirect back to avoid form re-submission issues
                    if(isset($_SERVER['REQUEST_URI']) && !empty($_SERVER['REQUEST_URI'])) {
                        wp_safe_redirect(wp_unslash($_SERVER['REQUEST_URI']));
                    } else {
                        // Handle case when REQUEST_URI is not set, perhaps redirect to default page or error message
                    }
                    exit;
                }
            }
        }

        if(isset($_POST['upsellar_sync_products']) && $_POST['upsellar_sync_products'] == '1'){
            if (check_admin_referer('upsellar_sync_products_nonce', 'upsellar_sync_products_nonce_field')){
                if  (current_user_can('manage_options')){            

                    // Get all products
                    $args = array(
                        'post_type' => 'product',
                        'posts_per_page' => -1,
                        'post_status' => 'publish',
                        'fields' => 'ids',
                        );

                    $products  = get_posts($args);

                    // Send to Upsellar API
                    $api_url = 'https://upsellar.com/api/v1/sync-products';
                    
                    $apiKey = $this->settingsManager->upsellar_get_api_key(); 

                    // Prepare the
                    $payload = array(
                        'products' => $products,
                    );

                    // Get response
                    $response = wp_remote_post($api_url, array(
                        'headers' => array(
                            'Content-Type' => 'application/json; charset=utf-8', 
                            'Authorization' => 'Bearer ' . $apiKey,
                            'accept' => 'application/json'
                        ),
                        'body' => wp_json_encode($payload)
                    ));

                    // Assuming the response is a WP_Error or WP_HTTP_Response object
                    if (!is_wp_error($response)) {
                        // Set a transient for showing a success message
                        set_transient('upsellar_sync_products_success', true, 10);
                    } else {
                        // Handle errors, maybe set a different transient for error message
                    }
    
                    // Redirect back to avoid form re-submission issues
                    if(isset($_SERVER['REQUEST_URI']) && !empty($_SERVER['REQUEST_URI'])) {
                        wp_safe_redirect(wp_unslash($_SERVER['REQUEST_URI']));
                    } else {
                        // Handle case when REQUEST_URI is not set, perhaps redirect to default page or error message
                    }
                    exit;
                }
            }
        }
    
        // Display success message if set
        if (get_transient('upsellar_import_success')) {
            echo '<div class="notice notice-success is-dismissible"><p>Existing sales data imported successfully.</p></div>';
            delete_transient('upsellar_import_success');
        }
    
        // Retrieve the current API key
        $apiKey = $this->settingsManager->upsellar_get_api_key();
    
        // Test the API key and get the result
        $testResult = $this->settingsManager->upsellar_test_api_key($apiKey);
    
        // Store the result in a transient for display
        set_transient('upsellar_api_key_test_result', $testResult, 60);
    
        include_once plugin_dir_path(__FILE__) . 'views/admin-settings.php';
    }
    
    /**
     * Check user permissions before displaying the settings page.
     * 
     * @return void
     * @since  1.0.0
     * 
     */
    private function upsellar_check_permissions() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'upsellar'));
        }
    }
}
