<?php
/**
 * Plugin Name: Update Alert
 * Description: Mostra una notifica nella bacheca per gli aggiornamenti di WordPress, plugin, temi e traduzioni.
 * Version: 1.2.1
 * Author: smalnet
 * Author URI: https://profiles.wordpress.org/smalnet/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Evita l'accesso diretto
}

add_action( 'admin_enqueue_scripts', 'update_alert_enqueue_dns' );

function update_alert_enqueue_dns() {
    $plugin_url = plugin_dir_url( __FILE__ );
    $wp_version = get_bloginfo( 'version' ); // Ottieni la versione corrente di WordPress

    wp_enqueue_style( 'toastr-css', $plugin_url . 'assets/css/toastr.min.css', array(), $wp_version );
    wp_enqueue_style( 'font-awesome-css', $plugin_url . 'assets/css/fontawesome.min.css', array(), $wp_version );
    wp_enqueue_script( 'toastr-js', $plugin_url . 'assets/js/toastr.min.js', array( 'jquery' ), $wp_version, true );
}


function update_alert_enqueue_scripts( $hook ) {
    
    // ✔️ Mostra le notifiche solo nella BACHECA
    if ( $hook !== 'index.php' ) {
        return; // Se non siamo nella bacheca, esci subito!
    }
    
    $wp_version = get_bloginfo( 'version' );

 

    $custom_css = "
        #toast-container > div {
            opacity: 0.95;
            font-size: 14px;
            border-radius: 10px;
        }
        #toast-container .toast-title {
            font-weight: bold;
        }
        #toast-container > .toast-info {
            background: linear-gradient(135deg, #f0f4f8, #e0eafc) !important;
            color: #333 !important;
            border: 1px solid #d0d8e0;
            border-radius: 12px !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            position: relative;
            padding-left: 40px !important;
            pointer-events: none; /* Rende la notifica non cliccabile */
        }
         #toast-container > .toast-info .toast-message {
            margin-left: 0;
        }

        #toast-container > .toast-info .button-primary {
            pointer-events: auto; /* Solo il bottone è cliccabile */
            background-color: #ffffff !important;
            color: #333 !important;
            border-color: #ccc !important;
            border-radius: 6px;
            display: inline-block;
            margin-top: 10px;
        }

        #toast-container > .toast-info .button-primary:hover {
            background-color: #f7f7f7 !important;
            color: #222 !important;
        }
    ";
    wp_add_inline_style( 'toastr-css', $custom_css );

    $custom_js = "
        jQuery(document).ready(function($) {
            function fetch_update_alerts() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    dataType: 'json',
                    data: {
                        action: 'update_alert_check_updates'
                    },
                    success: function(response) {
                        if (response.success && response.data.length > 0) {
                            response.data.forEach(function(alert) {
                                toastr.options = {
                                     'closeButton': true,
                'debug': false,
                'newestOnTop': true,
                'progressBar': true,
                'positionClass': 'toast-bottom-right',
                'preventDuplicates': true,
                'tapToDismiss': false,
                'onclick': null,
                'showDuration': '300',
                'hideDuration': '1000',
                'timeOut': '60000',
                'extendedTimeOut': '10000',
                'showEasing': 'swing',
                'hideEasing': 'linear',
                'showMethod': 'fadeIn',
                'hideMethod': 'fadeOut'
                                };
                                toastr.info(alert.message, alert.title);
                            });
                        }
                    }
                });
            }

            fetch_update_alerts();
            setInterval(fetch_update_alerts, 60000);
        });
    ";
    wp_add_inline_script( 'toastr-js', $custom_js );
}
add_action( 'admin_enqueue_scripts', 'update_alert_enqueue_scripts' );

function update_alert_check_updates() {
    $alerts = array();

    $update_data = get_site_transient( 'update_core' );
    $plugin_updates = get_site_transient( 'update_plugins' );
    $theme_updates = get_site_transient( 'update_themes' );

    // ✅ Controllo aggiornamento WordPress
    if ( ! empty( $update_data->updates ) && is_array( $update_data->updates ) ) {
        foreach ( $update_data->updates as $update ) {
            if ( isset( $update->response ) && $update->response === 'upgrade' ) {
                $alerts[] = array(
                    'title'   => '<i class=\"fab fa-wordpress\"></i> Aggiornamento WordPress',
                    'message' => 'È disponibile WordPress ' . esc_html( $update->version ) . '. <br/><a href=\'/wp-admin/update-core.php\' class="button button-primary">Aggiorna ora</a>.'
                );
            }
        }
    }

    // ✅ Aggiornamenti Plugin con nomi corretti
    if ( ! empty( $plugin_updates->response ) ) {
        $plugin_names = array();

        $installed_plugins = get_plugins(); // Ottieni tutti i plugin installati

        foreach ( $plugin_updates->response as $plugin_file => $plugin ) {
            $plugin_name = isset( $installed_plugins[ $plugin_file ]['Name'] ) ? $installed_plugins[ $plugin_file ]['Name'] : $plugin_file;
            $plugin_names[] = $plugin_name . ' (' . $plugin->new_version . ')';
        }

        if ( count( $plugin_names ) > 0 ) {
            $alerts[] = array(
                'title'   => '<i class=\"fas fa-plug\"></i> Aggiornamenti Plugin',
                'message' => 'Sono disponibili aggiornamenti per i seguenti plugin:<br/>' . implode( ',<br/>', $plugin_names ) . '. <br/><a href=\'/wp-admin/update-core.php\' class="button button-primary">Aggiorna ora</a>.'
            );
        }
    }

    // ✅ Aggiornamenti Temi con nomi corretti
    if ( ! empty( $theme_updates->response ) ) {
        $theme_names = array();
        $installed_themes = wp_get_themes();

        foreach ( $theme_updates->response as $theme_slug => $theme ) {
            if ( isset( $installed_themes[ $theme_slug ] ) ) {
                $theme_name = $installed_themes[ $theme_slug ]->get( 'Name' );
                $theme_names[] = $theme_name . ' (' . $theme['new_version'] . ')';
            } else {
                $theme_names[] = $theme_slug . ' (' . $theme['new_version'] . ')';
            }
        }

        if ( count( $theme_names ) > 0 ) {
            $alerts[] = array(
                'title'   => '<i class=\"fas fa-paint-brush\"></i> Aggiornamenti Temi',
                'message' => 'Sono disponibili aggiornamenti per i seguenti temi: ' . implode( ', ', $theme_names ) . '. <br/><a href=\'/wp-admin/update-core.php\' class="button button-primary">Aggiorna ora</a>.'
            );
        }
    }

    // ✅ Aggiornamenti Traduzioni
    $translations = wp_get_translation_updates();
    if ( ! empty( $translations ) ) {
        $alerts[] = array(
            'title'   => '<i class=\"fas fa-language\"></i> Aggiornamenti Traduzioni',
            'message' => 'Sono disponibili aggiornamenti per le traduzioni. <a href=\'/wp-admin/update-core.php\' class="button button-primary">Aggiorna ora</a>.'
        );
    }

    wp_send_json_success( $alerts );
}
add_action( 'wp_ajax_update_alert_check_updates', 'update_alert_check_updates' );
