jQuery(function ($) {
	/**
	 * i18n messages.
	 *
	 * @return {string} Search beer result message.
	 */
	function searchBeerLanguages() {
		return {
			language: {
				errorLoading: function () {
					return wc_enhanced_select_params.i18n_searching; // Workaround for https://github.com/select2/select2/issues/4355 instead of i18n_ajax_error.
				},
				inputTooLong: function (args) {
					const overChars = args.input.length - args.maximum;

					if (overChars === 1) {
						return wc_enhanced_select_params.i18n_input_too_long_1;
					}

					return wc_enhanced_select_params.i18n_input_too_long_n.replace(
						'%qty%',
						overChars
					);
				},
				inputTooShort: function (args) {
					const remainingChars = args.minimum - args.input.length;

					if (remainingChars === 1) {
						return wc_enhanced_select_params.i18n_input_too_short_1;
					}

					return wc_enhanced_select_params.i18n_input_too_short_n.replace(
						'%qty%',
						remainingChars
					);
				},
				inputIntegerTooShort: function (args) {
					const remainingChars = args.minimum - args.input.length;

					if (remainingChars === 1) {
						return urwc_enhanced_select_params.i18n_input_integer_too_short_1;
					}

					return urwc_enhanced_select_params.i18n_input_integer_too_short_n.replace(
						'%qty%',
						remainingChars
					);
				},
				loadingMore: function () {
					return wc_enhanced_select_params.i18n_load_more;
				},
				maximumSelected: function (args) {
					if (args.maximum === 1) {
						return wc_enhanced_select_params.i18n_selection_too_long_1;
					}

					return wc_enhanced_select_params.i18n_selection_too_long_n.replace(
						'%qty%',
						args.maximum
					);
				},
				noResults: function () {
					return wc_enhanced_select_params.i18n_no_matches;
				},
				searching: function () {
					return wc_enhanced_select_params.i18n_searching;
				},
			},
		};
	}

	/**
	 * i18n messages.
	 *
	 * @return {string} Search brewery result message.
	 */
	function searchBreweryLanguages() {
		return {
			language: {
				errorLoading: function () {
					return wc_enhanced_select_params.i18n_searching; // Workaround for https://github.com/select2/select2/issues/4355 instead of i18n_ajax_error.
				},
				inputTooLong: function (args) {
					const overChars = args.input.length - args.maximum;

					if (overChars === 1) {
						return wc_enhanced_select_params.i18n_input_too_long_1;
					}

					return wc_enhanced_select_params.i18n_input_too_long_n.replace(
						'%qty%',
						overChars
					);
				},
				inputTooShort: function (args) {
					const remainingChars = args.minimum - args.input.length;

					if (remainingChars === 1) {
						return wc_enhanced_select_params.i18n_input_too_short_1;
					}

					return wc_enhanced_select_params.i18n_input_too_short_n.replace(
						'%qty%',
						remainingChars
					);
				},
				inputIntegerTooShort: function (args) {
					const remainingChars = args.minimum - args.input.length;

					if (remainingChars === 1) {
						return urwc_enhanced_select_params.i18n_input_integer_too_short_1;
					}

					return urwc_enhanced_select_params.i18n_input_integer_too_short_n.replace(
						'%qty%',
						remainingChars
					);
				},
				loadingMore: function () {
					return wc_enhanced_select_params.i18n_load_more;
				},
				maximumSelected: function (args) {
					if (args.maximum === 1) {
						return wc_enhanced_select_params.i18n_selection_too_long_1;
					}

					return wc_enhanced_select_params.i18n_selection_too_long_n.replace(
						'%qty%',
						args.maximum
					);
				},
				noResults: function () {
					return wc_enhanced_select_params.i18n_no_matches;
				},
				searching: function () {
					return wc_enhanced_select_params.i18n_searching;
				},
			},
		};
	}

	/**
	 * Format search result.
	 *
	 * All HTML data is generated by jQuery.
	 *
	 * @param {Object} searchResult
	 * @return {*} jQuery Object or text.
	 */
	function formatState(searchResult) {
		if (!searchResult.id || !searchResult.text) {
			return null;
		}

		if (
			searchResult.label &&
			searchResult.name &&
			searchResult.beer_count
		) {
			const resultLabel = $('<img>').attr({ src: searchResult.label });

			const resultContainerLabelContainer = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__label' })
				.append(resultLabel);

			const resultText = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__name' })
				.append($('<span>').text(searchResult.name))
				.append($('<b>').text(' (ID#' + searchResult.id + ')'));

			const resultStatsBeerCount = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__beer_count' })
				.append(
					$('<span>').attr({
						class: 'dashicons-before dashicons-beer',
					})
				)
				.append($('<span>').text(searchResult.beer_count));

			const resultBreweryCountry = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__brewery_country' })
				.append(
					$('<span>').attr({
						class: 'dashicons-before dashicons-flag',
					})
				)
				.append($('<span>').text(searchResult.country_name));

			const resultBreweryPageUrl = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__brewery_page_url' })
				.append(
					$('<span>')
						.attr({
							class: 'dashicons-before dashicons-admin-links',
						})
						.html(
							$('<a>')
								.attr({
									class: 'urwc-search-brewery-link',
									href:
										'https://untappd.com' +
										searchResult.brewery_page_url,
									rel: 'noopener noreferer',
									target: '_blank',
								})
								.text(searchResult.brewery_page_url)
						)
				);

			const resultContainerStats = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__stats' })
				.append(resultStatsBeerCount)
				.append(resultBreweryCountry)
				.append(resultBreweryPageUrl);

			const resultContainerMeta = $('<div>')
				.attr({ class: 'urwc-search-brewery-result__meta' })
				.append(resultText)
				.append(resultContainerStats);

			const resultContainer = $('<div>')
				.attr({
					class: 'urwc-search-brewery-result-container clearfix',
				})
				.append(resultContainerLabelContainer)
				.append(resultContainerMeta);

			return resultContainer;
		}

		return searchResult.text;
	}

	try {
		$(document.body)
			.on('urwc-enhanced-select-init', function () {
				let allowSearchByID = false;

				// Override the MinimumInputLength.prototype.query function
				// Check length for strngs not integers to allow input both ID or a search term.
				$.fn.select2.amd.require(
					'select2/data/minimumInputLength'
				).prototype.query = function (decorated, params, callback) {
					params.term = params.term || '';

					if (
						!allowSearchByID &&
						(params.term.length < 1 ||
							params.term.length < this.minimumInputLength)
					) {
						this.trigger('results:message', {
							message: 'inputTooShort',
							args: {
								minimum: this.minimumInputLength,
								input: params.term,
								params,
							},
						});
						return;
					}

					if (allowSearchByID && params.term.length < 1) {
						this.trigger('results:message', {
							message: 'inputIntegerTooShort',
							args: {
								minimum: 1,
								input: params.term,
								params,
							},
						});

						return;
					}

					decorated.call(this, params, callback);
				};

				$(':input.urwc-beer-search')
					.filter(':not(.enhanced)')
					.each(function () {
						allowSearchByID = true;
						let select2Args = {
							allowClear: !!$(this).data('allowClear'),
							placeholder: $(this).data('placeholder'),
							minimumInputLength: $(this).data(
								'minimumInputLength'
							)
								? $(this).data('minimumInputLength')
								: '3',
							escapeMarkup: function (markup) {
								return markup;
							},
							ajax: {
								url: urwc_enhanced_select_params.ajax_url,
								dataType: 'json',
								delay: 250,
								data: function (params) {
									return {
										term: params.term,
										action:
											$(this).data('action') ||
											'urwc_beer_search',
										security:
											urwc_enhanced_select_params.search_beers_nonce,
										limit: $(this).data('limit'),
									};
								},
								processResults: function (beerSearchResponse) {
									const terms = [];

									if (beerSearchResponse.error) {
										window.console.error(
											beerSearchResponse.error
										);
									} else {
										$.each(
											beerSearchResponse.items,
											function (id, text) {
												terms.push({ id, text });
											}
										);
									}
									return {
										results: terms,
									};
								},
								cache: true,
							},
						};

						select2Args = $.extend(
							select2Args,
							searchBeerLanguages()
						);

						$(this).selectWoo(select2Args).addClass('enhanced');
					});

				// Make a request to find the beer name on select to prevent too many calls to Unatppd API.
				$('.urwc-beer-search').on('select2:select', function (event) {
					const beerSearchData = event.params.data;

					if (beerSearchData.id) {
						if (!isNaN(beerSearchData.text)) {
							const beerSearchcontainer = this;
							const request = $.ajax({
								url: urwc_enhanced_select_params.ajax_url,
								dataType: 'json',
								type: 'GET',
								data: {
									action: 'urwc_beer_info',
									security:
										urwc_enhanced_select_params.beer_info_nonce,
									beer_id: beerSearchData.id,
								},
							});
							request.done(function (beerSearchResponse) {
								if (beerSearchResponse.error) {
									$(beerSearchcontainer)
										.val(null)
										.trigger('change');
									showTooltip(
										0,
										0,
										beerSearchResponse.error,
										$(beerSearchcontainer).parent(
											'._urwc_beer_id_field'
										)
									);
								} else {
									$(beerSearchcontainer)
										.empty()
										.append(
											new Option(
												beerSearchResponse.beer_name,
												beerSearchData.id,
												true,
												true
											)
										)
										.trigger('change');
								}
							});
							request.fail(function (jqXHR, textStatus) {
								showTooltip(
									0,
									0,
									textStatus,
									$(beerSearchcontainer).parent(
										'._urwc_beer_id_field'
									)
								);
							});
						}
					}
				});

				// Ajax product search box
				$(':input.urwc-brewery-search')
					.filter(':not(.enhanced)')
					.each(function () {
						allowSearchByID = false;
						let select2Args = {
							allowClear: !!$(this).data('allow_clear'),
							placeholder: $(this).data('placeholder'),
							minimumInputLength: $(this).data(
								'minimum_input_length'
							)
								? $(this).data('minimum_input_length')
								: '3',
							templateResult: formatState,
							escapeMarkup: function (markup) {
								return markup;
							},
							ajax: {
								url: urwc_enhanced_select_params.ajax_url,
								dataType: 'json',
								delay: 250,
								data: function (params) {
									return {
										term: params.term,
										action:
											$(this).data('action') ||
											'urwc_brewery_search',
										security:
											urwc_enhanced_select_params.brewery_search_nonce,
										limit: $(this).data('limit'),
									};
								},
								processResults: function (
									brewerySearchResponse
								) {
									const terms = [];
									if (brewerySearchResponse.items) {
										// Be convert here the only allowed html entitie, single quote '. Thereafter jQuery will only use text().
										$.each(
											brewerySearchResponse.items,
											function (id, breweryObject) {
												terms.push({
													id,
													text: breweryObject.text
														.split('&#039;')
														.join("'"),
													label: breweryObject.brewery_label,
													name: breweryObject.brewery_name
														.split('&#039;')
														.join("'"),
													beer_count:
														breweryObject.beer_count,
													country_name:
														breweryObject.country_name
															.split('&#039;')
															.join("'"),
													brewery_page_url:
														breweryObject.brewery_page_url,
												});
											}
										);
									}
									return {
										results: terms,
									};
								},
								cache: true,
							},
						};

						select2Args = $.extend(
							select2Args,
							searchBreweryLanguages()
						);

						$(this).selectWoo(select2Args).addClass('enhanced');
					});

				/**
				 * Write brewery id to clipboard.
				 *
				 * TODO: Create shortcode:
				 *
				 * 	let shortCode = '[urwc_untappd_map height="%height%" api_key="%api_key%" brewery_id="%brewery_id%" max_checkins="%brewery_id%"]';
				 *  shortCode = shortCode.replace(/%brewery_id%/g, args.data.id );
				 *
				 * @param {Object} args
				 */
				function writeClipboardText(args) {
					try {
						navigator.clipboard.writeText(args.data.id).then(
							() => {
								showTooltip(
									10,
									12,
									urwc_enhanced_select_params.i18n_copied +
										' #' +
										args.data.id,
									args.originalEvent.delegateTarget.parentNode
								);
							},
							(error) => {
								window.console.error(error);
							}
						);
					} catch (error) {
						window.console.error(error);
					}
				}

				/**
				 * Show tooltip to display a message inside select2.
				 *
				 * Based on Woo chart-tooltip.
				 *
				 * @param {number} x
				 * @param {number} y
				 * @param {string} contents
				 * @param {Object} container
				 */
				function showTooltip(x, y, contents, container) {
					let toolTip = $(container).find('.chart-tooltip');
					if (!toolTip.length) {
						toolTip = $('<div>')
							.attr({ class: 'chart-tooltip' })
							.css({ top: y, left: x })
							.appendTo(container);
					}
					toolTip.text(contents).stop().fadeIn(200).fadeOut(2000);
				}

				$('.urwc-brewery-search').on(
					'select2:selecting',
					function (event) {
						event.preventDefault();
						writeClipboardText(event.params.args);
					}
				);
			})
			.trigger('urwc-enhanced-select-init');

		$('html').on('click', function (event) {
			if (this === event.target) {
				$(':input.urwc-beer-search, :input.urwc-brewery-search')
					.filter('.select2-hidden-accessible')
					.selectWoo('close');
			}
		});
	} catch (err) {
		// If select2 failed (conflict?) log the error but don't stop other scripts breaking.
		window.console.log(err);
	}
});
