<?php
/**
 * The plugin bootstrap file
 *
 * This file is read by WordPress to generate the plugin information in the plugin
 * admin area. This file also includes all of the dependencies used by the plugin,
 * registers the activation and deactivation functions, and defines a function
 * that starts the plugin.
 *
 * @link              https://mulberrytech.ca/unplug
 * @since             1.0.0
 * @package           Unplug
 *
 * @wordpress-plugin
 * Plugin Name: Unplug - Site Add-On Optimizer
 * Plugin URI: https://mulberrytech.ca/unplug
 * Description: Intelligently analyze and optimize your WordPress add-ons. Identify unused add-ons, measure performance impact, and safely remove bloat to boost your site's speed and security.
 * Version: 1.3.0
 * Author: Mulberry
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: unplug
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Currently plugin version.
 * Start at version 1.0.0 and use SemVer - https://semver.org
 * Rename this for your plugin and update it as you release new versions.
 */
define( 'UNPLUG_VERSION', '1.3.0' );

/**
 * The code that runs during plugin activation.
 * This action is documented in includes/class-unplug-activator.php
 */
function unplug_activate_plugin() {
    require_once plugin_dir_path(__FILE__) . 'includes/class-unplug-activator.php';
    UNPLUG_Activator::activate();
}

/**
 * The code that runs during plugin deactivation.
 * This action is documented in includes/class-unplug-deactivator.php
 */
function unplug_deactivate_plugin() {
    require_once plugin_dir_path(__FILE__) . 'includes/class-unplug-deactivator.php';
    UNPLUG_Deactivator::deactivate();
}

register_activation_hook(__FILE__, 'unplug_activate_plugin');
register_deactivation_hook(__FILE__, 'unplug_deactivate_plugin');

/**
 * The core plugin class that is used to define internationalization,
 * admin-specific hooks, and public-facing site hooks.
 */
require plugin_dir_path(__FILE__) . 'includes/class-unplug.php';

/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * then kicking off the plugin from this point in the file does
 * not affect the page life cycle.
 *
 * @since    1.0.0
 */
function unplug_run_plugin() {
    $plugin = new UNPLUG_Plugin();
    $plugin->run();
}
unplug_run_plugin();

add_action('wp_ajax_unplug_get_progress', 'unplug_ajax_get_progress');
function unplug_ajax_get_progress() {
    if ( ! current_user_can('manage_options') ) {
        wp_send_json_error(['message' => 'Insufficient permissions.']);
        return;
    }
    
    global $wpdb;
    require_once __DIR__ . '/includes/class-unplug-database.php';
    $table = UNPLUG_Database::get_table_name('queue');
    // Get the most recent processing or pending task
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific queue table query
    $task = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT * FROM %s WHERE status IN ('processing','pending') ORDER BY started_at DESC, created_at DESC LIMIT 1",
            $table
        ),
        ARRAY_A
    );
    if (!$task) {
        wp_send_json_success([
            'percent' => 0,
            'status' => 'Idle',
            'log' => ''
        ]);
        return;
    }
    wp_send_json_success([
        'percent' => intval($task['progress_percent']),
        'status' => $task['status'],
        'log' => maybe_unserialize($task['progress_data'])
    ]);
}

add_action('wp_ajax_unplug_remove_queue_task', 'unplug_ajax_remove_queue_task');
function unplug_ajax_remove_queue_task() {
    if ( ! current_user_can('manage_options') ) {
        wp_send_json_error(['message' => 'Insufficient permissions.']);
        return;
    }
    
    // Verify nonce
    if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'unplug_remove_task' ) ) {
        wp_send_json_error(['message' => 'Security check failed.']);
        return;
    }
    
    $task_id = isset($_POST['task_id']) ? intval($_POST['task_id']) : 0;
    if (!$task_id) {
        wp_send_json_error(['message' => 'No task ID provided.']);
        return;
    }
    require_once __DIR__ . '/includes/class-unplug-database.php';
    global $wpdb;
    $table = UNPLUG_Database::get_table_name('queue');
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific queue table deletion
    $deleted = $wpdb->delete($table, ['id' => $task_id], ['%d']);
    if ($deleted) {
        wp_send_json_success(['message' => 'Task removed.']);
    } else {
        wp_send_json_error(['message' => 'Task not found or could not be removed.']);
    }
}

add_action('wp_ajax_unplug_export_plugins_csv', 'unplug_ajax_export_plugins_csv');
function unplug_ajax_export_plugins_csv() {
    if ( ! current_user_can('manage_options') ) {
        wp_die('Insufficient permissions.');
    }
    
    // Verify nonce
    if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['nonce'] ) ), 'unplug_export_csv' ) ) {
        wp_die('Security check failed.');
    }
    
    if ( ! function_exists( 'get_plugins' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    $plugins = get_plugins();
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="unplug-plugins-export.csv"');
    
    // Use WordPress file system functions
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        require_once ABSPATH . '/wp-admin/includes/file.php';
        WP_Filesystem();
    }
    
    $csv_content = '';
    $csv_content .= '"Plugin Name","Status","Version","Author","Description"' . "\n";
    
    foreach ($plugins as $plugin_file => $plugin_data) {
        $is_active = is_plugin_active($plugin_file) ? 'Active' : 'Inactive';
        $csv_content .= '"' . str_replace('"', '""', $plugin_data['Name']) . '",';
        $csv_content .= '"' . str_replace('"', '""', $is_active) . '",';
        $csv_content .= '"' . str_replace('"', '""', $plugin_data['Version']) . '",';
        $csv_content .= '"' . str_replace('"', '""', $plugin_data['Author']) . '",';
        $csv_content .= '"' . str_replace('"', '""', $plugin_data['Description']) . '"' . "\n";
    }
    
    echo esc_html($csv_content);
    exit;
} 