<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Fired when the plugin is uninstalled.
 *
 * When populating this file, consider the following flow
 * of control:
 *
 * - This method should be static
 * - Check if the $_REQUEST content actually is the plugin name
 * - Run an admin referrer check to make sure it goes through authentication
 * - Verify the output of $_GET makes sense
 * - Repeat with other user roles. Best directly by using the links/query string parameters.
 * - Repeat things for multisite. Once for a single site in the network, once sitewide.
 *
 * This file may be updated more in future version of the Boilerplate; however, this is the
 * general skeleton and outline for how the file should work.
 *
 * For more information, see the following discussion:
 * https://github.com/tommcfarlin/WordPress-Plugin-Boilerplate/pull/123#issuecomment-28541913
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 */

// If uninstall not called from WordPress, then exit.
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

/**
 * Delete all plugin options from wp_options table
 */
// Load options helper class for comprehensive cleanup
require_once plugin_dir_path( __FILE__ ) . 'includes/class-unplug-options.php';

// Delete all plugin options using the options manager
Unplug_Options::delete_all_options();

/**
 * Delete all plugin transients
 */
delete_transient('unplug_plugin_list');
delete_transient('unplug_scan_progress');
delete_transient('unplug_api_validation');

/**
 * Remove custom database tables
 */
global $wpdb;

// Delete scan results table
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
$wpdb->query( $wpdb->prepare( "DROP TABLE IF EXISTS %s", $wpdb->prefix . 'unplug_scan_results' ) );

// Delete queue table
$wpdb->query( $wpdb->prepare( "DROP TABLE IF EXISTS %s", $wpdb->prefix . 'unplug_queue' ) );

// Delete locations table
$wpdb->query( $wpdb->prepare( "DROP TABLE IF EXISTS %s", $wpdb->prefix . 'unplug_locations' ) );
// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange

/**
 * Clear any scheduled events
 */
// Load cleanup class for proper event cleanup
require_once plugin_dir_path( __FILE__ ) . 'includes/class-unplug-cleanup.php';

// Unschedule all cleanup events
Unplug_Cleanup::unschedule_cleanup_events();

// Clear any legacy scheduled events
wp_clear_scheduled_hook('unplug_cleanup_old_scans');
wp_clear_scheduled_hook('unplug_process_queue');

/**
 * Delete user meta for all users
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
$wpdb->query( $wpdb->prepare( "DELETE FROM %s WHERE meta_key LIKE %s", $wpdb->usermeta, 'unplug_%' ) );
// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

/**
 * For multisite installations, clean up network-wide options
 */
if (is_multisite()) {
    delete_site_option('unplug_network_settings');
    delete_site_option('unplug_network_api_token');
}

/**
 * Clean up any uploaded files or directories
 */
$upload_dir = wp_upload_dir();
$unplug_dir = trailingslashit($upload_dir['basedir']) . 'unplug';

if (is_dir($unplug_dir)) {
    // Initialize WordPress filesystem
    global $wp_filesystem;
    if (empty($wp_filesystem)) {
        require_once(ABSPATH . '/wp-admin/includes/file.php');
        WP_Filesystem();
    }
    
    // Remove all files in the directory
    $files = glob($unplug_dir . '/*');
    foreach ($files as $file) {
        if (is_file($file)) {
            wp_delete_file($file);
        }
    }
    // Remove the directory using WordPress filesystem
    $wp_filesystem->rmdir($unplug_dir);
} 