<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Plugin {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Unplug_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'UNPLUG_VERSION' ) ) {
            $this->version = UNPLUG_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'unplug';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->define_database_hooks();
        $this->define_cleanup_hooks();
        $this->define_security_hooks();
        $this->define_scanner_hooks();
        $this->define_activity_scan_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Unplug_Loader. Orchestrates the hooks of the plugin.
     * - Unplug_i18n. Defines internationalization functionality.
     * - Unplug_Admin. Defines all hooks for the admin area.
     * - Unplug_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-i18n.php';

        /**
         * The class responsible for database management and schema operations.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';

        /**
         * The class responsible for plugin options management and validation.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        /**
         * The class responsible for database cleanup and maintenance.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-cleanup.php';

        /**
         * The class responsible for database migrations and upgrade handling.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-migrator.php';

        /**
         * The class responsible for security, nonce verification, and input sanitization.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';

        /**
         * The class responsible for queue management and task processing.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-queue.php';

        /**
         * The class responsible for comprehensive plugin scanning and metadata extraction.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-scanner.php';

        /**
         * The class responsible for conflict pattern identification and analysis.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-conflict-engine.php';

        /**
         * The class responsible for activity scanning and plugin usage detection.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-activity-scan-engine.php';

        /**
         * The class responsible for comprehensive Gutenberg block analysis.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-gutenberg-block-analyzer.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-unplug-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-unplug-public.php';

        $this->loader = new Unplug_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Unplug_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Unplug_i18n();
        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new Unplug_Admin( $this->get_plugin_name(), $this->get_version() );

        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
        $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
        $this->loader->add_action( 'admin_init', $plugin_admin, 'init_settings' );
        $this->loader->add_action( 'admin_notices', $plugin_admin, 'show_migration_notices' );

        // AJAX handlers
        $this->loader->add_action( 'wp_ajax_unplug_refresh_plugins', $plugin_admin, 'ajax_refresh_plugins' );
        $this->loader->add_action( 'wp_ajax_unplug_start_scan', $plugin_admin, 'ajax_start_scan' );
        $this->loader->add_action( 'wp_ajax_unplug_scan_progress', $plugin_admin, 'ajax_scan_progress' );
        $this->loader->add_action( 'wp_ajax_unplug_export_csv', $plugin_admin, 'ajax_export_csv' );
        $this->loader->add_action( 'wp_ajax_unplug_generate_report', $plugin_admin, 'ajax_generate_report' );
        $this->loader->add_action( 'wp_ajax_unplug_clear_location_data', $plugin_admin, 'ajax_clear_location_data' );
        $this->loader->add_action( 'wp_ajax_unplug_export_location_data', $plugin_admin, 'ajax_export_location_data' );
        $this->loader->add_action( 'wp_ajax_unplug_get_confirmation_token', $plugin_admin, 'ajax_get_confirmation_token' );
        		$this->loader->add_action( 'wp_ajax_unplug_get_plugin_inventory', $plugin_admin, 'ajax_get_plugin_inventory' );
		$this->loader->add_action( 'wp_ajax_unplug_detect_plugin_conflicts', $plugin_admin, 'ajax_detect_plugin_conflicts' );
		$this->loader->add_action( 'wp_ajax_unplug_get_plugins_by_category', $plugin_admin, 'ajax_get_plugins_by_category' );
		        $this->loader->add_action( 'wp_ajax_unplug_analyze_conflict_patterns', $plugin_admin, 'ajax_analyze_conflict_patterns' );
        $this->loader->add_action( 'wp_ajax_unplug_export_conflict_analysis', $plugin_admin, 'ajax_export_conflict_analysis' );
        $this->loader->add_action( 'wp_ajax_unplug_get_warnings', $plugin_admin, 'ajax_get_warnings' );
        $this->loader->add_action( 'wp_ajax_unplug_update_warning_status', $plugin_admin, 'ajax_update_warning_status' );
        $this->loader->add_action( 'wp_ajax_unplug_dismiss_warning', $plugin_admin, 'ajax_dismiss_warning' );
        $this->loader->add_action( 'wp_ajax_unplug_generate_warnings', $plugin_admin, 'ajax_generate_warnings' );
        
        // Safe mode AJAX actions
        $this->loader->add_action( 'wp_ajax_unplug_get_safe_mode_status', $plugin_admin, 'ajax_get_safe_mode_status' );
        $this->loader->add_action( 'wp_ajax_unplug_activate_safe_mode', $plugin_admin, 'ajax_activate_safe_mode' );
        $this->loader->add_action( 'wp_ajax_unplug_deactivate_safe_mode', $plugin_admin, 'ajax_deactivate_safe_mode' );
        $this->loader->add_action( 'wp_ajax_unplug_restore_from_backup', $plugin_admin, 'ajax_restore_from_backup' );
        $this->loader->add_action( 'wp_ajax_unplug_get_backup_states', $plugin_admin, 'ajax_get_backup_states' );
        $this->loader->add_action( 'wp_ajax_unplug_delete_backup_state', $plugin_admin, 'ajax_delete_backup_state' );
        $this->loader->add_action( 'wp_ajax_unplug_run_activity_test', $plugin_admin, 'ajax_run_activity_test' );
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $plugin_public = new Unplug_Public( $this->get_plugin_name(), $this->get_version() );

        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
    }

    /**
     * Register all of the hooks related to the database functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_database_hooks() {
        // Check for database upgrades on plugin load
        $this->loader->add_action( 'plugins_loaded', 'Unplug_Database', 'check_upgrade' );
    }

    /**
     * Register all of the hooks related to the cleanup functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_cleanup_hooks() {
        // Initialize cleanup system
        $this->loader->add_action( 'init', 'Unplug_Cleanup', 'init' );
    }

    /**
     * Register all of the hooks related to the security functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_security_hooks() {
        // Initialize capabilities on plugin activation
        register_activation_hook( __FILE__, array( 'Unplug_Security', 'initialize_capabilities' ) );
        
        // Remove capabilities on plugin deactivation
        register_deactivation_hook( __FILE__, array( 'Unplug_Security', 'remove_capabilities' ) );
        
        		// Initialize security system
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
		
		// Check capabilities on admin init
		$this->loader->add_action( 'admin_init', 'Unplug_Security', 'verify_admin_capabilities' );

		// Initialize plugin inventory system
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';
		Unplug_Plugin_Inventory::init();
    }

    /**
     * Register all of the hooks related to the scanner functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_scanner_hooks() {
        // Initialize scanner system with required dependencies
        add_action( 'init', array( $this, 'initialize_scanner_system' ) );
        
        // Schedule background plugin scanning
        if ( ! wp_next_scheduled( 'unplug_background_scan' ) ) {
            wp_schedule_event( time(), 'daily', 'unplug_background_scan' );
        }
        
        add_action( 'unplug_background_scan', array( $this, 'run_background_scan' ) );
        
        // Clean up scan results periodically
        if ( ! wp_next_scheduled( 'unplug_cleanup_scan_results' ) ) {
            wp_schedule_event( time(), 'weekly', 'unplug_cleanup_scan_results' );
        }
        
        add_action( 'unplug_cleanup_scan_results', array( $this, 'cleanup_old_scan_results' ) );
    }

    /**
     * Register all of the hooks related to the activity scan functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_activity_scan_hooks() {
        // Initialize activity scan engine
        add_action( 'init', [ 'Unplug_Activity_Scan_Engine', 'init' ] );
        
        // Initialize activity scan system with required dependencies
        add_action( 'init', array( $this, 'initialize_activity_scan_system' ) );
    }

    /**
     * Initialize the scanner system with required dependencies.
     *
     * @since    1.0.0
     */
    public function initialize_scanner_system() {
        // Only initialize if we have required capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Initialize database connection
        $database = new Unplug_Database();
        
        // Initialize security system
        $security = new Unplug_Security();
        
        // Initialize queue system
        $queue = new Unplug_Queue( $database );
        
        // Initialize scanner with dependencies
        $scanner = new Unplug_Scanner( $database, $security, $queue );
        
        // Initialize conflict engine
        $conflict_engine = new Unplug_Conflict_Engine( $database, $security );
        
        // Initialize warning system
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-warning-system.php';
        
        // Initialize safe mode system
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
        Unplug_Safe_Mode::init();
        
        // Store instances for later use
        $GLOBALS['unplug_scanner'] = $scanner;
        $GLOBALS['unplug_conflict_engine'] = $conflict_engine;
    }

    /**
     * Initialize the activity scan system with required dependencies.
     *
     * @since    1.0.0
     */
    public function initialize_activity_scan_system() {
        // Only initialize if we have required capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Initialize required dependencies
        $database = new Unplug_Database();
        $security = new Unplug_Security();
        $queue = new Unplug_Queue( $database, $security, null );
        
        // Create activity scan engine instance
        $activity_scan_engine = new Unplug_Activity_Scan_Engine( $database, $security, $queue );
        
        // Store instance for later use
        $GLOBALS['unplug_activity_scan_engine'] = $activity_scan_engine;
        
        return $activity_scan_engine;
    }

    /**
     * Run background plugin scanning.
     *
     * @since    1.0.0
     */
    public function run_background_scan() {
        if ( isset( $GLOBALS['unplug_scanner'] ) ) {
            $scanner = $GLOBALS['unplug_scanner'];
            $scanner->scan_all_plugins();
        }
    }

    /**
     * Clean up old scan results.
     *
     * @since    1.0.0
     */
    public function cleanup_old_scan_results() {
        $database = new Unplug_Database();
        $database->cleanup_old_scan_results( 30 ); // Keep results for 30 days
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Unplug_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
} 