<?php

/**
 * The warning system functionality of the plugin.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * The warning system class.
 *
 * This class manages user notifications and warnings for detected conflicts,
 * providing appropriate user interaction flows and persistent warning management.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Warning_System {

    /**
     * Warning types constants
     */
    const WARNING_TYPE_BLOCKING = 'blocking';
    const WARNING_TYPE_ADVISORY = 'advisory';
    const WARNING_TYPE_INFORMATIONAL = 'informational';

    /**
     * Warning severity constants
     */
    const SEVERITY_CRITICAL = 'critical';
    const SEVERITY_HIGH = 'high';
    const SEVERITY_MEDIUM = 'medium';
    const SEVERITY_LOW = 'low';

    /**
     * Warning status constants
     */
    const STATUS_ACTIVE = 'active';
    const STATUS_ACKNOWLEDGED = 'acknowledged';
    const STATUS_DISMISSED = 'dismissed';
    const STATUS_RESOLVED = 'resolved';

    /**
     * Option name for storing warnings
     */
    const WARNINGS_OPTION = 'unplug_warnings';

    /**
     * Option name for storing user warning preferences
     */
    const WARNING_PREFERENCES_OPTION = 'unplug_warning_preferences';

    /**
     * Maximum number of warnings to store
     */
    const MAX_WARNINGS = 100;

    /**
     * Warning retention period in days
     */
    const WARNING_RETENTION_DAYS = 30;

    /**
     * Create a new warning
     *
     * @param array $warning_data Warning data
     * @return string|false Warning ID on success, false on failure
     */
    public static function create_warning( $warning_data ) {
        $defaults = array(
            'type' => self::WARNING_TYPE_ADVISORY,
            'severity' => self::SEVERITY_MEDIUM,
            'title' => '',
            'message' => '',
            'details' => '',
            'actions' => array(),
            'related_plugins' => array(),
            'category' => 'general',
            'dismissible' => true,
            'auto_dismiss' => false,
            'auto_dismiss_delay' => 0,
            'created_at' => current_time( 'timestamp' ),
            'status' => self::STATUS_ACTIVE,
            'context' => array()
        );

        $warning = wp_parse_args( $warning_data, $defaults );

        // Validate required fields
        if ( empty( $warning['title'] ) || empty( $warning['message'] ) ) {
            return false;
        }

        // Validate warning type
        if ( ! in_array( $warning['type'], array( self::WARNING_TYPE_BLOCKING, self::WARNING_TYPE_ADVISORY, self::WARNING_TYPE_INFORMATIONAL ) ) ) {
            $warning['type'] = self::WARNING_TYPE_ADVISORY;
        }

        // Validate severity
        if ( ! in_array( $warning['severity'], array( self::SEVERITY_CRITICAL, self::SEVERITY_HIGH, self::SEVERITY_MEDIUM, self::SEVERITY_LOW ) ) ) {
            $warning['severity'] = self::SEVERITY_MEDIUM;
        }

        // Generate unique warning ID
        $warning_id = wp_generate_uuid4();
        $warning['id'] = $warning_id;

        // Get existing warnings
        $warnings = self::get_warnings();

        // Add new warning
        $warnings[ $warning_id ] = $warning;

        // Cleanup old warnings if necessary
        $warnings = self::cleanup_warnings( $warnings );

        // Save warnings
        if ( update_option( self::WARNINGS_OPTION, $warnings ) ) {
            return $warning_id;
        }

        return false;
    }

    /**
     * Get all warnings
     *
     * @param array $filters Optional filters
     * @return array Array of warnings
     */
    public static function get_warnings( $filters = array() ) {
        $warnings = get_option( self::WARNINGS_OPTION, array() );

        if ( empty( $filters ) ) {
            return $warnings;
        }

        $filtered_warnings = array();

        foreach ( $warnings as $warning_id => $warning ) {
            $include = true;

            // Filter by type
            if ( isset( $filters['type'] ) && $warning['type'] !== $filters['type'] ) {
                $include = false;
            }

            // Filter by severity
            if ( isset( $filters['severity'] ) && $warning['severity'] !== $filters['severity'] ) {
                $include = false;
            }

            // Filter by status
            if ( isset( $filters['status'] ) && $warning['status'] !== $filters['status'] ) {
                $include = false;
            }

            // Filter by category
            if ( isset( $filters['category'] ) && $warning['category'] !== $filters['category'] ) {
                $include = false;
            }

            // Filter by plugin
            if ( isset( $filters['plugin'] ) && ! in_array( $filters['plugin'], $warning['related_plugins'] ) ) {
                $include = false;
            }

            // Filter by active only
            if ( isset( $filters['active_only'] ) && $filters['active_only'] && $warning['status'] !== self::STATUS_ACTIVE ) {
                $include = false;
            }

            if ( $include ) {
                $filtered_warnings[ $warning_id ] = $warning;
            }
        }

        return $filtered_warnings;
    }

    /**
     * Get a specific warning by ID
     *
     * @param string $warning_id Warning ID
     * @return array|false Warning data or false if not found
     */
    public static function get_warning( $warning_id ) {
        $warnings = self::get_warnings();
        return isset( $warnings[ $warning_id ] ) ? $warnings[ $warning_id ] : false;
    }

    /**
     * Update warning status
     *
     * @param string $warning_id Warning ID
     * @param string $status New status
     * @param array $additional_data Additional data to update
     * @return bool True on success, false on failure
     */
    public static function update_warning_status( $warning_id, $status, $additional_data = array() ) {
        $warnings = self::get_warnings();

        if ( ! isset( $warnings[ $warning_id ] ) ) {
            return false;
        }

        // Validate status
        if ( ! in_array( $status, array( self::STATUS_ACTIVE, self::STATUS_ACKNOWLEDGED, self::STATUS_DISMISSED, self::STATUS_RESOLVED ) ) ) {
            return false;
        }

        $warnings[ $warning_id ]['status'] = $status;
        $warnings[ $warning_id ]['updated_at'] = current_time( 'timestamp' );

        // Add additional data
        if ( ! empty( $additional_data ) ) {
            $warnings[ $warning_id ] = array_merge( $warnings[ $warning_id ], $additional_data );
        }

        return update_option( self::WARNINGS_OPTION, $warnings );
    }

    /**
     * Delete a warning
     *
     * @param string $warning_id Warning ID
     * @return bool True on success, false on failure
     */
    public static function delete_warning( $warning_id ) {
        $warnings = self::get_warnings();

        if ( ! isset( $warnings[ $warning_id ] ) ) {
            return false;
        }

        unset( $warnings[ $warning_id ] );

        return update_option( self::WARNINGS_OPTION, $warnings );
    }

    /**
     * Generate warnings from conflict analysis
     *
     * @param array $conflict_analysis Conflict analysis results
     * @return array Array of created warning IDs
     */
    public static function generate_warnings_from_conflicts( $conflict_analysis ) {
        $warning_ids = array();

        if ( empty( $conflict_analysis['summary'] ) ) {
            return $warning_ids;
        }

        $summary = $conflict_analysis['summary'];

        // Generate warnings for critical conflicts
        if ( $summary['critical_conflicts'] > 0 ) {
            $warning_id = self::create_warning( array(
                'type' => self::WARNING_TYPE_BLOCKING,
                'severity' => self::SEVERITY_CRITICAL,
                'title' => 'Critical Plugin Conflicts Detected',
                'message' => sprintf( '%d critical conflicts detected that may cause site instability or data loss.', $summary['critical_conflicts'] ),
                'details' => 'Critical conflicts typically involve database table collisions, security vulnerabilities, or core functionality interference. Immediate action is required.',
                'actions' => array(
                    array(
                        'label' => 'View Details',
                        'action' => 'view_conflict_details',
                        'data' => array( 'severity' => 'critical' )
                    ),
                    array(
                        'label' => 'Deactivate Conflicting Plugins',
                        'action' => 'deactivate_plugins',
                        'data' => array( 'severity' => 'critical' )
                    )
                ),
                'category' => 'conflict',
                'dismissible' => false,
                'related_plugins' => self::extract_plugins_from_conflicts( $conflict_analysis, 'critical' )
            ) );

            if ( $warning_id ) {
                $warning_ids[] = $warning_id;
            }
        }

        // Generate warnings for high severity conflicts
        if ( $summary['high_conflicts'] > 0 ) {
            $warning_id = self::create_warning( array(
                'type' => self::WARNING_TYPE_ADVISORY,
                'severity' => self::SEVERITY_HIGH,
                'title' => 'High Priority Plugin Conflicts',
                'message' => sprintf( '%d high priority conflicts detected that may affect site performance or functionality.', $summary['high_conflicts'] ),
                'details' => 'High priority conflicts can cause performance degradation, broken features, or user experience issues. Review and resolve these conflicts soon.',
                'actions' => array(
                    array(
                        'label' => 'View Details',
                        'action' => 'view_conflict_details',
                        'data' => array( 'severity' => 'high' )
                    ),
                    array(
                        'label' => 'Run Compatibility Test',
                        'action' => 'run_compatibility_test',
                        'data' => array( 'severity' => 'high' )
                    )
                ),
                'category' => 'conflict',
                'dismissible' => true,
                'related_plugins' => self::extract_plugins_from_conflicts( $conflict_analysis, 'high' )
            ) );

            if ( $warning_id ) {
                $warning_ids[] = $warning_id;
            }
        }

        // Generate warnings for medium severity conflicts
        if ( $summary['medium_conflicts'] > 0 ) {
            $warning_id = self::create_warning( array(
                'type' => self::WARNING_TYPE_INFORMATIONAL,
                'severity' => self::SEVERITY_MEDIUM,
                'title' => 'Medium Priority Plugin Conflicts',
                'message' => sprintf( '%d medium priority conflicts detected that may cause minor issues.', $summary['medium_conflicts'] ),
                'details' => 'Medium priority conflicts typically involve styling issues, minor functionality overlaps, or compatibility concerns. Monitor these conflicts for potential issues.',
                'actions' => array(
                    array(
                        'label' => 'View Details',
                        'action' => 'view_conflict_details',
                        'data' => array( 'severity' => 'medium' )
                    ),
                    array(
                        'label' => 'Schedule Review',
                        'action' => 'schedule_review',
                        'data' => array( 'severity' => 'medium' )
                    )
                ),
                'category' => 'conflict',
                'dismissible' => true,
                'auto_dismiss' => false,
                'related_plugins' => self::extract_plugins_from_conflicts( $conflict_analysis, 'medium' )
            ) );

            if ( $warning_id ) {
                $warning_ids[] = $warning_id;
            }
        }

        // Generate specific warnings for known problematic patterns
        if ( isset( $conflict_analysis['by_type']['known_patterns'] ) ) {
            $warning_ids = array_merge( $warning_ids, self::generate_pattern_warnings( $conflict_analysis['by_type']['known_patterns'] ) );
        }

        // Generate warnings for resource conflicts
        if ( isset( $conflict_analysis['by_type']['resource'] ) ) {
            $warning_ids = array_merge( $warning_ids, self::generate_resource_warnings( $conflict_analysis['by_type']['resource'] ) );
        }

        return $warning_ids;
    }

    /**
     * Extract plugins from conflict analysis for specific severity
     *
     * @param array $conflict_analysis Conflict analysis results
     * @param string $severity Severity level
     * @return array Array of plugin names
     */
    private static function extract_plugins_from_conflicts( $conflict_analysis, $severity ) {
        $plugins = array();

        if ( empty( $conflict_analysis['by_type'] ) ) {
            return $plugins;
        }

        foreach ( $conflict_analysis['by_type'] as $type => $conflicts ) {
            if ( is_array( $conflicts ) ) {
                foreach ( $conflicts as $conflict ) {
                    if ( isset( $conflict['severity'] ) && $conflict['severity'] >= self::get_severity_score( $severity ) ) {
                        if ( isset( $conflict['plugins'] ) && is_array( $conflict['plugins'] ) ) {
                            $plugins = array_merge( $plugins, $conflict['plugins'] );
                        }
                    }
                }
            }
        }

        return array_unique( $plugins );
    }

    /**
     * Generate warnings for known problematic patterns
     *
     * @param array $patterns Known patterns
     * @return array Array of warning IDs
     */
    private static function generate_pattern_warnings( $patterns ) {
        $warning_ids = array();

        foreach ( $patterns as $pattern_key => $pattern_data ) {
            if ( isset( $pattern_data['severity'] ) && $pattern_data['severity'] >= 7 ) {
                $warning_id = self::create_warning( array(
                    'type' => self::WARNING_TYPE_ADVISORY,
                    'severity' => $pattern_data['severity'] >= 9 ? self::SEVERITY_CRITICAL : self::SEVERITY_HIGH,
                    'title' => 'Known Plugin Pattern Conflict',
                    'message' => $pattern_data['description'] ?? 'A known problematic plugin pattern has been detected.',
                    'details' => $pattern_data['resolution'] ?? 'This pattern is known to cause issues. Please review the affected plugins.',
                    'actions' => array(
                        array(
                            'label' => 'View Pattern Details',
                            'action' => 'view_pattern_details',
                            'data' => array( 'pattern' => $pattern_key )
                        )
                    ),
                    'category' => 'pattern',
                    'dismissible' => true,
                    'related_plugins' => $pattern_data['plugins'] ?? array()
                ) );

                if ( $warning_id ) {
                    $warning_ids[] = $warning_id;
                }
            }
        }

        return $warning_ids;
    }

    /**
     * Generate warnings for resource conflicts
     *
     * @param array $resource_conflicts Resource conflicts
     * @return array Array of warning IDs
     */
    private static function generate_resource_warnings( $resource_conflicts ) {
        $warning_ids = array();

        foreach ( $resource_conflicts as $resource_key => $resource_data ) {
            if ( isset( $resource_data['severity'] ) && $resource_data['severity'] >= 6 ) {
                $warning_id = self::create_warning( array(
                    'type' => self::WARNING_TYPE_INFORMATIONAL,
                    'severity' => $resource_data['severity'] >= 8 ? self::SEVERITY_HIGH : self::SEVERITY_MEDIUM,
                    'title' => 'Resource Conflict Detected',
                    'message' => $resource_data['description'] ?? 'A resource conflict has been detected between plugins.',
                    'details' => $resource_data['resolution'] ?? 'This conflict may cause performance issues or resource contention.',
                    'actions' => array(
                        array(
                            'label' => 'View Resource Details',
                            'action' => 'view_resource_details',
                            'data' => array( 'resource' => $resource_key )
                        )
                    ),
                    'category' => 'resource',
                    'dismissible' => true,
                    'related_plugins' => $resource_data['plugins'] ?? array()
                ) );

                if ( $warning_id ) {
                    $warning_ids[] = $warning_id;
                }
            }
        }

        return $warning_ids;
    }

    /**
     * Get severity score for severity level
     *
     * @param string $severity Severity level
     * @return int Severity score
     */
    private static function get_severity_score( $severity ) {
        switch ( $severity ) {
            case 'critical':
                return 9;
            case 'high':
                return 7;
            case 'medium':
                return 5;
            case 'low':
                return 3;
            default:
                return 5;
        }
    }

    /**
     * Cleanup old warnings
     *
     * @param array $warnings Current warnings
     * @return array Cleaned warnings
     */
    private static function cleanup_warnings( $warnings ) {
        $current_time = current_time( 'timestamp' );
        $retention_period = self::WARNING_RETENTION_DAYS * DAY_IN_SECONDS;
        $cleaned_warnings = array();

        // Sort by creation time (newest first)
        uasort( $warnings, function( $a, $b ) {
            return $b['created_at'] - $a['created_at'];
        });

        $count = 0;
        foreach ( $warnings as $warning_id => $warning ) {
            // Keep warnings within retention period and limit
            if ( $count < self::MAX_WARNINGS && 
                 ( $current_time - $warning['created_at'] ) < $retention_period ) {
                $cleaned_warnings[ $warning_id ] = $warning;
                $count++;
            }
        }

        return $cleaned_warnings;
    }

    /**
     * Get warning statistics
     *
     * @return array Warning statistics
     */
    public static function get_warning_stats() {
        $warnings = self::get_warnings();
        
        $stats = array(
            'total' => count( $warnings ),
            'active' => 0,
            'acknowledged' => 0,
            'dismissed' => 0,
            'resolved' => 0,
            'by_severity' => array(
                'critical' => 0,
                'high' => 0,
                'medium' => 0,
                'low' => 0
            ),
            'by_type' => array(
                'blocking' => 0,
                'advisory' => 0,
                'informational' => 0
            )
        );

        foreach ( $warnings as $warning ) {
            // Count by status
            if ( isset( $stats[ $warning['status'] ] ) ) {
                $stats[ $warning['status'] ]++;
            }

            // Count by severity
            if ( isset( $stats['by_severity'][ $warning['severity'] ] ) ) {
                $stats['by_severity'][ $warning['severity'] ]++;
            }

            // Count by type
            if ( isset( $stats['by_type'][ $warning['type'] ] ) ) {
                $stats['by_type'][ $warning['type'] ]++;
            }
        }

        return $stats;
    }

    /**
     * Get user warning preferences
     *
     * @param int $user_id User ID (optional, defaults to current user)
     * @return array User warning preferences
     */
    public static function get_user_preferences( $user_id = null ) {
        if ( $user_id === null ) {
            $user_id = get_current_user_id();
        }

        $preferences = get_user_meta( $user_id, self::WARNING_PREFERENCES_OPTION, true );

        if ( empty( $preferences ) ) {
            $preferences = array(
                'email_notifications' => false,
                'show_informational' => true,
                'show_advisory' => true,
                'show_blocking' => true,
                'auto_acknowledge_low' => false,
                'notification_frequency' => 'immediate'
            );
        }

        return $preferences;
    }

    /**
     * Update user warning preferences
     *
     * @param array $preferences New preferences
     * @param int $user_id User ID (optional, defaults to current user)
     * @return bool True on success, false on failure
     */
    public static function update_user_preferences( $preferences, $user_id = null ) {
        if ( $user_id === null ) {
            $user_id = get_current_user_id();
        }

        $current_preferences = self::get_user_preferences( $user_id );
        $new_preferences = array_merge( $current_preferences, $preferences );

        return update_user_meta( $user_id, self::WARNING_PREFERENCES_OPTION, $new_preferences );
    }

    /**
     * Check if user should see specific warning
     *
     * @param array $warning Warning data
     * @param int $user_id User ID (optional, defaults to current user)
     * @return bool True if user should see warning, false otherwise
     */
    public static function should_show_warning( $warning, $user_id = null ) {
        $preferences = self::get_user_preferences( $user_id );

        // Always show blocking warnings
        if ( $warning['type'] === self::WARNING_TYPE_BLOCKING ) {
            return true;
        }

        // Check type preferences
        if ( $warning['type'] === self::WARNING_TYPE_ADVISORY && ! $preferences['show_advisory'] ) {
            return false;
        }

        if ( $warning['type'] === self::WARNING_TYPE_INFORMATIONAL && ! $preferences['show_informational'] ) {
            return false;
        }

        // Auto-acknowledge low severity warnings if enabled
        if ( $warning['severity'] === self::SEVERITY_LOW && $preferences['auto_acknowledge_low'] ) {
            self::update_warning_status( $warning['id'], self::STATUS_ACKNOWLEDGED );
            return false;
        }

        return true;
    }

    /**
     * Get warnings for display
     *
     * @param int $user_id User ID (optional, defaults to current user)
     * @return array Array of warnings to display
     */
    public static function get_warnings_for_display( $user_id = null ) {
        $active_warnings = self::get_warnings( array( 'active_only' => true ) );
        $display_warnings = array();

        foreach ( $active_warnings as $warning_id => $warning ) {
            if ( self::should_show_warning( $warning, $user_id ) ) {
                $display_warnings[ $warning_id ] = $warning;
            }
        }

        // Sort by severity and creation time
        uasort( $display_warnings, function( $a, $b ) {
            $severity_order = array( 'critical' => 4, 'high' => 3, 'medium' => 2, 'low' => 1 );
            
            if ( $severity_order[ $a['severity'] ] !== $severity_order[ $b['severity'] ] ) {
                return $severity_order[ $b['severity'] ] - $severity_order[ $a['severity'] ];
            }
            
            return $b['created_at'] - $a['created_at'];
        });

        return $display_warnings;
    }

    /**
     * Clear all warnings
     *
     * @return bool True on success, false on failure
     */
    public static function clear_all_warnings() {
        return delete_option( self::WARNINGS_OPTION );
    }

    /**
     * Clear resolved warnings
     *
     * @return bool True on success, false on failure
     */
    public static function clear_resolved_warnings() {
        $warnings = self::get_warnings();
        $active_warnings = array();

        foreach ( $warnings as $warning_id => $warning ) {
            if ( $warning['status'] !== self::STATUS_RESOLVED ) {
                $active_warnings[ $warning_id ] = $warning;
            }
        }

        return update_option( self::WARNINGS_OPTION, $active_warnings );
    }
} 