<?php
/**
 * Unplug Speed Categorizer
 * Categorizes performance impacts into severity levels and provides actionable recommendations
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Speed_Categorizer {
    
    /**
     * Impact thresholds for categorization
     *
     * @var array
     */
    private static $impact_thresholds = array(
        'load_time' => array(
            'low' => 100,      // < 100ms
            'medium' => 500,   // 100-500ms
            'high' => 1000,    // 500-1000ms
            'critical' => 2000 // > 1000ms
        ),
        'render_delay' => array(
            'low' => 50,       // < 50ms
            'medium' => 200,   // 50-200ms
            'high' => 500,     // 200-500ms
            'critical' => 1000 // > 500ms
        ),
        'asset_size' => array(
            'low' => 50000,     // < 50KB
            'medium' => 200000, // 50-200KB
            'high' => 500000,   // 200-500KB
            'critical' => 1000000 // > 500KB
        ),
        'asset_count' => array(
            'low' => 10,      // < 10 assets
            'medium' => 25,   // 10-25 assets
            'high' => 50,     // 25-50 assets
            'critical' => 100 // > 50 assets
        )
    );
    
    /**
     * Categorize performance impacts
     *
     * @param array $performance_data Performance impact data
     * @return array Categorized impacts with recommendations
     */
    public static function categorize_impacts( $performance_data ) {
        $categorization = array(
            'overall_category' => array(),
            'plugin_categories' => array(),
            'theme_categories' => array(),
            'asset_categories' => array(),
            'priority_actions' => array(),
            'optimization_roadmap' => array()
        );
        
        // Categorize overall impact
        $categorization['overall_category'] = self::categorize_overall_impact( $performance_data['overall_impact'] );
        
        // Categorize plugin impacts
        if ( isset( $performance_data['plugin_impacts'] ) ) {
            $categorization['plugin_categories'] = self::categorize_plugin_impacts( $performance_data['plugin_impacts'] );
        }
        
        // Categorize theme impacts
        if ( isset( $performance_data['theme_impacts'] ) ) {
            $categorization['theme_categories'] = self::categorize_theme_impacts( $performance_data['theme_impacts'] );
        }
        
        // Categorize individual asset impacts
        if ( isset( $performance_data['asset_impacts'] ) ) {
            $categorization['asset_categories'] = self::categorize_asset_impacts( $performance_data['asset_impacts'] );
        }
        
        // Generate priority actions
        $categorization['priority_actions'] = self::generate_priority_actions( $categorization );
        
        // Create optimization roadmap
        $categorization['optimization_roadmap'] = self::create_optimization_roadmap( $categorization );
        
        return $categorization;
    }
    
    /**
     * Categorize overall performance impact
     *
     * @param array $overall_impact Overall impact data
     * @return array Categorized overall impact
     */
    private static function categorize_overall_impact( $overall_impact ) {
        $category = array(
            'load_time_category' => self::categorize_metric( $overall_impact['estimated_load_time'], 'load_time' ),
            'render_delay_category' => self::categorize_metric( $overall_impact['estimated_render_delay'], 'render_delay' ),
            'asset_size_category' => self::categorize_metric( $overall_impact['total_size'], 'asset_size' ),
            'asset_count_category' => self::categorize_metric( $overall_impact['total_assets'], 'asset_count' ),
            'overall_severity' => $overall_impact['impact_level'],
            'performance_score' => $overall_impact['performance_score'],
            'primary_issues' => array(),
            'recommendations' => array()
        );
        
        // Identify primary issues
        $category['primary_issues'] = self::identify_primary_issues( $category );
        
        // Generate specific recommendations
        $category['recommendations'] = self::generate_overall_recommendations( $category, $overall_impact );
        
        return $category;
    }
    
    /**
     * Categorize plugin impacts
     *
     * @param array $plugin_impacts Plugin impact data
     * @return array Categorized plugin impacts
     */
    private static function categorize_plugin_impacts( $plugin_impacts ) {
        $categorized = array();
        
        foreach ( $plugin_impacts as $plugin_slug => $plugin_impact ) {
            $category = array(
                'plugin_slug' => $plugin_slug,
                'plugin_name' => $plugin_impact['plugin_name'],
                'severity' => $plugin_impact['impact_level'],
                'performance_score' => $plugin_impact['performance_score'],
                'load_time_category' => self::categorize_metric( $plugin_impact['total_load_time_impact'], 'load_time' ),
                'render_delay_category' => self::categorize_metric( $plugin_impact['total_render_delay_impact'], 'render_delay' ),
                'asset_size_category' => self::categorize_metric( $plugin_impact['total_size'], 'asset_size' ),
                'asset_count_category' => self::categorize_metric( $plugin_impact['total_assets'], 'asset_count' ),
                'optimization_potential' => self::assess_plugin_optimization_potential( $plugin_impact ),
                'action_priority' => self::determine_plugin_action_priority( $plugin_impact ),
                'recommended_actions' => self::generate_plugin_recommendations( $plugin_impact )
            );
            
            $categorized[ $plugin_slug ] = $category;
        }
        
        // Sort by action priority
        uasort( $categorized, function( $a, $b ) {
            $priority_order = array( 'critical' => 4, 'high' => 3, 'medium' => 2, 'low' => 1 );
            return $priority_order[ $b['action_priority'] ] - $priority_order[ $a['action_priority'] ];
        });
        
        return $categorized;
    }
    
    /**
     * Categorize theme impacts
     *
     * @param array $theme_impacts Theme impact data
     * @return array Categorized theme impacts
     */
    private static function categorize_theme_impacts( $theme_impacts ) {
        $categorized = array();
        
        foreach ( $theme_impacts as $theme_slug => $theme_impact ) {
            $category = array(
                'theme_slug' => $theme_slug,
                'theme_name' => $theme_impact['theme_name'],
                'severity' => $theme_impact['impact_level'],
                'performance_score' => $theme_impact['performance_score'],
                'load_time_category' => self::categorize_metric( $theme_impact['total_load_time_impact'], 'load_time' ),
                'render_delay_category' => self::categorize_metric( $theme_impact['total_render_delay_impact'], 'render_delay' ),
                'asset_size_category' => self::categorize_metric( $theme_impact['total_size'], 'asset_size' ),
                'asset_count_category' => self::categorize_metric( $theme_impact['total_assets'], 'asset_count' ),
                'optimization_potential' => self::assess_theme_optimization_potential( $theme_impact ),
                'action_priority' => self::determine_theme_action_priority( $theme_impact ),
                'recommended_actions' => self::generate_theme_recommendations( $theme_impact )
            );
            
            $categorized[ $theme_slug ] = $category;
        }
        
        return $categorized;
    }
    
    /**
     * Categorize individual asset impacts
     *
     * @param array $asset_impacts Asset impact data
     * @return array Categorized asset impacts
     */
    private static function categorize_asset_impacts( $asset_impacts ) {
        $categorized = array();
        
        foreach ( $asset_impacts as $asset_type => $assets ) {
            foreach ( $assets as $handle => $asset ) {
                $category = array(
                    'asset_type' => $asset_type,
                    'handle' => $handle,
                    'source' => $asset['source'],
                    'attribution_type' => $asset['attribution_type'],
                    'attributed_to' => $asset['attributed_to'],
                    'severity' => $asset['impact_level'],
                    'performance_score' => $asset['performance_score'],
                    'load_time_category' => self::categorize_metric( $asset['load_time_impact'], 'load_time' ),
                    'render_delay_category' => self::categorize_metric( $asset['render_delay_impact'], 'render_delay' ),
                    'size_category' => self::categorize_metric( $asset['size'], 'asset_size' ),
                    'optimization_priority' => $asset['optimization_priority'],
                    'recommended_actions' => self::generate_asset_recommendations( $asset )
                );
                
                $categorized[ $asset_type ][ $handle ] = $category;
            }
        }
        
        return $categorized;
    }
    
    /**
     * Categorize a metric based on thresholds
     *
     * @param float $value Metric value
     * @param string $metric_type Type of metric
     * @return string Category (low, medium, high, critical)
     */
    private static function categorize_metric( $value, $metric_type ) {
        if ( ! isset( self::$impact_thresholds[ $metric_type ] ) ) {
            return 'unknown';
        }
        
        $thresholds = self::$impact_thresholds[ $metric_type ];
        
        if ( $value < $thresholds['low'] ) {
            return 'low';
        } elseif ( $value < $thresholds['medium'] ) {
            return 'medium';
        } elseif ( $value < $thresholds['high'] ) {
            return 'high';
        } else {
            return 'critical';
        }
    }
    
    /**
     * Identify primary performance issues
     *
     * @param array $category Categorized data
     * @return array Primary issues
     */
    private static function identify_primary_issues( $category ) {
        $issues = array();
        
        if ( $category['load_time_category'] === 'critical' || $category['load_time_category'] === 'high' ) {
            $issues[] = array(
                'type' => 'load_time',
                'severity' => $category['load_time_category'],
                'description' => 'Slow page load times affecting user experience'
            );
        }
        
        if ( $category['render_delay_category'] === 'critical' || $category['render_delay_category'] === 'high' ) {
            $issues[] = array(
                'type' => 'render_delay',
                'severity' => $category['render_delay_category'],
                'description' => 'Render-blocking resources delaying page display'
            );
        }
        
        if ( $category['asset_size_category'] === 'critical' || $category['asset_size_category'] === 'high' ) {
            $issues[] = array(
                'type' => 'asset_size',
                'severity' => $category['asset_size_category'],
                'description' => 'Large asset sizes increasing load times'
            );
        }
        
        if ( $category['asset_count_category'] === 'critical' || $category['asset_count_category'] === 'high' ) {
            $issues[] = array(
                'type' => 'asset_count',
                'severity' => $category['asset_count_category'],
                'description' => 'Too many assets creating request overhead'
            );
        }
        
        return $issues;
    }
    
    /**
     * Generate overall recommendations
     *
     * @param array $category Categorized data
     * @param array $overall_impact Overall impact data
     * @return array Recommendations
     */
    private static function generate_overall_recommendations( $category, $overall_impact ) {
        $recommendations = array();
        
        foreach ( $category['primary_issues'] as $issue ) {
            switch ( $issue['type'] ) {
                case 'load_time':
                    $recommendations[] = array(
                        'action' => 'optimize_load_time',
                        'title' => 'Optimize Page Load Time',
                        'description' => 'Implement caching, compress assets, and optimize images to reduce load times',
                        'priority' => $issue['severity'],
                        'estimated_impact' => 'High',
                        'difficulty' => 'Medium'
                    );
                    break;
                    
                case 'render_delay':
                    $recommendations[] = array(
                        'action' => 'reduce_render_blocking',
                        'title' => 'Reduce Render-Blocking Resources',
                        'description' => 'Defer non-critical JavaScript and CSS to improve initial page rendering',
                        'priority' => $issue['severity'],
                        'estimated_impact' => 'High',
                        'difficulty' => 'Medium'
                    );
                    break;
                    
                case 'asset_size':
                    $recommendations[] = array(
                        'action' => 'optimize_asset_sizes',
                        'title' => 'Optimize Asset Sizes',
                        'description' => 'Minify CSS/JS, compress images, and remove unused code',
                        'priority' => $issue['severity'],
                        'estimated_impact' => 'Medium',
                        'difficulty' => 'Low'
                    );
                    break;
                    
                case 'asset_count':
                    $recommendations[] = array(
                        'action' => 'reduce_asset_count',
                        'title' => 'Reduce Asset Count',
                        'description' => 'Combine files, remove unused plugins, and use asset bundling',
                        'priority' => $issue['severity'],
                        'estimated_impact' => 'Medium',
                        'difficulty' => 'High'
                    );
                    break;
            }
        }
        
        return $recommendations;
    }
    
    /**
     * Assess plugin optimization potential
     *
     * @param array $plugin_impact Plugin impact data
     * @return string Optimization potential (low, medium, high)
     */
    private static function assess_plugin_optimization_potential( $plugin_impact ) {
        $score = 0;
        
        // High impact plugins have high optimization potential
        if ( $plugin_impact['impact_level'] === 'critical' ) {
            $score += 3;
        } elseif ( $plugin_impact['impact_level'] === 'high' ) {
            $score += 2;
        }
        
        // Many assets indicate optimization opportunities
        if ( $plugin_impact['total_assets'] > 5 ) {
            $score += 2;
        } elseif ( $plugin_impact['total_assets'] > 2 ) {
            $score += 1;
        }
        
        // Large render delay indicates blocking optimization potential
        if ( $plugin_impact['total_render_delay_impact'] > 200 ) {
            $score += 2;
        } elseif ( $plugin_impact['total_render_delay_impact'] > 100 ) {
            $score += 1;
        }
        
        if ( $score >= 4 ) {
            return 'high';
        } elseif ( $score >= 2 ) {
            return 'medium';
        } else {
            return 'low';
        }
    }
    
    /**
     * Assess theme optimization potential
     *
     * @param array $theme_impact Theme impact data
     * @return string Optimization potential (low, medium, high)
     */
    private static function assess_theme_optimization_potential( $theme_impact ) {
        // Similar to plugin assessment but themes are generally harder to optimize
        $potential = self::assess_plugin_optimization_potential( $theme_impact );
        
        // Reduce potential by one level for themes (harder to modify)
        if ( $potential === 'high' ) {
            return 'medium';
        } elseif ( $potential === 'medium' ) {
            return 'low';
        } else {
            return 'low';
        }
    }
    
    /**
     * Determine plugin action priority
     *
     * @param array $plugin_impact Plugin impact data
     * @return string Action priority (low, medium, high, critical)
     */
    private static function determine_plugin_action_priority( $plugin_impact ) {
        // Base priority on impact level and optimization potential
        $impact_level = $plugin_impact['impact_level'];
        $optimization_potential = self::assess_plugin_optimization_potential( $plugin_impact );
        
        if ( $impact_level === 'critical' && $optimization_potential === 'high' ) {
            return 'critical';
        } elseif ( $impact_level === 'high' || ( $impact_level === 'critical' && $optimization_potential === 'medium' ) ) {
            return 'high';
        } elseif ( $impact_level === 'medium' || ( $impact_level === 'high' && $optimization_potential === 'low' ) ) {
            return 'medium';
        } else {
            return 'low';
        }
    }
    
    /**
     * Determine theme action priority
     *
     * @param array $theme_impact Theme impact data
     * @return string Action priority (low, medium, high, critical)
     */
    private static function determine_theme_action_priority( $theme_impact ) {
        // Similar to plugin but generally lower priority (harder to change themes)
        $plugin_priority = self::determine_plugin_action_priority( $theme_impact );
        
        // Reduce priority by one level for themes
        if ( $plugin_priority === 'critical' ) {
            return 'high';
        } elseif ( $plugin_priority === 'high' ) {
            return 'medium';
        } else {
            return 'low';
        }
    }
    
    /**
     * Generate plugin-specific recommendations
     *
     * @param array $plugin_impact Plugin impact data
     * @return array Recommendations
     */
    private static function generate_plugin_recommendations( $plugin_impact ) {
        $recommendations = array();
        
        if ( $plugin_impact['total_render_delay_impact'] > 100 ) {
            $recommendations[] = array(
                'action' => 'defer_plugin_assets',
                'title' => 'Defer Plugin Assets',
                'description' => 'Move plugin CSS/JS to load after critical content',
                'difficulty' => 'Medium',
                'estimated_impact' => 'High'
            );
        }
        
        if ( $plugin_impact['total_assets'] > 3 ) {
            $recommendations[] = array(
                'action' => 'combine_plugin_assets',
                'title' => 'Combine Plugin Assets',
                'description' => 'Merge multiple plugin files to reduce HTTP requests',
                'difficulty' => 'Low',
                'estimated_impact' => 'Medium'
            );
        }
        
        if ( $plugin_impact['total_size'] > 200000 ) {
            $recommendations[] = array(
                'action' => 'minify_plugin_assets',
                'title' => 'Minify Plugin Assets',
                'description' => 'Compress plugin CSS and JavaScript files',
                'difficulty' => 'Low',
                'estimated_impact' => 'Medium'
            );
        }
        
        if ( $plugin_impact['impact_level'] === 'critical' ) {
            $recommendations[] = array(
                'action' => 'consider_plugin_alternatives',
                'title' => 'Consider Plugin Alternatives',
                'description' => 'Evaluate lighter alternatives or custom solutions',
                'difficulty' => 'High',
                'estimated_impact' => 'High'
            );
        }
        
        return $recommendations;
    }
    
    /**
     * Generate theme-specific recommendations
     *
     * @param array $theme_impact Theme impact data
     * @return array Recommendations
     */
    private static function generate_theme_recommendations( $theme_impact ) {
        $recommendations = array();
        
        if ( $theme_impact['total_render_delay_impact'] > 150 ) {
            $recommendations[] = array(
                'action' => 'optimize_theme_css',
                'title' => 'Optimize Theme CSS',
                'description' => 'Implement critical CSS and defer non-critical styles',
                'difficulty' => 'High',
                'estimated_impact' => 'High'
            );
        }
        
        if ( $theme_impact['total_size'] > 300000 ) {
            $recommendations[] = array(
                'action' => 'optimize_theme_assets',
                'title' => 'Optimize Theme Assets',
                'description' => 'Compress images, minify CSS/JS, and remove unused code',
                'difficulty' => 'Medium',
                'estimated_impact' => 'Medium'
            );
        }
        
        if ( $theme_impact['impact_level'] === 'critical' ) {
            $recommendations[] = array(
                'action' => 'consider_theme_alternatives',
                'title' => 'Consider Theme Alternatives',
                'description' => 'Evaluate performance-optimized themes or custom development',
                'difficulty' => 'High',
                'estimated_impact' => 'High'
            );
        }
        
        return $recommendations;
    }
    
    /**
     * Generate asset-specific recommendations
     *
     * @param array $asset Asset data
     * @return array Recommendations
     */
    private static function generate_asset_recommendations( $asset ) {
        $recommendations = array();
        
        if ( $asset['render_delay_impact'] > 50 ) {
            $recommendations[] = array(
                'action' => 'defer_asset',
                'title' => 'Defer Asset Loading',
                'description' => 'Load this asset after critical content',
                'difficulty' => 'Low',
                'estimated_impact' => 'Medium'
            );
        }
        
        if ( $asset['size'] > 100000 ) {
            $recommendations[] = array(
                'action' => 'compress_asset',
                'title' => 'Compress Asset',
                'description' => 'Reduce file size through compression or minification',
                'difficulty' => 'Low',
                'estimated_impact' => 'Medium'
            );
        }
        
        if ( $asset['source'] === 'external' ) {
            $recommendations[] = array(
                'action' => 'host_locally',
                'title' => 'Host Locally',
                'description' => 'Serve from your domain to reduce external dependencies',
                'difficulty' => 'Medium',
                'estimated_impact' => 'Low'
            );
        }
        
        return $recommendations;
    }
    
    /**
     * Generate priority actions
     *
     * @param array $categorization Categorization data
     * @return array Priority actions
     */
    private static function generate_priority_actions( $categorization ) {
        $actions = array();
        
        // Add critical plugin actions
        foreach ( $categorization['plugin_categories'] as $plugin_category ) {
            if ( $plugin_category['action_priority'] === 'critical' ) {
                $actions[] = array(
                    'type' => 'plugin',
                    'target' => $plugin_category['plugin_slug'],
                    'title' => 'Optimize Critical Plugin: ' . $plugin_category['plugin_name'],
                    'priority' => 'critical',
                    'estimated_impact' => 'High',
                    'actions' => $plugin_category['recommended_actions']
                );
            }
        }
        
        // Add high priority actions
        foreach ( $categorization['plugin_categories'] as $plugin_category ) {
            if ( $plugin_category['action_priority'] === 'high' ) {
                $actions[] = array(
                    'type' => 'plugin',
                    'target' => $plugin_category['plugin_slug'],
                    'title' => 'Optimize High-Impact Plugin: ' . $plugin_category['plugin_name'],
                    'priority' => 'high',
                    'estimated_impact' => 'Medium',
                    'actions' => $plugin_category['recommended_actions']
                );
            }
        }
        
        // Add theme actions if high priority
        foreach ( $categorization['theme_categories'] as $theme_category ) {
            if ( $theme_category['action_priority'] === 'high' ) {
                $actions[] = array(
                    'type' => 'theme',
                    'target' => $theme_category['theme_slug'],
                    'title' => 'Optimize Theme: ' . $theme_category['theme_name'],
                    'priority' => 'high',
                    'estimated_impact' => 'Medium',
                    'actions' => $theme_category['recommended_actions']
                );
            }
        }
        
        // Sort by priority
        usort( $actions, function( $a, $b ) {
            $priority_order = array( 'critical' => 4, 'high' => 3, 'medium' => 2, 'low' => 1 );
            return $priority_order[ $b['priority'] ] - $priority_order[ $a['priority'] ];
        });
        
        return array_slice( $actions, 0, 5 ); // Return top 5 actions
    }
    
    /**
     * Create optimization roadmap
     *
     * @param array $categorization Categorization data
     * @return array Optimization roadmap
     */
    private static function create_optimization_roadmap( $categorization ) {
        $roadmap = array(
            'immediate' => array(), // 0-1 week
            'short_term' => array(), // 1-4 weeks
            'medium_term' => array(), // 1-3 months
            'long_term' => array() // 3+ months
        );
        
        // Immediate actions (critical priority, low difficulty)
        foreach ( $categorization['priority_actions'] as $action ) {
            if ( $action['priority'] === 'critical' ) {
                $roadmap['immediate'][] = $action;
            }
        }
        
        // Short-term actions (high priority)
        foreach ( $categorization['priority_actions'] as $action ) {
            if ( $action['priority'] === 'high' ) {
                $roadmap['short_term'][] = $action;
            }
        }
        
        // Medium-term actions (medium priority or high difficulty)
        foreach ( $categorization['plugin_categories'] as $plugin_category ) {
            if ( $plugin_category['action_priority'] === 'medium' ) {
                $roadmap['medium_term'][] = array(
                    'type' => 'plugin',
                    'target' => $plugin_category['plugin_slug'],
                    'title' => 'Optimize Plugin: ' . $plugin_category['plugin_name'],
                    'priority' => 'medium',
                    'estimated_impact' => 'Medium',
                    'actions' => $plugin_category['recommended_actions']
                );
            }
        }
        
        // Long-term actions (low priority, high difficulty, or major changes)
        $roadmap['long_term'][] = array(
            'type' => 'infrastructure',
            'target' => 'overall',
            'title' => 'Implement Performance Monitoring',
            'priority' => 'low',
            'estimated_impact' => 'High',
            'description' => 'Set up continuous performance monitoring and alerting'
        );
        
        return $roadmap;
    }
    
    /**
     * Get category color for display
     *
     * @param string $category Category level
     * @return string CSS color class
     */
    public static function get_category_color( $category ) {
        switch ( $category ) {
            case 'critical':
                return 'danger';
            case 'high':
                return 'warning';
            case 'medium':
                return 'info';
            case 'low':
                return 'success';
            default:
                return 'secondary';
        }
    }
    
    /**
     * Get category icon for display
     *
     * @param string $category Category level
     * @return string Dashicon class
     */
    public static function get_category_icon( $category ) {
        switch ( $category ) {
            case 'critical':
                return 'dashicons-warning';
            case 'high':
                return 'dashicons-flag';
            case 'medium':
                return 'dashicons-info';
            case 'low':
                return 'dashicons-yes';
            default:
                return 'dashicons-minus';
        }
    }
} 