<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The security-specific functionality of the plugin.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * The security-specific functionality of the plugin.
 *
 * Defines nonce verification, capability checks, and input sanitization.
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Security {

    /**
     * Available nonce actions for the plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $nonce_actions    The available nonce actions.
     */
    private static $nonce_actions = array(
        'admin'         => 'unplug_admin_nonce',
        'settings'      => 'unplug_settings_nonce',
        'scan'          => 'unplug_scan_nonce',
        'export'        => 'unplug_export_nonce',
        'plugin_action' => 'unplug_plugin_action_nonce',
        'ajax'          => 'unplug_ajax_nonce'
    );

    /**
     * Create a nonce for a specific action.
     *
     * @since    1.0.0
     * @param    string    $action    The action type (admin, settings, scan, export, plugin_action, ajax).
     * @return   string               The nonce value.
     */
    public static function create_nonce( $action = 'admin' ) {
        if ( ! isset( self::$nonce_actions[ $action ] ) ) {
            $action = 'admin';
        }
        
        return wp_create_nonce( self::$nonce_actions[ $action ] );
    }

    /**
     * Verify a nonce for a specific action.
     *
     * @since    1.0.0
     * @param    string    $nonce     The nonce value to verify.
     * @param    string    $action    The action type (admin, settings, scan, export, plugin_action, ajax).
     * @param    bool      $ajax      Whether this is an AJAX request (affects error handling).
     * @return   bool                 True if valid, false otherwise.
     */
    public static function verify_nonce( $nonce, $action = 'admin', $ajax = false ) {
        if ( ! isset( self::$nonce_actions[ $action ] ) ) {
            $action = 'admin';
        }
        
        $is_valid = wp_verify_nonce( $nonce, self::$nonce_actions[ $action ] );
        
        if ( ! $is_valid ) {
            if ( $ajax ) {
                wp_send_json_error( array( 
                    'message' => __( 'Security check failed. Please refresh the page and try again.', 'unplug' ),
                    'code' => 'nonce_verification_failed'
                ) );
            } else {
                wp_die( 
                    esc_html( __( 'Security check failed. Please refresh the page and try again.', 'unplug' ) ), 
                    esc_html( __( 'Security Error', 'unplug' ) ), 
                    array( 'response' => 403 )
                );
            }
        }
        
        return $is_valid;
    }

    /**
     * Get the nonce field HTML for a specific action.
     *
     * @since    1.0.0
     * @param    string    $action    The action type (admin, settings, scan, export, plugin_action, ajax).
     * @param    string    $name      The name attribute for the nonce field.
     * @param    bool      $referer   Whether to include the referer field.
     * @param    bool      $echo      Whether to echo the field or return it.
     * @return   string               The nonce field HTML.
     */
    public static function nonce_field( $action = 'admin', $name = '_wpnonce', $referer = true, $echo = true ) {
        if ( ! isset( self::$nonce_actions[ $action ] ) ) {
            $action = 'admin';
        }
        
        return wp_nonce_field( self::$nonce_actions[ $action ], $name, $referer, $echo );
    }

    /**
     * Get the nonce URL for a specific action.
     *
     * @since    1.0.0
     * @param    string    $actionurl    The URL to add the nonce to.
     * @param    string    $action       The action type (admin, settings, scan, export, plugin_action, ajax).
     * @param    string    $name         The name for the nonce parameter.
     * @return   string                  The URL with nonce parameter.
     */
    public static function nonce_url( $actionurl, $action = 'admin', $name = '_wpnonce' ) {
        if ( ! isset( self::$nonce_actions[ $action ] ) ) {
            $action = 'admin';
        }
        
        return wp_nonce_url( $actionurl, self::$nonce_actions[ $action ], $name );
    }

    /**
     * Check if the current user has the required capability.
     *
     * @since    1.0.0
     * @param    string    $capability    The capability to check.
     * @param    bool      $ajax          Whether this is an AJAX request (affects error handling).
     * @return   bool                     True if user has capability, false otherwise.
     */
    public static function check_capability( $capability = 'manage_options', $ajax = false ) {
        $has_capability = current_user_can( $capability );
        
        if ( ! $has_capability ) {
            if ( $ajax ) {
                wp_send_json_error( array( 
                    'message' => __( 'You do not have sufficient permissions to perform this action.', 'unplug' ),
                    'code' => 'insufficient_permissions'
                ) );
            } else {
                wp_die( 
                    esc_html( __( 'You do not have sufficient permissions to perform this action.', 'unplug' ) ), 
                    esc_html( __( 'Permission Error', 'unplug' ) ), 
                    array( 'response' => 403 )
                );
            }
        }
        
        return $has_capability;
    }

    /**
     * Comprehensive security check for AJAX requests.
     *
     * @since    1.0.0
     * @param    string    $action       The nonce action type.
     * @param    string    $capability   The required capability.
     * @param    string    $nonce_key    The POST key containing the nonce.
     * @return   bool                    True if all checks pass.
     */
    public static function verify_ajax_request( $action = 'ajax', $capability = 'manage_options', $nonce_key = 'nonce' ) {
        // Check if nonce is present
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- This function handles nonce verification internally
        if ( ! isset( $_POST[ $nonce_key ] ) ) {
            wp_send_json_error( array( 
                'message' => __( 'Security token missing. Please refresh the page and try again.', 'unplug' ),
                'code' => 'nonce_missing'
            ) );
        }
        
        // Verify nonce - direct call for WordPress plugin checker compliance
        $nonce_action = isset( self::$nonce_actions[ $action ] ) ? self::$nonce_actions[ $action ] : self::$nonce_actions['admin'];
        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST[ $nonce_key ] ) ), $nonce_action ) ) {
            wp_send_json_error( array( 
                'message' => __( 'Security check failed. Please refresh the page and try again.', 'unplug' ),
                'code' => 'nonce_verification_failed'
            ) );
        }
        
        // Check user capabilities
        self::check_capability( $capability, true );
        
        return true;
    }

    /**
     * Sanitize text input with additional security measures.
     *
     * @since    1.0.0
     * @param    mixed     $input         The input to sanitize.
     * @param    string    $type          The type of sanitization (text, email, url, textarea, html).
     * @param    array     $allowed_tags  Allowed HTML tags for 'html' type.
     * @return   mixed                    The sanitized input.
     */
    public static function sanitize_input( $input, $type = 'text', $allowed_tags = array() ) {
        if ( is_array( $input ) ) {
            return array_map( function( $item ) use ( $type, $allowed_tags ) {
                return self::sanitize_input( $item, $type, $allowed_tags );
            }, $input );
        }
        
        switch ( $type ) {
            case 'email':
                return sanitize_email( $input );
                
            case 'url':
                return esc_url_raw( $input );
                
            case 'textarea':
                return sanitize_textarea_field( $input );
                
            case 'html':
                if ( empty( $allowed_tags ) ) {
                    $allowed_tags = array(
                        'p' => array(),
                        'br' => array(),
                        'strong' => array(),
                        'em' => array(),
                        'a' => array( 'href' => array(), 'title' => array() )
                    );
                }
                return wp_kses( $input, $allowed_tags );
                
            case 'int':
                return absint( $input );
                
            case 'float':
                return floatval( $input );
                
            case 'bool':
                return (bool) $input;
                
            case 'text':
            default:
                return sanitize_text_field( $input );
        }
    }

    /**
     * Validate and sanitize settings input.
     *
     * @since    1.0.0
     * @param    array    $input    The input array to validate.
     * @return   array              The validated and sanitized input.
     */
    public static function validate_settings( $input ) {
        $sanitized = array();
        
        // API Token
        if ( isset( $input['api_token'] ) ) {
            $token = self::sanitize_input( $input['api_token'], 'text' );
            if ( ! empty( $token ) && self::validate_api_token_format( $token ) ) {
                $sanitized['api_token'] = $token;
            } elseif ( ! empty( $token ) ) {
                // Log invalid token format
                self::log_security_event( 'api_token_validation_failed', 'API token format validation failed during settings update' );
                $sanitized['api_token'] = ''; // Clear invalid token
            } else {
                $sanitized['api_token'] = '';
            }
        }
        
        // Scan Timeout
        if ( isset( $input['scan_timeout'] ) ) {
            $timeout = absint( $input['scan_timeout'] );
            $sanitized['scan_timeout'] = max( 30, min( 600, $timeout ) );
        }
        
        // Cleanup Retention
        if ( isset( $input['cleanup_retention'] ) ) {
            $retention = absint( $input['cleanup_retention'] );
            $valid_retention = array( 7, 30, 90, 365 );
            $sanitized['cleanup_retention'] = in_array( $retention, $valid_retention ) ? $retention : 30;
        }
        
        // Enable Logging
        if ( isset( $input['enable_logging'] ) ) {
            $sanitized['enable_logging'] = (bool) $input['enable_logging'];
        }
        
        return $sanitized;
    }

    /**
     * Sanitize and validate JSON input.
     *
     * @since    1.0.0
     * @param    string    $json_string    The JSON string to sanitize.
     * @param    int       $max_depth      Maximum depth for JSON parsing.
     * @return   mixed                     The sanitized JSON data or false on failure.
     */
    public static function sanitize_json( $json_string, $max_depth = 512 ) {
        // Basic string sanitization
        $json_string = sanitize_textarea_field( $json_string );
        
        // Validate JSON structure
        $data = json_decode( $json_string, true, $max_depth );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            self::log_security_event( 'json_parse_error', 'Invalid JSON input received', array( 'error' => json_last_error_msg() ) );
            return false;
        }
        
        // Recursively sanitize the data
        return self::sanitize_input( $data );
    }

    /**
     * Validate file upload.
     *
     * @since    1.0.0
     * @param    array     $file           The file upload array.
     * @param    array     $allowed_types  Allowed MIME types.
     * @param    int       $max_size       Maximum file size in bytes.
     * @return   array                     Validation result with sanitized data.
     */
    public static function validate_file_upload( $file, $allowed_types = array(), $max_size = 2097152 ) {
        $result = array(
            'valid' => false,
            'error' => '',
            'file' => null
        );
        
        // Check if file was uploaded
        if ( ! isset( $file['tmp_name'] ) || ! is_uploaded_file( $file['tmp_name'] ) ) {
            $result['error'] = __( 'No file was uploaded or file upload failed.', 'unplug' );
            return $result;
        }
        
        // Check file size
        if ( $file['size'] > $max_size ) {
            // translators: %s: maximum allowed file size in bytes
            $result['error'] = sprintf( 
                /* translators: %s: maximum allowed file size in bytes */
                __( 'File size exceeds maximum allowed size of %s bytes.', 'unplug' ), 
                number_format( $max_size ) 
            );
            return $result;
        }
        
        // Validate MIME type
        $file_type = wp_check_filetype( $file['name'] );
        if ( ! empty( $allowed_types ) && ! in_array( $file_type['type'], $allowed_types ) ) {
            $result['error'] = __( 'File type is not allowed.', 'unplug' );
            return $result;
        }
        
        // Additional security checks
        if ( ! self::is_safe_filename( $file['name'] ) ) {
            $result['error'] = __( 'Filename contains invalid characters.', 'unplug' );
            return $result;
        }
        
        // Sanitize filename
        $file['name'] = sanitize_file_name( $file['name'] );
        
        $result['valid'] = true;
        $result['file'] = $file;
        
        return $result;
    }

    /**
     * Check if filename is safe.
     *
     * @since    1.0.0
     * @param    string    $filename    The filename to check.
     * @return   bool                   True if safe, false otherwise.
     */
    public static function is_safe_filename( $filename ) {
        // Check for null bytes
        if ( strpos( $filename, "\0" ) !== false ) {
            return false;
        }
        
        // Check for directory traversal
        if ( strpos( $filename, '..' ) !== false || strpos( $filename, '/' ) !== false || strpos( $filename, '\\' ) !== false ) {
            return false;
        }
        
        // Check for executable extensions
        $dangerous_extensions = array( 'php', 'php3', 'php4', 'php5', 'phtml', 'pl', 'py', 'jsp', 'asp', 'sh', 'cgi', 'exe', 'bat', 'com', 'scr', 'vbs', 'js' );
        $extension = strtolower( pathinfo( $filename, PATHINFO_EXTENSION ) );
        
        if ( in_array( $extension, $dangerous_extensions ) ) {
            return false;
        }
        
        return true;
    }

    /**
     * Sanitize database query parameters.
     *
     * @since    1.0.0
     * @param    array     $params    The parameters to sanitize.
     * @param    array     $types     Parameter types (string, int, float, etc.).
     * @return   array                Sanitized parameters.
     */
    public static function sanitize_db_params( $params, $types = array() ) {
        $sanitized = array();
        
        foreach ( $params as $key => $value ) {
            $type = isset( $types[ $key ] ) ? $types[ $key ] : 'string';
            
            switch ( $type ) {
                case 'int':
                    $sanitized[ $key ] = absint( $value );
                    break;
                case 'float':
                    $sanitized[ $key ] = floatval( $value );
                    break;
                case 'bool':
                    $sanitized[ $key ] = (bool) $value;
                    break;
                case 'email':
                    $sanitized[ $key ] = sanitize_email( $value );
                    break;
                case 'url':
                    $sanitized[ $key ] = esc_url_raw( $value );
                    break;
                case 'sql':
                    $sanitized[ $key ] = esc_sql( $value );
                    break;
                case 'string':
                default:
                    $sanitized[ $key ] = sanitize_text_field( $value );
                    break;
            }
        }
        
        return $sanitized;
    }

    /**
     * Validate form data with comprehensive rules.
     *
     * @since    1.0.0
     * @param    array     $data      The form data to validate.
     * @param    array     $rules     Validation rules.
     * @return   array                Validation result.
     */
    public static function validate_form_data( $data, $rules ) {
        $result = array(
            'valid' => true,
            'errors' => array(),
            'sanitized' => array()
        );
        
        foreach ( $rules as $field => $rule ) {
            $value = isset( $data[ $field ] ) ? $data[ $field ] : '';
            
            // Check required fields
            if ( isset( $rule['required'] ) && $rule['required'] && empty( $value ) ) {
                // translators: %s: field label or name
                $result['errors'][ $field ] = sprintf( 
                    /* translators: %s: field name that is required */
                    __( '%s is required.', 'unplug' ), 
                    $rule['label'] ?? $field 
                );
                $result['valid'] = false;
                continue;
            }
            
            // Skip validation for empty non-required fields
            if ( empty( $value ) && ! ( isset( $rule['required'] ) && $rule['required'] ) ) {
                $result['sanitized'][ $field ] = '';
                continue;
            }
            
            // Validate based on type
            $type = $rule['type'] ?? 'text';
            $sanitized_value = self::sanitize_input( $value, $type );
            
            // Additional validation rules
            if ( isset( $rule['min_length'] ) && strlen( $sanitized_value ) < $rule['min_length'] ) {
                // translators: %1$s: field label or name, %2$d: minimum length
                $result['errors'][ $field ] = sprintf( 
                    /* translators: %1$s: field name, %2$d: minimum character length */
                    __( '%1$s must be at least %2$d characters long.', 'unplug' ), 
                    $rule['label'] ?? $field, 
                    $rule['min_length'] 
                );
                $result['valid'] = false;
            }
            
            if ( isset( $rule['max_length'] ) && strlen( $sanitized_value ) > $rule['max_length'] ) {
                // translators: %1$s: field label or name, %2$d: maximum length
                $result['errors'][ $field ] = sprintf( 
                    /* translators: %1$s: field name, %2$d: maximum character length */
                    __( '%1$s must be no more than %2$d characters long.', 'unplug' ), 
                    $rule['label'] ?? $field, 
                    $rule['max_length'] 
                );
                $result['valid'] = false;
            }
            
            if ( isset( $rule['pattern'] ) && ! preg_match( $rule['pattern'], $sanitized_value ) ) {
                // translators: %s: field label or name
                $result['errors'][ $field ] = sprintf( 
                    /* translators: %s: field name with invalid format */
                    __( '%s format is invalid.', 'unplug' ), 
                    $rule['label'] ?? $field 
                );
                $result['valid'] = false;
            }
            
            if ( isset( $rule['options'] ) && ! in_array( $sanitized_value, $rule['options'] ) ) {
                // translators: %s: field label or name
                $result['errors'][ $field ] = sprintf( 
                    /* translators: %s: field name with invalid option */
                    __( '%s contains an invalid option.', 'unplug' ), 
                    $rule['label'] ?? $field 
                );
                $result['valid'] = false;
            }
            
            $result['sanitized'][ $field ] = $sanitized_value;
        }
        
        return $result;
    }

    /**
     * Escape output for safe display.
     *
     * @since    1.0.0
     * @param    mixed     $value     The value to escape.
     * @param    string    $context   The context (html, attr, js, css, url).
     * @return   string               The escaped value.
     */
    public static function escape_output( $value, $context = 'html' ) {
        switch ( $context ) {
            case 'attr':
                return esc_attr( $value );
            case 'js':
                return esc_js( $value );
            case 'css':
                return esc_css( $value );
            case 'url':
                return esc_url( $value );
            case 'textarea':
                return esc_textarea( $value );
            case 'html':
            default:
                return esc_html( $value );
        }
    }

    /**
     * Validate and sanitize admin form data.
     *
     * @since    1.0.0
     * @param    array     $data    The form data.
     * @return   array              Validation result.
     */
    public static function validate_admin_form( $data ) {
        $rules = array(
            'action' => array(
                'required' => true,
                'type' => 'text',
                'options' => array( 'refresh_plugins', 'start_scan', 'export_csv', 'generate_report' ),
                'label' => __( 'Action', 'unplug' )
            ),
            'scan_id' => array(
                'required' => false,
                'type' => 'text',
                'pattern' => '/^[a-f0-9\-]{36}$/',
                'label' => __( 'Scan ID', 'unplug' )
            )
        );
        
        return self::validate_form_data( $data, $rules );
    }

    /**
     * Create prepared statement parameters.
     *
     * @since    1.0.0
     * @param    array     $params    The parameters to prepare.
     * @return   array                Prepared parameters with types.
     */
    public static function prepare_db_params( $params ) {
        $prepared = array(
            'values' => array(),
            'format' => array()
        );
        
        foreach ( $params as $param ) {
            if ( is_int( $param ) ) {
                $prepared['values'][] = $param;
                $prepared['format'][] = '%d';
            } elseif ( is_float( $param ) ) {
                $prepared['values'][] = $param;
                $prepared['format'][] = '%f';
            } else {
                $prepared['values'][] = (string) $param;
                $prepared['format'][] = '%s';
            }
        }
        
                 return $prepared;
    }

    /**
     * Encryption key for API tokens.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $encryption_key    The encryption key.
     */
    private static $encryption_key = null;

    /**
     * Get or generate encryption key.
     *
     * @since    1.0.0
     * @return   string    The encryption key.
     */
    private static function get_encryption_key() {
        if ( self::$encryption_key === null ) {
            // Try to get existing key from WordPress constants or options
            if ( defined( 'UNPLUG_ENCRYPTION_KEY' ) ) {
                self::$encryption_key = UNPLUG_ENCRYPTION_KEY;
            } else {
                $stored_key = get_option( 'unplug_encryption_key' );
                if ( empty( $stored_key ) ) {
                    // Generate new key
                    self::$encryption_key = self::generate_encryption_key();
                    // Store it securely
                    update_option( 'unplug_encryption_key', self::$encryption_key );
                } else {
                    self::$encryption_key = $stored_key;
                }
            }
        }
        
        return self::$encryption_key;
    }

    /**
     * Generate a secure encryption key.
     *
     * @since    1.0.0
     * @return   string    The generated encryption key.
     */
    private static function generate_encryption_key() {
        // Use WordPress's wp_generate_password for cryptographically secure random key
        return wp_generate_password( 64, true, true );
    }

    /**
     * Encrypt API token.
     *
     * @since    1.0.0
     * @param    string    $token    The API token to encrypt.
     * @return   string              The encrypted token.
     */
    public static function encrypt_api_token( $token ) {
        if ( empty( $token ) ) {
            return '';
        }

        $key = self::get_encryption_key();
        
        // Use WordPress's built-in encryption if available (WP 4.4+)
        if ( function_exists( 'wp_hash' ) ) {
            // Create a unique salt for this token
            $salt = wp_generate_password( 16, false );
            $hashed_key = wp_hash( $key . $salt );
            
            // Use OpenSSL for encryption if available
            if ( function_exists( 'openssl_encrypt' ) ) {
                $method = 'AES-256-CBC';
                $iv = openssl_random_pseudo_bytes( openssl_cipher_iv_length( $method ) );
                $encrypted = openssl_encrypt( $token, $method, $hashed_key, 0, $iv );
                
                // Combine salt, IV, and encrypted data
                $result = base64_encode( $salt . '::' . base64_encode( $iv ) . '::' . $encrypted );
                
                self::log_security_event( 'api_token_encrypted', 'API token encrypted successfully' );
                return $result;
            }
        }
        
        // Fallback to base64 encoding (not secure, but better than plain text)
        self::log_security_event( 'api_token_encrypted_fallback', 'API token encrypted using fallback method' );
        return base64_encode( $token );
    }

    /**
     * Decrypt API token.
     *
     * @since    1.0.0
     * @param    string    $encrypted_token    The encrypted API token.
     * @return   string                        The decrypted token.
     */
    public static function decrypt_api_token( $encrypted_token ) {
        if ( empty( $encrypted_token ) ) {
            return '';
        }

        $key = self::get_encryption_key();
        
        // Check if this is OpenSSL encrypted data
        if ( strpos( $encrypted_token, '::' ) !== false ) {
            $decoded = base64_decode( $encrypted_token );
            if ( $decoded !== false ) {
                $parts = explode( '::', $decoded );
                if ( count( $parts ) === 3 ) {
                    $salt = $parts[0];
                    $iv = base64_decode( $parts[1] );
                    $encrypted = $parts[2];
                    
                    $hashed_key = wp_hash( $key . $salt );
                    
                    if ( function_exists( 'openssl_decrypt' ) ) {
                        $method = 'AES-256-CBC';
                        $decrypted = openssl_decrypt( $encrypted, $method, $hashed_key, 0, $iv );
                        
                        if ( $decrypted !== false ) {
                            self::log_security_event( 'api_token_decrypted', 'API token decrypted successfully' );
                            return $decrypted;
                        }
                    }
                }
            }
        }
        
        // Fallback to base64 decoding
        $decoded = base64_decode( $encrypted_token );
        if ( $decoded !== false ) {
            self::log_security_event( 'api_token_decrypted_fallback', 'API token decrypted using fallback method' );
            return $decoded;
        }
        
        // If all else fails, return the original (might be unencrypted)
        self::log_security_event( 'api_token_decrypt_failed', 'API token decryption failed', array( 'token_length' => strlen( $encrypted_token ) ) );
        return $encrypted_token;
    }

    /**
     * Validate API token format.
     *
     * @since    1.0.0
     * @param    string    $token    The API token to validate.
     * @return   bool                True if valid format, false otherwise.
     */
    public static function validate_api_token_format( $token ) {
        if ( empty( $token ) ) {
            return false;
        }
        
        // API tokens should be at least 16 characters long
        if ( strlen( $token ) < 16 ) {
            return false;
        }
        
        // Check for basic format (alphanumeric with possible dashes/underscores)
        if ( ! preg_match( '/^[a-zA-Z0-9_-]+$/', $token ) ) {
            return false;
        }
        
        return true;
    }

    /**
     * Securely store API token.
     *
     * @since    1.0.0
     * @param    string    $token    The API token to store.
     * @return   bool                True if successful, false otherwise.
     */
    public static function store_api_token( $token ) {
        if ( ! self::validate_api_token_format( $token ) ) {
            self::log_security_event( 'api_token_invalid_format', 'API token rejected due to invalid format' );
            return false;
        }
        
        $encrypted_token = self::encrypt_api_token( $token );
        $result = update_option( 'unplug_api_token_encrypted', $encrypted_token );
        
        if ( $result ) {
            self::log_security_event( 'api_token_stored', 'API token stored successfully' );
            
            // Clear the old unencrypted token if it exists
            delete_option( 'unplug_api_token' );
        }
        
        return $result;
    }

    /**
     * Retrieve API token.
     *
     * @since    1.0.0
     * @return   string    The decrypted API token.
     */
    public static function retrieve_api_token() {
        $encrypted_token = get_option( 'unplug_api_token_encrypted', '' );
        
        if ( empty( $encrypted_token ) ) {
            // Check for legacy unencrypted token
            $legacy_token = get_option( 'unplug_api_token', '' );
            if ( ! empty( $legacy_token ) ) {
                // Migrate to encrypted storage
                if ( self::store_api_token( $legacy_token ) ) {
                    self::log_security_event( 'api_token_migrated', 'Legacy API token migrated to encrypted storage' );
                    return $legacy_token;
                }
            }
            return '';
        }
        
        return self::decrypt_api_token( $encrypted_token );
    }

    /**
     * Delete API token.
     *
     * @since    1.0.0
     * @return   bool    True if successful, false otherwise.
     */
    public static function delete_api_token() {
        $result1 = delete_option( 'unplug_api_token_encrypted' );
        $result2 = delete_option( 'unplug_api_token' ); // Also delete legacy token
        
        if ( $result1 || $result2 ) {
            self::log_security_event( 'api_token_deleted', 'API token deleted successfully' );
        }
        
        return $result1 || $result2;
    }

    /**
     * Check if API token is encrypted.
     *
     * @since    1.0.0
     * @return   bool    True if encrypted, false otherwise.
     */
    public static function is_api_token_encrypted() {
        return ! empty( get_option( 'unplug_api_token_encrypted' ) );
    }

    /**
     * Rotate encryption key (for security maintenance).
     *
     * @since    1.0.0
     * @return   bool    True if successful, false otherwise.
     */
    public static function rotate_encryption_key() {
        // Get current token
        $current_token = self::retrieve_api_token();
        
        if ( empty( $current_token ) ) {
            return true; // Nothing to rotate
        }
        
        // Generate new key
        $new_key = self::generate_encryption_key();
        
        // Update key
        update_option( 'unplug_encryption_key', $new_key );
        self::$encryption_key = $new_key;
        
        // Re-encrypt token with new key
        $result = self::store_api_token( $current_token );
        
        if ( $result ) {
            self::log_security_event( 'encryption_key_rotated', 'Encryption key rotated successfully' );
        }
        
                 return $result;
    }

    /**
     * Available critical operations that require double confirmation.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $critical_operations    The critical operations.
     */
    private static $critical_operations = array(
        'plugin_deactivation_test' => array(
            'title' => 'Plugin Deactivation Test',
            'warning' => 'This will temporarily deactivate plugins on your live site. Make sure you have a recent backup before proceeding.',
            'risk_level' => 'high',
            'backup_required' => true
        ),
        'plugin_deletion' => array(
            'title' => 'Plugin Deletion',
            'warning' => 'This will permanently delete plugins from your site. This action cannot be undone.',
            'risk_level' => 'high',
            'backup_required' => true
        ),
        'settings_reset' => array(
            'title' => 'Settings Reset',
            'warning' => 'This will reset all Unplug settings to default values. Your API token and custom configuration will be lost.',
            'risk_level' => 'medium',
            'backup_required' => false
        ),
        'api_token_deletion' => array(
            'title' => 'API Token Deletion',
            'warning' => 'This will remove your API token and revert your account to the free tier. You will lose access to Pro/Agency features.',
            'risk_level' => 'medium',
            'backup_required' => false
        ),
        'scan_data_cleanup' => array(
            'title' => 'Scan Data Cleanup',
            'warning' => 'This will permanently delete all scan results and analysis data. You will need to run new scans to get plugin recommendations.',
            'risk_level' => 'low',
            'backup_required' => false
        )
    );

    /**
     * Generate confirmation token for critical operation.
     *
     * @since    1.0.0
     * @param    string    $operation    The operation requiring confirmation.
     * @param    int       $user_id      The user ID (optional).
     * @return   string                  The confirmation token.
     */
    public static function generate_confirmation_token( $operation, $user_id = null ) {
        if ( ! isset( self::$critical_operations[ $operation ] ) ) {
            return false;
        }
        
        $user_id = $user_id ?: get_current_user_id();
        $token = wp_generate_password( 32, false );
        $expiry = time() + ( 5 * MINUTE_IN_SECONDS ); // 5 minute expiry
        
        $confirmation_data = array(
            'operation' => $operation,
            'user_id' => $user_id,
            'token' => $token,
            'expiry' => $expiry,
            'created' => time(),
            'ip_address' => sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ?? 'unknown' ) )
        );
        
        // Store in transient for security
        set_transient( 'unplug_confirmation_' . $token, $confirmation_data, 5 * MINUTE_IN_SECONDS );
        
        self::log_security_event( 'confirmation_token_generated', 'Confirmation token generated for critical operation', array( 
            'operation' => $operation,
            'user_id' => $user_id,
            'expiry' => $expiry
        ) );
        
        return $token;
    }

    /**
     * Verify confirmation token.
     *
     * @since    1.0.0
     * @param    string    $token        The confirmation token.
     * @param    string    $operation    The operation to verify.
     * @return   bool                    True if valid, false otherwise.
     */
    public static function verify_confirmation_token( $token, $operation ) {
        if ( empty( $token ) || empty( $operation ) ) {
            return false;
        }
        
        $confirmation_data = get_transient( 'unplug_confirmation_' . $token );
        
        if ( ! $confirmation_data ) {
            self::log_security_event( 'confirmation_token_not_found', 'Confirmation token not found or expired', array( 'token' => substr( $token, 0, 8 ) . '...' ) );
            return false;
        }
        
        // Verify operation matches (using hash_equals to prevent timing attacks)
        if ( ! hash_equals( $confirmation_data['operation'], $operation ) ) {
            self::log_security_event( 'confirmation_token_operation_mismatch', 'Confirmation token operation mismatch', array( 
                'expected' => $operation,
                'actual' => $confirmation_data['operation']
            ) );
            return false;
        }
        
        // Verify user
        if ( $confirmation_data['user_id'] !== get_current_user_id() ) {
            self::log_security_event( 'confirmation_token_user_mismatch', 'Confirmation token user mismatch', array( 
                'token_user' => $confirmation_data['user_id'],
                'current_user' => get_current_user_id()
            ) );
            return false;
        }
        
        // Verify expiry
        if ( time() > $confirmation_data['expiry'] ) {
            self::log_security_event( 'confirmation_token_expired', 'Confirmation token expired', array( 'token' => substr( $token, 0, 8 ) . '...' ) );
            return false;
        }
        
        // Delete the token after successful verification (single use)
        delete_transient( 'unplug_confirmation_' . $token );
        
        self::log_security_event( 'confirmation_token_verified', 'Confirmation token verified successfully', array( 
            'operation' => $operation,
            'user_id' => $confirmation_data['user_id']
        ) );
        
        return true;
    }

    /**
     * Get operation details for confirmation dialog.
     *
     * @since    1.0.0
     * @param    string    $operation    The operation name.
     * @return   array                   Operation details or false.
     */
    public static function get_operation_details( $operation ) {
        return isset( self::$critical_operations[ $operation ] ) ? self::$critical_operations[ $operation ] : false;
    }

    /**
     * Check if operation requires backup.
     *
     * @since    1.0.0
     * @param    string    $operation    The operation name.
     * @return   bool                    True if backup required, false otherwise.
     */
    public static function operation_requires_backup( $operation ) {
        $details = self::get_operation_details( $operation );
        return $details ? $details['backup_required'] : false;
    }

    /**
     * Generate confirmation dialog HTML.
     *
     * @since    1.0.0
     * @param    string    $operation    The operation requiring confirmation.
     * @param    string    $token        The confirmation token.
     * @return   string                  The confirmation dialog HTML.
     */
    public static function get_confirmation_dialog_html( $operation, $token ) {
        $details = self::get_operation_details( $operation );
        
        if ( ! $details ) {
            return '';
        }
        
        $backup_warning = '';
        if ( $details['backup_required'] ) {
            $backup_warning = '<div class="unplug-backup-warning">
                <p><strong>' . __( '⚠️ BACKUP REQUIRED', 'unplug' ) . '</strong></p>
                <p>' . __( 'Before proceeding, ensure you have a recent backup of your website. This operation may affect your site\'s functionality.', 'unplug' ) . '</p>
            </div>';
        }
        
        $risk_class = 'unplug-risk-' . $details['risk_level'];
        
        $html = '<div class="unplug-confirmation-overlay" id="unplug-confirmation-' . esc_attr( $operation ) . '">
            <div class="unplug-confirmation-dialog ' . esc_attr( $risk_class ) . '">
                <div class="unplug-confirmation-header">
                    <h3>' . esc_html( $details['title'] ) . '</h3>
                    <button type="button" class="unplug-confirmation-close">&times;</button>
                </div>
                <div class="unplug-confirmation-content">
                    ' . $backup_warning . '
                    <div class="unplug-operation-warning">
                        <p>' . esc_html( $details['warning'] ) . '</p>
                    </div>
                    <div class="unplug-confirmation-steps">
                        <label class="unplug-confirmation-step">
                            <input type="checkbox" id="unplug-understand-risks" required>
                            <span>' . __( 'I understand the risks and have read the warning above', 'unplug' ) . '</span>
                        </label>';
        
        if ( $details['backup_required'] ) {
            $html .= '<label class="unplug-confirmation-step">
                            <input type="checkbox" id="unplug-confirm-backup" required>
                            <span>' . __( 'I confirm that I have a recent backup of my website', 'unplug' ) . '</span>
                        </label>';
        }
        
        $html .= '                        <label class="unplug-confirmation-step">
                            <input type="checkbox" id="unplug-final-confirmation" required>
                            <span>' . sprintf( 
                                // translators: %s: operation title
                                __( 'I want to proceed with %s', 'unplug' ), 
                                $details['title'] 
                            ) . '</span>
                        </label>
                    </div>
                </div>
                <div class="unplug-confirmation-actions">
                    <button type="button" class="button" id="unplug-confirmation-cancel">' . __( 'Cancel', 'unplug' ) . '</button>
                    <button type="button" class="button button-primary" id="unplug-confirmation-proceed" disabled>' . __( 'Proceed', 'unplug' ) . '</button>
                </div>
                <input type="hidden" id="unplug-confirmation-token" value="' . esc_attr( $token ) . '">
                <input type="hidden" id="unplug-confirmation-operation" value="' . esc_attr( $operation ) . '">
            </div>
        </div>';
        
        return $html;
    }

    /**
     * Validate confirmation request.
     *
     * @since    1.0.0
     * @param    string    $operation    The operation name.
     * @param    string    $token        The confirmation token.
     * @param    bool      $ajax         Whether this is an AJAX request.
     * @return   bool                    True if valid, false otherwise.
     */
    public static function validate_confirmation_request( $operation, $token, $ajax = false ) {
        // Verify nonce first
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- This function handles nonce verification internally
        if ( $ajax && ! self::verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'admin', true ) ) {
            return false;
        }
        
        // Verify capability
        if ( ! self::check_capability( 'manage_options', $ajax ) ) {
            return false;
        }
        
        // Verify confirmation token
        if ( ! self::verify_confirmation_token( $token, $operation ) ) {
            if ( $ajax ) {
                wp_send_json_error( array( 
                    'message' => __( 'Confirmation token is invalid or expired. Please try again.', 'unplug' ),
                    'code' => 'invalid_confirmation_token'
                ) );
            }
            return false;
        }
        
        self::log_security_event( 'confirmation_request_validated', 'Confirmation request validated successfully', array( 
            'operation' => $operation,
            'ajax' => $ajax
        ) );
        
        return true;
    }

    /**
     * Create admin notice for confirmation requirement.
     *
     * @since    1.0.0
     * @param    string    $operation    The operation name.
     * @return   string                  The admin notice HTML.
     */
    public static function get_confirmation_admin_notice( $operation ) {
        $details = self::get_operation_details( $operation );
        
        if ( ! $details ) {
            return '';
        }
        
        $notice_class = 'notice-warning';
        if ( $details['risk_level'] === 'high' ) {
            $notice_class = 'notice-error';
        }
        
        $html = '<div class="notice ' . $notice_class . ' unplug-confirmation-notice">
            <p><strong>' . sprintf( 
                /* translators: %s: operation title that requires confirmation */
                __( '%s requires confirmation', 'unplug' ), 
                $details['title'] 
            ) . '</strong></p>
            <p>' . esc_html( $details['warning'] ) . '</p>';
        
        if ( $details['backup_required'] ) {
            $html .= '<p><strong>' . __( 'Backup Required:', 'unplug' ) . '</strong> ' . __( 'Please ensure you have a recent backup before proceeding.', 'unplug' ) . '</p>';
        }
        
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Log security events for debugging and monitoring.
     *
     * @since    1.0.0
     * @param    string    $event        The security event type.
     * @param    string    $message      The event message.
     * @param    array     $context      Additional context data.
     * @return   void
     */
    public static function log_security_event( $event, $message, $context = array() ) {
        // Only log if debug logging is enabled
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-options.php';
        
        if ( ! UNPLUG_Options::get_enable_logging() ) {
            return;
        }
        
        $log_entry = array(
            'timestamp' => current_time( 'mysql' ),
            'event' => $event,
            'message' => $message,
            'user_id' => get_current_user_id(),
            'ip_address' => sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ?? 'unknown' ) ),
            'user_agent' => sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ?? 'unknown' ) ),
            'context' => $context
        );
        
        // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log( 
            sprintf( 
                '[UNPLUG_SECURITY] %s: %s | User: %d | IP: %s | Context: %s', 
                $event, 
                $message, 
                $log_entry['user_id'],
                $log_entry['ip_address'],
                wp_json_encode( $context )
            ) 
        );
        // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
    }

    /**
     * Get available nonce actions.
     *
     * @since    1.0.0
     * @return   array    The available nonce actions.
     */
    public static function get_nonce_actions() {
        return self::$nonce_actions;
    }

    /**
     * Check if a nonce action is valid.
     *
     * @since    1.0.0
     * @param    string    $action    The action to check.
     * @return   bool                 True if valid, false otherwise.
     */
    public static function is_valid_nonce_action( $action ) {
        return isset( self::$nonce_actions[ $action ] );
    }

    /**
     * Plugin-specific capabilities mapping.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $plugin_capabilities    The plugin-specific capabilities.
     */
    private static $plugin_capabilities = array(
        'manage_unplug_settings' => 'manage_options',
        'run_unplug_scans' => 'manage_options',
        'export_unplug_data' => 'manage_options',
        'view_unplug_reports' => 'manage_options',
        'manage_unplug_cleanup' => 'manage_options'
    );

    /**
     * Check if user has plugin-specific capability.
     *
     * @since    1.0.0
     * @param    string    $capability    The plugin capability to check.
     * @param    bool      $ajax          Whether this is an AJAX request.
     * @return   bool                     True if user has capability, false otherwise.
     */
    public static function check_plugin_capability( $capability, $ajax = false ) {
        if ( ! isset( self::$plugin_capabilities[ $capability ] ) ) {
            // Unknown capability, deny access
            if ( $ajax ) {
                wp_send_json_error( array( 
                    'message' => __( 'Invalid capability requested.', 'unplug' ),
                    'code' => 'invalid_capability'
                ) );
            } else {
                wp_die( 
                    esc_html( __( 'Invalid capability requested.', 'unplug' ) ), 
                    esc_html( __( 'Permission Error', 'unplug' ) ), 
                    array( 'response' => 403 )
                );
            }
        }
        
        $required_capability = self::$plugin_capabilities[ $capability ];
        return self::check_capability( $required_capability, $ajax );
    }

    /**
     * Check if user can perform specific admin action.
     *
     * @since    1.0.0
     * @param    string    $action    The action to check (settings, scan, export, etc.).
     * @param    bool      $ajax      Whether this is an AJAX request.
     * @return   bool                 True if user can perform action, false otherwise.
     */
    public static function can_perform_action( $action, $ajax = false ) {
        $action_mapping = array(
            'settings' => 'manage_unplug_settings',
            'scan' => 'run_unplug_scans',
            'export' => 'export_unplug_data',
            'reports' => 'view_unplug_reports',
            'cleanup' => 'manage_unplug_cleanup'
        );
        
        if ( isset( $action_mapping[ $action ] ) ) {
            return self::check_plugin_capability( $action_mapping[ $action ], $ajax );
        }
        
        // Default to basic admin capability
        return self::check_capability( 'manage_options', $ajax );
    }

    /**
     * Get required capability for a specific action.
     *
     * @since    1.0.0
     * @param    string    $action    The action to check.
     * @return   string               The required capability.
     */
    public static function get_required_capability( $action ) {
        $action_mapping = array(
            'settings' => 'manage_unplug_settings',
            'scan' => 'run_unplug_scans',
            'export' => 'export_unplug_data',
            'reports' => 'view_unplug_reports',
            'cleanup' => 'manage_unplug_cleanup'
        );
        
        if ( isset( $action_mapping[ $action ] ) ) {
            $plugin_capability = $action_mapping[ $action ];
            return self::$plugin_capabilities[ $plugin_capability ];
        }
        
        return 'manage_options';
    }

    /**
     * Initialize plugin capabilities by adding them to appropriate roles.
     *
     * @since    1.0.0
     * @return   void
     */
    public static function initialize_capabilities() {
        // Add custom capabilities to administrator role
        $admin_role = get_role( 'administrator' );
        
        if ( $admin_role ) {
            foreach ( self::$plugin_capabilities as $plugin_cap => $required_cap ) {
                $admin_role->add_cap( $plugin_cap );
            }
        }
        
        // Log capability initialization
        self::log_security_event( 'capabilities_initialized', 'Plugin capabilities added to administrator role' );
    }

    /**
     * Remove plugin capabilities from roles (cleanup on plugin deactivation).
     *
     * @since    1.0.0
     * @return   void
     */
    public static function remove_capabilities() {
        // Remove custom capabilities from all roles
        $roles = wp_roles();
        
        foreach ( $roles->roles as $role_name => $role_data ) {
            $role = get_role( $role_name );
            if ( $role ) {
                foreach ( self::$plugin_capabilities as $plugin_cap => $required_cap ) {
                    $role->remove_cap( $plugin_cap );
                }
            }
        }
        
        // Log capability removal
        self::log_security_event( 'capabilities_removed', 'Plugin capabilities removed from all roles' );
    }

    /**
     * Check if current user has minimum role.
     *
     * @since    1.0.0
     * @param    string    $minimum_role    The minimum role required.
     * @param    bool      $ajax            Whether this is an AJAX request.
     * @return   bool                       True if user has minimum role, false otherwise.
     */
    public static function check_minimum_role( $minimum_role = 'administrator', $ajax = false ) {
        $current_user = wp_get_current_user();
        
        if ( ! $current_user || ! $current_user->ID ) {
            if ( $ajax ) {
                wp_send_json_error( array( 
                    'message' => __( 'You must be logged in to perform this action.', 'unplug' ),
                    'code' => 'not_logged_in'
                ) );
            } else {
                wp_die( 
                    esc_html( __( 'You must be logged in to perform this action.', 'unplug' ) ), 
                    esc_html( __( 'Authentication Error', 'unplug' ) ), 
                    array( 'response' => 403 )
                );
            }
        }
        
        $has_role = in_array( $minimum_role, $current_user->roles );
        
        if ( ! $has_role ) {
            if ( $ajax ) {
                wp_send_json_error( array( 
                    'message' => sprintf( 
                        /* translators: %s: minimum required user role */
                        __( 'You must have %s role to perform this action.', 'unplug' ), 
                        $minimum_role 
                    ),
                    'code' => 'insufficient_role'
                ) );
            } else {
                wp_die( 
                    esc_html( sprintf( 
                        /* translators: %s: minimum required user role */
                        __( 'You must have %s role to perform this action.', 'unplug' ), 
                        $minimum_role 
                    ) ), 
                    esc_html( __( 'Permission Error', 'unplug' ) ), 
                    array( 'response' => 403 )
                );
            }
        }
        
        return $has_role;
    }

    /**
     * Get all plugin capabilities.
     *
     * @since    1.0.0
     * @return   array    The plugin capabilities.
     */
    public static function get_plugin_capabilities() {
        return self::$plugin_capabilities;
    }

    /**
     * Verify admin capabilities on admin_init.
     *
     * @since    1.0.0
     * @return   void
     */
    public static function verify_admin_capabilities() {
        // Only check on Unplug admin pages
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This function handles admin capability checks and is called from verified contexts
        if ( ! isset( $_GET['page'] ) || strpos( sanitize_text_field( wp_unslash( $_GET['page'] ) ), 'unplug' ) !== 0 ) {
            return;
        }

        // Check basic capability
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 
                esc_html( __( 'You do not have sufficient permissions to access this page.', 'unplug' ) ), 
                esc_html( __( 'Permission Denied', 'unplug' ) ), 
                array( 'response' => 403 )
            );
        }

        // Log admin access
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This function handles admin capability checks and is called from verified contexts
        self::log_security_event( 'admin_access', 'User accessed Unplug admin area', array( 'page' => sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) );
    }

    /**
     * Add capabilities to user when they are assigned admin role.
     *
     * @since    1.0.0
     * @param    int    $user_id    The user ID.
     * @return   void
     */
    public static function maybe_add_capabilities_to_user( $user_id ) {
        $user = get_user_by( 'id', $user_id );
        
        if ( $user && in_array( 'administrator', $user->roles ) ) {
            foreach ( self::$plugin_capabilities as $plugin_cap => $required_cap ) {
                $user->add_cap( $plugin_cap );
            }
            
            self::log_security_event( 'user_capabilities_added', 'Plugin capabilities added to new admin user', array( 'user_id' => $user_id ) );
        }
    }

    /**
     * Enhanced capability check with detailed logging.
     *
     * @since    1.0.0
     * @param    string    $capability    The capability to check.
     * @param    int       $user_id       The user ID to check (optional).
     * @param    bool      $ajax          Whether this is an AJAX request.
     * @return   bool                     True if user has capability, false otherwise.
     */
    public static function verify_user_capability( $capability, $user_id = null, $ajax = false ) {
        $user_id = $user_id ?: get_current_user_id();
        $user = get_user_by( 'id', $user_id );
        
        if ( ! $user ) {
            self::log_security_event( 'capability_check_failed', 'User not found during capability check', array( 'user_id' => $user_id, 'capability' => $capability ) );
            return false;
        }
        
        $has_capability = user_can( $user, $capability );
        
        if ( ! $has_capability ) {
            self::log_security_event( 'capability_denied', 'User capability check failed', array( 
                'user_id' => $user_id, 
                'capability' => $capability, 
                'user_roles' => $user->roles 
            ) );
        }
        
        return $has_capability;
    }

    /**
     * Check if current page requires specific capability.
     *
     * @since    1.0.0
     * @param    string    $page    The page slug.
     * @return   string             The required capability.
     */
    public static function get_page_required_capability( $page ) {
        $page_capabilities = array(
            'unplug' => 'run_unplug_scans',
            'unplug-settings' => 'manage_unplug_settings'
        );
        
        return isset( $page_capabilities[ $page ] ) ? $page_capabilities[ $page ] : 'manage_options';
    }

    /**
     * Comprehensive admin access validation.
     *
     * @since    1.0.0
     * @param    string    $page    The page being accessed.
     * @return   bool               True if access is allowed, false otherwise.
     */
    public static function validate_admin_access( $page = '' ) {
        // Get required capability for the page
        $required_capability = self::get_page_required_capability( $page );
        
        // Check if user has the required capability
        if ( ! self::verify_user_capability( $required_capability ) ) {
            self::log_security_event( 'admin_access_denied', 'Admin access denied', array( 
                'page' => $page, 
                'required_capability' => $required_capability 
            ) );
            return false;
        }
        
        return true;
    }

    /**
     * Validate API token using JWT with caching.
     *
     * @since 1.0.0
     * @param string $token The JWT token to validate.
     * @return array|false Decoded claims on success, false on failure.
     */
    public static function validate_jwt_token($token) {
        if (empty($token)) return false;
        $cache_key = 'unplug_jwt_' . md5($token);
        $cached = get_transient($cache_key);
        if ($cached !== false) return $cached;
        $publicKey = get_option('unplug_jwt_public_key');
        if (empty($publicKey)) {
            self::log_security_event('jwt_validation_failed', 'No public key set for JWT validation');
            return false;
        }
        if (!class_exists('Firebase\JWT\JWT')) {
            self::log_security_event('jwt_validation_failed', 'firebase/php-jwt library not found');
            return false;
        }
        try {
            $decoded = \Firebase\JWT\JWT::decode($token, new \Firebase\JWT\Key($publicKey, 'RS256'));
            $claims = (array) $decoded;
            set_transient($cache_key, $claims, 15 * MINUTE_IN_SECONDS);
            return $claims;
        } catch (\Exception $e) {
            self::log_security_event('jwt_validation_failed', 'JWT validation error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if site has a valid API token.
     * Temporary behavior: treat a specific test token as always valid until real validation exists.
     *
     * @since 1.0.0
     * @return bool
     */
    public static function has_valid_api_token() {
        // Get stored token
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-options.php';
        $token = UNPLUG_Options::get_api_token();

        if ( empty( $token ) ) {
            return false;
        }

        // Test token that is always valid during development
        if ( $token === 'UNPLUG_TEST_TOKEN_123456' ) {
            return true;
        }

        // If JWT validation is configured, attempt verification
        $claims = self::validate_jwt_token( $token );
        if ( $claims !== false ) {
            return true;
        }

        // Fallback: basic format check for now (non-empty and passes format rule)
        return self::validate_api_token_format( $token );
    }
} 