<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
/**
 * Plugin Scanner - Comprehensive plugin detection and metadata extraction
 *
 * This class handles scanning all installed plugins and extracting detailed
 * metadata about their functionality, dependencies, and potential conflicts.
 *
 * @link       https://github.com/stnchrch/unplug-wp
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * The plugin scanner class.
 *
 * This class provides comprehensive plugin detection and metadata extraction
 * capabilities for the Unplug WordPress plugin.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Your Name <email@example.com>
 */
class UNPLUG_Scanner {

    /**
     * The database instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Database    $db    The database instance.
     */
    private $db;

    /**
     * The security instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Security    $security    The security instance.
     */
    private $security;

    /**
     * The queue instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Queue    $queue    The queue instance.
     */
    private $queue;

    /**
     * Resource limits for scanning.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $limits    Resource limits configuration.
     */
    private $limits = array(
        'memory_threshold' => 100 * 1024 * 1024, // 100MB
        'time_threshold' => 30,                   // 30 seconds
        'file_size_limit' => 5 * 1024 * 1024,    // 5MB per file
        'scan_depth' => 10,                       // Max directory depth
        'max_files_per_plugin' => 1000            // Max files to scan per plugin
    );

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    UNPLUG_Database    $db         The database instance.
     * @param    UNPLUG_Security    $security   The security instance.
     * @param    UNPLUG_Queue       $queue      The queue instance.
     */
    public function __construct( $db, $security, $queue ) {
        $this->db = $db;
        $this->security = $security;
        $this->queue = $queue;
    }

    /**
     * Scan all installed plugins and add them to the processing queue.
     *
     * @since    1.0.0
     * @return   int    Number of plugins found and queued for scanning.
     */
    public function scan_all_plugins() {
        // Security check
        if ( ! current_user_can( 'manage_options' ) ) {
            return new WP_Error( 'insufficient_permissions', 'You do not have permission to scan plugins.' );
        }

        // Load plugin functions if not available
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        // Get all plugins
        $plugins = get_plugins();
        $active_plugins = get_option( 'active_plugins', array() );

        // For multisite, also get network activated plugins
        if ( is_multisite() ) {
            $network_plugins = array_keys( get_site_option( 'active_sitewide_plugins', array() ) );
            $active_plugins = array_merge( $active_plugins, $network_plugins );
        }

        $queued_count = 0;
        $start_time = microtime( true );

        foreach ( $plugins as $plugin_path => $plugin_data ) {
            // Check resource limits
            if ( $this->check_resource_limits( $start_time ) ) {
                // Log that we're pausing the scan
                $this->security->log_security_event( 'plugin_scan_paused', 'Plugin scan paused due to resource limits' );
                break;
            }

            // Determine if plugin is active
            $is_active = in_array( $plugin_path, $active_plugins );

            // Add to queue with appropriate priority
            $priority = $is_active ? 5 : 10; // Higher priority for active plugins
            $this->queue->add_task( 'plugin_scan', array(
                'plugin_path' => $plugin_path,
                'plugin_data' => $plugin_data,
                'is_active' => $is_active,
                'scan_timestamp' => current_time( 'timestamp' )
            ), $priority );

            $queued_count++;
        }

        // Log the scan initiation
        $this->security->log_security_event( 'plugin_scan_initiated', 'Plugin scan initiated', array(
            'total_plugins' => count( $plugins ),
            'queued_plugins' => $queued_count,
            'active_plugins' => count( $active_plugins )
        ) );

        return $queued_count;
    }

    /**
     * Process a single plugin scan from the queue.
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @param    array   $plugin_data  The plugin data.
     * @param    bool    $is_active    Whether the plugin is active.
     * @return   array|WP_Error        The scan results or error.
     */
    public function process_plugin_scan( $plugin_path, $plugin_data, $is_active = false ) {
        $start_time = microtime( true );

        try {
            // Extract comprehensive metadata
            $metadata = $this->extract_plugin_metadata( $plugin_path, $plugin_data, $is_active );

            // Detect API usage patterns
            $api_usage = $this->detect_api_usage( $plugin_path );

            // Analyze dependencies
            $dependencies = $this->analyze_dependencies( $plugin_path );

            // Analyze plugin hooks
            $hooks = $this->analyze_plugin_hooks( $plugin_path );

            // Detect custom database tables
            $custom_tables = $this->detect_custom_tables( $plugin_path );

            // Calculate resource usage
            $resource_usage = $this->calculate_resource_usage( $plugin_path );

            // Compile comprehensive scan results
            $scan_results = array(
                'metadata' => $metadata,
                'api_usage' => $api_usage,
                'dependencies' => $dependencies,
                'hooks' => $hooks,
                'custom_tables' => $custom_tables,
                'resource_usage' => $resource_usage,
                'scan_duration' => microtime( true ) - $start_time,
                'scan_timestamp' => current_time( 'timestamp' )
            );

            // Store results in database
            $this->db->store_scan_result( $plugin_path, 'comprehensive', $scan_results, 100 );

            // Log successful scan
            $this->security->log_security_event( 'plugin_scan_completed', 'Plugin scan completed successfully', array(
                'plugin_path' => $plugin_path,
                'plugin_name' => $metadata['name'],
                'scan_duration' => $scan_results['scan_duration']
            ) );

            return $scan_results;

        } catch ( Exception $e ) {
            // Log the error
            $this->security->log_security_event( 'plugin_scan_error', 'Plugin scan failed', array(
                'plugin_path' => $plugin_path,
                'error' => $e->getMessage()
            ) );

            return new WP_Error( 'scan_failed', 'Plugin scan failed: ' . $e->getMessage() );
        }
    }

    /**
     * Extract comprehensive plugin metadata.
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @param    array   $plugin_data  The plugin data from WordPress.
     * @param    bool    $is_active    Whether the plugin is active.
     * @return   array                 The extracted metadata.
     */
    private function extract_plugin_metadata( $plugin_path, $plugin_data, $is_active = false ) {
        // Start with WordPress plugin data
        $metadata = $plugin_data;
        $metadata['is_active'] = $is_active;
        $metadata['plugin_path'] = $plugin_path;

        // Get plugin directory
        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_path );
        $plugin_file = WP_PLUGIN_DIR . '/' . $plugin_path;

        // Additional metadata from file system
        if ( file_exists( $plugin_file ) ) {
            $metadata['file_size'] = filesize( $plugin_file );
            $metadata['last_modified'] = filemtime( $plugin_file );
        }

        // Extract from readme.txt if available
        if ( file_exists( $plugin_dir . '/readme.txt' ) ) {
            $readme = file_get_contents( $plugin_dir . '/readme.txt' );
            
            // Extract stable tag
            if ( preg_match( '/Stable tag:\s*(.+)$/m', $readme, $stable_tag ) ) {
                $metadata['stable_tag'] = trim( $stable_tag[1] );
            }
            
            // Extract tested up to
            if ( preg_match( '/Tested up to:\s*(.+)$/m', $readme, $tested_up_to ) ) {
                $metadata['tested_up_to'] = trim( $tested_up_to[1] );
            }
            
            // Extract requires PHP
            if ( preg_match( '/Requires PHP:\s*(.+)$/m', $readme, $requires_php ) ) {
                $metadata['requires_php'] = trim( $requires_php[1] );
            }
        }

        // Check for composer.json
        if ( file_exists( $plugin_dir . '/composer.json' ) ) {
            $composer_data = json_decode( file_get_contents( $plugin_dir . '/composer.json' ), true );
            if ( $composer_data && is_array( $composer_data ) ) {
                $metadata['composer_data'] = $composer_data;
                $metadata['has_composer'] = true;
            }
        } else {
            $metadata['has_composer'] = false;
        }

        // Count files and calculate total size
        $metadata['file_count'] = 0;
        $metadata['total_size'] = 0;
        $metadata['file_types'] = array();

        if ( is_dir( $plugin_dir ) ) {
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator( $plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
                RecursiveIteratorIterator::LEAVES_ONLY
            );

            foreach ( $files as $file ) {
                if ( $file->isFile() ) {
                    $metadata['file_count']++;
                    $metadata['total_size'] += $file->getSize();
                    
                    // Track file types
                    $extension = strtolower( $file->getExtension() );
                    if ( ! isset( $metadata['file_types'][$extension] ) ) {
                        $metadata['file_types'][$extension] = 0;
                    }
                    $metadata['file_types'][$extension]++;
                }
            }
        }

        return $metadata;
    }

    /**
     * Detect API usage patterns in the plugin.
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @return   array                 The detected API usage patterns.
     */
    private function detect_api_usage( $plugin_path ) {
        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_path );
        $api_usage = array(
            'rest_api' => false,
            'shortcodes' => array(),
            'widgets' => false,
            'custom_post_types' => array(),
            'custom_taxonomies' => array(),
            'gutenberg_blocks' => false,
            'options_api' => false,
            'transients' => false,
            'cron_jobs' => false,
            'admin_menus' => false,
            'customizer' => false,
            'theme_support' => false,
            'ajax_handlers' => array(),
            'rewrite_rules' => false,
            'meta_boxes' => false,
            'user_roles' => false,
            'post_meta' => false,
            'term_meta' => false,
            'comment_meta' => false
        );

        if ( ! is_dir( $plugin_dir ) ) {
            return $api_usage;
        }

        // Get all PHP files in the plugin directory
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        $files_processed = 0;
        foreach ( $files as $file ) {
            if ( $file->getExtension() !== 'php' ) {
                continue;
            }

            // Check resource limits
            if ( $files_processed > $this->limits['max_files_per_plugin'] ||
                 $file->getSize() > $this->limits['file_size_limit'] ) {
                continue;
            }

            $content = file_get_contents( $file->getPathname() );
            if ( $content === false ) {
                continue;
            }

            // Check for REST API usage
            if ( preg_match( '/register_rest_route\s*\(/', $content ) ) {
                $api_usage['rest_api'] = true;
            }

            // Extract shortcodes
            if ( preg_match_all( '/add_shortcode\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $shortcode_matches ) ) {
                $api_usage['shortcodes'] = array_merge( $api_usage['shortcodes'], $shortcode_matches[1] );
            }

            // Check for widget registration
            if ( preg_match( '/extends\s+WP_Widget/', $content ) || 
                 preg_match( '/register_widget\s*\(/', $content ) ) {
                $api_usage['widgets'] = true;
            }

            // Extract custom post types
            if ( preg_match_all( '/register_post_type\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $cpt_matches ) ) {
                $api_usage['custom_post_types'] = array_merge( $api_usage['custom_post_types'], $cpt_matches[1] );
            }

            // Extract custom taxonomies
            if ( preg_match_all( '/register_taxonomy\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $tax_matches ) ) {
                $api_usage['custom_taxonomies'] = array_merge( $api_usage['custom_taxonomies'], $tax_matches[1] );
            }

            // Check for Gutenberg blocks
            if ( preg_match( '/registerBlockType\s*\(/', $content ) || 
                 preg_match( '/register_block_type\s*\(/', $content ) ) {
                $api_usage['gutenberg_blocks'] = true;
            }

            // Check for options API
            if ( preg_match( '/get_option\s*\(/', $content ) || 
                 preg_match( '/update_option\s*\(/', $content ) ) {
                $api_usage['options_api'] = true;
            }

            // Check for transients
            if ( preg_match( '/set_transient\s*\(/', $content ) || 
                 preg_match( '/get_transient\s*\(/', $content ) ) {
                $api_usage['transients'] = true;
            }

            // Check for cron jobs
            if ( preg_match( '/wp_schedule_event\s*\(/', $content ) || 
                 preg_match( '/wp_schedule_single_event\s*\(/', $content ) ) {
                $api_usage['cron_jobs'] = true;
            }

            // Check for admin menus
            if ( preg_match( '/add_menu_page\s*\(/', $content ) || 
                 preg_match( '/add_submenu_page\s*\(/', $content ) ) {
                $api_usage['admin_menus'] = true;
            }

            // Check for customizer
            if ( preg_match( '/add_setting\s*\(/', $content ) || 
                 preg_match( '/add_control\s*\(/', $content ) ) {
                $api_usage['customizer'] = true;
            }

            // Check for theme support
            if ( preg_match( '/add_theme_support\s*\(/', $content ) ) {
                $api_usage['theme_support'] = true;
            }

            // Extract AJAX handlers
            if ( preg_match_all( '/wp_ajax_([a-zA-Z0-9_]+)/', $content, $ajax_matches ) ) {
                $api_usage['ajax_handlers'] = array_merge( $api_usage['ajax_handlers'], $ajax_matches[1] );
            }

            // Check for rewrite rules
            if ( preg_match( '/add_rewrite_rule\s*\(/', $content ) || 
                 preg_match( '/flush_rewrite_rules\s*\(/', $content ) ) {
                $api_usage['rewrite_rules'] = true;
            }

            // Check for meta boxes
            if ( preg_match( '/add_meta_box\s*\(/', $content ) ) {
                $api_usage['meta_boxes'] = true;
            }

            // Check for user roles
            if ( preg_match( '/add_role\s*\(/', $content ) || 
                 preg_match( '/add_cap\s*\(/', $content ) ) {
                $api_usage['user_roles'] = true;
            }

            // Check for post meta
            if ( preg_match( '/add_post_meta\s*\(/', $content ) || 
                 preg_match( '/update_post_meta\s*\(/', $content ) ) {
                $api_usage['post_meta'] = true;
            }

            // Check for term meta
            if ( preg_match( '/add_term_meta\s*\(/', $content ) || 
                 preg_match( '/update_term_meta\s*\(/', $content ) ) {
                $api_usage['term_meta'] = true;
            }

            // Check for comment meta
            if ( preg_match( '/add_comment_meta\s*\(/', $content ) || 
                 preg_match( '/update_comment_meta\s*\(/', $content ) ) {
                $api_usage['comment_meta'] = true;
            }

            $files_processed++;
        }

        // Remove duplicates from arrays
        $api_usage['shortcodes'] = array_unique( $api_usage['shortcodes'] );
        $api_usage['custom_post_types'] = array_unique( $api_usage['custom_post_types'] );
        $api_usage['custom_taxonomies'] = array_unique( $api_usage['custom_taxonomies'] );
        $api_usage['ajax_handlers'] = array_unique( $api_usage['ajax_handlers'] );

        return $api_usage;
    }

    /**
     * Analyze plugin dependencies.
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @return   array                 The detected dependencies.
     */
    private function analyze_dependencies( $plugin_path ) {
        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_path );
        $dependencies = array(
            'wordpress' => array(),
            'php' => array(),
            'plugins' => array(),
            'themes' => array(),
            'composer' => array(),
            'detected' => array()
        );

        // Check composer.json for dependencies
        if ( file_exists( $plugin_dir . '/composer.json' ) ) {
            $composer_data = json_decode( file_get_contents( $plugin_dir . '/composer.json' ), true );
            if ( $composer_data && isset( $composer_data['require'] ) ) {
                foreach ( $composer_data['require'] as $package => $version ) {
                    if ( strpos( $package, 'wpackagist-plugin/' ) === 0 ) {
                        $plugin_slug = str_replace( 'wpackagist-plugin/', '', $package );
                        $dependencies['plugins'][$plugin_slug] = $version;
                    } elseif ( strpos( $package, 'wpackagist-theme/' ) === 0 ) {
                        $theme_slug = str_replace( 'wpackagist-theme/', '', $package );
                        $dependencies['themes'][$theme_slug] = $version;
                    } else {
                        $dependencies['composer'][$package] = $version;
                    }
                }
            }
        }

        // Analyze PHP files for plugin dependencies
        if ( is_dir( $plugin_dir ) ) {
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator( $plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
                RecursiveIteratorIterator::LEAVES_ONLY
            );

            $files_processed = 0;
            foreach ( $files as $file ) {
                if ( $file->getExtension() !== 'php' ) {
                    continue;
                }

                // Check resource limits
                if ( $files_processed > $this->limits['max_files_per_plugin'] ||
                     $file->getSize() > $this->limits['file_size_limit'] ) {
                    continue;
                }

                $content = file_get_contents( $file->getPathname() );
                if ( $content === false ) {
                    continue;
                }

                // Check for WooCommerce dependency
                if ( preg_match( '/class_exists\s*\(\s*[\'"]WooCommerce[\'"]/', $content ) ||
                     preg_match( '/function_exists\s*\(\s*[\'"]wc_/', $content ) ||
                     preg_match( '/WC\s*::\s*/', $content ) ) {
                    $dependencies['detected']['woocommerce'] = 'WooCommerce';
                }

                // Check for ACF dependency
                if ( preg_match( '/class_exists\s*\(\s*[\'"]ACF[\'"]/', $content ) ||
                     preg_match( '/function_exists\s*\(\s*[\'"]get_field[\'"]/', $content ) ) {
                    $dependencies['detected']['advanced-custom-fields'] = 'Advanced Custom Fields';
                }

                // Check for Yoast SEO dependency
                if ( preg_match( '/class_exists\s*\(\s*[\'"]WPSEO_/', $content ) ||
                     preg_match( '/function_exists\s*\(\s*[\'"]wpseo_/', $content ) ) {
                    $dependencies['detected']['wordpress-seo'] = 'Yoast SEO';
                }

                // Check for Elementor dependency
                if ( preg_match( '/class_exists\s*\(\s*[\'"]\\\\Elementor\\\\/', $content ) ||
                     preg_match( '/did_action\s*\(\s*[\'"]elementor\/loaded[\'"]/', $content ) ) {
                    $dependencies['detected']['elementor'] = 'Elementor';
                }

                // Check for Contact Form 7 dependency
                if ( preg_match( '/class_exists\s*\(\s*[\'"]WPCF7_/', $content ) ||
                     preg_match( '/function_exists\s*\(\s*[\'"]wpcf7_/', $content ) ) {
                    $dependencies['detected']['contact-form-7'] = 'Contact Form 7';
                }

                // Check for Gravity Forms dependency
                if ( preg_match( '/class_exists\s*\(\s*[\'"]GFForms[\'"]/', $content ) ||
                     preg_match( '/function_exists\s*\(\s*[\'"]gf_/', $content ) ) {
                    $dependencies['detected']['gravityforms'] = 'Gravity Forms';
                }

                $files_processed++;
            }
        }

        return $dependencies;
    }

    /**
     * Analyze plugin hooks (actions and filters).
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @return   array                 The detected hooks.
     */
    private function analyze_plugin_hooks( $plugin_path ) {
        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_path );
        $hooks = array(
            'actions' => array(),
            'filters' => array(),
            'custom_actions' => array(),
            'custom_filters' => array()
        );

        if ( ! is_dir( $plugin_dir ) ) {
            return $hooks;
        }

        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        $files_processed = 0;
        foreach ( $files as $file ) {
            if ( $file->getExtension() !== 'php' ) {
                continue;
            }

            // Check resource limits
            if ( $files_processed > $this->limits['max_files_per_plugin'] ||
                 $file->getSize() > $this->limits['file_size_limit'] ) {
                continue;
            }

            $content = file_get_contents( $file->getPathname() );
            if ( $content === false ) {
                continue;
            }

            // Extract add_action calls
            if ( preg_match_all( '/add_action\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $action_matches ) ) {
                foreach ( $action_matches[1] as $hook ) {
                    if ( ! isset( $hooks['actions'][$hook] ) ) {
                        $hooks['actions'][$hook] = 0;
                    }
                    $hooks['actions'][$hook]++;
                }
            }

            // Extract add_filter calls
            if ( preg_match_all( '/add_filter\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $filter_matches ) ) {
                foreach ( $filter_matches[1] as $hook ) {
                    if ( ! isset( $hooks['filters'][$hook] ) ) {
                        $hooks['filters'][$hook] = 0;
                    }
                    $hooks['filters'][$hook]++;
                }
            }

            // Extract do_action calls (custom actions)
            if ( preg_match_all( '/do_action\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $custom_action_matches ) ) {
                foreach ( $custom_action_matches[1] as $hook ) {
                    if ( ! isset( $hooks['custom_actions'][$hook] ) ) {
                        $hooks['custom_actions'][$hook] = 0;
                    }
                    $hooks['custom_actions'][$hook]++;
                }
            }

            // Extract apply_filters calls (custom filters)
            if ( preg_match_all( '/apply_filters\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $custom_filter_matches ) ) {
                foreach ( $custom_filter_matches[1] as $hook ) {
                    if ( ! isset( $hooks['custom_filters'][$hook] ) ) {
                        $hooks['custom_filters'][$hook] = 0;
                    }
                    $hooks['custom_filters'][$hook]++;
                }
            }

            $files_processed++;
        }

        return $hooks;
    }

    /**
     * Detect custom database tables created by the plugin.
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @return   array                 The detected custom tables.
     */
    private function detect_custom_tables( $plugin_path ) {
        global $wpdb;

        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_path );
        $plugin_name = basename( dirname( $plugin_path ) );
        $potential_tables = array();
        $confirmed_tables = array();

        // Generate possible table prefixes
        $possible_prefixes = array(
            $plugin_name,
            str_replace( '-', '_', $plugin_name ),
            substr( str_replace( '-', '_', $plugin_name ), 0, 8 )
        );

        if ( ! is_dir( $plugin_dir ) ) {
            return $confirmed_tables;
        }

        // Scan plugin files for table references
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        $files_processed = 0;
        foreach ( $files as $file ) {
            if ( $file->getExtension() !== 'php' ) {
                continue;
            }

            // Check resource limits
            if ( $files_processed > $this->limits['max_files_per_plugin'] ||
                 $file->getSize() > $this->limits['file_size_limit'] ) {
                continue;
            }

            $content = file_get_contents( $file->getPathname() );
            if ( $content === false ) {
                continue;
            }

            // Look for dbDelta calls
            if ( preg_match( '/dbDelta\s*\(/', $content ) ) {
                if ( preg_match_all( '/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?(?:`|\')?([^`\'\s]+)(?:`|\')?/i', $content, $matches ) ) {
                    foreach ( $matches[1] as $table ) {
                        $table = trim( $table );
                        // Remove wpdb prefix if present
                        $table = str_replace( array( '$wpdb->prefix . ', '$wpdb->prefix.' ), '', $table );
                        $table = str_replace( array( '\'', '"', '`' ), '', $table );
                        $potential_tables[] = $table;
                    }
                }
            }

            // Look for direct table references
            foreach ( $possible_prefixes as $prefix ) {
                if ( preg_match_all( '/\$wpdb->(?:prefix\s*\.\s*|prefix\.)?[\'"](' . preg_quote( $prefix, '/' ) . '.*?)[\'"]/', $content, $matches ) ) {
                    $potential_tables = array_merge( $potential_tables, $matches[1] );
                }
            }

            $files_processed++;
        }

        // Remove duplicates
        $potential_tables = array_unique( $potential_tables );

        // Check if tables actually exist in the database
        if ( ! empty( $potential_tables ) ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific table existence check for scan
            $existing_tables = $wpdb->get_results( "SHOW TABLES", ARRAY_N );
            $existing_tables = array_map( 'current', $existing_tables );

            foreach ( $potential_tables as $table ) {
                $full_table_name = $wpdb->prefix . $table;
                if ( in_array( $full_table_name, $existing_tables ) ) {
                    $confirmed_tables[] = $full_table_name;
                }
            }
        }

        return $confirmed_tables;
    }

    /**
     * Calculate resource usage of the plugin.
     *
     * @since    1.0.0
     * @param    string  $plugin_path  The plugin path.
     * @return   array                 The resource usage statistics.
     */
    private function calculate_resource_usage( $plugin_path ) {
        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_path );
        $usage = array(
            'disk_space' => 0,
            'file_count' => 0,
            'php_files' => 0,
            'js_files' => 0,
            'css_files' => 0,
            'image_files' => 0,
            'largest_file' => 0,
            'memory_estimate' => 0,
            'complexity_score' => 0
        );

        if ( ! is_dir( $plugin_dir ) ) {
            return $usage;
        }

        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        $php_lines = 0;
        foreach ( $files as $file ) {
            if ( ! $file->isFile() ) {
                continue;
            }

            $size = $file->getSize();
            $usage['disk_space'] += $size;
            $usage['file_count']++;

            if ( $size > $usage['largest_file'] ) {
                $usage['largest_file'] = $size;
            }

            $extension = strtolower( $file->getExtension() );
            switch ( $extension ) {
                case 'php':
                    $usage['php_files']++;
                    // Count lines of code for complexity
                    if ( $size < $this->limits['file_size_limit'] ) {
                        $content = file_get_contents( $file->getPathname() );
                        if ( $content !== false ) {
                            $php_lines += substr_count( $content, "\n" );
                        }
                    }
                    break;
                case 'js':
                    $usage['js_files']++;
                    break;
                case 'css':
                    $usage['css_files']++;
                    break;
                case 'jpg':
                case 'jpeg':
                case 'png':
                case 'gif':
                case 'svg':
                    $usage['image_files']++;
                    break;
            }
        }

        // Estimate memory usage (rough calculation)
        $usage['memory_estimate'] = $usage['disk_space'] * 0.3; // Assume 30% of disk space is loaded into memory

        // Calculate complexity score based on various factors
        $complexity_factors = array(
            'file_count' => min( $usage['file_count'] / 10, 3 ),           // Max 3 points
            'php_lines' => min( $php_lines / 1000, 4 ),                    // Max 4 points
            'php_files' => min( $usage['php_files'] / 5, 2 ),              // Max 2 points
            'size' => min( $usage['disk_space'] / (1024 * 1024), 1 )       // Max 1 point (per MB)
        );

        $usage['complexity_score'] = array_sum( $complexity_factors );

        return $usage;
    }

    /**
     * Check if we're approaching resource limits.
     *
     * @since    1.0.0
     * @param    float  $start_time  The start time of the operation.
     * @return   bool                Whether we're approaching limits.
     */
    private function check_resource_limits( $start_time ) {
        $current_memory = memory_get_usage();
        $current_time = microtime( true ) - $start_time;

        return ( $current_memory > $this->limits['memory_threshold'] ||
                 $current_time > $this->limits['time_threshold'] );
    }

    /**
     * Get scan statistics for reporting.
     *
     * @since    1.0.0
     * @return   array  The scan statistics.
     */
    public function get_scan_statistics() {
        // Get scan results from database
        $results = $this->db->get_scan_results();
        $stats = array(
            'total_plugins' => 0,
            'active_plugins' => 0,
            'inactive_plugins' => 0,
            'total_size' => 0,
            'average_complexity' => 0,
            'api_usage_summary' => array(),
            'dependency_summary' => array(),
            'most_complex_plugins' => array(),
            'newest_scans' => array()
        );

        foreach ( $results as $result ) {
            $data = json_decode( $result['scan_data'], true );
            if ( ! $data ) {
                continue;
            }

            $stats['total_plugins']++;
            $stats['total_size'] += $data['resource_usage']['disk_space'] ?? 0;
            $stats['average_complexity'] += $data['resource_usage']['complexity_score'] ?? 0;

            if ( $data['metadata']['is_active'] ?? false ) {
                $stats['active_plugins']++;
            } else {
                $stats['inactive_plugins']++;
            }

            // Collect API usage patterns
            if ( isset( $data['api_usage'] ) ) {
                foreach ( $data['api_usage'] as $api => $usage ) {
                    if ( $usage === true || ( is_array( $usage ) && ! empty( $usage ) ) ) {
                        if ( ! isset( $stats['api_usage_summary'][$api] ) ) {
                            $stats['api_usage_summary'][$api] = 0;
                        }
                        $stats['api_usage_summary'][$api]++;
                    }
                }
            }

            // Collect dependency information
            if ( isset( $data['dependencies']['detected'] ) ) {
                foreach ( $data['dependencies']['detected'] as $plugin => $name ) {
                    if ( ! isset( $stats['dependency_summary'][$plugin] ) ) {
                        $stats['dependency_summary'][$plugin] = 0;
                    }
                    $stats['dependency_summary'][$plugin]++;
                }
            }

            // Track most complex plugins
            $complexity = $data['resource_usage']['complexity_score'] ?? 0;
            $stats['most_complex_plugins'][] = array(
                'plugin' => $result['plugin_slug'],
                'name' => $data['metadata']['name'] ?? $result['plugin_slug'],
                'complexity' => $complexity,
                'size' => $data['resource_usage']['disk_space'] ?? 0
            );
        }

        // Calculate averages
        if ( $stats['total_plugins'] > 0 ) {
            $stats['average_complexity'] = $stats['average_complexity'] / $stats['total_plugins'];
        }

        // Sort and limit most complex plugins
        usort( $stats['most_complex_plugins'], function( $a, $b ) {
            return $b['complexity'] <=> $a['complexity'];
        } );
        $stats['most_complex_plugins'] = array_slice( $stats['most_complex_plugins'], 0, 10 );

        return $stats;
    }

} 