<?php

/**
 * The Safe Mode System functionality of the plugin.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * The Safe Mode System class.
 *
 * This class manages safe mode operation when critical plugin conflicts are detected,
 * providing graceful degradation with minimal functionality and rollback capabilities.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Safe_Mode {

    /**
     * Safe mode status constants
     */
    const MODE_NORMAL = 'normal';
    const MODE_SAFE = 'safe';
    const MODE_EMERGENCY = 'emergency';

    /**
     * Safe mode trigger constants
     */
    const TRIGGER_CRITICAL_CONFLICT = 'critical_conflict';
    const TRIGGER_USER_ACTIVATED = 'user_activated';
    const TRIGGER_PLUGIN_ERROR = 'plugin_error';
    const TRIGGER_EMERGENCY = 'emergency';

    /**
     * Plugin state constants
     */
    const STATE_ACTIVE = 'active';
    const STATE_INACTIVE = 'inactive';
    const STATE_QUARANTINED = 'quarantined';
    const STATE_DISABLED = 'disabled';

    /**
     * Option names for safe mode data
     */
    const SAFE_MODE_OPTION = 'unplug_safe_mode_status';
    const BACKUP_STATES_OPTION = 'unplug_plugin_backup_states';
    const SAFE_MODE_CONFIG_OPTION = 'unplug_safe_mode_config';
    const SAFE_MODE_LOG_OPTION = 'unplug_safe_mode_log';

    /**
     * Maximum backup states to store
     */
    const MAX_BACKUP_STATES = 10;

    /**
     * Safe mode log retention period in days
     */
    const LOG_RETENTION_DAYS = 30;

    /**
     * Essential plugins that should never be disabled
     */
    private static $essential_plugins = array(
    //'unplug/unplug.php', // Remove this line to allow deactivation
    'wordpress-importer/wordpress-importer.php',
    'akismet/akismet.php'
);

    /**
     * Get current safe mode status
     *
     * @return array Safe mode status information
     */
    public static function get_safe_mode_status() {
        $status = get_option( self::SAFE_MODE_OPTION, array(
            'mode' => self::MODE_NORMAL,
            'trigger' => null,
            'activated_time' => null,
            'activated_by' => null,
            'quarantined_plugins' => array(),
            'disabled_plugins' => array(),
            'backup_id' => null,
            'can_restore' => false,
            'restrictions' => array()
        ) );

        // Ensure all required fields are present
        $status = array_merge( array(
            'mode' => self::MODE_NORMAL,
            'trigger' => null,
            'activated_time' => null,
            'activated_by' => null,
            'quarantined_plugins' => array(),
            'disabled_plugins' => array(),
            'backup_id' => null,
            'can_restore' => false,
            'restrictions' => array()
        ), $status );

        return $status;
    }

    /**
     * Check if safe mode is active
     *
     * @return bool True if safe mode is active
     */
    public static function is_safe_mode_active() {
        $status = self::get_safe_mode_status();
        return $status['mode'] !== self::MODE_NORMAL;
    }

    /**
     * Check if emergency mode is active
     *
     * @return bool True if emergency mode is active
     */
    public static function is_emergency_mode_active() {
        $status = self::get_safe_mode_status();
        return $status['mode'] === self::MODE_EMERGENCY;
    }

    /**
     * Activate safe mode
     *
     * @param string $trigger Trigger reason for safe mode activation
     * @param array $conflicting_plugins Array of conflicting plugin paths
     * @param array $options Additional options for safe mode
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public static function activate_safe_mode( $trigger, $conflicting_plugins = array(), $options = array() ) {
        // Verify user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return new WP_Error( 'insufficient_permissions', 'Insufficient permissions to activate safe mode' );
        }

        // Check if safe mode is already active
        if ( self::is_safe_mode_active() ) {
            return new WP_Error( 'safe_mode_active', 'Safe mode is already active' );
        }

        // Create backup of current plugin states
        $backup_id = self::create_plugin_state_backup();
        if ( is_wp_error( $backup_id ) ) {
            return $backup_id;
        }

        // Determine safe mode type
        $mode = self::MODE_SAFE;
        if ( $trigger === self::TRIGGER_EMERGENCY ) {
            $mode = self::MODE_EMERGENCY;
        }

        // Quarantine conflicting plugins
        $quarantined_plugins = array();
        $disabled_plugins = array();

        if ( ! empty( $conflicting_plugins ) ) {
            foreach ( $conflicting_plugins as $plugin_path ) {
                if ( self::is_plugin_essential( $plugin_path ) ) {
                    continue; // Skip essential plugins
                }

                if ( is_plugin_active( $plugin_path ) ) {
                    // Quarantine the plugin
                    $result = self::quarantine_plugin( $plugin_path );
                    if ( ! is_wp_error( $result ) ) {
                        $quarantined_plugins[] = $plugin_path;
                    }
                } else {
                    // Disable inactive plugins from being activated
                    $disabled_plugins[] = $plugin_path;
                }
            }
        }

        // Apply safe mode restrictions
        $restrictions = self::get_safe_mode_restrictions( $mode, $options );

        // Update safe mode status
        $status = array(
            'mode' => $mode,
            'trigger' => $trigger,
            'activated_time' => current_time( 'timestamp' ),
            'activated_by' => get_current_user_id(),
            'quarantined_plugins' => $quarantined_plugins,
            'disabled_plugins' => $disabled_plugins,
            'backup_id' => $backup_id,
            'can_restore' => true,
            'restrictions' => $restrictions
        );

        $updated = update_option( self::SAFE_MODE_OPTION, $status );

        if ( ! $updated ) {
            return new WP_Error( 'update_failed', 'Failed to update safe mode status' );
        }

        // Log safe mode activation
        self::log_safe_mode_event( 'activated', $trigger, array(
            'quarantined_plugins' => $quarantined_plugins,
            'disabled_plugins' => $disabled_plugins,
            'backup_id' => $backup_id
        ) );

        // Trigger safe mode activation hooks
        do_action( 'unplug_safe_mode_activated', $status );

        return true;
    }

    /**
     * Deactivate safe mode and restore normal operation
     *
     * @param bool $restore_plugins Whether to restore quarantined plugins
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public static function deactivate_safe_mode( $restore_plugins = false ) {
        // Verify user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return new WP_Error( 'insufficient_permissions', 'Insufficient permissions to deactivate safe mode' );
        }

        // Check if safe mode is active
        if ( ! self::is_safe_mode_active() ) {
            return new WP_Error( 'safe_mode_inactive', 'Safe mode is not active' );
        }

        $current_status = self::get_safe_mode_status();

        // Restore plugins if requested
        if ( $restore_plugins && ! empty( $current_status['quarantined_plugins'] ) ) {
            foreach ( $current_status['quarantined_plugins'] as $plugin_path ) {
                self::restore_plugin( $plugin_path );
            }
        }

        // Reset safe mode status
        $status = array(
            'mode' => self::MODE_NORMAL,
            'trigger' => null,
            'activated_time' => null,
            'activated_by' => null,
            'quarantined_plugins' => array(),
            'disabled_plugins' => array(),
            'backup_id' => null,
            'can_restore' => false,
            'restrictions' => array()
        );

        $updated = update_option( self::SAFE_MODE_OPTION, $status );

        if ( ! $updated ) {
            return new WP_Error( 'update_failed', 'Failed to update safe mode status' );
        }

        // Log safe mode deactivation
        self::log_safe_mode_event( 'deactivated', $current_status['trigger'], array(
            'restored_plugins' => $restore_plugins ? $current_status['quarantined_plugins'] : array(),
            'backup_id' => $current_status['backup_id']
        ) );

        // Trigger safe mode deactivation hooks
        do_action( 'unplug_safe_mode_deactivated', $current_status );

        return true;
    }

    /**
     * Create backup of current plugin states
     *
     * @return string|WP_Error Backup ID on success, WP_Error on failure
     */
    private static function create_plugin_state_backup() {
        $backup_id = 'backup_' . time() . '_' . wp_generate_uuid4();
        
        // Get current plugin states
        $all_plugins = get_plugins();
        $active_plugins = get_option( 'active_plugins', array() );
        $network_active_plugins = is_multisite() ? get_site_option( 'active_sitewide_plugins', array() ) : array();

        $backup_data = array(
            'id' => $backup_id,
            'created' => current_time( 'timestamp' ),
            'created_by' => get_current_user_id(),
            'all_plugins' => $all_plugins,
            'active_plugins' => $active_plugins,
            'network_active_plugins' => $network_active_plugins,
            'description' => 'Pre-safe mode backup'
        );

        // Get existing backups
        $backups = get_option( self::BACKUP_STATES_OPTION, array() );

        // Add new backup
        $backups[$backup_id] = $backup_data;

        // Maintain maximum backup limit
        if ( count( $backups ) > self::MAX_BACKUP_STATES ) {
            // Remove oldest backups
            uasort( $backups, function( $a, $b ) {
                return $a['created'] - $b['created'];
            } );
            $backups = array_slice( $backups, -self::MAX_BACKUP_STATES, null, true );
        }

        // Save backups
        $updated = update_option( self::BACKUP_STATES_OPTION, $backups );

        if ( ! $updated ) {
            return new WP_Error( 'backup_failed', 'Failed to create plugin state backup' );
        }

        return $backup_id;
    }

    /**
     * Restore plugin states from backup
     *
     * @param string $backup_id Backup ID to restore
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public static function restore_from_backup( $backup_id ) {
        // Verify user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return new WP_Error( 'insufficient_permissions', 'Insufficient permissions to restore from backup' );
        }

        // Get backup data
        $backups = get_option( self::BACKUP_STATES_OPTION, array() );
        
        if ( ! isset( $backups[$backup_id] ) ) {
            return new WP_Error( 'backup_not_found', 'Backup not found' );
        }

        $backup_data = $backups[$backup_id];

        // Log backup restoration request (WordPress guidelines prevent automatic plugin state changes)
        if ( isset( $backup_data['active_plugins'] ) ) {
            self::log_safe_mode_event( 'backup_restoration_requested', 'user_action', array(
                'backup_id' => $backup_id,
                'backup_created' => $backup_data['created'],
                'plugins_to_restore' => $backup_data['active_plugins'],
                'action_required' => 'Please manually activate/deactivate plugins to match the backup state'
            ) );
        }

        // Log network plugin restoration request (if multisite)
        if ( is_multisite() && isset( $backup_data['network_active_plugins'] ) ) {
            self::log_safe_mode_event( 'network_backup_restoration_requested', 'user_action', array(
                'backup_id' => $backup_id,
                'backup_created' => $backup_data['created'],
                'network_plugins_to_restore' => $backup_data['network_active_plugins'],
                'action_required' => 'Please manually activate/deactivate network plugins to match the backup state'
            ) );
        }

        // Log restoration
        self::log_safe_mode_event( 'backup_restored', 'user_action', array(
            'backup_id' => $backup_id,
            'backup_created' => $backup_data['created']
        ) );

        return true;
    }

    /**
     * Log problematic plugin for manual review (WordPress guidelines prevent automatic deactivation)
     *
     * @param string $plugin_path Plugin path
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    private static function quarantine_plugin( $plugin_path ) {
        if ( ! is_plugin_active( $plugin_path ) ) {
            return true; // Already inactive
        }

        // Log the problematic plugin for manual review
        // WordPress guidelines prevent plugins from automatically deactivating other plugins
        self::log_safe_mode_event( 'plugin_flagged_for_review', 'automatic', array(
            'plugin_path' => $plugin_path,
            'reason' => 'Plugin flagged as problematic and requires manual review',
            'action_required' => 'Please manually deactivate this plugin if needed'
        ) );

        // Return success but note that no automatic action was taken
        return true;
    }

    /**
     * Log plugin restoration request for manual review (WordPress guidelines prevent automatic activation)
     *
     * @param string $plugin_path Plugin path
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    private static function restore_plugin( $plugin_path ) {
        if ( is_plugin_active( $plugin_path ) ) {
            return true; // Already active
        }

        // Log the plugin restoration request for manual review
        // WordPress guidelines prevent plugins from automatically activating other plugins
        self::log_safe_mode_event( 'plugin_restoration_requested', 'automatic', array(
            'plugin_path' => $plugin_path,
            'reason' => 'Plugin restoration requested but requires manual activation',
            'action_required' => 'Please manually activate this plugin if desired'
        ) );

        // Return success but note that no automatic action was taken
        return true;
    }

    /**
     * Check if a plugin is essential and should not be disabled
     *
     * @param string $plugin_path Plugin path
     * @return bool True if plugin is essential
     */
    private static function is_plugin_essential( $plugin_path ) {
        $essential_plugins = apply_filters( 'unplug_essential_plugins', self::$essential_plugins );
        return in_array( $plugin_path, $essential_plugins );
    }

    /**
     * Get safe mode restrictions based on mode and options
     *
     * @param string $mode Safe mode type
     * @param array $options Additional options
     * @return array Restrictions array
     */
    private static function get_safe_mode_restrictions( $mode, $options = array() ) {
        $restrictions = array();

        if ( $mode === self::MODE_SAFE ) {
            $restrictions = array(
                'disable_plugin_activation' => true,
                'disable_plugin_installation' => true,
                'disable_theme_changes' => false,
                'disable_updates' => false,
                'show_admin_notices' => true,
                'limit_admin_access' => false
            );
        } elseif ( $mode === self::MODE_EMERGENCY ) {
            $restrictions = array(
                'disable_plugin_activation' => true,
                'disable_plugin_installation' => true,
                'disable_theme_changes' => true,
                'disable_updates' => true,
                'show_admin_notices' => true,
                'limit_admin_access' => true
            );
        }

        // Apply custom restrictions from options
        if ( ! empty( $options['restrictions'] ) ) {
            $restrictions = array_merge( $restrictions, $options['restrictions'] );
        }

        return apply_filters( 'unplug_safe_mode_restrictions', $restrictions, $mode, $options );
    }

    /**
     * Get list of all backup states
     *
     * @return array Array of backup states
     */
    public static function get_backup_states() {
        $backups = get_option( self::BACKUP_STATES_OPTION, array() );
        
        // Sort by creation time (newest first)
        uasort( $backups, function( $a, $b ) {
            return $b['created'] - $a['created'];
        } );

        return $backups;
    }

    /**
     * Delete a backup state
     *
     * @param string $backup_id Backup ID to delete
     * @return bool True on success, false on failure
     */
    public static function delete_backup_state( $backup_id ) {
        // Verify user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return false;
        }

        $backups = get_option( self::BACKUP_STATES_OPTION, array() );
        
        if ( ! isset( $backups[$backup_id] ) ) {
            return false;
        }

        unset( $backups[$backup_id] );
        
        return update_option( self::BACKUP_STATES_OPTION, $backups );
    }

    /**
     * Get safe mode configuration
     *
     * @return array Safe mode configuration
     */
    public static function get_safe_mode_config() {
        return get_option( self::SAFE_MODE_CONFIG_OPTION, array(
            'auto_activate_on_critical' => true,
            'auto_activate_on_plugin_error' => true,
            'max_quarantine_duration' => 7 * DAY_IN_SECONDS, // 7 days
            'notification_email' => get_option( 'admin_email' ),
            'emergency_mode_triggers' => array(
                'multiple_critical_conflicts',
                'site_unavailable',
                'database_errors'
            )
        ) );
    }

    /**
     * Update safe mode configuration
     *
     * @param array $config Configuration array
     * @return bool True on success, false on failure
     */
    public static function update_safe_mode_config( $config ) {
        // Verify user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return false;
        }

        $current_config = self::get_safe_mode_config();
        $updated_config = array_merge( $current_config, $config );

        return update_option( self::SAFE_MODE_CONFIG_OPTION, $updated_config );
    }

    /**
     * Log safe mode event
     *
     * @param string $event Event type
     * @param string $trigger Trigger reason
     * @param array $data Additional event data
     */
    private static function log_safe_mode_event( $event, $trigger, $data = array() ) {
        $log_entry = array(
            'event' => $event,
            'trigger' => $trigger,
            'timestamp' => current_time( 'timestamp' ),
            'user_id' => get_current_user_id(),
            'data' => $data
        );

        $log = get_option( self::SAFE_MODE_LOG_OPTION, array() );
        $log[] = $log_entry;

        // Maintain log size (keep last 100 entries)
        if ( count( $log ) > 100 ) {
            $log = array_slice( $log, -100 );
        }

        update_option( self::SAFE_MODE_LOG_OPTION, $log );

        // Also log to security system if available
        if ( class_exists( 'UNPLUG_Security' ) ) {
            UNPLUG_Security::log_security_event( 'safe_mode_' . $event, $trigger, $data );
        }
    }

    /**
     * Get safe mode log
     *
     * @param int $limit Number of entries to return
     * @return array Log entries
     */
    public static function get_safe_mode_log( $limit = 50 ) {
        $log = get_option( self::SAFE_MODE_LOG_OPTION, array() );
        
        // Sort by timestamp (newest first)
        usort( $log, function( $a, $b ) {
            return $b['timestamp'] - $a['timestamp'];
        } );

        return array_slice( $log, 0, $limit );
    }

    /**
     * Check if automatic safe mode activation should occur
     *
     * @param array $conflict_analysis Conflict analysis results
     * @return bool True if safe mode should be activated
     */
    public static function should_auto_activate_safe_mode( $conflict_analysis ) {
        if ( self::is_safe_mode_active() ) {
            return false; // Already active
        }

        $config = self::get_safe_mode_config();

        // Check for critical conflicts
        if ( $config['auto_activate_on_critical'] ) {
            $critical_conflicts = $conflict_analysis['summary']['critical_conflicts'] ?? 0;
            if ( $critical_conflicts > 0 ) {
                return true;
            }
        }

        // Check for multiple high-severity conflicts
        $high_conflicts = $conflict_analysis['summary']['high_conflicts'] ?? 0;
        if ( $high_conflicts >= 3 ) {
            return true;
        }

        return false;
    }

    /**
     * Extract conflicting plugins from conflict analysis
     *
     * @param array $conflict_analysis Conflict analysis results
     * @return array Array of conflicting plugin paths
     */
    public static function extract_conflicting_plugins( $conflict_analysis ) {
        $conflicting_plugins = array();

        if ( empty( $conflict_analysis['by_plugin'] ) ) {
            return $conflicting_plugins;
        }

        foreach ( $conflict_analysis['by_plugin'] as $plugin_path => $plugin_data ) {
            if ( $plugin_data['max_severity'] >= 9 ) { // Critical or high severity
                $conflicting_plugins[] = $plugin_path;
            }
        }

        return $conflicting_plugins;
    }

    /**
     * Get safe mode dashboard data
     *
     * @return array Dashboard data
     */
    public static function get_dashboard_data() {
        $status = self::get_safe_mode_status();
        $config = self::get_safe_mode_config();
        $backups = self::get_backup_states();
        $log = self::get_safe_mode_log( 10 );

        return array(
            'status' => $status,
            'config' => $config,
            'backups' => array_slice( $backups, 0, 5 ), // Latest 5 backups
            'recent_log' => $log,
            'is_active' => self::is_safe_mode_active(),
            'is_emergency' => self::is_emergency_mode_active()
        );
    }

    /**
     * Initialize safe mode hooks and filters
     */
    public static function init() {
        // Add hooks for safe mode restrictions
        if ( self::is_safe_mode_active() ) {
            $status = self::get_safe_mode_status();
            $restrictions = $status['restrictions'];

            // Prevent plugin activation if restricted
            if ( ! empty( $restrictions['disable_plugin_activation'] ) ) {
                add_filter( 'plugin_action_links', array( __CLASS__, 'filter_plugin_action_links' ), 10, 2 );
                add_action( 'admin_init', array( __CLASS__, 'prevent_plugin_activation' ) );
            }

            // Prevent plugin installation if restricted
            if ( ! empty( $restrictions['disable_plugin_installation'] ) ) {
                add_action( 'admin_init', array( __CLASS__, 'prevent_plugin_installation' ) );
            }

            // Show admin notices if enabled
            if ( ! empty( $restrictions['show_admin_notices'] ) ) {
                add_action( 'admin_notices', array( __CLASS__, 'show_safe_mode_notices' ) );
            }
        }

        // Add cleanup hooks
        add_action( 'unplug_daily_cleanup', array( __CLASS__, 'cleanup_old_backups' ) );
    }

    /**
     * Filter plugin action links in safe mode
     *
     * @param array $actions Plugin action links
     * @param string $plugin_file Plugin file path
     * @return array Modified action links
     */
    public static function filter_plugin_action_links( $actions, $plugin_file ) {
        $status = self::get_safe_mode_status();
        
        // Remove activation link for quarantined plugins
        if ( in_array( $plugin_file, $status['quarantined_plugins'] ) ) {
            unset( $actions['activate'] );
            $actions['quarantined'] = '<span style="color: #d63638;">Quarantined</span>';
        }

        // Remove activation link for disabled plugins
        if ( in_array( $plugin_file, $status['disabled_plugins'] ) ) {
            unset( $actions['activate'] );
            $actions['disabled'] = '<span style="color: #d63638;">Disabled by Safe Mode</span>';
        }

        return $actions;
    }

    /**
     * Prevent plugin activation in safe mode
     */
    public static function prevent_plugin_activation() {
        if ( isset( $_GET['action'] ) && $_GET['action'] === 'activate' && isset( $_GET['plugin'] ) ) {
            // Verify nonce for security
            $plugin = sanitize_text_field( wp_unslash( $_GET['plugin'] ) );
            if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'activate-plugin_' . $plugin ) ) {
                wp_die( 'Security check failed.', 'Safe Mode Active', array( 'back_link' => true ) );
            }
            $status = self::get_safe_mode_status();

            if ( in_array( $plugin, $status['quarantined_plugins'] ) || in_array( $plugin, $status['disabled_plugins'] ) ) {
                wp_die( 
                    'Plugin activation is prevented in safe mode. Please resolve conflicts first.',
                    'Safe Mode Active',
                    array( 'back_link' => true )
                );
            }
        }
    }

    /**
     * Prevent plugin installation in safe mode
     */
    public static function prevent_plugin_installation() {
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'plugin-install' ) {
            // Verify nonce for security
            if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'install-plugin' ) ) {
                wp_die( 'Security check failed.', 'Safe Mode Active', array( 'back_link' => true ) );
            }
            
            $restrictions = self::get_safe_mode_status()['restrictions'];
            if ( ! empty( $restrictions['disable_plugin_installation'] ) ) {
                wp_die( 
                    'Plugin installation is disabled in safe mode. Please resolve conflicts first.',
                    'Safe Mode Active',
                    array( 'back_link' => true )
                );
            }
        }
    }

    /**
     * Show safe mode admin notices
     */
    public static function show_safe_mode_notices() {
        $status = self::get_safe_mode_status();
        $mode_text = $status['mode'] === self::MODE_EMERGENCY ? 'Emergency Mode' : 'Safe Mode';
        $class = $status['mode'] === self::MODE_EMERGENCY ? 'notice-error' : 'notice-warning';
        
        echo '<div class="notice ' . esc_attr( $class ) . ' is-dismissible">';
        echo '<p><strong>' . esc_html( $mode_text ) . ' is active.</strong> ';
        echo 'Some plugins have been quarantined due to conflicts. ';
        echo '<a href="' . esc_url( admin_url( 'admin.php?page=unplug' ) ) . '">Manage Safe Mode</a>';
        echo '</p>';
        echo '</div>';
    }

    /**
     * Cleanup old backups
     */
    public static function cleanup_old_backups() {
        $backups = get_option( self::BACKUP_STATES_OPTION, array() );
        $cutoff_time = current_time( 'timestamp' ) - ( self::LOG_RETENTION_DAYS * DAY_IN_SECONDS );
        
        $cleaned_backups = array();
        foreach ( $backups as $backup_id => $backup_data ) {
            if ( $backup_data['created'] > $cutoff_time ) {
                $cleaned_backups[$backup_id] = $backup_data;
            }
        }

        if ( count( $cleaned_backups ) !== count( $backups ) ) {
            update_option( self::BACKUP_STATES_OPTION, $cleaned_backups );
        }
    }

    /**
     * Handle automatic safe mode activation on critical conflicts
     *
     * @param array $conflict_analysis Conflict analysis results
     */
    public static function handle_auto_activation( $conflict_analysis ) {
        if ( ! self::should_auto_activate_safe_mode( $conflict_analysis ) ) {
            return;
        }

        $conflicting_plugins = self::extract_conflicting_plugins( $conflict_analysis );
        
        if ( empty( $conflicting_plugins ) ) {
            return;
        }

        $result = self::activate_safe_mode( 
            self::TRIGGER_CRITICAL_CONFLICT, 
            $conflicting_plugins,
            array(
                'auto_activated' => true,
                'conflict_analysis' => $conflict_analysis
            )
        );

        if ( is_wp_error( $result ) ) {
            // Log error
            // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( 'Unplug: Failed to auto-activate safe mode: ' . $result->get_error_message() );
            // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
        }
    }
} 