<?php
/**
 * Unplug Render Blocking Analyzer
 * Identifies scripts and stylesheets that block page rendering
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Render_Blocking_Analyzer {
    
    /**
     * Critical CSS patterns that should not be deferred
     *
     * @var array
     */
    private static $critical_css_patterns = array(
        'style.css',
        'main.css',
        'theme.css',
        'critical.css',
        'above-fold.css',
        'bootstrap.css',
        'foundation.css'
    );
    
    /**
     * Scripts that should not be deferred
     *
     * @var array
     */
    private static $critical_scripts = array(
        'jquery-core',
        'jquery',
        'modernizr',
        'html5shiv',
        'polyfill'
    );
    
    /**
     * Analyze assets for render-blocking behavior
     *
     * @param array $assets Assets from UNPLUG_Asset_Detector
     * @return array Render-blocking analysis results
     */
    public static function analyze_render_blocking( $assets ) {
        $analysis = array(
            'blocking_scripts' => array(),
            'blocking_styles' => array(),
            'non_blocking_scripts' => array(),
            'non_blocking_styles' => array(),
            'recommendations' => array(),
            'performance_impact' => array()
        );
        
        // Analyze scripts
        if ( isset( $assets['scripts'] ) ) {
            foreach ( $assets['scripts'] as $handle => $script ) {
                $blocking_info = self::analyze_script_blocking( $script );
                
                if ( $blocking_info['is_blocking'] ) {
                    $analysis['blocking_scripts'][ $handle ] = array_merge( $script, $blocking_info );
                } else {
                    $analysis['non_blocking_scripts'][ $handle ] = array_merge( $script, $blocking_info );
                }
            }
        }
        
        // Analyze styles
        if ( isset( $assets['styles'] ) ) {
            foreach ( $assets['styles'] as $handle => $style ) {
                $blocking_info = self::analyze_style_blocking( $style );
                
                if ( $blocking_info['is_blocking'] ) {
                    $analysis['blocking_styles'][ $handle ] = array_merge( $style, $blocking_info );
                } else {
                    $analysis['non_blocking_styles'][ $handle ] = array_merge( $style, $blocking_info );
                }
            }
        }
        
        // Generate recommendations
        $analysis['recommendations'] = self::generate_recommendations( $analysis );
        
        // Calculate performance impact
        $analysis['performance_impact'] = self::calculate_performance_impact( $analysis );
        
        return $analysis;
    }
    
    /**
     * Analyze script for render-blocking behavior
     *
     * @param array $script Script asset data
     * @return array Blocking analysis
     */
    private static function analyze_script_blocking( $script ) {
        $analysis = array(
            'is_blocking' => false,
            'blocking_reason' => '',
            'blocking_score' => 0,
            'optimization_potential' => 'low',
            'recommendations' => array()
        );
        
        // Check if script is already optimized
        if ( $script['async'] || $script['defer'] ) {
            $analysis['is_blocking'] = false;
            $analysis['blocking_reason'] = $script['async'] ? 'Has async attribute' : 'Has defer attribute';
            return $analysis;
        }
        
        // Check if script is in footer
        if ( $script['in_footer'] ) {
            $analysis['is_blocking'] = false;
            $analysis['blocking_reason'] = 'Loaded in footer';
            return $analysis;
        }
        
        // Check if script is critical
        if ( self::is_critical_script( $script ) ) {
            $analysis['is_blocking'] = true;
            $analysis['blocking_reason'] = 'Critical script in header';
            $analysis['blocking_score'] = 2; // Lower score for critical scripts
            $analysis['optimization_potential'] = 'low';
            $analysis['recommendations'][] = 'Consider inline critical portions';
            return $analysis;
        }
        
        // Script is render-blocking
        $analysis['is_blocking'] = true;
        $analysis['blocking_reason'] = 'Script in header without async/defer';
        $analysis['blocking_score'] = self::calculate_script_blocking_score( $script );
        $analysis['optimization_potential'] = self::get_optimization_potential( $analysis['blocking_score'] );
        $analysis['recommendations'] = self::get_script_recommendations( $script );
        
        return $analysis;
    }
    
    /**
     * Analyze style for render-blocking behavior
     *
     * @param array $style Style asset data
     * @return array Blocking analysis
     */
    private static function analyze_style_blocking( $style ) {
        $analysis = array(
            'is_blocking' => false,
            'blocking_reason' => '',
            'blocking_score' => 0,
            'optimization_potential' => 'low',
            'recommendations' => array()
        );
        
        // Check media attribute
        if ( isset( $style['media'] ) && $style['media'] !== 'all' && $style['media'] !== 'screen' ) {
            $analysis['is_blocking'] = false;
            $analysis['blocking_reason'] = 'Non-blocking media: ' . $style['media'];
            return $analysis;
        }
        
        // Check if style is critical
        if ( isset( $style['critical'] ) && $style['critical'] ) {
            $analysis['is_blocking'] = true;
            $analysis['blocking_reason'] = 'Critical CSS for above-fold content';
            $analysis['blocking_score'] = 2; // Lower score for critical CSS
            $analysis['optimization_potential'] = 'low';
            $analysis['recommendations'][] = 'Consider inlining critical CSS';
            return $analysis;
        }
        
        // All other CSS is render-blocking by default
        $analysis['is_blocking'] = true;
        $analysis['blocking_reason'] = 'CSS blocks rendering by default';
        $analysis['blocking_score'] = self::calculate_style_blocking_score( $style );
        $analysis['optimization_potential'] = self::get_optimization_potential( $analysis['blocking_score'] );
        $analysis['recommendations'] = self::get_style_recommendations( $style );
        
        return $analysis;
    }
    
    /**
     * Check if script is critical and should not be deferred
     *
     * @param array $script Script asset data
     * @return bool True if critical
     */
    private static function is_critical_script( $script ) {
        // Check handle against critical scripts
        foreach ( self::$critical_scripts as $critical ) {
            if ( strpos( $script['handle'], $critical ) !== false ) {
                return true;
            }
        }
        
        // Check source for critical indicators
        $critical_sources = array(
            'jquery',
            'modernizr',
            'polyfill',
            'html5shiv'
        );
        
        foreach ( $critical_sources as $critical_source ) {
            if ( strpos( $script['src'], $critical_source ) !== false ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Calculate blocking score for script
     *
     * @param array $script Script asset data
     * @return int Blocking score (1-10)
     */
    private static function calculate_script_blocking_score( $script ) {
        $score = 5; // Base score
        
        // Increase score based on file size
        if ( $script['size'] > 100000 ) { // > 100KB
            $score += 3;
        } elseif ( $script['size'] > 50000 ) { // > 50KB
            $score += 2;
        } elseif ( $script['size'] > 20000 ) { // > 20KB
            $score += 1;
        }
        
        // Increase score for external scripts
        if ( $script['source'] === 'external' ) {
            $score += 2;
        }
        
        // Increase score for plugin scripts (easier to optimize)
        if ( strpos( $script['source'], 'plugin:' ) === 0 ) {
            $score += 1;
        }
        
        // Decrease score for theme scripts (harder to optimize)
        if ( strpos( $script['source'], 'theme:' ) === 0 ) {
            $score -= 1;
        }
        
        return max( 1, min( 10, $score ) );
    }
    
    /**
     * Calculate blocking score for style
     *
     * @param array $style Style asset data
     * @return int Blocking score (1-10)
     */
    private static function calculate_style_blocking_score( $style ) {
        $score = 4; // Base score (CSS generally less problematic than JS)
        
        // Increase score based on file size
        if ( $style['size'] > 200000 ) { // > 200KB
            $score += 3;
        } elseif ( $style['size'] > 100000 ) { // > 100KB
            $score += 2;
        } elseif ( $style['size'] > 50000 ) { // > 50KB
            $score += 1;
        }
        
        // Increase score for external styles
        if ( $style['source'] === 'external' ) {
            $score += 2;
        }
        
        // Increase score for plugin styles (easier to optimize)
        if ( strpos( $style['source'], 'plugin:' ) === 0 ) {
            $score += 1;
        }
        
        return max( 1, min( 10, $score ) );
    }
    
    /**
     * Get optimization potential based on blocking score
     *
     * @param int $score Blocking score
     * @return string Optimization potential
     */
    private static function get_optimization_potential( $score ) {
        if ( $score >= 7 ) {
            return 'high';
        } elseif ( $score >= 4 ) {
            return 'medium';
        } else {
            return 'low';
        }
    }
    
    /**
     * Get script optimization recommendations
     *
     * @param array $script Script asset data
     * @return array Recommendations
     */
    private static function get_script_recommendations( $script ) {
        $recommendations = array();
        
        // Size-based recommendations
        if ( $script['size'] > 100000 ) {
            $recommendations[] = 'Consider code splitting for large script';
        }
        
        // Source-based recommendations
        if ( $script['source'] === 'external' ) {
            $recommendations[] = 'Consider hosting externally loaded script locally';
        }
        
        if ( strpos( $script['source'], 'plugin:' ) === 0 ) {
            $recommendations[] = 'Add defer attribute to plugin script';
            $recommendations[] = 'Move script to footer if possible';
        }
        
        // General recommendations
        if ( ! $script['in_footer'] ) {
            $recommendations[] = 'Move script to footer';
        }
        
        if ( ! $script['async'] && ! $script['defer'] ) {
            $recommendations[] = 'Add defer attribute';
        }
        
        return $recommendations;
    }
    
    /**
     * Get style optimization recommendations
     *
     * @param array $style Style asset data
     * @return array Recommendations
     */
    private static function get_style_recommendations( $style ) {
        $recommendations = array();
        
        // Size-based recommendations
        if ( $style['size'] > 200000 ) {
            $recommendations[] = 'Consider splitting large CSS file';
        }
        
        // Source-based recommendations
        if ( $style['source'] === 'external' ) {
            $recommendations[] = 'Consider hosting external CSS locally';
        }
        
        if ( strpos( $style['source'], 'plugin:' ) === 0 ) {
            $recommendations[] = 'Load plugin CSS asynchronously if not critical';
        }
        
        // Media-based recommendations
        if ( ! isset( $style['media'] ) || $style['media'] === 'all' ) {
            $recommendations[] = 'Consider using specific media queries';
        }
        
        $recommendations[] = 'Consider critical CSS inlining';
        
        return $recommendations;
    }
    
    /**
     * Generate overall recommendations
     *
     * @param array $analysis Analysis results
     * @return array Recommendations
     */
    private static function generate_recommendations( $analysis ) {
        $recommendations = array();
        
        $blocking_scripts_count = count( $analysis['blocking_scripts'] );
        $blocking_styles_count = count( $analysis['blocking_styles'] );
        
        if ( $blocking_scripts_count > 5 ) {
            $recommendations[] = array(
                'type' => 'critical',
                'message' => sprintf( 'You have %d render-blocking scripts. Consider optimizing the most impactful ones first.', $blocking_scripts_count ),
                'priority' => 'high'
            );
        }
        
        if ( $blocking_styles_count > 3 ) {
            $recommendations[] = array(
                'type' => 'warning',
                'message' => sprintf( 'You have %d render-blocking stylesheets. Consider implementing critical CSS.', $blocking_styles_count ),
                'priority' => 'medium'
            );
        }
        
        // Find highest impact assets
        $high_impact_scripts = array_filter( $analysis['blocking_scripts'], function( $script ) {
            return $script['blocking_score'] >= 7;
        });
        
        if ( ! empty( $high_impact_scripts ) ) {
            $recommendations[] = array(
                'type' => 'optimization',
                'message' => 'Focus on optimizing high-impact scripts: ' . implode( ', ', array_keys( $high_impact_scripts ) ),
                'priority' => 'high'
            );
        }
        
        return $recommendations;
    }
    
    /**
     * Calculate performance impact
     *
     * @param array $analysis Analysis results
     * @return array Performance impact data
     */
    private static function calculate_performance_impact( $analysis ) {
        $impact = array(
            'total_blocking_resources' => 0,
            'total_blocking_size' => 0,
            'estimated_delay' => 0,
            'impact_level' => 'low',
            'potential_savings' => 0
        );
        
        // Count blocking resources
        $impact['total_blocking_resources'] = count( $analysis['blocking_scripts'] ) + count( $analysis['blocking_styles'] );
        
        // Calculate total blocking size
        foreach ( $analysis['blocking_scripts'] as $script ) {
            $impact['total_blocking_size'] += $script['size'];
        }
        
        foreach ( $analysis['blocking_styles'] as $style ) {
            $impact['total_blocking_size'] += $style['size'];
        }
        
        // Estimate delay (rough calculation)
        $impact['estimated_delay'] = ( $impact['total_blocking_resources'] * 50 ) + ( $impact['total_blocking_size'] / 10000 );
        
        // Determine impact level
        if ( $impact['estimated_delay'] > 1000 ) {
            $impact['impact_level'] = 'high';
        } elseif ( $impact['estimated_delay'] > 500 ) {
            $impact['impact_level'] = 'medium';
        } else {
            $impact['impact_level'] = 'low';
        }
        
        // Calculate potential savings
        $impact['potential_savings'] = $impact['estimated_delay'] * 0.7; // Assume 70% improvement possible
        
        return $impact;
    }
    
    /**
     * Get render-blocking summary
     *
     * @param array $analysis Analysis results
     * @return array Summary data
     */
    public static function get_summary( $analysis ) {
        return array(
            'blocking_scripts_count' => count( $analysis['blocking_scripts'] ),
            'blocking_styles_count' => count( $analysis['blocking_styles'] ),
            'total_blocking_resources' => $analysis['performance_impact']['total_blocking_resources'],
            'total_blocking_size' => $analysis['performance_impact']['total_blocking_size'],
            'estimated_delay' => $analysis['performance_impact']['estimated_delay'],
            'impact_level' => $analysis['performance_impact']['impact_level'],
            'potential_savings' => $analysis['performance_impact']['potential_savings'],
            'top_recommendations' => array_slice( $analysis['recommendations'], 0, 3 )
        );
    }
    
    /**
     * Format file size for display
     *
     * @param int $size Size in bytes
     * @return string Formatted size
     */
    public static function format_file_size( $size ) {
        if ( $size >= 1048576 ) {
            return round( $size / 1048576, 2 ) . ' MB';
        } elseif ( $size >= 1024 ) {
            return round( $size / 1024, 2 ) . ' KB';
        } else {
            return $size . ' B';
        }
    }
} 