<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
/**
 * Queue Management System (Database-backed Priority Queue)
 *
 * Implements a robust, priority-based queue using the wp_unplug_queue table.
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

class UNPLUG_Queue {
    /**
     * The database instance.
     * @var UNPLUG_Database
     */
    private $db;

    /**
     * Initialize the class and set its properties.
     * @param UNPLUG_Database $db
     */
    public function __construct( $db ) {
        $this->db = $db;
    }

    /**
     * Enqueue a new task into the priority queue.
     * @param string $test_type
     * @param array $task_data
     * @param int $priority (lower = higher priority)
     * @return int|false Inserted task ID or false on failure
     */
    public function enqueue( $test_type, $task_data, $priority = 5 ) {
        global $wpdb;
        $max_queue_size = get_option('unplug_queue_max_size', 1000); // Default 1000
        if ($this->size() >= $max_queue_size) {
            // Optionally log or handle overflow here
            return false;
        }
        $table = $this->db->get_table_name('queue');
        $data = array(
            'test_type'    => $test_type,
            'task_data'    => maybe_serialize($task_data),
            'status'       => 'pending',
            'priority'     => intval($priority),
            'attempts'     => 0,
            'max_attempts' => 3,
            'created_at'   => current_time('mysql'),
        );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific queue insert
        $result = $wpdb->insert($table, $data, array('%s','%s','%s','%d','%d','%d','%s'));
        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Atomically dequeue the next highest-priority, oldest pending task.
     * @return array|false Task row or false if none available
     */
    public function dequeue() {
        global $wpdb;
        $table = $this->db->get_table_name('queue');
        $now = current_time('mysql');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific transaction
        $wpdb->query('START TRANSACTION');
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific get row
        $task = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM `%s` WHERE status = %s AND (next_attempt_at IS NULL OR next_attempt_at <= %s) ORDER BY priority ASC, created_at ASC LIMIT 1 FOR UPDATE",
                $table, 'pending', $now
            ),
            ARRAY_A
        );
        if ($task) {
            // Check if attempts >= max_attempts
            if ($task['attempts'] >= $task['max_attempts']) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific queue task status update
                $wpdb->update(
                    $table,
                    array('status' => 'dead_letter', 'error_message' => 'Max attempts reached', 'last_update' => $now),
                    array('id' => $task['id'], 'status' => 'pending'),
                    array('%s','%s','%s'),
                    array('%d','%s')
                );
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific transaction
                $wpdb->query('COMMIT');
                // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                return false;
            }
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific queue task status update
            $updated = $wpdb->update(
                $table,
                array('status' => 'processing', 'started_at' => $now, 'attempts' => $task['attempts'] + 1, 'next_attempt_at' => null),
                array('id' => $task['id'], 'status' => 'pending'),
                array('%s','%s','%d','%s'),
                array('%d','%s')
            );
            if ($updated) {
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific transaction
                $wpdb->query('COMMIT');
                // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $task['status'] = 'processing';
                $task['started_at'] = $now;
                $task['attempts']++;
                $task['next_attempt_at'] = null;
                $task['task_data'] = maybe_unserialize($task['task_data']);
                return $task;
            }
        }
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific transaction
        $wpdb->query('ROLLBACK');
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return false;
    }

    /**
     * Peek at the next task to be processed (does not change status).
     * @return array|false Task row or false if none available
     */
    public function peek() {
        global $wpdb;
        $table = $this->db->get_table_name('queue');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific get row
        $task = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM `%s` WHERE status = %s ORDER BY priority ASC, created_at ASC LIMIT 1",
                $table, 'pending'
            ),
            ARRAY_A
        );
        if ($task) {
            $task['task_data'] = maybe_unserialize($task['task_data']);
        }
        return $task;
    }

    /**
     * Get the number of pending tasks in the queue.
     * @return int
     */
    public function size() {
        global $wpdb;
        $table = $this->db->get_table_name('queue');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific get var
        return (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM `%s` WHERE status = %s",
                $table, 'pending'
            )
        );
    }

    /**
     * Clear completed tasks from the queue.
     * @return int Number of tasks cleared
     */
    public function clear_completed_tasks() {
        global $wpdb;
        $table = $this->db->get_table_name('queue');
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM `%s` WHERE status = %s",
                $table, 'completed'
            )
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    }
} 