<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
/**
 * Plugin Inventory Analysis System
 *
 * Comprehensive plugin discovery and cataloging system for conflict detection.
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 * @since      1.0.0
 */

/**
 * The plugin inventory analysis class.
 *
 * This class is responsible for scanning, cataloging, and analyzing all installed
 * WordPress plugins to extract metadata and characteristics for conflict detection.
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Your Name <email@example.com>
 * @since      1.0.0
 */
class UNPLUG_Plugin_Inventory {

    /**
     * The plugin inventory cache.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $inventory    The cached plugin inventory data.
     */
    private static $inventory = null;

    /**
     * Plugin categories for conflict analysis.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $plugin_categories    Plugin type categories.
     */
    private static $plugin_categories = array(
        'security' => array(
            'wordfence', 'ithemes-security', 'all-in-one-wp-security', 'sucuri-scanner',
            'bulletproof-security', 'wp-security-audit-log', 'limit-login-attempts',
            'defender-security', 'wp-fail2ban', 'cerber-security'
        ),
        'caching' => array(
            'w3-total-cache', 'wp-super-cache', 'wp-fastest-cache', 'wp-rocket',
            'litespeed-cache', 'hummingbird-performance', 'autoptimize', 'swift-performance',
            'wp-optimize', 'comet-cache', 'hyper-cache', 'cache-enabler'
        ),
        'optimization' => array(
            'wp-optimize', 'autoptimize', 'wp-smush', 'shortpixel-image-optimiser',
            'ewww-image-optimizer', 'hummingbird-performance', 'swift-performance',
            'perfmatters', 'wp-rocket', 'clearfy', 'wp-fastest-cache'
        ),
        'seo' => array(
            'wordpress-seo', 'all-in-one-seo-pack', 'the-seo-framework', 'rankmath',
            'seo-by-rank-math', 'wp-seopress', 'squirrly-seo', 'platinum-seo-pack'
        ),
        'backup' => array(
            'updraftplus', 'backupwordpress', 'backup-guard', 'duplicator',
            'all-in-one-wp-migration', 'wp-db-backup', 'backwpup', 'jetpack-backup'
        ),
        'maintenance' => array(
            'wp-maintenance-mode', 'coming-soon', 'maintenance', 'wp-under-construction',
            'seedprod-coming-soon-pro', 'ultimate-maintenance-mode'
        ),
        'multisite' => array(
            'wordpress-mu-domain-mapping', 'multisite-enhancements', 'wp-multi-network',
            'multisite-toolbar-additions', 'unconfirmed'
        )
    );

    /**
     * Common plugin conflict patterns.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $conflict_patterns    Known conflict patterns.
     */
    private static $conflict_patterns = array(
        'multiple_caching' => array(
            'severity' => 'high',
            'description' => 'Multiple caching plugins detected',
            'recommendation' => 'Keep only one caching plugin active to avoid conflicts'
        ),
        'multiple_security' => array(
            'severity' => 'medium',
            'description' => 'Multiple security plugins detected',
            'recommendation' => 'Consider consolidating security features to avoid overlapping protection'
        ),
        'multiple_seo' => array(
            'severity' => 'high',
            'description' => 'Multiple SEO plugins detected',
            'recommendation' => 'Keep only one SEO plugin active to avoid duplicate meta tags'
        ),
        'multiple_backup' => array(
            'severity' => 'low',
            'description' => 'Multiple backup plugins detected',
            'recommendation' => 'Consider using one primary backup solution to avoid resource conflicts'
        ),
        'optimization_conflicts' => array(
            'severity' => 'medium',
            'description' => 'Conflicting optimization plugins detected',
            'recommendation' => 'Review optimization settings to avoid duplicate processing'
        )
    );

    /**
     * Enhanced: Detect plugin removal (admin or FTP) and sync inventory/cleanup.
     */
    public static function enhanced_removal_detection() {
        // Compare current plugins to cached inventory
        $cached = get_transient('unplug_plugin_inventory');
        if (!$cached || !isset($cached['plugins'])) {
            return;
        }
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $current_plugins = array_keys(get_plugins());
        $cached_plugins = array_keys($cached['plugins']);
        $missing = array_diff($cached_plugins, $current_plugins);
        if (!empty($missing)) {
            // Plugins deleted via FTP or otherwise missing
            self::update_inventory_cache();
            self::trigger_orphaned_data_cleanup($missing);
            self::trigger_dashboard_refresh();
        }
    }

    /**
     * Trigger orphaned scan/queue cleanup for removed plugins.
     * @param array $removed_plugins
     */
    public static function trigger_orphaned_data_cleanup($removed_plugins) {
        // Load cleanup class
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-unplug-cleanup.php';
        // Run orphaned scan/queue cleanup
        UNPLUG_Cleanup::cleanup_old_scan_results();
        UNPLUG_Cleanup::cleanup_expired_queue_items();
        UNPLUG_Cleanup::cleanup_orphaned_locations();
        // Optionally, pass $removed_plugins for more targeted cleanup
    }

    /**
     * Trigger dashboard refresh (set transient/option flag).
     */
    public static function trigger_dashboard_refresh() {
        set_transient('unplug_dashboard_refresh', 1, 60);
    }

    /**
     * Initialize the plugin inventory system.
     *
     * @since    1.0.0
     */
    public static function init() {
        // Clear inventory cache on plugin activation/deactivation
        add_action( 'activated_plugin', array( __CLASS__, 'clear_inventory_cache' ) );
        add_action( 'deactivated_plugin', array( __CLASS__, 'clear_inventory_cache' ) );
        add_action( 'delete_plugin', array( __CLASS__, 'clear_inventory_cache' ) );
        
        // Schedule periodic inventory updates
        if ( ! wp_next_scheduled( 'unplug_update_plugin_inventory' ) ) {
            wp_schedule_event( time(), 'daily', 'unplug_update_plugin_inventory' );
        }
        
        add_action( 'unplug_update_plugin_inventory', array( __CLASS__, 'update_inventory_cache' ) );
        add_action('admin_init', array(__CLASS__, 'enhanced_removal_detection'));
    }

    /**
     * Get comprehensive plugin inventory.
     *
     * @since    1.0.0
     * @param    bool    $force_refresh    Force refresh of cached data.
     * @return   array                     Complete plugin inventory data.
     */
    public static function get_plugin_inventory( $force_refresh = false ) {
        if ( $force_refresh || self::$inventory === null ) {
            self::$inventory = self::build_plugin_inventory();
        }
        
        return self::$inventory;
    }

    /**
     * Build comprehensive plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @return   array    Complete plugin inventory data.
     */
    private static function build_plugin_inventory() {
        $inventory = array(
            'plugins' => array(),
            'active_plugins' => array(),
            'inactive_plugins' => array(),
            'categories' => array(),
            'conflicts' => array(),
            'metadata' => array(
                'last_updated' => current_time( 'timestamp' ),
                'total_plugins' => 0,
                'active_count' => 0,
                'inactive_count' => 0
            )
        );

        // Get all plugins
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $active_plugins = get_option( 'active_plugins', array() );
        
        // Network active plugins for multisite
        if ( is_multisite() ) {
            $network_active = get_site_option( 'active_sitewide_plugins', array() );
            $network_active = array_keys( $network_active );
            $active_plugins = array_merge( $active_plugins, $network_active );
        }

        foreach ( $all_plugins as $plugin_file => $plugin_data ) {
            $plugin_analysis = self::analyze_plugin( $plugin_file, $plugin_data );
            $is_active = in_array( $plugin_file, $active_plugins );
            
            $plugin_analysis['active'] = $is_active;
            $plugin_analysis['file'] = $plugin_file;
            
            // Add to inventory
            $inventory['plugins'][$plugin_file] = $plugin_analysis;
            
            if ( $is_active ) {
                $inventory['active_plugins'][$plugin_file] = $plugin_analysis;
            } else {
                $inventory['inactive_plugins'][$plugin_file] = $plugin_analysis;
            }
            
            // Categorize plugin
            $category = self::categorize_plugin( $plugin_file, $plugin_data );
            if ( $category ) {
                if ( ! isset( $inventory['categories'][$category] ) ) {
                    $inventory['categories'][$category] = array();
                }
                $inventory['categories'][$category][$plugin_file] = $plugin_analysis;
            }
        }

        // Update metadata
        $inventory['metadata']['total_plugins'] = count( $all_plugins );
        $inventory['metadata']['active_count'] = count( $inventory['active_plugins'] );
        $inventory['metadata']['inactive_count'] = count( $inventory['inactive_plugins'] );

        // Analyze conflicts
        $inventory['conflicts'] = self::analyze_conflicts( $inventory );

        return $inventory;
    }

    /**
     * Analyze individual plugin characteristics.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $plugin_file    Plugin file path.
     * @param    array     $plugin_data    Plugin header data.
     * @return   array                     Plugin analysis data.
     */
    private static function analyze_plugin( $plugin_file, $plugin_data ) {
        $analysis = array(
            'name' => $plugin_data['Name'],
            'version' => $plugin_data['Version'],
            'author' => $plugin_data['Author'],
            'description' => $plugin_data['Description'],
            'plugin_uri' => $plugin_data['PluginURI'],
            'text_domain' => $plugin_data['TextDomain'],
            'domain_path' => $plugin_data['DomainPath'],
            'network' => $plugin_data['Network'],
            'requires_wp' => $plugin_data['RequiresWP'],
            'tested_wp' => $plugin_data['TestedUpTo'],
            'requires_php' => $plugin_data['RequiresPHP'],
            'update_uri' => $plugin_data['UpdateURI'],
            'slug' => dirname( $plugin_file ),
            'main_file' => basename( $plugin_file ),
            'path' => plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file ) . basename( $plugin_file ),
            'hooks' => array(),
            'api_usage' => array(),
            'resource_requirements' => array(),
            'security_features' => array(),
            'performance_impact' => 'unknown',
            'compatibility' => array(),
            'last_updated' => '',
            'size' => 0
        );

        // Get plugin directory information
        $plugin_dir = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        if ( is_dir( $plugin_dir ) ) {
            $analysis['size'] = self::get_directory_size( $plugin_dir );
            $analysis['file_count'] = self::count_files_in_directory( $plugin_dir );
        }

        // Analyze plugin code for hooks and API usage
        if ( file_exists( $analysis['path'] ) ) {
            $analysis['hooks'] = self::extract_hooks_from_plugin( $analysis['path'] );
            $analysis['api_usage'] = self::analyze_api_usage( $analysis['path'] );
            $analysis['security_features'] = self::analyze_security_features( $analysis['path'] );
        }

        // Estimate performance impact
        $analysis['performance_impact'] = self::estimate_performance_impact( $analysis );

        // Check compatibility
        $analysis['compatibility'] = self::check_compatibility( $plugin_data );

        return $analysis;
    }

    /**
     * Categorize plugin by type.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $plugin_file    Plugin file path.
     * @param    array     $plugin_data    Plugin header data.
     * @return   string|null               Plugin category or null.
     */
    private static function categorize_plugin( $plugin_file, $plugin_data ) {
        $plugin_slug = dirname( $plugin_file );
        $plugin_name = strtolower( $plugin_data['Name'] );
        $plugin_description = strtolower( $plugin_data['Description'] );

        foreach ( self::$plugin_categories as $category => $plugins ) {
            if ( in_array( $plugin_slug, $plugins ) ) {
                return $category;
            }
        }

        // Fallback: categorize by keywords in name/description
        $keywords = array(
            'security' => array( 'security', 'firewall', 'malware', 'brute force', 'login attempts' ),
            'caching' => array( 'cache', 'caching', 'speed', 'performance', 'optimization' ),
            'optimization' => array( 'optimize', 'compress', 'minify', 'image optimization' ),
            'seo' => array( 'seo', 'search engine', 'sitemap', 'meta', 'schema' ),
            'backup' => array( 'backup', 'restore', 'migration', 'clone' ),
            'maintenance' => array( 'maintenance', 'coming soon', 'under construction' )
        );

        foreach ( $keywords as $category => $category_keywords ) {
            foreach ( $category_keywords as $keyword ) {
                if ( strpos( $plugin_name, $keyword ) !== false || 
                     strpos( $plugin_description, $keyword ) !== false ) {
                    return $category;
                }
            }
        }

        return null;
    }

    /**
     * Analyze potential conflicts between plugins.
     *
     * @since    1.0.0
     * @access   private
     * @param    array    $inventory    Plugin inventory data.
     * @return   array                  Detected conflicts.
     */
    private static function analyze_conflicts( $inventory ) {
        $conflicts = array();

        // Check for multiple plugins in same category
        foreach ( $inventory['categories'] as $category => $plugins ) {
            $active_in_category = array_filter( $plugins, function( $plugin ) {
                return $plugin['active'];
            });

            if ( count( $active_in_category ) > 1 ) {
                $conflict_key = 'multiple_' . $category;
                if ( isset( self::$conflict_patterns[$conflict_key] ) ) {
                    $conflicts[$conflict_key] = array_merge(
                        self::$conflict_patterns[$conflict_key],
                        array(
                            'plugins' => array_keys( $active_in_category ),
                            'count' => count( $active_in_category )
                        )
                    );
                }
            }
        }

        // Check for specific known conflicts
        $conflicts = array_merge( $conflicts, self::check_known_conflicts( $inventory ) );

        return $conflicts;
    }

    /**
     * Check for known specific plugin conflicts.
     *
     * @since    1.0.0
     * @access   private
     * @param    array    $inventory    Plugin inventory data.
     * @return   array                  Known conflicts.
     */
    private static function check_known_conflicts( $inventory ) {
        $conflicts = array();
        $active_plugins = array_keys( $inventory['active_plugins'] );

        // Define specific known conflicts
        $known_conflicts = array(
            array(
                'plugins' => array( 'w3-total-cache/w3-total-cache.php', 'wp-super-cache/wp-cache.php' ),
                'conflict_id' => 'w3tc_wp_super_cache',
                'severity' => 'high',
                'description' => 'W3 Total Cache and WP Super Cache conflict',
                'recommendation' => 'Deactivate one of these caching plugins'
            ),
            array(
                'plugins' => array( 'wordfence/wordfence.php', 'ithemes-security-pro/ithemes-security-pro.php' ),
                'conflict_id' => 'wordfence_ithemes',
                'severity' => 'medium',
                'description' => 'Wordfence and iThemes Security may have overlapping features',
                'recommendation' => 'Review security settings to avoid duplicate protection'
            ),
            array(
                'plugins' => array( 'wordpress-seo/wp-seo.php', 'all-in-one-seo-pack/all_in_one_seo_pack.php' ),
                'conflict_id' => 'yoast_aioseo',
                'severity' => 'high',
                'description' => 'Yoast SEO and All in One SEO Pack conflict',
                'recommendation' => 'Keep only one SEO plugin active'
            )
        );

        foreach ( $known_conflicts as $conflict ) {
            $conflict_plugins = array_intersect( $conflict['plugins'], $active_plugins );
            if ( count( $conflict_plugins ) > 1 ) {
                $conflicts[$conflict['conflict_id']] = array(
                    'severity' => $conflict['severity'],
                    'description' => $conflict['description'],
                    'recommendation' => $conflict['recommendation'],
                    'plugins' => $conflict_plugins,
                    'count' => count( $conflict_plugins )
                );
            }
        }

        return $conflicts;
    }

    /**
     * Extract hooks used by a plugin.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $plugin_file    Plugin main file path.
     * @return   array                     Array of hooks found.
     */
    private static function extract_hooks_from_plugin( $plugin_file ) {
        $hooks = array(
            'actions' => array(),
            'filters' => array(),
            'admin_actions' => array(),
            'ajax_actions' => array()
        );

        if ( ! file_exists( $plugin_file ) ) {
            return $hooks;
        }

        $content = file_get_contents( $plugin_file );
        
        // Find action hooks
        if ( preg_match_all( '/add_action\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
            $hooks['actions'] = array_unique( $matches[1] );
        }
        
        // Find filter hooks
        if ( preg_match_all( '/add_filter\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
            $hooks['filters'] = array_unique( $matches[1] );
        }
        
        // Find admin actions
        if ( preg_match_all( '/wp_ajax_([a-zA-Z0-9_]+)/', $content, $matches ) ) {
            $hooks['ajax_actions'] = array_unique( $matches[1] );
        }

        return $hooks;
    }

    /**
     * Analyze API usage patterns.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $plugin_file    Plugin main file path.
     * @return   array                     API usage analysis.
     */
    private static function analyze_api_usage( $plugin_file ) {
        $api_usage = array(
            'wp_remote_get' => false,
            'wp_remote_post' => false,
            'curl' => false,
            'file_get_contents' => false,
            'mysql_query' => false,
            'wp_enqueue_script' => false,
            'wp_enqueue_style' => false,
            'wp_register_script' => false,
            'wp_register_style' => false
        );

        if ( ! file_exists( $plugin_file ) ) {
            return $api_usage;
        }

        $content = file_get_contents( $plugin_file );
        
        foreach ( $api_usage as $function => $used ) {
            if ( strpos( $content, $function ) !== false ) {
                $api_usage[$function] = true;
            }
        }

        return $api_usage;
    }

    /**
     * Analyze security features.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $plugin_file    Plugin main file path.
     * @return   array                     Security features analysis.
     */
    private static function analyze_security_features( $plugin_file ) {
        $security_features = array(
            'nonce_verification' => false,
            'capability_checks' => false,
            'input_sanitization' => false,
            'sql_preparation' => false,
            'output_escaping' => false
        );

        if ( ! file_exists( $plugin_file ) ) {
            return $security_features;
        }

        $content = file_get_contents( $plugin_file );
        
        // Check for nonce verification
        if ( preg_match( '/wp_verify_nonce|check_ajax_referer|wp_nonce_field/', $content ) ) {
            $security_features['nonce_verification'] = true;
        }
        
        // Check for capability checks
        if ( preg_match( '/current_user_can|user_can/', $content ) ) {
            $security_features['capability_checks'] = true;
        }
        
        // Check for input sanitization
        if ( preg_match( '/sanitize_|wp_kses|stripslashes/', $content ) ) {
            $security_features['input_sanitization'] = true;
        }
        
        // Check for SQL preparation
        if ( preg_match( '/\$wpdb->prepare/', $content ) ) {
            $security_features['sql_preparation'] = true;
        }
        
        // Check for output escaping
        if ( preg_match( '/esc_html|esc_attr|esc_url/', $content ) ) {
            $security_features['output_escaping'] = true;
        }

        return $security_features;
    }

    /**
     * Estimate performance impact.
     *
     * @since    1.0.0
     * @access   private
     * @param    array    $analysis    Plugin analysis data.
     * @return   string               Performance impact level.
     */
    private static function estimate_performance_impact( $analysis ) {
        $score = 0;

        // Size factor
        if ( $analysis['size'] > 10485760 ) { // 10MB
            $score += 3;
        } elseif ( $analysis['size'] > 5242880 ) { // 5MB
            $score += 2;
        } elseif ( $analysis['size'] > 1048576 ) { // 1MB
            $score += 1;
        }

        // Hook count factor
        $total_hooks = count( $analysis['hooks']['actions'] ) + count( $analysis['hooks']['filters'] );
        if ( $total_hooks > 20 ) {
            $score += 2;
        } elseif ( $total_hooks > 10 ) {
            $score += 1;
        }

        // API usage factor
        $api_usage_count = count( array_filter( $analysis['api_usage'] ) );
        if ( $api_usage_count > 5 ) {
            $score += 2;
        } elseif ( $api_usage_count > 2 ) {
            $score += 1;
        }

        // Return impact level
        if ( $score >= 5 ) {
            return 'high';
        } elseif ( $score >= 3 ) {
            return 'medium';
        } else {
            return 'low';
        }
    }

    /**
     * Check plugin compatibility.
     *
     * @since    1.0.0
     * @access   private
     * @param    array    $plugin_data    Plugin header data.
     * @return   array                    Compatibility information.
     */
    private static function check_compatibility( $plugin_data ) {
        global $wp_version;

        $compatibility = array(
            'wp_version_compatible' => true,
            'php_version_compatible' => true,
            'requires_wp' => $plugin_data['RequiresWP'],
            'tested_up_to' => $plugin_data['TestedUpTo'],
            'requires_php' => $plugin_data['RequiresPHP'],
            'current_wp' => $wp_version,
            'current_php' => PHP_VERSION
        );

        // Check WordPress version compatibility
        if ( ! empty( $plugin_data['RequiresWP'] ) && version_compare( $wp_version, $plugin_data['RequiresWP'], '<' ) ) {
            $compatibility['wp_version_compatible'] = false;
        }

        // Check PHP version compatibility
        if ( ! empty( $plugin_data['RequiresPHP'] ) && version_compare( PHP_VERSION, $plugin_data['RequiresPHP'], '<' ) ) {
            $compatibility['php_version_compatible'] = false;
        }

        return $compatibility;
    }

    /**
     * Get directory size.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $directory    Directory path.
     * @return   int                     Directory size in bytes.
     */
    private static function get_directory_size( $directory ) {
        $size = 0;
        
        if ( is_dir( $directory ) ) {
            $files = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $directory ) );
            
            foreach ( $files as $file ) {
                if ( $file->isFile() ) {
                    $size += $file->getSize();
                }
            }
        }
        
        return $size;
    }

    /**
     * Count files in directory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string    $directory    Directory path.
     * @return   int                     Number of files.
     */
    private static function count_files_in_directory( $directory ) {
        $count = 0;
        
        if ( is_dir( $directory ) ) {
            $files = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $directory ) );
            
            foreach ( $files as $file ) {
                if ( $file->isFile() ) {
                    $count++;
                }
            }
        }
        
        return $count;
    }

    /**
     * Get plugin summary for quick analysis.
     *
     * @since    1.0.0
     * @return   array    Plugin summary data.
     */
    public static function get_plugin_summary() {
        $inventory = self::get_plugin_inventory();
        
        return array(
            'total_plugins' => $inventory['metadata']['total_plugins'],
            'active_plugins' => $inventory['metadata']['active_count'],
            'inactive_plugins' => $inventory['metadata']['inactive_count'],
            'categories' => array_map( 'count', $inventory['categories'] ),
            'conflicts' => count( $inventory['conflicts'] ),
            'high_severity_conflicts' => count( array_filter( $inventory['conflicts'], function( $conflict ) {
                return $conflict['severity'] === 'high';
            }))
        );
    }

    /**
     * Get plugin conflicts.
     *
     * @since    1.0.0
     * @return   array    Detected plugin conflicts.
     */
    public static function get_plugin_conflicts() {
        $inventory = self::get_plugin_inventory();
        return $inventory['conflicts'];
    }

    /**
     * Get plugins by category.
     *
     * @since    1.0.0
     * @param    string    $category    Plugin category.
     * @return   array                  Plugins in category.
     */
    public static function get_plugins_by_category( $category ) {
        $inventory = self::get_plugin_inventory();
        return isset( $inventory['categories'][$category] ) ? $inventory['categories'][$category] : array();
    }

    /**
     * Clear inventory cache.
     *
     * @since    1.0.0
     */
    public static function clear_inventory_cache() {
        self::$inventory = null;
        delete_transient( 'unplug_plugin_inventory' );
    }

    /**
     * Update inventory cache.
     *
     * @since    1.0.0
     */
    public static function update_inventory_cache() {
        self::get_plugin_inventory( true );
        set_transient( 'unplug_plugin_inventory', self::$inventory, DAY_IN_SECONDS );
    }

    /**
     * Get plugin analysis by file.
     *
     * @since    1.0.0
     * @param    string    $plugin_file    Plugin file path.
     * @return   array|null               Plugin analysis or null if not found.
     */
    public static function get_plugin_analysis( $plugin_file ) {
        $inventory = self::get_plugin_inventory();
        return isset( $inventory['plugins'][$plugin_file] ) ? $inventory['plugins'][$plugin_file] : null;
    }

    /**
     * Check if plugin has high performance impact.
     *
     * @since    1.0.0
     * @param    string    $plugin_file    Plugin file path.
     * @return   bool                      True if high impact.
     */
    public static function has_high_performance_impact( $plugin_file ) {
        $analysis = self::get_plugin_analysis( $plugin_file );
        return $analysis && $analysis['performance_impact'] === 'high';
    }

    /**
     * Get plugin category.
     *
     * @since    1.0.0
     * @param    string    $plugin_file    Plugin file path.
     * @return   string|null               Plugin category or null.
     */
    public static function get_plugin_category( $plugin_file ) {
        $inventory = self::get_plugin_inventory();
        
        foreach ( $inventory['categories'] as $category => $plugins ) {
            if ( isset( $plugins[$plugin_file] ) ) {
                return $category;
            }
        }
        
        return null;
    }

    /**
     * Check if plugin is compatible with current environment.
     *
     * @since    1.0.0
     * @param    string    $plugin_file    Plugin file path.
     * @return   bool                      True if compatible.
     */
    public static function is_plugin_compatible( $plugin_file ) {
        $analysis = self::get_plugin_analysis( $plugin_file );
        
        if ( ! $analysis ) {
            return true; // Unknown, assume compatible
        }
        
        return $analysis['compatibility']['wp_version_compatible'] && 
               $analysis['compatibility']['php_version_compatible'];
    }

    /**
     * Get plugin security score.
     *
     * @since    1.0.0
     * @param    string    $plugin_file    Plugin file path.
     * @return   int                       Security score (0-100).
     */
    public static function get_plugin_security_score( $plugin_file ) {
        $analysis = self::get_plugin_analysis( $plugin_file );
        
        if ( ! $analysis ) {
            return 0;
        }
        
        $score = 0;
        $features = $analysis['security_features'];
        
        // Each security feature adds 20 points
        foreach ( $features as $feature => $implemented ) {
            if ( $implemented ) {
                $score += 20;
            }
        }
        
        return min( $score, 100 );
    }

    /**
     * Export plugin inventory data.
     *
     * @since    1.0.0
     * @param    string    $format    Export format (json, csv, xml).
     * @return   string               Exported data.
     */
    public static function export_inventory( $format = 'json' ) {
        $inventory = self::get_plugin_inventory();
        
        switch ( $format ) {
            case 'json':
                return json_encode( $inventory, JSON_PRETTY_PRINT );
                
            case 'csv':
                return self::export_inventory_csv( $inventory );
                
            case 'xml':
                return self::export_inventory_xml( $inventory );
                
            default:
                return json_encode( $inventory, JSON_PRETTY_PRINT );
        }
    }

    /**
     * Export inventory as CSV.
     *
     * @since    1.0.0
     * @access   private
     * @param    array    $inventory    Plugin inventory data.
     * @return   string                CSV data.
     */
    private static function export_inventory_csv( $inventory ) {
        $csv = "Plugin Name,Version,Active,Category,Performance Impact,Security Score,Conflicts\n";
        
        foreach ( $inventory['plugins'] as $plugin_file => $plugin ) {
            $category = self::get_plugin_category( $plugin_file );
            $security_score = self::get_plugin_security_score( $plugin_file );
            $conflicts = array();
            
            foreach ( $inventory['conflicts'] as $conflict_id => $conflict ) {
                if ( in_array( $plugin_file, $conflict['plugins'] ) ) {
                    $conflicts[] = $conflict_id;
                }
            }
            
            $csv .= sprintf(
                '"%s","%s","%s","%s","%s","%d","%s"' . "\n",
                str_replace( '"', '""', $plugin['name'] ),
                $plugin['version'],
                $plugin['active'] ? 'Yes' : 'No',
                $category ?: 'Unknown',
                $plugin['performance_impact'],
                $security_score,
                implode( ', ', $conflicts )
            );
        }
        
        return $csv;
    }

    /**
     * Export inventory as XML.
     *
     * @since    1.0.0
     * @access   private
     * @param    array    $inventory    Plugin inventory data.
     * @return   string                XML data.
     */
    private static function export_inventory_xml( $inventory ) {
        $xml = new SimpleXMLElement( '<plugin_inventory></plugin_inventory>' );
        
        $metadata = $xml->addChild( 'metadata' );
        $metadata->addChild( 'last_updated', gmdate( 'Y-m-d H:i:s', $inventory['metadata']['last_updated'] ) );
        $metadata->addChild( 'total_plugins', $inventory['metadata']['total_plugins'] );
        $metadata->addChild( 'active_count', $inventory['metadata']['active_count'] );
        $metadata->addChild( 'inactive_count', $inventory['metadata']['inactive_count'] );
        
        $plugins = $xml->addChild( 'plugins' );
        foreach ( $inventory['plugins'] as $plugin_file => $plugin ) {
            $plugin_element = $plugins->addChild( 'plugin' );
            $plugin_element->addChild( 'file', htmlspecialchars( $plugin_file ) );
            $plugin_element->addChild( 'name', htmlspecialchars( $plugin['name'] ) );
            $plugin_element->addChild( 'version', htmlspecialchars( $plugin['version'] ) );
            $plugin_element->addChild( 'active', $plugin['active'] ? 'true' : 'false' );
            $plugin_element->addChild( 'performance_impact', $plugin['performance_impact'] );
        }
        
        $conflicts = $xml->addChild( 'conflicts' );
        foreach ( $inventory['conflicts'] as $conflict_id => $conflict ) {
            $conflict_element = $conflicts->addChild( 'conflict' );
            $conflict_element->addChild( 'id', $conflict_id );
            $conflict_element->addChild( 'severity', $conflict['severity'] );
            $conflict_element->addChild( 'description', htmlspecialchars( $conflict['description'] ) );
        }
        
        return $xml->asXML();
    }

} 