<?php
/**
 * Unplug Performance Impact Measurement
 * Measures the performance impact of identified assets on page load speed
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Performance_Impact {
    
    /**
     * Performance metrics collected during measurement
     *
     * @var array
     */
    private static $performance_metrics = array();
    
    /**
     * Timing data for performance measurement
     *
     * @var array
     */
    private static $timing_data = array();
    
    /**
     * Performance measurement hooks initialized
     *
     * @var bool
     */
    private static $hooks_initialized = false;
    
    /**
     * Initialize performance measurement hooks
     */
    public static function init() {
        if ( self::$hooks_initialized ) {
            return;
        }
        
        // Hook into WordPress timing points
        add_action( 'init', array( __CLASS__, 'start_timing' ), 1 );
        add_action( 'wp_head', array( __CLASS__, 'mark_head_complete' ), 999 );
        add_action( 'wp_footer', array( __CLASS__, 'mark_footer_start' ), 1 );
        add_action( 'wp_footer', array( __CLASS__, 'mark_footer_complete' ), 999 );
        add_action( 'shutdown', array( __CLASS__, 'finalize_timing' ), 1 );
        
        // Hook into asset loading
        add_action( 'wp_enqueue_scripts', array( __CLASS__, 'track_asset_timing' ), 1 );
        add_action( 'wp_enqueue_scripts', array( __CLASS__, 'track_asset_timing' ), 999 );
        
        self::$hooks_initialized = true;
    }
    
    /**
     * Start timing measurement
     */
    public static function start_timing() {
        self::$timing_data['start_time'] = microtime( true );
        self::$timing_data['memory_start'] = memory_get_usage();
        self::$timing_data['memory_peak_start'] = memory_get_peak_usage();
    }
    
    /**
     * Mark head section complete
     */
    public static function mark_head_complete() {
        self::$timing_data['head_complete'] = microtime( true );
        self::$timing_data['memory_after_head'] = memory_get_usage();
    }
    
    /**
     * Mark footer start
     */
    public static function mark_footer_start() {
        self::$timing_data['footer_start'] = microtime( true );
        self::$timing_data['memory_before_footer'] = memory_get_usage();
    }
    
    /**
     * Mark footer complete
     */
    public static function mark_footer_complete() {
        self::$timing_data['footer_complete'] = microtime( true );
        self::$timing_data['memory_after_footer'] = memory_get_usage();
    }
    
    /**
     * Finalize timing measurement
     */
    public static function finalize_timing() {
        self::$timing_data['end_time'] = microtime( true );
        self::$timing_data['memory_end'] = memory_get_usage();
        self::$timing_data['memory_peak_end'] = memory_get_peak_usage();
        
        // Calculate derived metrics
        self::calculate_derived_metrics();
    }
    
    /**
     * Track asset timing
     */
    public static function track_asset_timing() {
        global $wp_scripts, $wp_styles;
        
        $current_time = microtime( true );
        $current_memory = memory_get_usage();
        
        // Track script timing
        if ( $wp_scripts && ! empty( $wp_scripts->queue ) ) {
            foreach ( $wp_scripts->queue as $handle ) {
                if ( ! isset( self::$timing_data['assets']['scripts'][ $handle ] ) ) {
                    self::$timing_data['assets']['scripts'][ $handle ] = array(
                        'enqueue_time' => $current_time,
                        'memory_at_enqueue' => $current_memory
                    );
                }
            }
        }
        
        // Track style timing
        if ( $wp_styles && ! empty( $wp_styles->queue ) ) {
            foreach ( $wp_styles->queue as $handle ) {
                if ( ! isset( self::$timing_data['assets']['styles'][ $handle ] ) ) {
                    self::$timing_data['assets']['styles'][ $handle ] = array(
                        'enqueue_time' => $current_time,
                        'memory_at_enqueue' => $current_memory
                    );
                }
            }
        }
    }
    
    /**
     * Calculate derived performance metrics
     */
    private static function calculate_derived_metrics() {
        if ( empty( self::$timing_data['start_time'] ) || empty( self::$timing_data['end_time'] ) ) {
            return;
        }
        
        $start = self::$timing_data['start_time'];
        $end = self::$timing_data['end_time'];
        
        self::$performance_metrics = array(
            'total_time' => ( $end - $start ) * 1000, // Convert to milliseconds
            'head_time' => isset( self::$timing_data['head_complete'] ) ? 
                ( self::$timing_data['head_complete'] - $start ) * 1000 : 0,
            'footer_time' => isset( self::$timing_data['footer_complete'] ) && isset( self::$timing_data['footer_start'] ) ? 
                ( self::$timing_data['footer_complete'] - self::$timing_data['footer_start'] ) * 1000 : 0,
            'memory_usage' => self::$timing_data['memory_end'] - self::$timing_data['memory_start'],
            'memory_peak' => self::$timing_data['memory_peak_end'],
            'timestamp' => current_time( 'mysql' )
        );
    }
    
    /**
     * Measure performance impact of assets
     *
     * @param array $assets Assets with attribution data
     * @param array $render_blocking_analysis Render blocking analysis
     * @return array Performance impact analysis
     */
    public static function measure_asset_impact( $assets, $render_blocking_analysis ) {
        $impact_analysis = array(
            'overall_impact' => array(),
            'asset_impacts' => array(),
            'plugin_impacts' => array(),
            'theme_impacts' => array(),
            'recommendations' => array()
        );
        
        // Calculate overall impact
        $impact_analysis['overall_impact'] = self::calculate_overall_impact( $assets, $render_blocking_analysis );
        
        // Calculate individual asset impacts
        $impact_analysis['asset_impacts'] = self::calculate_asset_impacts( $assets, $render_blocking_analysis );
        
        // Group impacts by plugin/theme
        $impact_analysis['plugin_impacts'] = self::calculate_plugin_impacts( $impact_analysis['asset_impacts'] );
        $impact_analysis['theme_impacts'] = self::calculate_theme_impacts( $impact_analysis['asset_impacts'] );
        
        // Generate performance recommendations
        $impact_analysis['recommendations'] = self::generate_performance_recommendations( $impact_analysis );
        
        return $impact_analysis;
    }
    
    /**
     * Calculate overall performance impact
     *
     * @param array $assets Assets data
     * @param array $render_blocking_analysis Render blocking analysis
     * @return array Overall impact metrics
     */
    private static function calculate_overall_impact( $assets, $render_blocking_analysis ) {
        $overall = array(
            'total_assets' => 0,
            'total_size' => 0,
            'blocking_assets' => 0,
            'blocking_size' => 0,
            'estimated_load_time' => 0,
            'estimated_render_delay' => 0,
            'performance_score' => 100,
            'impact_level' => 'low'
        );
        
        // Count all assets and calculate total size
        foreach ( $assets as $asset_type => $asset_list ) {
            foreach ( $asset_list as $asset ) {
                $overall['total_assets']++;
                $overall['total_size'] += $asset['size'];
            }
        }
        
        // Count blocking assets
        if ( isset( $render_blocking_analysis['blocking_scripts'] ) ) {
            foreach ( $render_blocking_analysis['blocking_scripts'] as $script ) {
                $overall['blocking_assets']++;
                $overall['blocking_size'] += $script['size'];
            }
        }
        
        if ( isset( $render_blocking_analysis['blocking_styles'] ) ) {
            foreach ( $render_blocking_analysis['blocking_styles'] as $style ) {
                $overall['blocking_assets']++;
                $overall['blocking_size'] += $style['size'];
            }
        }
        
        // Estimate load time based on asset size and count
        $overall['estimated_load_time'] = self::estimate_load_time( $overall['total_size'], $overall['total_assets'] );
        
        // Estimate render delay from blocking assets
        $overall['estimated_render_delay'] = self::estimate_render_delay( $overall['blocking_size'], $overall['blocking_assets'] );
        
        // Calculate performance score
        $overall['performance_score'] = self::calculate_performance_score( $overall );
        
        // Determine impact level
        $overall['impact_level'] = self::determine_impact_level( $overall['performance_score'] );
        
        return $overall;
    }
    
    /**
     * Calculate individual asset impacts
     *
     * @param array $assets Assets data
     * @param array $render_blocking_analysis Render blocking analysis
     * @return array Individual asset impacts
     */
    private static function calculate_asset_impacts( $assets, $render_blocking_analysis ) {
        $asset_impacts = array();
        
        foreach ( $assets as $asset_type => $asset_list ) {
            foreach ( $asset_list as $handle => $asset ) {
                $impact = self::calculate_single_asset_impact( $asset, $render_blocking_analysis );
                $asset_impacts[ $asset_type ][ $handle ] = array_merge( $asset, $impact );
            }
        }
        
        return $asset_impacts;
    }
    
    /**
     * Calculate impact for a single asset
     *
     * @param array $asset Asset data
     * @param array $render_blocking_analysis Render blocking analysis
     * @return array Asset impact data
     */
    private static function calculate_single_asset_impact( $asset, $render_blocking_analysis ) {
        $impact = array(
            'load_time_impact' => 0,
            'render_delay_impact' => 0,
            'memory_impact' => 0,
            'performance_score' => 100,
            'impact_level' => 'low',
            'optimization_priority' => 'low'
        );
        
        // Calculate load time impact based on size
        $impact['load_time_impact'] = self::calculate_load_time_impact( $asset['size'] );
        
        // Calculate render delay impact if asset is blocking
        if ( self::is_asset_blocking( $asset, $render_blocking_analysis ) ) {
            $impact['render_delay_impact'] = self::calculate_render_delay_impact( $asset['size'] );
        }
        
        // Estimate memory impact
        $impact['memory_impact'] = self::estimate_memory_impact( $asset );
        
        // Calculate performance score for this asset
        $impact['performance_score'] = self::calculate_asset_performance_score( $impact );
        
        // Determine impact level
        $impact['impact_level'] = self::determine_impact_level( $impact['performance_score'] );
        
        // Determine optimization priority
        $impact['optimization_priority'] = self::determine_optimization_priority( $asset, $impact );
        
        return $impact;
    }
    
    /**
     * Check if asset is blocking
     *
     * @param array $asset Asset data
     * @param array $render_blocking_analysis Render blocking analysis
     * @return bool True if asset is blocking
     */
    private static function is_asset_blocking( $asset, $render_blocking_analysis ) {
        $handle = $asset['handle'];
        
        return isset( $render_blocking_analysis['blocking_scripts'][ $handle ] ) ||
               isset( $render_blocking_analysis['blocking_styles'][ $handle ] );
    }
    
    /**
     * Calculate plugin performance impacts
     *
     * @param array $asset_impacts Asset impacts
     * @return array Plugin impacts
     */
    private static function calculate_plugin_impacts( $asset_impacts ) {
        $plugin_impacts = array();
        
        foreach ( $asset_impacts as $asset_type => $assets ) {
            foreach ( $assets as $asset ) {
                if ( $asset['attribution_type'] === 'plugin' ) {
                    $plugin_slug = $asset['attributed_to'];
                    
                    if ( ! isset( $plugin_impacts[ $plugin_slug ] ) ) {
                        $plugin_impacts[ $plugin_slug ] = array(
                            'plugin_name' => $asset['attribution_name'],
                            'plugin_version' => $asset['attribution_version'],
                            'plugin_author' => $asset['attribution_author'],
                            'total_assets' => 0,
                            'total_size' => 0,
                            'total_load_time_impact' => 0,
                            'total_render_delay_impact' => 0,
                            'total_memory_impact' => 0,
                            'performance_score' => 100,
                            'impact_level' => 'low',
                            'assets' => array()
                        );
                    }
                    
                    $plugin_impacts[ $plugin_slug ]['total_assets']++;
                    $plugin_impacts[ $plugin_slug ]['total_size'] += $asset['size'];
                    $plugin_impacts[ $plugin_slug ]['total_load_time_impact'] += $asset['load_time_impact'];
                    $plugin_impacts[ $plugin_slug ]['total_render_delay_impact'] += $asset['render_delay_impact'];
                    $plugin_impacts[ $plugin_slug ]['total_memory_impact'] += $asset['memory_impact'];
                    $plugin_impacts[ $plugin_slug ]['assets'][] = $asset;
                }
            }
        }
        
        // Calculate performance scores for plugins
        foreach ( $plugin_impacts as $plugin_slug => &$plugin_impact ) {
            $plugin_impact['performance_score'] = self::calculate_plugin_performance_score( $plugin_impact );
            $plugin_impact['impact_level'] = self::determine_impact_level( $plugin_impact['performance_score'] );
        }
        
        return $plugin_impacts;
    }
    
    /**
     * Calculate theme performance impacts
     *
     * @param array $asset_impacts Asset impacts
     * @return array Theme impacts
     */
    private static function calculate_theme_impacts( $asset_impacts ) {
        $theme_impacts = array();
        
        foreach ( $asset_impacts as $asset_type => $assets ) {
            foreach ( $assets as $asset ) {
                if ( $asset['attribution_type'] === 'theme' ) {
                    $theme_slug = $asset['attributed_to'];
                    
                    if ( ! isset( $theme_impacts[ $theme_slug ] ) ) {
                        $theme_impacts[ $theme_slug ] = array(
                            'theme_name' => $asset['attribution_name'],
                            'theme_version' => $asset['attribution_version'],
                            'theme_author' => $asset['attribution_author'],
                            'total_assets' => 0,
                            'total_size' => 0,
                            'total_load_time_impact' => 0,
                            'total_render_delay_impact' => 0,
                            'total_memory_impact' => 0,
                            'performance_score' => 100,
                            'impact_level' => 'low',
                            'assets' => array()
                        );
                    }
                    
                    $theme_impacts[ $theme_slug ]['total_assets']++;
                    $theme_impacts[ $theme_slug ]['total_size'] += $asset['size'];
                    $theme_impacts[ $theme_slug ]['total_load_time_impact'] += $asset['load_time_impact'];
                    $theme_impacts[ $theme_slug ]['total_render_delay_impact'] += $asset['render_delay_impact'];
                    $theme_impacts[ $theme_slug ]['total_memory_impact'] += $asset['memory_impact'];
                    $theme_impacts[ $theme_slug ]['assets'][] = $asset;
                }
            }
        }
        
        // Calculate performance scores for themes
        foreach ( $theme_impacts as $theme_slug => &$theme_impact ) {
            $theme_impact['performance_score'] = self::calculate_theme_performance_score( $theme_impact );
            $theme_impact['impact_level'] = self::determine_impact_level( $theme_impact['performance_score'] );
        }
        
        return $theme_impacts;
    }
    
    /**
     * Estimate load time based on asset size and count
     *
     * @param int $total_size Total size in bytes
     * @param int $asset_count Number of assets
     * @return float Estimated load time in milliseconds
     */
    private static function estimate_load_time( $total_size, $asset_count ) {
        // Base calculation: size impact + request overhead
        $size_impact = $total_size / 10000; // ~10KB per 1ms (rough estimate)
        $request_overhead = $asset_count * 50; // ~50ms per request (rough estimate)
        
        return $size_impact + $request_overhead;
    }
    
    /**
     * Estimate render delay from blocking assets
     *
     * @param int $blocking_size Size of blocking assets
     * @param int $blocking_count Number of blocking assets
     * @return float Estimated render delay in milliseconds
     */
    private static function estimate_render_delay( $blocking_size, $blocking_count ) {
        // Blocking assets have more severe impact
        $size_impact = $blocking_size / 5000; // ~5KB per 1ms delay
        $blocking_overhead = $blocking_count * 100; // ~100ms per blocking request
        
        return $size_impact + $blocking_overhead;
    }
    
    /**
     * Calculate load time impact for single asset
     *
     * @param int $size Asset size in bytes
     * @return float Load time impact in milliseconds
     */
    private static function calculate_load_time_impact( $size ) {
        return $size / 10000; // ~10KB per 1ms
    }
    
    /**
     * Calculate render delay impact for single asset
     *
     * @param int $size Asset size in bytes
     * @return float Render delay impact in milliseconds
     */
    private static function calculate_render_delay_impact( $size ) {
        return $size / 5000; // ~5KB per 1ms delay for blocking assets
    }
    
    /**
     * Estimate memory impact for asset
     *
     * @param array $asset Asset data
     * @return int Estimated memory impact in bytes
     */
    private static function estimate_memory_impact( $asset ) {
        // Rough estimate based on asset type and size
        $multiplier = 1;
        
        if ( strpos( $asset['src'], '.js' ) !== false ) {
            $multiplier = 2; // JS typically uses more memory
        } elseif ( strpos( $asset['src'], '.css' ) !== false ) {
            $multiplier = 1.5; // CSS moderate memory usage
        }
        
        return $asset['size'] * $multiplier;
    }
    
    /**
     * Calculate performance score (0-100)
     *
     * @param array $metrics Performance metrics
     * @return int Performance score
     */
    private static function calculate_performance_score( $metrics ) {
        $score = 100;
        
        // Deduct points for load time
        if ( $metrics['estimated_load_time'] > 3000 ) {
            $score -= 30;
        } elseif ( $metrics['estimated_load_time'] > 1000 ) {
            $score -= 20;
        } elseif ( $metrics['estimated_load_time'] > 500 ) {
            $score -= 10;
        }
        
        // Deduct points for render delay
        if ( $metrics['estimated_render_delay'] > 1000 ) {
            $score -= 40;
        } elseif ( $metrics['estimated_render_delay'] > 500 ) {
            $score -= 25;
        } elseif ( $metrics['estimated_render_delay'] > 200 ) {
            $score -= 15;
        }
        
        // Deduct points for asset count
        if ( $metrics['total_assets'] > 50 ) {
            $score -= 20;
        } elseif ( $metrics['total_assets'] > 25 ) {
            $score -= 10;
        }
        
        return max( 0, min( 100, $score ) );
    }
    
    /**
     * Calculate performance score for single asset
     *
     * @param array $impact Asset impact data
     * @return int Performance score
     */
    private static function calculate_asset_performance_score( $impact ) {
        $score = 100;
        
        // Deduct points for load time impact
        if ( $impact['load_time_impact'] > 100 ) {
            $score -= 30;
        } elseif ( $impact['load_time_impact'] > 50 ) {
            $score -= 20;
        } elseif ( $impact['load_time_impact'] > 20 ) {
            $score -= 10;
        }
        
        // Deduct points for render delay impact
        if ( $impact['render_delay_impact'] > 100 ) {
            $score -= 40;
        } elseif ( $impact['render_delay_impact'] > 50 ) {
            $score -= 25;
        } elseif ( $impact['render_delay_impact'] > 20 ) {
            $score -= 15;
        }
        
        return max( 0, min( 100, $score ) );
    }
    
    /**
     * Calculate performance score for plugin
     *
     * @param array $plugin_impact Plugin impact data
     * @return int Performance score
     */
    private static function calculate_plugin_performance_score( $plugin_impact ) {
        $score = 100;
        
        // Deduct points based on total impact
        if ( $plugin_impact['total_load_time_impact'] > 500 ) {
            $score -= 30;
        } elseif ( $plugin_impact['total_load_time_impact'] > 200 ) {
            $score -= 20;
        } elseif ( $plugin_impact['total_load_time_impact'] > 100 ) {
            $score -= 10;
        }
        
        if ( $plugin_impact['total_render_delay_impact'] > 300 ) {
            $score -= 40;
        } elseif ( $plugin_impact['total_render_delay_impact'] > 150 ) {
            $score -= 25;
        } elseif ( $plugin_impact['total_render_delay_impact'] > 50 ) {
            $score -= 15;
        }
        
        // Deduct points for asset count
        if ( $plugin_impact['total_assets'] > 10 ) {
            $score -= 20;
        } elseif ( $plugin_impact['total_assets'] > 5 ) {
            $score -= 10;
        }
        
        return max( 0, min( 100, $score ) );
    }
    
    /**
     * Calculate performance score for theme
     *
     * @param array $theme_impact Theme impact data
     * @return int Performance score
     */
    private static function calculate_theme_performance_score( $theme_impact ) {
        // Similar to plugin scoring but slightly more lenient for themes
        return self::calculate_plugin_performance_score( $theme_impact );
    }
    
    /**
     * Determine impact level based on performance score
     *
     * @param int $score Performance score
     * @return string Impact level
     */
    private static function determine_impact_level( $score ) {
        if ( $score >= 80 ) {
            return 'low';
        } elseif ( $score >= 60 ) {
            return 'medium';
        } elseif ( $score >= 40 ) {
            return 'high';
        } else {
            return 'critical';
        }
    }
    
    /**
     * Determine optimization priority
     *
     * @param array $asset Asset data
     * @param array $impact Impact data
     * @return string Optimization priority
     */
    private static function determine_optimization_priority( $asset, $impact ) {
        // High priority if high impact and easy to optimize
        if ( $impact['impact_level'] === 'high' || $impact['impact_level'] === 'critical' ) {
            if ( $asset['attribution_type'] === 'plugin' ) {
                return 'high'; // Plugin assets are usually easier to optimize
            } else {
                return 'medium';
            }
        }
        
        // Medium priority for moderate impact
        if ( $impact['impact_level'] === 'medium' ) {
            return 'medium';
        }
        
        return 'low';
    }
    
    /**
     * Generate performance recommendations
     *
     * @param array $impact_analysis Impact analysis data
     * @return array Recommendations
     */
    private static function generate_performance_recommendations( $impact_analysis ) {
        $recommendations = array();
        
        // Overall recommendations
        $overall = $impact_analysis['overall_impact'];
        
        if ( $overall['impact_level'] === 'critical' ) {
            $recommendations[] = array(
                'type' => 'critical',
                'category' => 'overall',
                'message' => 'Critical performance issues detected. Immediate optimization required.',
                'priority' => 'high',
                'estimated_savings' => $overall['estimated_render_delay'] * 0.7 . 'ms'
            );
        }
        
        if ( $overall['blocking_assets'] > 5 ) {
            $recommendations[] = array(
                'type' => 'optimization',
                'category' => 'blocking',
                'message' => sprintf( 'You have %d render-blocking assets. Consider optimizing the largest ones first.', $overall['blocking_assets'] ),
                'priority' => 'high',
                'estimated_savings' => $overall['estimated_render_delay'] * 0.5 . 'ms'
            );
        }
        
        // Plugin-specific recommendations
        foreach ( $impact_analysis['plugin_impacts'] as $plugin_slug => $plugin_impact ) {
            if ( $plugin_impact['impact_level'] === 'high' || $plugin_impact['impact_level'] === 'critical' ) {
                $recommendations[] = array(
                    'type' => 'plugin_optimization',
                    'category' => 'plugin',
                    'message' => sprintf( 'Plugin "%s" has high performance impact. Consider optimizing or finding alternatives.', $plugin_impact['plugin_name'] ),
                    'priority' => 'high',
                    'plugin_slug' => $plugin_slug,
                    'estimated_savings' => $plugin_impact['total_render_delay_impact'] * 0.6 . 'ms'
                );
            }
        }
        
        return $recommendations;
    }
    
    /**
     * Get current performance metrics
     *
     * @return array Performance metrics
     */
    public static function get_performance_metrics() {
        return self::$performance_metrics;
    }
    
    /**
     * Get timing data
     *
     * @return array Timing data
     */
    public static function get_timing_data() {
        return self::$timing_data;
    }
    
    /**
     * Format time for display
     *
     * @param float $time Time in milliseconds
     * @return string Formatted time
     */
    public static function format_time( $time ) {
        if ( $time >= 1000 ) {
            return round( $time / 1000, 2 ) . 's';
        } else {
            return round( $time, 2 ) . 'ms';
        }
    }
    
    /**
     * Format file size for display
     *
     * @param int $size Size in bytes
     * @return string Formatted size
     */
    public static function format_file_size( $size ) {
        if ( $size >= 1048576 ) {
            return round( $size / 1048576, 2 ) . ' MB';
        } elseif ( $size >= 1024 ) {
            return round( $size / 1024, 2 ) . ' KB';
        } else {
            return $size . ' B';
        }
    }
    
    /**
     * Reset performance data
     */
    public static function reset_performance_data() {
        self::$performance_metrics = array();
        self::$timing_data = array();
    }
} 