<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Plugin options management class
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * Plugin options management class.
 *
 * Handles all WordPress options for the plugin including database settings,
 * configuration management, and option validation.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Options {

    /**
     * Option names constants
     */
    const SETTINGS_OPTION = 'unplug_settings';
    const DB_VERSION_OPTION = 'unplug_db_version';
    const PLUGIN_VERSION_OPTION = 'unplug_version';
    const USER_TIER_OPTION = 'unplug_user_tier';
    const API_TOKEN_OPTION = 'unplug_api_token';
    const CACHE_OPTION = 'unplug_cache_settings';

    /**
     * Default settings array
     *
     * @since    1.0.0
     * @var      array    $default_settings    Default plugin settings
     */
    private static $default_settings = array(
        'api_token' => '',
        'user_tier' => 'free',
        'scan_timeout' => 300,
        'queue_batch_size' => 5,
        'max_queue_items' => 100,
        'enable_logging' => false,
        'log_level' => 'error',
        'cleanup_retention_days' => 30,
        'auto_cleanup_enabled' => true,
        'conflict_detection_enabled' => true,
        'progress_update_interval' => 5,
        'api_cache_duration' => 3600,
        'last_cleanup_run' => 0,
        'installation_date' => '',
        'first_activation' => true
    );

    /**
     * Database configuration defaults
     *
     * @since    1.0.0
     * @var      array    $default_db_config    Default database configuration
     */
    private static $default_db_config = array(
        'version' => '1.0',
        'charset' => 'utf8mb4',
        'collate' => 'utf8mb4_unicode_ci',
        'tables_created' => false,
        'last_upgrade' => '',
        'migration_log' => array()
    );

    /**
     * Cache settings defaults
     *
     * @since    1.0.0
     * @var      array    $default_cache_settings    Default cache configuration
     */
    private static $default_cache_settings = array(
        'plugin_list_cache_duration' => 300,
        'scan_result_cache_duration' => 1800,
        'api_validation_cache_duration' => 3600,
        'enable_object_cache' => true,
        'enable_transient_cache' => true
    );

    /**
     * Initialize all plugin options with defaults
     *
     * @since    1.0.0
     */
    public static function init_options() {
        // Set installation date if this is first activation
        $settings = self::get_settings();
        if ( $settings['first_activation'] ) {
            $settings['installation_date'] = current_time( 'mysql' );
            $settings['first_activation'] = false;
            self::update_settings( $settings );
        }

        // Initialize database configuration
        self::init_db_config();
        
        // Initialize cache settings
        self::init_cache_settings();
        
        // Set initial plugin version
        self::update_plugin_version( UNPLUG_VERSION );
        
        // Set initial user tier
        self::update_user_tier( 'free' );
    }

    /**
     * Get plugin settings with defaults
     *
     * @since    1.0.0
     * @return   array    Plugin settings
     */
    public static function get_settings() {
        $settings = get_option( self::SETTINGS_OPTION, self::$default_settings );
        
        // Merge with defaults to ensure all keys exist
        return wp_parse_args( $settings, self::$default_settings );
    }

    /**
     * Update plugin settings with validation
     *
     * @since    1.0.0
     * @param    array    $settings    Settings to update
     * @return   boolean               True if successful, false otherwise
     */
    public static function update_settings( $settings ) {
        $validated_settings = self::validate_settings( $settings );
        return update_option( self::SETTINGS_OPTION, $validated_settings );
    }

    /**
     * Get specific setting value
     *
     * @since    1.0.0
     * @param    string    $key        Setting key
     * @param    mixed     $default    Default value if key doesn't exist
     * @return   mixed                 Setting value
     */
    public static function get_setting( $key, $default = null ) {
        $settings = self::get_settings();
        
        if ( isset( $settings[$key] ) ) {
            return $settings[$key];
        }
        
        return $default !== null ? $default : ( isset( self::$default_settings[$key] ) ? self::$default_settings[$key] : null );
    }

    /**
     * Update specific setting value
     *
     * @since    1.0.0
     * @param    string    $key      Setting key
     * @param    mixed     $value    Setting value
     * @return   boolean             True if successful, false otherwise
     */
    public static function update_setting( $key, $value ) {
        $settings = self::get_settings();
        $settings[$key] = $value;
        
        return self::update_settings( $settings );
    }

    /**
     * Get database configuration
     *
     * @since    1.0.0
     * @return   array    Database configuration
     */
    public static function get_db_config() {
        return get_option( 'unplug_db_config', self::$default_db_config );
    }

    /**
     * Update database configuration
     *
     * @since    1.0.0
     * @param    array    $config    Database configuration
     * @return   boolean             True if successful, false otherwise
     */
    public static function update_db_config( $config ) {
        $validated_config = wp_parse_args( $config, self::$default_db_config );
        return update_option( 'unplug_db_config', $validated_config );
    }

    /**
     * Get cache settings
     *
     * @since    1.0.0
     * @return   array    Cache settings
     */
    public static function get_cache_settings() {
        return get_option( self::CACHE_OPTION, self::$default_cache_settings );
    }

    /**
     * Update cache settings
     *
     * @since    1.0.0
     * @param    array    $settings    Cache settings
     * @return   boolean               True if successful, false otherwise
     */
    public static function update_cache_settings( $settings ) {
        $validated_settings = wp_parse_args( $settings, self::$default_cache_settings );
        return update_option( self::CACHE_OPTION, $validated_settings );
    }

    /**
     * Get plugin version
     *
     * @since    1.0.0
     * @return   string    Plugin version
     */
    public static function get_plugin_version() {
        return get_option( self::PLUGIN_VERSION_OPTION, UNPLUG_VERSION );
    }

    /**
     * Update plugin version
     *
     * @since    1.0.0
     * @param    string    $version    Plugin version
     * @return   boolean               True if successful, false otherwise
     */
    public static function update_plugin_version( $version ) {
        return update_option( self::PLUGIN_VERSION_OPTION, sanitize_text_field( $version ) );
    }

    /**
     * Get database version
     *
     * @since    1.0.0
     * @return   string    Database version
     */
    public static function get_db_version() {
        return get_option( self::DB_VERSION_OPTION, '0' );
    }

    /**
     * Update database version
     *
     * @since    1.0.0
     * @param    string    $version    Database version
     * @return   boolean               True if successful, false otherwise
     */
    public static function update_db_version( $version ) {
        return update_option( self::DB_VERSION_OPTION, sanitize_text_field( $version ) );
    }

    /**
     * Get user tier
     *
     * @since    1.0.0
     * @return   string    User tier (free, pro, agency)
     */
    public static function get_user_tier() {
        return get_option( self::USER_TIER_OPTION, 'free' );
    }

    /**
     * Update user tier
     *
     * @since    1.0.0
     * @param    string    $tier    User tier
     * @return   boolean            True if successful, false otherwise
     */
    public static function update_user_tier( $tier ) {
        $valid_tiers = array( 'free', 'pro', 'agency' );
        $tier = in_array( $tier, $valid_tiers ) ? $tier : 'free';
        
        return update_option( self::USER_TIER_OPTION, $tier );
    }

    /**
     * Get API token (decrypted)
     *
     * @since    1.0.0
     * @return   string    API token
     */
    public static function get_api_token() {
        // Load security class
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-security.php';
        
        return UNPLUG_Security::retrieve_api_token();
    }

    /**
     * Update API token (encrypted storage)
     *
     * @since    1.0.0
     * @param    string    $token    API token
     * @return   boolean             True if successful, false otherwise
     */
    public static function update_api_token( $token ) {
        // Load security class
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-security.php';
        
        // Sanitize the token first
        $sanitized_token = sanitize_text_field( $token );
        
        if ( empty( $sanitized_token ) ) {
            // Delete token if empty
            return UNPLUG_Security::delete_api_token();
        }
        
        // Store encrypted token
        return UNPLUG_Security::store_api_token( $sanitized_token );
    }

    /**
     * Get scan timeout setting
     *
     * @since    1.0.0
     * @return   int    Scan timeout in seconds
     */
    public static function get_scan_timeout() {
        return self::get_setting( 'scan_timeout', 120 );
    }

    /**
     * Get cleanup retention days setting
     *
     * @since    1.0.0
     * @return   int    Cleanup retention days
     */
    public static function get_cleanup_retention_days() {
        return self::get_setting( 'cleanup_retention_days', 30 );
    }

    /**
     * Get enable logging setting
     *
     * @since    1.0.0
     * @return   bool    Whether logging is enabled
     */
    public static function get_enable_logging() {
        return self::get_setting( 'enable_logging', false );
    }

    /**
     * Validate settings array
     *
     * @since    1.0.0
     * @param    array    $settings    Settings to validate
     * @return   array                 Validated settings
     */
    private static function validate_settings( $settings ) {
        $validated = array();
        
        foreach ( self::$default_settings as $key => $default_value ) {
            if ( isset( $settings[$key] ) ) {
                switch ( $key ) {
                    case 'scan_timeout':
                    case 'queue_batch_size':
                    case 'max_queue_items':
                    case 'cleanup_retention_days':
                    case 'progress_update_interval':
                    case 'api_cache_duration':
                    case 'last_cleanup_run':
                        $validated[$key] = absint( $settings[$key] );
                        break;
                        
                    case 'enable_logging':
                    case 'auto_cleanup_enabled':
                    case 'conflict_detection_enabled':
                    case 'first_activation':
                        $validated[$key] = (bool) $settings[$key];
                        break;
                        
                    case 'user_tier':
                        $valid_tiers = array( 'free', 'pro', 'agency' );
                        $validated[$key] = in_array( $settings[$key], $valid_tiers ) ? $settings[$key] : 'free';
                        break;
                        
                    case 'log_level':
                        $valid_levels = array( 'debug', 'info', 'warning', 'error' );
                        $validated[$key] = in_array( $settings[$key], $valid_levels ) ? $settings[$key] : 'error';
                        break;
                        
                    default:
                        $validated[$key] = sanitize_text_field( $settings[$key] );
                        break;
                }
            } else {
                $validated[$key] = $default_value;
            }
        }
        
        return $validated;
    }

    /**
     * Initialize database configuration
     *
     * @since    1.0.0
     */
    private static function init_db_config() {
        $config = self::get_db_config();
        
        if ( ! isset( $config['version'] ) ) {
            $config['version'] = '1.0';
            $config['last_upgrade'] = current_time( 'mysql' );
            self::update_db_config( $config );
        }
    }

    /**
     * Initialize cache settings
     *
     * @since    1.0.0
     */
    private static function init_cache_settings() {
        $cache_settings = self::get_cache_settings();
        
        if ( empty( $cache_settings ) ) {
            self::update_cache_settings( self::$default_cache_settings );
        }
    }

    /**
     * Delete all plugin options
     *
     * @since    1.0.0
     */
    public static function delete_all_options() {
        delete_option( self::SETTINGS_OPTION );
        delete_option( self::DB_VERSION_OPTION );
        delete_option( self::PLUGIN_VERSION_OPTION );
        delete_option( self::USER_TIER_OPTION );
        delete_option( self::API_TOKEN_OPTION );
        delete_option( self::CACHE_OPTION );
        delete_option( 'unplug_db_config' );
        
        // Delete encryption-related options
        delete_option( 'unplug_api_token_encrypted' );
        delete_option( 'unplug_encryption_key' );
    }

    /**
     * Get all plugin options for export/debugging
     *
     * @since    1.0.0
     * @return   array    All plugin options
     */
    public static function get_all_options() {
        // Load security class to check encryption status
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-security.php';
        
        return array(
            'settings' => self::get_settings(),
            'db_config' => self::get_db_config(),
            'cache_settings' => self::get_cache_settings(),
            'plugin_version' => self::get_plugin_version(),
            'db_version' => self::get_db_version(),
            'user_tier' => self::get_user_tier(),
            'api_token' => self::get_api_token() ? '[HIDDEN]' : '',
            'api_token_encrypted' => UNPLUG_Security::is_api_token_encrypted() ? 'YES' : 'NO'
        );
    }

    /**
     * Get the current license status array (from the last validation)
     *
     * @since 1.0.0
     * @return array
     */
    public static function get_license_status() {
        return get_option('unplug_license_status', array());
    }

    /**
     * Check if a feature is allowed for the current tier
     *
     * @since 1.0.0
     * @param string $feature Feature name (e.g., 'pro', 'agency')
     * @return bool
     */
    public static function is_feature_allowed( $feature ) {
        $tier = self::get_user_tier();
        if ( $feature === 'free' ) return true;
        if ( $feature === 'pro' ) return in_array( $tier, array( 'pro', 'agency' ), true );
        if ( $feature === 'agency' ) return $tier === 'agency';
        return false;
    }
} 