<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Database migration and upgrade handling class
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * Database migration and upgrade handling class.
 *
 * Handles version-based database migrations, schema updates, rollback capabilities,
 * and ensures data integrity during plugin upgrades.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Migrator {

    /**
     * Current database version
     */
    const CURRENT_DB_VERSION = '1.0';

    /**
     * Migration definitions
     *
     * @since    1.0.0
     * @var      array    $migrations    Array of migration definitions
     */
    private static $migrations = array(
        '1.0' => array(
            'description' => 'Initial database schema creation',
            'up' => 'migrate_to_1_0',
            'down' => 'rollback_from_1_0'
        )
        // Future migrations will be added here
        // '1.1' => array(
        //     'description' => 'Add new field to scan_results table',
        //     'up' => 'migrate_to_1_1',
        //     'down' => 'rollback_from_1_1'
        // )
    );

    /**
     * Run database migrations
     *
     * @since    1.0.0
     * @return   boolean    True on success, false on failure
     */
    public static function run_migrations() {
        // Load required classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';

        $current_version = UNPLUG_Options::get_db_version();
        $target_version = self::CURRENT_DB_VERSION;

        // If versions match, no migration needed
        if ( version_compare( $current_version, $target_version, '=' ) ) {
            return true;
        }

        // Start migration process
        $migration_start_time = microtime( true );
        $migration_log = array();

        try {
            // Create backup before migration
            $backup_result = self::create_migration_backup( $current_version );
            if ( ! $backup_result['success'] ) {
                throw new Exception( 'Failed to create migration backup: ' . $backup_result['error'] );
            }

            $migration_log[] = array(
                'timestamp' => current_time( 'mysql' ),
                'action' => 'backup_created',
                'details' => $backup_result['backup_id']
            );

            // Get migrations to run
            $migrations_to_run = self::get_migrations_to_run( $current_version, $target_version );

            foreach ( $migrations_to_run as $version => $migration ) {
                $migration_log[] = array(
                    'timestamp' => current_time( 'mysql' ),
                    'action' => 'migration_start',
                    'version' => $version,
                    'description' => $migration['description']
                );

                // Run the migration
                $result = self::run_single_migration( $version, $migration );

                if ( ! $result['success'] ) {
                    // Migration failed - attempt rollback
                    $rollback_result = self::rollback_failed_migration( $version, $migration_log );
                    throw new Exception( 'Migration to version ' . $version . ' failed: ' . $result['error'] );
                }

                $migration_log[] = array(
                    'timestamp' => current_time( 'mysql' ),
                    'action' => 'migration_complete',
                    'version' => $version,
                    'execution_time' => $result['execution_time']
                );

                // Update version after successful migration
                UNPLUG_Options::update_db_version( $version );
            }

            // Verify data integrity after all migrations
            $integrity_check = self::verify_data_integrity();
            if ( ! $integrity_check['success'] ) {
                throw new Exception( 'Data integrity check failed: ' . $integrity_check['error'] );
            }

            $migration_log[] = array(
                'timestamp' => current_time( 'mysql' ),
                'action' => 'integrity_verified',
                'details' => $integrity_check['details']
            );

            // Update migration history
            self::update_migration_history( $migration_log, microtime( true ) - $migration_start_time );

            return true;

        } catch ( Exception $e ) {
            // Log error and attempt cleanup
            $migration_log[] = array(
                'timestamp' => current_time( 'mysql' ),
                'action' => 'migration_failed',
                'error' => $e->getMessage()
            );

            self::update_migration_history( $migration_log, microtime( true ) - $migration_start_time );

            // Log error for admin notice
            update_option( 'unplug_migration_error', $e->getMessage() );

            return false;
        }
    }

    /**
     * Get migrations that need to be run
     *
     * @since    1.0.0
     * @param    string    $from_version    Current version
     * @param    string    $to_version      Target version
     * @return   array                      Migrations to run
     */
    private static function get_migrations_to_run( $from_version, $to_version ) {
        $migrations_to_run = array();

        foreach ( self::$migrations as $version => $migration ) {
            if ( version_compare( $version, $from_version, '>' ) && 
                 version_compare( $version, $to_version, '<=' ) ) {
                $migrations_to_run[$version] = $migration;
            }
        }

        // Sort by version
        uksort( $migrations_to_run, 'version_compare' );

        return $migrations_to_run;
    }

    /**
     * Run a single migration
     *
     * @since    1.0.0
     * @param    string    $version      Migration version
     * @param    array     $migration    Migration definition
     * @return   array                   Migration result
     */
    private static function run_single_migration( $version, $migration ) {
        $start_time = microtime( true );

        try {
            if ( method_exists( __CLASS__, $migration['up'] ) ) {
                $result = call_user_func( array( __CLASS__, $migration['up'] ) );
                
                if ( ! $result ) {
                    return array(
                        'success' => false,
                        'error' => 'Migration method returned false'
                    );
                }
            } else {
                return array(
                    'success' => false,
                    'error' => 'Migration method ' . $migration['up'] . ' not found'
                );
            }

            return array(
                'success' => true,
                'execution_time' => microtime( true ) - $start_time
            );

        } catch ( Exception $e ) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }

    /**
     * Migration to version 1.0 (initial schema)
     *
     * @since    1.0.0
     * @return   boolean    True on success, false on failure
     */
    private static function migrate_to_1_0() {
        // Load database class and create tables
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        
        try {
            UNPLUG_Database::create_tables();
            return true;
        } catch ( Exception $e ) {
            // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( 'Unplug Migration 1.0 Error: ' . $e->getMessage() );
            // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            return false;
        }
    }

    /**
     * Rollback from version 1.0
     *
     * @since    1.0.0
     * @return   boolean    True on success, false on failure
     */
    private static function rollback_from_1_0() {
        // Load database class and drop tables
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        
        try {
            UNPLUG_Database::drop_tables();
            return true;
        } catch ( Exception $e ) {
            // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( 'Unplug Rollback 1.0 Error: ' . $e->getMessage() );
            // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            return false;
        }
    }

    /**
     * Create migration backup
     *
     * @since    1.0.0
     * @param    string    $version    Current version
     * @return   array                Backup result
     */
    private static function create_migration_backup( $version ) {
        global $wpdb;

        try {
            // Load required classes
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';

            $backup_id = 'unplug_backup_' . $version . '_' . time();
            $table_names = UNPLUG_Database::get_table_names();
            $backup_data = array();

            foreach ( $table_names as $key => $table_name ) {
                // Check if table exists with caching
                $table_cache_key = 'unplug_table_exists_' . md5( $table_name );
                $table_exists = wp_cache_get( $table_cache_key );
                if ( false === $table_exists ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific table existence check for migration backup
                    $table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
                    wp_cache_set( $table_cache_key, $table_exists, '', 300 ); // Cache for 5 minutes
                }
                
                if ( $table_exists ) {
                    // Get table data
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific backup data retrieval
                    $table_data = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `%s`", $table_name ), ARRAY_A );
                    $backup_data[$key] = $table_data;
                }
            }

            // Store backup data in options table (for small datasets) or file system
            if ( strlen( serialize( $backup_data ) ) < 1000000 ) { // Less than 1MB
                update_option( $backup_id, $backup_data );
            } else {
                // Store in file system for larger backups
                $upload_dir = wp_upload_dir();
                $backup_dir = trailingslashit( $upload_dir['basedir'] ) . 'unplug/backups';
                
                if ( ! file_exists( $backup_dir ) ) {
                    wp_mkdir_p( $backup_dir );
                }
                
                $backup_file = trailingslashit( $backup_dir ) . $backup_id . '.json';
                
                // Initialize WordPress file system
                global $wp_filesystem;
                if ( empty( $wp_filesystem ) ) {
                    require_once ABSPATH . '/wp-admin/includes/file.php';
                    WP_Filesystem();
                }
                
                $wp_filesystem->put_contents( $backup_file, json_encode( $backup_data ) );
                
                // Store file path in options
                update_option( $backup_id . '_file', $backup_file );
            }

            return array(
                'success' => true,
                'backup_id' => $backup_id
            );

        } catch ( Exception $e ) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }

    /**
     * Rollback failed migration
     *
     * @since    1.0.0
     * @param    string    $failed_version    Version that failed
     * @param    array     $migration_log     Migration log
     * @return   boolean                      True on success, false on failure
     */
    private static function rollback_failed_migration( $failed_version, $migration_log ) {
        try {
            // Find the backup to restore from
            $backup_id = null;
            foreach ( $migration_log as $log_entry ) {
                if ( $log_entry['action'] === 'backup_created' ) {
                    $backup_id = $log_entry['details'];
                    break;
                }
            }

            if ( ! $backup_id ) {
                throw new Exception( 'No backup found for rollback' );
            }

            // Restore from backup
            $restore_result = self::restore_from_backup( $backup_id );
            
            if ( ! $restore_result['success'] ) {
                throw new Exception( 'Failed to restore from backup: ' . $restore_result['error'] );
            }

            return true;

        } catch ( Exception $e ) {
            // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( 'Unplug Migration Rollback Error: ' . $e->getMessage() );
            // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            return false;
        }
    }

    /**
     * Restore from backup
     *
     * @since    1.0.0
     * @param    string    $backup_id    Backup identifier
     * @return   array                   Restore result
     */
    private static function restore_from_backup( $backup_id ) {
        global $wpdb;

        try {
            // Load required classes
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';

            // Try to get backup from options first
            $backup_data = get_option( $backup_id );
            
            if ( ! $backup_data ) {
                // Try to get from file system
                $backup_file = get_option( $backup_id . '_file' );
                if ( $backup_file && file_exists( $backup_file ) ) {
                    $backup_data = json_decode( file_get_contents( $backup_file ), true );
                }
            }

            if ( ! $backup_data ) {
                throw new Exception( 'Backup data not found' );
            }

            $table_names = UNPLUG_Database::get_table_names();

            // Restore each table
            foreach ( $backup_data as $key => $table_data ) {
                if ( isset( $table_names[$key] ) ) {
                    $table_name = $table_names[$key];
                    
                    // Truncate table first
                    // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->query( $wpdb->prepare( "TRUNCATE TABLE `%s`", $table_name ) );
                    // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    
                    // Restore data
                    if ( ! empty( $table_data ) ) {
                        foreach ( $table_data as $row ) {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific backup restoration
                            $wpdb->insert( $table_name, $row );
                        }
                    }
                }
            }

            return array( 'success' => true );

        } catch ( Exception $e ) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }

    /**
     * Verify data integrity after migration
     *
     * @since    1.0.0
     * @return   array    Integrity check result
     */
    private static function verify_data_integrity() {
        global $wpdb;

        try {
            // Load required classes
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';

            $table_names = UNPLUG_Database::get_table_names();
            $integrity_issues = array();

            // Check that all required tables exist
            foreach ( $table_names as $key => $table_name ) {
                $table_cache_key = 'unplug_table_exists_' . md5( $table_name );
                $table_exists = wp_cache_get( $table_cache_key );
                if ( false === $table_exists ) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific table existence check
                    $table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
                    wp_cache_set( $table_cache_key, $table_exists, '', 300 ); // Cache for 5 minutes
                }
                if ( ! $table_exists ) {
                    $integrity_issues[] = "Table $table_name does not exist";
                }
            }

            // Check for orphaned data
            if ( isset( $table_names['locations'] ) && isset( $table_names['scan_results'] ) ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific data integrity check
                $orphaned_locations = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM `%s` l
                         LEFT JOIN `%s` s ON l.scan_result_id = s.id
                         WHERE s.id IS NULL",
                        $table_names['locations'],
                        $table_names['scan_results']
                    )
                );
                
                if ( $orphaned_locations > 0 ) {
                    $integrity_issues[] = "$orphaned_locations orphaned location records found";
                }
            }

            // Check for valid status values in queue table
            if ( isset( $table_names['queue'] ) ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific data integrity check
                $invalid_statuses = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM `%s` 
                         WHERE `status` NOT IN ('pending', 'in-progress', 'completed', 'failed', 'cancelled')",
                        $table_names['queue']
                    )
                );
                
                if ( $invalid_statuses > 0 ) {
                    $integrity_issues[] = "$invalid_statuses queue records with invalid status found";
                }
            }

            if ( ! empty( $integrity_issues ) ) {
                return array(
                    'success' => false,
                    'error' => 'Data integrity issues found',
                    'issues' => $integrity_issues
                );
            }

            return array(
                'success' => true,
                'details' => 'All integrity checks passed'
            );

        } catch ( Exception $e ) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }

    /**
     * Update migration history
     *
     * @since    1.0.0
     * @param    array     $migration_log      Migration log entries
     * @param    float     $total_time         Total execution time
     */
    private static function update_migration_history( $migration_log, $total_time ) {
        // Load options class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        $db_config = UNPLUG_Options::get_db_config();
        
        $migration_entry = array(
            'timestamp' => current_time( 'mysql' ),
            'total_execution_time' => round( $total_time, 2 ),
            'log' => $migration_log
        );

        $db_config['migration_log'][] = $migration_entry;
        
        // Keep only last 10 migration entries
        if ( count( $db_config['migration_log'] ) > 10 ) {
            $db_config['migration_log'] = array_slice( $db_config['migration_log'], -10 );
        }

        UNPLUG_Options::update_db_config( $db_config );
    }

    /**
     * Get migration history
     *
     * @since    1.0.0
     * @return   array    Migration history
     */
    public static function get_migration_history() {
        // Load options class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        $db_config = UNPLUG_Options::get_db_config();
        return isset( $db_config['migration_log'] ) ? $db_config['migration_log'] : array();
    }

    /**
     * Check if migration is needed
     *
     * @since    1.0.0
     * @return   boolean    True if migration is needed, false otherwise
     */
    public static function is_migration_needed() {
        // Load options class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        $current_version = UNPLUG_Options::get_db_version();
        return version_compare( $current_version, self::CURRENT_DB_VERSION, '<' );
    }

    /**
     * Get available migrations
     *
     * @since    1.0.0
     * @return   array    Available migrations
     */
    public static function get_available_migrations() {
        return self::$migrations;
    }

    /**
     * Manual rollback to specific version
     *
     * @since    1.0.0
     * @param    string    $target_version    Version to rollback to
     * @return   boolean                      True on success, false on failure
     */
    public static function rollback_to_version( $target_version ) {
        // Load options class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        $current_version = UNPLUG_Options::get_db_version();
        
        if ( version_compare( $current_version, $target_version, '<=' ) ) {
            return false; // Can't rollback to same or newer version
        }

        // Find migrations to rollback
        $rollback_migrations = array();
        foreach ( self::$migrations as $version => $migration ) {
            if ( version_compare( $version, $target_version, '>' ) && 
                 version_compare( $version, $current_version, '<=' ) ) {
                $rollback_migrations[$version] = $migration;
            }
        }

        // Sort in reverse order for rollback
        krsort( $rollback_migrations, SORT_NATURAL );

        try {
            foreach ( $rollback_migrations as $version => $migration ) {
                if ( method_exists( __CLASS__, $migration['down'] ) ) {
                    $result = call_user_func( array( __CLASS__, $migration['down'] ) );
                    if ( ! $result ) {
                        throw new Exception( 'Rollback from version ' . $version . ' failed' );
                    }
                }
            }

            // Update version
            UNPLUG_Options::update_db_version( $target_version );
            return true;

        } catch ( Exception $e ) {
            // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( 'Unplug Manual Rollback Error: ' . $e->getMessage() );
            // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            return false;
        }
    }
} 