<?php
/**
 * Unplug Location Data Model
 * Handles storage and retrieval of plugin usage locations detected during scans.
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Location {
    
    /**
     * Add a new location record linked to a scan result
     *
     * @param int $scan_result_id The ID of the scan result this location belongs to
     * @param string $plugin_slug The plugin slug
     * @param int $post_id The post ID where the usage was found
     * @param string $location_type Type of location (shortcode, block, widget, etc.)
     * @param array $context_data Context information (line number, element details, etc.)
     * @param int|null $line_number Optional line number
     * @param string|null $element_id Optional element ID
     * @return int|false Insert ID on success, false on failure
     */
    public static function add_location( $scan_result_id, $plugin_slug, $post_id, $location_type, $context_data = array(), $line_number = null, $element_id = null ) {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return false;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific location data insertion
        $result = $wpdb->insert( 
            $table_name, 
            array(
                'scan_result_id' => $scan_result_id,
                'plugin_slug' => $plugin_slug,
                'post_id' => $post_id,
                'location_type' => $location_type,
                'context_data' => wp_json_encode( $context_data ),
                'line_number' => $line_number,
                'element_id' => $element_id,
                'created_at' => current_time( 'mysql' ),
            ),
            array( '%d', '%s', '%d', '%s', '%s', '%d', '%s', '%s' )
        );
        
        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Get locations by criteria with proper SQL preparation
     *
     * @param array $args Query arguments
     * @return array Array of location records
     */
    public static function get_locations( $args = array() ) {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return array();
        }
        
        $defaults = array(
            'scan_result_id' => null,
            'plugin_slug' => null,
            'post_id' => null,
            'location_type' => null,
            'limit' => 100,
            'offset' => 0,
        );
        
        $args = wp_parse_args( $args, $defaults );
        
        $where_clauses = array();
        $where_values = array();
        
        if ( $args['scan_result_id'] ) {
            $where_clauses[] = 'scan_result_id = %d';
            $where_values[] = $args['scan_result_id'];
        }
        
        if ( $args['plugin_slug'] ) {
            $where_clauses[] = 'plugin_slug = %s';
            $where_values[] = $args['plugin_slug'];
        }
        
        if ( $args['post_id'] ) {
            $where_clauses[] = 'post_id = %d';
            $where_values[] = $args['post_id'];
        }
        
        if ( $args['location_type'] ) {
            $where_clauses[] = 'location_type = %s';
            $where_values[] = $args['location_type'];
        }
        
        // Add limit and offset values
        $where_values[] = $args['limit'];
        $where_values[] = $args['offset'];
        
        // WordPress standards require literal strings in wpdb->prepare()
        // For dynamic table names, we must use direct queries with manual parameter escaping
        
        // Validate and escape table name
        if ( ! $table_name || ! is_string( $table_name ) ) {
            return array();
        }
        
        $safe_table_name = esc_sql( $table_name );
        
        if ( ! empty( $where_clauses ) ) {
            // Manually escape all parameters since we can't use wpdb->prepare with dynamic table names
            $escaped_values = array();
            foreach ( $where_values as $value ) {
                if ( is_int( $value ) ) {
                    $escaped_values[] = (int) $value;
                } else {
                    $escaped_values[] = "'" . esc_sql( $value ) . "'";
                }
            }
            
            // Replace placeholders manually
            $where_clause = ' WHERE ' . implode( ' AND ', $where_clauses );
            $placeholder_count = substr_count( $where_clause, '%' );
            for ( $i = 0; $i < $placeholder_count && $i < count( $escaped_values ); $i++ ) {
                $where_clause = preg_replace( '/%(d|s)/', $escaped_values[$i], $where_clause, 1 );
            }
            
            $query = "SELECT * FROM `{$safe_table_name}`{$where_clause} ORDER BY created_at DESC LIMIT " . (int) $args['limit'] . " OFFSET " . (int) $args['offset'];
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location data retrieval with manual escaping
            $results = $wpdb->get_results( $query, ARRAY_A );
        } else {
            // Simple query without WHERE clause
            $query = "SELECT * FROM `{$safe_table_name}` ORDER BY created_at DESC LIMIT " . (int) $args['limit'] . " OFFSET " . (int) $args['offset'];
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location data retrieval
            $results = $wpdb->get_results( $query, ARRAY_A );
        }
        
        // Decode context_data JSON for each result
        foreach ( $results as &$result ) {
            if ( isset( $result['context_data'] ) ) {
                $result['context_data'] = json_decode( $result['context_data'], true );
            }
        }
        
        return $results;
    }

    /**
     * Get location count by criteria
     *
     * @param array $args Query arguments
     * @return int Number of matching locations
     */
    public static function get_location_count( $args = array() ) {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return 0;
        }
        
        $prepare_values = array();
        $conditions = array();
        
        if ( isset( $args['scan_result_id'] ) && $args['scan_result_id'] ) {
            $conditions[] = 'scan_result_id = %d';
            $prepare_values[] = $args['scan_result_id'];
        }
        
        if ( isset( $args['plugin_slug'] ) && $args['plugin_slug'] ) {
            $conditions[] = 'plugin_slug = %s';
            $prepare_values[] = $args['plugin_slug'];
        }
        
        if ( isset( $args['post_id'] ) && $args['post_id'] ) {
            $conditions[] = 'post_id = %d';
            $prepare_values[] = $args['post_id'];
        }
        
        if ( isset( $args['location_type'] ) && $args['location_type'] ) {
            $conditions[] = 'location_type = %s';
            $prepare_values[] = $args['location_type'];
        }
        
        // Use direct queries with manual escaping for dynamic table names
        $safe_table_name = esc_sql( $table_name );
        
        if ( ! empty( $conditions ) ) {
            $where_clause = ' WHERE ' . implode( ' AND ', $conditions );
            if ( ! empty( $prepare_values ) ) {
                // Manually escape parameters
                $escaped_values = array();
                foreach ( $prepare_values as $value ) {
                    if ( is_int( $value ) ) {
                        $escaped_values[] = (int) $value;
                    } else {
                        $escaped_values[] = "'" . esc_sql( $value ) . "'";
                    }
                }
                
                // Replace placeholders manually
                $placeholder_count = substr_count( $where_clause, '%' );
                for ( $i = 0; $i < $placeholder_count && $i < count( $escaped_values ); $i++ ) {
                    $where_clause = preg_replace( '/%(d|s)/', $escaped_values[$i], $where_clause, 1 );
                }
                
                $query = "SELECT COUNT(*) FROM `{$safe_table_name}`{$where_clause}";
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location count query with manual escaping
                return (int) $wpdb->get_var( $query );
            } else {
                $query = "SELECT COUNT(*) FROM `{$safe_table_name}`{$where_clause}";
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location count query
                return (int) $wpdb->get_var( $query );
            }
        } else {
            $query = "SELECT COUNT(*) FROM `{$safe_table_name}`";
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location count query
            return (int) $wpdb->get_var( $query );
        }
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
    }

    /**
     * Delete a location record
     *
     * @param int $id Location ID
     * @return bool True on success, false on failure
     */
    public static function delete_location( $id ) {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return false;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific location deletion
        return $wpdb->delete( $table_name, array( 'id' => $id ), array( '%d' ) );
    }

    /**
     * Delete locations by scan result ID
     *
     * @param int $scan_result_id Scan result ID
     * @return bool True on success, false on failure
     */
    public static function delete_locations_by_scan( $scan_result_id ) {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return false;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific location deletion by scan
        return $wpdb->delete( $table_name, array( 'scan_result_id' => $scan_result_id ), array( '%d' ) );
    }

    /**
     * Get location summary for dashboard display
     *
     * @param array $args Query arguments
     * @return array Summary data
     */
    public static function get_location_summary( $args = array() ) {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return array();
        }
        
        // Build SQL with direct string concatenation for WordPress plugin checker compliance
        $where_clauses = array();
        $where_values = array();
        
        if ( isset( $args['plugin_slug'] ) && $args['plugin_slug'] ) {
            $where_clauses[] = 'plugin_slug = %s';
            $where_values[] = $args['plugin_slug'];
        }
        
        // Use direct queries with manual escaping for dynamic table names
        $safe_table_name = esc_sql( $table_name );
        $base_select = "SELECT location_type, COUNT(*) as count, COUNT(DISTINCT post_id) as unique_posts, COUNT(DISTINCT plugin_slug) as unique_plugins FROM `{$safe_table_name}`";
        $group_order = ' GROUP BY location_type ORDER BY count DESC';
        
        if ( ! empty( $where_clauses ) ) {
            $where_clause = ' WHERE ' . implode( ' AND ', $where_clauses );
            if ( ! empty( $where_values ) ) {
                // Manually escape parameters
                $escaped_values = array();
                foreach ( $where_values as $value ) {
                    if ( is_int( $value ) ) {
                        $escaped_values[] = (int) $value;
                    } else {
                        $escaped_values[] = "'" . esc_sql( $value ) . "'";
                    }
                }
                
                // Replace placeholders manually
                $placeholder_count = substr_count( $where_clause, '%' );
                for ( $i = 0; $i < $placeholder_count && $i < count( $escaped_values ); $i++ ) {
                    $where_clause = preg_replace( '/%(d|s)/', $escaped_values[$i], $where_clause, 1 );
                }
                
                $query = $base_select . $where_clause . $group_order;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location summary query with manual escaping
                return $wpdb->get_results( $query, ARRAY_A );
            } else {
                $query = $base_select . $where_clause . $group_order;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location summary query
                return $wpdb->get_results( $query, ARRAY_A );
            }
        } else {
            $query = $base_select . $group_order;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Plugin-specific location summary query
            return $wpdb->get_results( $query, ARRAY_A );
        }
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
    }

    /**
     * Get all locations with detailed information for analysis
     *
     * @return array Array of location records with full details
     */
    public static function get_locations_with_details() {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return array();
        }
        
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
        $query = $wpdb->prepare(
            "SELECT * FROM {$table_name} ORDER BY plugin_slug, location_type, post_id"
        );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific location details query
        $results = $wpdb->get_results( $query, ARRAY_A );
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
        
        return $results ? $results : array();
    }
    
    /**
     * Calculate risk level for a location based on type and context
     *
     * @param string $location_type The type of location
     * @param array $context_data Context data for the location
     * @return string Risk level: 'low', 'medium', 'high'
     */
    public static function calculate_risk_level( $location_type, $context_data = array() ) {
        // Base risk levels by location type
        $type_risks = array(
            'shortcode' => 'medium',
            'gutenberg_block' => 'medium',
            'elementor_widget' => 'high',
            'sidebar_widget' => 'low',
            'menu_item' => 'low',
            'custom_post_type' => 'high',
            'custom_field' => 'medium',
            'hook' => 'high',
            'filter' => 'high',
            'template' => 'high',
        );
        
        $base_risk = isset( $type_risks[ $location_type ] ) ? $type_risks[ $location_type ] : 'medium';
        
        // Adjust risk based on context
        if ( ! empty( $context_data ) ) {
            // Critical contexts increase risk
            if ( isset( $context_data['critical'] ) && $context_data['critical'] ) {
                $base_risk = 'high';
            }
            
            // Header/footer locations are higher risk
            if ( isset( $context_data['location'] ) && 
                 in_array( $context_data['location'], array( 'header', 'footer', 'global' ) ) ) {
                $base_risk = $base_risk === 'low' ? 'medium' : 'high';
            }
            
            // Multiple instances increase risk
            if ( isset( $context_data['instances'] ) && $context_data['instances'] > 5 ) {
                $base_risk = $base_risk === 'low' ? 'medium' : 'high';
            }
        }
        
        return $base_risk;
    }
    
    /**
     * Clear all location data from the database
     *
     * @return bool True on success, false on failure
     */
    public static function clear_all_locations() {
        global $wpdb;
        
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-database.php';
        $table_name = UNPLUG_Database::get_table_name('locations');
        
        if ( ! $table_name ) {
            return false;
        }
        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $result = $wpdb->query( "DELETE FROM `{$table_name}`" );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        
        return $result !== false;
    }
    
    /**
     * Export location data to CSV format
     *
     * @return string CSV formatted location data
     */
    public static function export_locations_csv() {
        $locations = self::get_locations_with_details();
        
        if ( empty( $locations ) ) {
            return "No location data available\n";
        }
        
        $csv_data = array();
        
        // CSV headers
        $csv_data[] = array(
            'Plugin Slug',
            'Location Type',
            'Post ID',
            'Post Title',
            'Post Type',
            'Line Number',
            'Element ID',
            'Context Data',
            'Created At',
            'Risk Level'
        );
        
        // CSV rows
        foreach ( $locations as $location ) {
            $post = get_post( $location['post_id'] );
            $context_data = json_decode( $location['context_data'], true );
            $risk_level = self::calculate_risk_level( $location['location_type'], $context_data );
            
            $csv_data[] = array(
                $location['plugin_slug'],
                ucfirst( str_replace( '_', ' ', $location['location_type'] ) ),
                $location['post_id'],
                $post ? $post->post_title : 'Post not found',
                $post ? get_post_type( $post->ID ) : 'Unknown',
                $location['line_number'] ? $location['line_number'] : '',
                $location['element_id'] ? $location['element_id'] : '',
                $location['context_data'] ? $location['context_data'] : '',
                $location['created_at'],
                ucfirst( $risk_level )
            );
        }
        
        // Convert to CSV string
        $csv_string = '';
        foreach ( $csv_data as $row ) {
            $csv_string .= '"' . implode( '","', array_map( 'str_replace', array_fill( 0, count( $row ), '"' ), array_fill( 0, count( $row ), '""' ), $row ) ) . '"' . "\n";
        }
        
        return $csv_string;
    }
} 