<?php
/**
 * Unplug Licensing Integration
 * Handles communication with the external licensing server for license validation, activation, deactivation, and update checking.
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Licensing {
    const LICENSE_SERVER = 'https://mulberrytech.ca';
    const VALIDATE_ENDPOINT = '/api/license/validate';
    const ACTIVATE_ENDPOINT = '/api/license/activate';
    const DEACTIVATE_ENDPOINT = '/api/license/deactivate';
    const UPDATE_ENDPOINT = '/api/license/check-update';
    const CACHE_KEY = 'unplug_license_validation';
    const CACHE_TTL = 12 * HOUR_IN_SECONDS;

    /**
     * Validate the license key with the licensing server.
     *
     * @param string $license_key
     * @param bool $force_refresh
     * @return array|WP_Error
     */
    public static function validate_license( $license_key, $force_refresh = false ) {
        $rate = self::check_rate_limit('validate_license');
        if ( is_wp_error($rate) ) return $rate;

        if ( empty( $license_key ) ) {
            return new WP_Error( 'no_license_key', __( 'No license key provided.', 'unplug' ) );
        }

        $cache_key = self::CACHE_KEY . '_' . md5( $license_key );
        if ( ! $force_refresh ) {
            $cached = get_transient( $cache_key );
            if ( $cached !== false ) {
                return $cached;
            }
        }

        $response = wp_remote_post( self::LICENSE_SERVER . self::VALIDATE_ENDPOINT, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-Unplug-Plugin' => 'unplug',
                'X-Unplug-Version' => defined('UNPLUG_VERSION') ? UNPLUG_VERSION : '1.0.0',
            ),
            'body' => wp_json_encode( array( 'license_key' => $license_key ) ),
            'timeout' => 8,
            'sslverify' => true,
        ) );

        if ( is_wp_error( $response ) ) {
            // Graceful degradation: use last known valid state if available
            $cached = get_transient( $cache_key );
            if ( $cached !== false ) {
                $cached['error'] = $response->get_error_message();
                $cached['degraded'] = true;
                return $cached;
            }
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( $code !== 200 || ! is_array( $data ) ) {
            $error = isset( $data['error'] ) ? $data['error'] : __( 'Invalid response from license server.', 'unplug' );
            // Graceful fallback instead of fatal: cache minimal degraded state and return it
            $fallback = array(
                'status' => 'valid',
                'tier' => 'free',
                'degraded' => true,
                'error' => is_string($error) ? $error : 'unknown'
            );
            set_transient( $cache_key, $fallback, HOUR_IN_SECONDS );
            return $fallback;
        }

        // Cache the result
        set_transient( $cache_key, $data, self::CACHE_TTL );
        return $data;
    }

    /**
     * Activate the license key with the licensing server.
     *
     * @param string $license_key
     * @return array|WP_Error
     */
    public static function activate_license( $license_key ) {
        $rate = self::check_rate_limit('activate_license');
        if ( is_wp_error($rate) ) return $rate;

        if ( empty( $license_key ) ) {
            return new WP_Error( 'no_license_key', __( 'No license key provided.', 'unplug' ) );
        }
        $response = wp_remote_post( self::LICENSE_SERVER . self::ACTIVATE_ENDPOINT, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-Unplug-Plugin' => 'unplug',
                'X-Unplug-Version' => defined('UNPLUG_VERSION') ? UNPLUG_VERSION : '1.0.0',
            ),
            'body' => wp_json_encode( array( 'license_key' => $license_key ) ),
            'timeout' => 15,
            'sslverify' => true,
        ) );
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( $code !== 200 || ! is_array( $data ) ) {
            $error = isset( $data['error'] ) ? $data['error'] : __( 'Invalid response from license server.', 'unplug' );
            return new WP_Error( 'license_server_error', $error );
        }
        return $data;
    }

    /**
     * Deactivate the license key with the licensing server.
     *
     * @param string $license_key
     * @return array|WP_Error
     */
    public static function deactivate_license( $license_key ) {
        $rate = self::check_rate_limit('deactivate_license');
        if ( is_wp_error($rate) ) return $rate;

        if ( empty( $license_key ) ) {
            return new WP_Error( 'no_license_key', __( 'No license key provided.', 'unplug' ) );
        }
        $response = wp_remote_post( self::LICENSE_SERVER . self::DEACTIVATE_ENDPOINT, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-Unplug-Plugin' => 'unplug',
                'X-Unplug-Version' => defined('UNPLUG_VERSION') ? UNPLUG_VERSION : '1.0.0',
            ),
            'body' => wp_json_encode( array( 'license_key' => $license_key ) ),
            'timeout' => 15,
            'sslverify' => true,
        ) );
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( $code !== 200 || ! is_array( $data ) ) {
            $error = isset( $data['error'] ) ? $data['error'] : __( 'Invalid response from license server.', 'unplug' );
            return new WP_Error( 'license_server_error', $error );
        }
        return $data;
    }

    /**
     * Check for plugin updates via the licensing server.
     *
     * @param string $license_key
     * @return array|WP_Error
     */
    public static function check_update( $license_key ) {
        $rate = self::check_rate_limit('check_update');
        if ( is_wp_error($rate) ) return $rate;

        if ( empty( $license_key ) ) {
            return new WP_Error( 'no_license_key', __( 'No license key provided.', 'unplug' ) );
        }
        $response = wp_remote_post( self::LICENSE_SERVER . self::UPDATE_ENDPOINT, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-Unplug-Plugin' => 'unplug',
                'X-Unplug-Version' => defined('UNPLUG_VERSION') ? UNPLUG_VERSION : '1.0.0',
            ),
            'body' => wp_json_encode( array( 'license_key' => $license_key ) ),
            'timeout' => 15,
            'sslverify' => true,
        ) );
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        if ( $code !== 200 || ! is_array( $data ) ) {
            $error = isset( $data['error'] ) ? $data['error'] : __( 'Invalid response from license server.', 'unplug' );
            return new WP_Error( 'license_server_error', $error );
        }
        return $data;
    }

    /**
     * Check and enforce rate limiting for API calls
     *
     * @param string $key Unique key for the API endpoint
     * @param int $max_requests Maximum allowed requests per window
     * @param int $window_seconds Window duration in seconds
     * @return bool|WP_Error True if allowed, WP_Error if rate limited
     */
    public static function check_rate_limit($key = 'license_api', $max_requests = 10, $window_seconds = 3600) {
        $transient_key = 'unplug_rate_' . $key;
        $data = get_transient($transient_key);
        $now = time();
        if (!$data || $now > $data['window_start'] + $window_seconds) {
            $data = ['count' => 1, 'window_start' => $now];
            set_transient($transient_key, $data, $window_seconds);
            return true;
        }
        if ($data['count'] < $max_requests) {
            $data['count']++;
            set_transient($transient_key, $data, $window_seconds - ($now - $data['window_start']));
            return true;
        }
        return new WP_Error('rate_limited', __('API rate limit exceeded. Please try again later.', 'unplug'));
    }
} 