<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
/**
 * Gutenberg Block Analyzer - Advanced block analysis system
 *
 * This class provides comprehensive analysis of Gutenberg blocks including
 * proper parsing, relationship tracking, source detection, and dependency mapping.
 *
 * @link       https://github.com/stnchrch/unplug-wp
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * Advanced Gutenberg block analysis class.
 *
 * This class enhances the basic block detection with comprehensive parsing
 * using WordPress core functions, relationship tracking, and source detection.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Your Name <email@example.com>
 */
class UNPLUG_Gutenberg_Block_Analyzer {

    /**
     * Known block namespaces and their sources.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $known_namespaces    Known block namespaces.
     */
    private $known_namespaces = array(
        'core/' => 'WordPress Core',
        'woocommerce/' => 'WooCommerce',
        'jetpack/' => 'Jetpack',
        'gravityforms/' => 'Gravity Forms',
        'yoast-seo/' => 'Yoast SEO',
        'acf/' => 'Advanced Custom Fields',
        'kadence/' => 'Kadence Blocks',
        'ultimate-blocks/' => 'Ultimate Blocks',
        'atomic-blocks/' => 'Atomic Blocks',
        'coblocks/' => 'CoBlocks',
        'stackable/' => 'Stackable',
        'generateblocks/' => 'GenerateBlocks',
        'genesis-blocks/' => 'Genesis Blocks',
        'otter-blocks/' => 'Otter Blocks',
        'editorskit/' => 'EditorsKit',
        'getwid/' => 'Getwid',
        'kioken/' => 'Kioken Blocks',
        'advanced-gutenberg/' => 'Advanced Gutenberg',
        'qubely/' => 'Qubely',
        'blocks-kit/' => 'Blocks Kit',
        'premium-blocks-for-gutenberg/' => 'Premium Blocks for Gutenberg',
        'essential-blocks/' => 'Essential Blocks',
        'gutenberg-blocks/' => 'Gutenberg Blocks',
        'block-options/' => 'Block Options',
        'editor-blocks/' => 'Editor Blocks'
    );

    /**
     * Block registry cache.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $block_registry    Block registry cache.
     */
    private $block_registry = null;

    /**
     * Plugin data cache.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $plugin_data    Plugin data cache.
     */
    private $plugin_data = null;

    /**
     * Block analysis cache.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $analysis_cache    Block analysis cache.
     */
    private $analysis_cache = array();

    /**
     * Initialize the analyzer.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->initialize_block_registry();
        $this->load_plugin_data();
    }

    /**
     * Analyze all blocks in a post.
     *
     * @since    1.0.0
     * @param    int    $post_id    The post ID to analyze.
     * @return   array              Complete block analysis results.
     */
    public function analyze_post_content( $post_id ) {
        $cache_key = 'block_analysis_' . $post_id;
        
        if ( isset( $this->analysis_cache[$cache_key] ) ) {
            return $this->analysis_cache[$cache_key];
        }

        $post = get_post( $post_id );
        if ( ! $post || empty( $post->post_content ) ) {
            return array();
        }

        // Use WordPress core parse_blocks function for proper parsing
        $blocks = parse_blocks( $post->post_content );
        $results = $this->analyze_blocks_comprehensive( $blocks );

        // Cache the results
        $this->analysis_cache[$cache_key] = $results;

        return $results;
    }

    /**
     * Comprehensive block analysis with relationship tracking.
     *
     * @since    1.0.0
     * @param    array  $blocks  Array of blocks to analyze.
     * @return   array           Complete analysis results.
     */
    public function analyze_blocks_comprehensive( $blocks ) {
        $results = array(
            'blocks' => array(),
            'relationships' => array(),
            'sources' => array(),
            'custom_blocks' => array(),
            'reusable_blocks' => array(),
            'summary' => array(
                'total_blocks' => 0,
                'custom_blocks' => 0,
                'core_blocks' => 0,
                'third_party_blocks' => 0,
                'max_depth' => 0,
                'reusable_blocks' => 0
            )
        );

        $this->process_blocks_recursive( $blocks, $results );
        $this->build_source_summary( $results );
        $this->calculate_block_statistics( $results );

        return $results;
    }

    /**
     * Process blocks recursively with relationship tracking.
     *
     * @since    1.0.0
     * @access   private
     * @param    array  $blocks    Array of blocks to process.
     * @param    array  $results   Results array to populate.
     * @param    string $parent    Parent block ID.
     * @param    int    $depth     Current depth level.
     */
    private function process_blocks_recursive( $blocks, &$results, $parent = null, $depth = 0 ) {
        foreach ( $blocks as $index => $block ) {
            if ( empty( $block['blockName'] ) ) {
                continue; // Skip empty blocks
            }

            $block_id = $this->generate_block_id( $block, $index, $depth );
            
            // Comprehensive block analysis
            $block_info = $this->analyze_individual_block( $block, $block_id, $parent, $depth );
            
            // Add to results
            $results['blocks'][$block_id] = $block_info;

            // Track relationships
            if ( $parent ) {
                $results['relationships'][] = array(
                    'parent' => $parent,
                    'child' => $block_id,
                    'depth' => $depth
                );
            }

            // Update depth tracking
            $results['summary']['max_depth'] = max( $results['summary']['max_depth'], $depth );

            // Handle reusable blocks
            if ( $block['blockName'] === 'core/block' && ! empty( $block['attrs']['ref'] ) ) {
                $reusable_analysis = $this->process_reusable_block( $block['attrs']['ref'], $block_id, $depth );
                if ( $reusable_analysis ) {
                    $results['reusable_blocks'][$block_id] = $reusable_analysis;
                    $results['summary']['reusable_blocks']++;
                    
                    // Process reusable block content
                    $this->process_blocks_recursive( 
                        $reusable_analysis['blocks'], 
                        $results, 
                        $block_id, 
                        $depth + 1 
                    );
                }
            }

            // Process inner blocks recursively
            if ( ! empty( $block['innerBlocks'] ) ) {
                $this->process_blocks_recursive( 
                    $block['innerBlocks'], 
                    $results, 
                    $block_id, 
                    $depth + 1 
                );
            }
        }
    }

    /**
     * Analyze an individual block comprehensively.
     *
     * @since    1.0.0
     * @access   private
     * @param    array  $block     Block data.
     * @param    string $block_id  Block ID.
     * @param    string $parent    Parent block ID.
     * @param    int    $depth     Block depth.
     * @return   array             Block analysis results.
     */
    private function analyze_individual_block( $block, $block_id, $parent, $depth ) {
        $block_name = $block['blockName'];
        $is_custom = $this->is_custom_block( $block_name );

        // Basic block information
        $block_info = array(
            'id' => $block_id,
            'name' => $block_name,
            'parent' => $parent,
            'depth' => $depth,
            'is_custom' => $is_custom,
            'is_core' => strpos( $block_name, 'core/' ) === 0,
            'attributes' => $block['attrs'] ?? array(),
            'has_inner_blocks' => ! empty( $block['innerBlocks'] ),
            'inner_block_count' => count( $block['innerBlocks'] ?? array() ),
            'html_content' => $block['innerHTML'] ?? '',
            'fingerprint' => $this->generate_block_fingerprint( $block )
        );

        // Source detection
        $source = $this->detect_comprehensive_block_source( $block_name );
        $block_info['source'] = $source;

        // Registry information
        $registry_info = $this->get_block_registry_info( $block_name );
        if ( $registry_info ) {
            $block_info['registry'] = $registry_info;
        }

        // Asset dependencies
        $dependencies = $this->get_block_dependencies( $block_name );
        if ( $dependencies ) {
            $block_info['dependencies'] = $dependencies;
        }

        // Block variations and capabilities
        $capabilities = $this->analyze_block_capabilities( $block, $block_name );
        $block_info['capabilities'] = $capabilities;

        return $block_info;
    }

    /**
     * Process reusable block content.
     *
     * @since    1.0.0
     * @access   private
     * @param    int    $ref       Reusable block reference ID.
     * @param    string $block_id  Parent block ID.
     * @param    int    $depth     Current depth.
     * @return   array|null        Reusable block analysis or null.
     */
    private function process_reusable_block( $ref, $block_id, $depth ) {
        $reusable_post = get_post( $ref );
        if ( ! $reusable_post || $reusable_post->post_type !== 'wp_block' ) {
            return null;
        }

        $reusable_blocks = parse_blocks( $reusable_post->post_content );
        
        return array(
            'ref' => $ref,
            'title' => $reusable_post->post_title,
            'content' => $reusable_post->post_content,
            'blocks' => $reusable_blocks,
            'parent_block' => $block_id,
            'depth' => $depth
        );
    }

    /**
     * Detect comprehensive block source information.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   array               Source information.
     */
    private function detect_comprehensive_block_source( $block_name ) {
        // Try registry lookup first
        $registry_info = $this->get_block_source_from_registry( $block_name );
        if ( $registry_info ) {
            return $registry_info;
        }

        // Try known namespaces
        $namespace_info = $this->get_source_from_namespace( $block_name );
        if ( $namespace_info ) {
            return $namespace_info;
        }

        // Try asset analysis
        $asset_info = $this->get_source_from_assets( $block_name );
        if ( $asset_info ) {
            return $asset_info;
        }

        // Default unknown source
        return array(
            'type' => 'unknown',
            'name' => 'Unknown',
            'confidence' => 'low',
            'method' => 'none'
        );
    }

    /**
     * Get block source from WordPress registry.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   array|null          Source information or null.
     */
    private function get_block_source_from_registry( $block_name ) {
        if ( ! $this->block_registry ) {
            return null;
        }

        $registry = WP_Block_Type_Registry::get_instance();
        $block_type = $registry->get_registered( $block_name );
        
        if ( ! $block_type || empty( $block_type->file ) ) {
            return null;
        }

        return $this->analyze_file_path( $block_type->file );
    }

    /**
     * Get block source from known namespaces.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   array|null          Source information or null.
     */
    private function get_source_from_namespace( $block_name ) {
        foreach ( $this->known_namespaces as $namespace => $plugin_name ) {
            if ( strpos( $block_name, $namespace ) === 0 ) {
                return array(
                    'type' => $namespace === 'core/' ? 'core' : 'plugin',
                    'name' => $plugin_name,
                    'namespace' => $namespace,
                    'confidence' => 'high',
                    'method' => 'namespace'
                );
            }
        }

        // Try to guess from namespace
        $parts = explode( '/', $block_name );
        if ( count( $parts ) >= 2 ) {
            $potential_plugin = str_replace( '-', ' ', $parts[0] );
            return array(
                'type' => 'plugin',
                'name' => ucwords( $potential_plugin ) . ' (guessed)',
                'namespace' => $parts[0] . '/',
                'confidence' => 'medium',
                'method' => 'namespace_guess'
            );
        }

        return null;
    }

    /**
     * Get block source from asset analysis.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   array|null          Source information or null.
     */
    private function get_source_from_assets( $block_name ) {
        $registry = WP_Block_Type_Registry::get_instance();
        $block_type = $registry->get_registered( $block_name );
        
        if ( ! $block_type ) {
            return null;
        }

        $scripts = array();
        if ( ! empty( $block_type->editor_script ) ) {
            $scripts[] = $block_type->editor_script;
        }
        if ( ! empty( $block_type->script ) ) {
            $scripts[] = $block_type->script;
        }

        global $wp_scripts;
        
        foreach ( $scripts as $script_handle ) {
            if ( isset( $wp_scripts->registered[$script_handle] ) ) {
                $script = $wp_scripts->registered[$script_handle];
                $source = $this->analyze_asset_url( $script->src );
                if ( $source ) {
                    return $source;
                }
            }
        }

        return null;
    }

    /**
     * Analyze file path to determine source.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $file_path  File path to analyze.
     * @return   array              Source information.
     */
    private function analyze_file_path( $file_path ) {
        // Check if it's from a plugin
        $plugin_dir = wp_normalize_path( WP_PLUGIN_DIR );
        if ( strpos( wp_normalize_path( $file_path ), $plugin_dir ) === 0 ) {
            $relative_path = substr( $file_path, strlen( $plugin_dir ) + 1 );
            $plugin_folder = explode( '/', $relative_path )[0];
            
            $plugin_info = $this->get_plugin_info( $plugin_folder );
            if ( $plugin_info ) {
                return array(
                    'type' => 'plugin',
                    'name' => $plugin_info['name'],
                    'file' => $plugin_info['file'],
                    'folder' => $plugin_folder,
                    'confidence' => 'high',
                    'method' => 'file_path'
                );
            }
        }

        // Check if it's from a theme
        $theme_dir = get_theme_root();
        if ( strpos( $file_path, $theme_dir ) === 0 ) {
            $relative_path = substr( $file_path, strlen( $theme_dir ) + 1 );
            $theme_folder = explode( '/', $relative_path )[0];
            
            $theme = wp_get_theme( $theme_folder );
            return array(
                'type' => 'theme',
                'name' => $theme->get( 'Name' ),
                'folder' => $theme_folder,
                'confidence' => 'high',
                'method' => 'file_path'
            );
        }

        return array(
            'type' => 'unknown',
            'file' => $file_path,
            'confidence' => 'low',
            'method' => 'file_path'
        );
    }

    /**
     * Analyze asset URL to determine source.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $url  Asset URL to analyze.
     * @return   array|null   Source information or null.
     */
    private function analyze_asset_url( $url ) {
        // Check for plugin URL pattern
        $plugins_base_url = plugins_url();
        if ( strpos( $url, $plugins_base_url ) === 0 ) {
            $path = str_replace( $plugins_base_url, '', $url );
            $parts = explode( '/', ltrim( $path, '/' ) );
            if ( ! empty( $parts[0] ) ) {
                $plugin_folder = $parts[0];
                $plugin_info = $this->get_plugin_info( $plugin_folder );
                if ( $plugin_info ) {
                    return array(
                        'type' => 'plugin',
                        'name' => $plugin_info['name'],
                        'folder' => $plugin_folder,
                        'confidence' => 'high',
                        'method' => 'asset_url'
                    );
                }
            }
        }

        // Check for theme URL pattern
        if ( strpos( $url, get_theme_root_uri() ) === 0 ) {
            $path = str_replace( get_theme_root_uri(), '', $url );
            $parts = explode( '/', ltrim( $path, '/' ) );
            if ( ! empty( $parts[0] ) ) {
                $theme_folder = $parts[0];
                $theme = wp_get_theme( $theme_folder );
                return array(
                    'type' => 'theme',
                    'name' => $theme->get( 'Name' ),
                    'folder' => $theme_folder,
                    'confidence' => 'high',
                    'method' => 'asset_url'
                );
            }
        }

        return null;
    }

    /**
     * Get plugin information by folder.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $plugin_folder  Plugin folder name.
     * @return   array|null             Plugin information or null.
     */
    private function get_plugin_info( $plugin_folder ) {
        if ( ! $this->plugin_data ) {
            return null;
        }

        foreach ( $this->plugin_data as $plugin_file => $plugin_data ) {
            if ( strpos( $plugin_file, $plugin_folder . '/' ) === 0 ) {
                return array(
                    'name' => $plugin_data['Name'],
                    'file' => $plugin_file,
                    'version' => $plugin_data['Version'],
                    'folder' => $plugin_folder
                );
            }
        }

        return null;
    }

    /**
     * Get block registry information.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   array|null          Registry information or null.
     */
    private function get_block_registry_info( $block_name ) {
        if ( ! $this->block_registry ) {
            return null;
        }

        $registry = WP_Block_Type_Registry::get_instance();
        $block_type = $registry->get_registered( $block_name );
        
        if ( ! $block_type ) {
            return null;
        }

        return array(
            'file' => $block_type->file ?? null,
            'editor_script' => $block_type->editor_script ?? null,
            'editor_style' => $block_type->editor_style ?? null,
            'script' => $block_type->script ?? null,
            'style' => $block_type->style ?? null,
            'render_callback' => is_callable( $block_type->render_callback ) ? 'callable' : null,
            'attributes' => $block_type->attributes ?? array(),
            'supports' => $block_type->supports ?? array()
        );
    }

    /**
     * Get block dependencies.
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   array|null          Dependencies or null.
     */
    private function get_block_dependencies( $block_name ) {
        $registry = WP_Block_Type_Registry::get_instance();
        $block_type = $registry->get_registered( $block_name );
        
        if ( ! $block_type ) {
            return null;
        }

        global $wp_scripts, $wp_styles;
        $dependencies = array();

        // Script dependencies
        if ( ! empty( $block_type->editor_script ) && isset( $wp_scripts->registered[$block_type->editor_script] ) ) {
            $script = $wp_scripts->registered[$block_type->editor_script];
            $dependencies['editor_script'] = array(
                'handle' => $block_type->editor_script,
                'src' => $script->src,
                'deps' => $script->deps,
                'version' => $script->ver
            );
        }

        // Style dependencies
        if ( ! empty( $block_type->editor_style ) && isset( $wp_styles->registered[$block_type->editor_style] ) ) {
            $style = $wp_styles->registered[$block_type->editor_style];
            $dependencies['editor_style'] = array(
                'handle' => $block_type->editor_style,
                'src' => $style->src,
                'deps' => $style->deps,
                'version' => $style->ver
            );
        }

        return ! empty( $dependencies ) ? $dependencies : null;
    }

    /**
     * Analyze block capabilities.
     *
     * @since    1.0.0
     * @access   private
     * @param    array  $block      Block data.
     * @param    string $block_name Block name.
     * @return   array              Block capabilities.
     */
    private function analyze_block_capabilities( $block, $block_name ) {
        $capabilities = array(
            'has_attributes' => ! empty( $block['attrs'] ),
            'has_inner_content' => ! empty( $block['innerHTML'] ),
            'has_inner_blocks' => ! empty( $block['innerBlocks'] ),
            'is_dynamic' => false,
            'supports_alignment' => false,
            'supports_color' => false,
            'supports_typography' => false
        );

        // Check registry for supports
        $registry = WP_Block_Type_Registry::get_instance();
        $block_type = $registry->get_registered( $block_name );
        
        if ( $block_type ) {
            $capabilities['is_dynamic'] = is_callable( $block_type->render_callback );
            
            $supports = $block_type->supports ?? array();
            $capabilities['supports_alignment'] = ! empty( $supports['align'] );
            $capabilities['supports_color'] = ! empty( $supports['color'] );
            $capabilities['supports_typography'] = ! empty( $supports['typography'] );
        }

        return $capabilities;
    }

    /**
     * Generate unique block ID.
     *
     * @since    1.0.0
     * @access   private
     * @param    array  $block  Block data.
     * @param    int    $index  Block index.
     * @param    int    $depth  Block depth.
     * @return   string         Unique block ID.
     */
    private function generate_block_id( $block, $index, $depth ) {
        $attrs_hash = md5( wp_json_encode( $block['attrs'] ?? array() ) );
        return $block['blockName'] . '_' . $attrs_hash . '_' . $index . '_' . $depth;
    }

    /**
     * Generate block fingerprint.
     *
     * @since    1.0.0
     * @access   private
     * @param    array $block  Block data.
     * @return   string        Block fingerprint.
     */
    private function generate_block_fingerprint( $block ) {
        $fingerprint = array(
            'name' => $block['blockName'],
            'attribute_keys' => array_keys( $block['attrs'] ?? array() ),
            'has_inner_blocks' => ! empty( $block['innerBlocks'] ),
            'has_content' => ! empty( $block['innerHTML'] )
        );
        
        return md5( wp_json_encode( $fingerprint ) );
    }

    /**
     * Check if block is custom (non-core).
     *
     * @since    1.0.0
     * @access   private
     * @param    string $block_name  Block name.
     * @return   bool                True if custom block.
     */
    private function is_custom_block( $block_name ) {
        if ( empty( $block_name ) ) {
            return false;
        }
        
        return strpos( $block_name, 'core/' ) !== 0;
    }

    /**
     * Build source summary.
     *
     * @since    1.0.0
     * @access   private
     * @param    array $results  Results array to populate.
     */
    private function build_source_summary( &$results ) {
        $sources = array();
        
        foreach ( $results['blocks'] as $block ) {
            $source = $block['source'];
            $source_key = $source['type'] . '::' . $source['name'];
            
            if ( ! isset( $sources[$source_key] ) ) {
                $sources[$source_key] = array(
                    'type' => $source['type'],
                    'name' => $source['name'],
                    'blocks' => array(),
                    'count' => 0
                );
            }
            
            $sources[$source_key]['blocks'][] = $block['name'];
            $sources[$source_key]['count']++;
        }
        
        $results['sources'] = $sources;
    }

    /**
     * Calculate block statistics.
     *
     * @since    1.0.0
     * @access   private
     * @param    array $results  Results array to populate.
     */
    private function calculate_block_statistics( &$results ) {
        $results['summary']['total_blocks'] = count( $results['blocks'] );
        
        foreach ( $results['blocks'] as $block ) {
            if ( $block['is_core'] ) {
                $results['summary']['core_blocks']++;
            } elseif ( $block['is_custom'] ) {
                $results['summary']['custom_blocks']++;
            } else {
                $results['summary']['third_party_blocks']++;
            }
        }
    }

    /**
     * Initialize block registry.
     *
     * @since    1.0.0
     * @access   private
     */
    private function initialize_block_registry() {
        if ( class_exists( 'WP_Block_Type_Registry' ) ) {
            $this->block_registry = true;
        }
    }

    /**
     * Load plugin data.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_plugin_data() {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $this->plugin_data = get_plugins();
    }

    /**
     * Get comprehensive block analysis for a site.
     *
     * @since    1.0.0
     * @param    array $options  Analysis options.
     * @return   array           Site-wide block analysis.
     */
    public function analyze_site_blocks( $options = array() ) {
        $default_options = array(
            'post_types' => array( 'post', 'page' ),
            'posts_per_page' => 50,
            'include_custom_post_types' => true,
            'include_reusable_blocks' => true
        );
        
        $options = wp_parse_args( $options, $default_options );
        
        if ( $options['include_custom_post_types'] ) {
            $custom_post_types = get_post_types( array( 'public' => true, '_builtin' => false ) );
            $options['post_types'] = array_merge( $options['post_types'], $custom_post_types );
        }
        
        $site_analysis = array(
            'posts_analyzed' => 0,
            'blocks_found' => array(),
            'sources' => array(),
            'custom_blocks' => array(),
            'summary' => array()
        );
        
        $query_args = array(
            'post_type' => $options['post_types'],
            'post_status' => 'publish',
            'posts_per_page' => $options['posts_per_page'],
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false
        );
        
        $query = new WP_Query( $query_args );
        
        while ( $query->have_posts() ) {
            $query->the_post();
            $post_id = get_the_ID();
            
            $post_analysis = $this->analyze_post_content( $post_id );
            
            if ( ! empty( $post_analysis['blocks'] ) ) {
                $site_analysis['posts_analyzed']++;
                
                foreach ( $post_analysis['blocks'] as $block ) {
                    $block_name = $block['name'];
                    
                    if ( ! isset( $site_analysis['blocks_found'][$block_name] ) ) {
                        $site_analysis['blocks_found'][$block_name] = array(
                            'count' => 0,
                            'posts' => array(),
                            'source' => $block['source']
                        );
                    }
                    
                    $site_analysis['blocks_found'][$block_name]['count']++;
                    
                    if ( ! in_array( $post_id, $site_analysis['blocks_found'][$block_name]['posts'] ) ) {
                        $site_analysis['blocks_found'][$block_name]['posts'][] = $post_id;
                    }
                    
                    if ( $block['is_custom'] ) {
                        $site_analysis['custom_blocks'][$block_name] = $block['source'];
                    }
                }
                
                // Merge sources
                foreach ( $post_analysis['sources'] as $source_key => $source_data ) {
                    if ( ! isset( $site_analysis['sources'][$source_key] ) ) {
                        $site_analysis['sources'][$source_key] = $source_data;
                    } else {
                        $site_analysis['sources'][$source_key]['count'] += $source_data['count'];
                        $site_analysis['sources'][$source_key]['blocks'] = array_merge(
                            $site_analysis['sources'][$source_key]['blocks'],
                            $source_data['blocks']
                        );
                    }
                }
            }
        }
        
        wp_reset_postdata();
        
        // Calculate summary statistics
        $site_analysis['summary'] = array(
            'total_block_types' => count( $site_analysis['blocks_found'] ),
            'custom_block_types' => count( $site_analysis['custom_blocks'] ),
            'total_sources' => count( $site_analysis['sources'] ),
            'posts_with_blocks' => $site_analysis['posts_analyzed']
        );
        
        return $site_analysis;
    }

    /**
     * Clear analysis cache.
     *
     * @since    1.0.0
     */
    public function clear_cache() {
        $this->analysis_cache = array();
    }

    /**
     * Get cached analysis results.
     *
     * @since    1.0.0
     * @param    int $post_id  Post ID.
     * @return   array|null    Cached results or null.
     */
    public function get_cached_analysis( $post_id ) {
        $cache_key = 'block_analysis_' . $post_id;
        return $this->analysis_cache[$cache_key] ?? null;
    }
} 