<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Fired during plugin deactivation
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * Fired during plugin deactivation.
 *
 * This class defines all code necessary to run during the plugin's deactivation.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Deactivator {

    /**
     * Short Description. (use period)
     *
     * Long Description.
     *
     * @since    1.0.0
     */
    public static function deactivate() {
        // Clear any scheduled events
        self::clear_scheduled_events();
        
        // Clear transients
        self::clear_transients();
        
        // Cleanup temporary files
        self::cleanup_temporary_files();
    }

    /**
     * Clear all scheduled events.
     *
     * @since    1.0.0
     */
    private static function clear_scheduled_events() {
        // Load cleanup class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-cleanup.php';
        
        // Unschedule cleanup events using the cleanup class
        UNPLUG_Cleanup::unschedule_cleanup_events();
        
        // Clear any other scheduled events
        wp_clear_scheduled_hook( 'unplug_cleanup_old_scans' );
        wp_clear_scheduled_hook( 'unplug_process_queue' );
        wp_clear_scheduled_hook( 'unplug_validate_api_token' );
    }

    /**
     * Clear all plugin transients.
     *
     * @since    1.0.0
     */
    private static function clear_transients() {
        delete_transient( 'unplug_plugin_list' );
        delete_transient( 'unplug_scan_progress' );
        delete_transient( 'unplug_api_validation' );
        delete_transient( 'unplug_user_tier' );
        delete_transient( 'unplug_queue_status' );
    }

    /**
     * Cleanup temporary files.
     *
     * @since    1.0.0
     */
    private static function cleanup_temporary_files() {
        $upload_dir = wp_upload_dir();
        $unplug_dir = trailingslashit( $upload_dir['basedir'] ) . 'unplug';
        
        if ( is_dir( $unplug_dir ) ) {
            // Remove temporary files but keep the directory structure
            $temp_files = glob( $unplug_dir . '/temp_*' );
            foreach ( $temp_files as $file ) {
                if ( is_file( $file ) ) {
                    wp_delete_file( $file );
                }
            }
            
            // Remove scan cache files
            $cache_files = glob( $unplug_dir . '/cache_*' );
            foreach ( $cache_files as $file ) {
                if ( is_file( $file ) ) {
                    wp_delete_file( $file );
                }
            }
        }
    }
} 