<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Database management and utility class
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * Database management and utility class.
 *
 * Handles database table creation, upgrades, and schema management
 * for the Unplug plugin.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Database {

    /**
     * Database version constant
     */
    const DB_VERSION = '1.0';

    /**
     * Check if database upgrade is needed and run it
     *
     * @since    1.0.0
     */
    public static function check_upgrade() {
        global $wpdb;
        $table = self::get_table_name('queue');
        // Check if next_attempt_at column exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific schema check
        $column = $wpdb->get_results($wpdb->prepare(
            "SHOW COLUMNS FROM `%s` LIKE %s",
            $table,
            'next_attempt_at'
        ));
        if (empty($column)) {
            // Add the next_attempt_at column
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Plugin-specific schema modification
            $wpdb->query( $wpdb->prepare( "ALTER TABLE `%s` ADD COLUMN `next_attempt_at` datetime DEFAULT NULL AFTER `completed_at`", $table ) );
            // Add index for next_attempt_at
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Plugin-specific schema modification
            $wpdb->query( $wpdb->prepare( "ALTER TABLE `%s` ADD KEY `next_attempt_at` (`next_attempt_at`)", $table ) );
        }
        // Load migrator class for comprehensive migration handling
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-migrator.php';
        
        // Check if migration is needed
        if ( UNPLUG_Migrator::is_migration_needed() ) {
            // Run comprehensive migration system
            $migration_result = UNPLUG_Migrator::run_migrations();
            
            if ( ! $migration_result ) {
                // Migration failed - log error for admin notice
                // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log( 'Unplug database migration failed. Check migration logs for details.' );
                // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
                
                // Set admin notice flag
                update_option( 'unplug_migration_failed', true );
            } else {
                // Migration successful - clear any previous error flags
                delete_option( 'unplug_migration_failed' );
                delete_option( 'unplug_migration_error' );
            }
        }
    }

    /**
     * Create all plugin database tables
     *
     * @since    1.0.0
     */
    public static function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

        $tables = self::get_table_schemas( $charset_collate );

        foreach ( $tables as $table_name => $sql ) {
            dbDelta( $sql );
        }
    }

    /**
     * Get all table schemas
     *
     * @since    1.0.0
     * @param    string    $charset_collate    The charset collation string
     * @return   array                         Array of table schemas
     */
    private static function get_table_schemas( $charset_collate ) {
        global $wpdb;

        return array(
            'scan_results' => self::get_scan_results_schema( $wpdb->prefix, $charset_collate ),
            'queue' => self::get_queue_schema( $wpdb->prefix, $charset_collate ),
            'locations' => self::get_locations_schema( $wpdb->prefix, $charset_collate )
        );
    }

    /**
     * Get scan results table schema
     *
     * @since    1.0.0
     * @param    string    $prefix             Database table prefix
     * @param    string    $charset_collate    The charset collation string
     * @return   string                        Table creation SQL
     */
    private static function get_scan_results_schema( $prefix, $charset_collate ) {
        $table_name = $prefix . 'unplug_scan_results';
        
        return "CREATE TABLE `$table_name` (
            `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            `plugin_slug` varchar(100) NOT NULL,
            `test_type` varchar(50) NOT NULL,
            `result_data` longtext NOT NULL,
            `confidence_level` tinyint(3) unsigned NOT NULL DEFAULT 0,
            `scan_timestamp` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY  (`id`),
            KEY `plugin_slug` (`plugin_slug`),
            KEY `test_type` (`test_type`),
            KEY `scan_timestamp` (`scan_timestamp`),
            KEY `plugin_test` (`plugin_slug`, `test_type`)
        ) $charset_collate;";
    }

    /**
     * Get queue table schema
     *
     * @since    1.0.0
     * @param    string    $prefix             Database table prefix
     * @param    string    $charset_collate    The charset collation string
     * @return   string                        Table creation SQL
     */
    private static function get_queue_schema( $prefix, $charset_collate ) {
        $table_name = $prefix . 'unplug_queue';
        
        return "CREATE TABLE `$table_name` (
            `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            `plugin_slug` varchar(100) NOT NULL,
            `test_type` varchar(50) NOT NULL,
            `task_data` longtext DEFAULT NULL,
            `status` varchar(20) NOT NULL DEFAULT 'pending',
            `priority` tinyint(3) unsigned NOT NULL DEFAULT 5,
            `attempts` tinyint(3) unsigned NOT NULL DEFAULT 0,
            `max_attempts` tinyint(3) unsigned NOT NULL DEFAULT 3,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `started_at` datetime DEFAULT NULL,
            `completed_at` datetime DEFAULT NULL,
            `next_attempt_at` datetime DEFAULT NULL,
            `error_message` text DEFAULT NULL,
            `progress_data` longtext DEFAULT NULL,
            `progress_percent` int(3) unsigned NOT NULL DEFAULT 0,
            `last_update` datetime DEFAULT NULL,
            PRIMARY KEY  (`id`),
            KEY `status_priority` (`status`, `priority`),
            KEY `plugin_slug` (`plugin_slug`),
            KEY `test_type` (`test_type`),
            KEY `created_at` (`created_at`),
            KEY `next_attempt_at` (`next_attempt_at`)
        ) $charset_collate;";
    }

    /**
     * Get locations table schema
     *
     * @since    1.0.0
     * @param    string    $prefix             Database table prefix
     * @param    string    $charset_collate    The charset collation string
     * @return   string                        Table creation SQL
     */
    private static function get_locations_schema( $prefix, $charset_collate ) {
        $table_name = $prefix . 'unplug_locations';
        
        return "CREATE TABLE `$table_name` (
            `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            `scan_result_id` bigint(20) unsigned NOT NULL,
            `plugin_slug` varchar(100) NOT NULL,
            `post_id` bigint(20) unsigned NOT NULL,
            `location_type` varchar(50) NOT NULL,
            `context_data` text NOT NULL,
            `line_number` int(11) DEFAULT NULL,
            `element_id` varchar(255) DEFAULT NULL,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (`id`),
            KEY `scan_result_id` (`scan_result_id`),
            KEY `plugin_slug` (`plugin_slug`),
            KEY `post_id` (`post_id`),
            KEY `location_type` (`location_type`)
        ) $charset_collate;";
    }

    /**
     * Get table names
     *
     * @since    1.0.0
     * @return   array    Array of table names
     */
    public static function get_table_names() {
        global $wpdb;
        
        return array(
            'scan_results' => $wpdb->prefix . 'unplug_scan_results',
            'queue' => $wpdb->prefix . 'unplug_queue',
            'locations' => $wpdb->prefix . 'unplug_locations'
        );
    }

    /**
     * Get a single table name by key
     *
     * @since    1.0.0
     * @param    string    $key    The table key (e.g., 'queue', 'scan_results', 'locations')
     * @return   string|null       The table name or null if not found
     */
    public static function get_table_name($key) {
        $tables = self::get_table_names();
        return isset($tables[$key]) ? $tables[$key] : null;
    }

    /**
     * Check if tables exist
     *
     * @since    1.0.0
     * @return   boolean    True if all tables exist, false otherwise
     */
    public static function tables_exist() {
        // Check cache first
        $cache_key = 'unplug_tables_exist';
        $tables_exist = wp_cache_get( $cache_key );
        if ( false !== $tables_exist ) {
            return (bool) $tables_exist;
        }
        
        global $wpdb;
        
        $tables = self::get_table_names();
        
        foreach ( $tables as $table_name ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific table existence check
            $result = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
            if ( $result !== $table_name ) {
                wp_cache_set( $cache_key, false, '', 300 ); // Cache for 5 minutes
                return false;
            }
        }
        
        wp_cache_set( $cache_key, true, '', 300 ); // Cache for 5 minutes
        return true;
    }

    /**
     * Drop all plugin tables
     *
     * @since    1.0.0
     */
    public static function drop_tables() {
        global $wpdb;
        
        $tables = self::get_table_names();
        
        foreach ( $tables as $table_name ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Plugin-specific table cleanup
            $wpdb->query( $wpdb->prepare( "DROP TABLE IF EXISTS `%s`", $table_name ) );
        }
    }

    /**
     * Truncate all plugin tables (clear data but keep structure)
     *
     * @since    1.0.0
     */
    public static function truncate_tables() {
        global $wpdb;
        
        $tables = self::get_table_names();
        
        foreach ( $tables as $table_name ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Plugin-specific table cleanup
            $wpdb->query( $wpdb->prepare( "TRUNCATE TABLE `%s`", $table_name ) );
        }
    }

    /**
     * Get database statistics
     *
     * @since    1.0.0
     * @return   array    Array of table statistics
     */
    public static function get_stats() {
        global $wpdb;
        
        $tables = self::get_table_names();
        $stats = array();
        
        foreach ( $tables as $key => $table_name ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific statistics query
            $count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `%s`", $table_name ) );
            $stats[$key] = array(
                'table_name' => $table_name,
                'row_count' => (int) $count
            );
        }
        
        return $stats;
    }
} 