<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Database cleanup and maintenance class
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * Database cleanup and maintenance class.
 *
 * Handles automated cleanup routines for maintaining database performance
 * and storage with configurable retention policies.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Cleanup {

    /**
     * Hook name for scheduled cleanup
     */
    const CLEANUP_HOOK = 'unplug_scheduled_cleanup';

    /**
     * Hook name for queue cleanup
     */
    const QUEUE_CLEANUP_HOOK = 'unplug_queue_cleanup';

    /**
     * Initialize cleanup system
     *
     * @since    1.0.0
     */
    public static function init() {
        // Schedule cleanup events if not already scheduled
        self::schedule_cleanup_events();
        
        // Register cleanup hooks
        add_action( self::CLEANUP_HOOK, array( __CLASS__, 'run_scheduled_cleanup' ) );
        add_action( self::QUEUE_CLEANUP_HOOK, array( __CLASS__, 'cleanup_queue' ) );
        
        // Register admin cleanup actions
        add_action( 'wp_ajax_unplug_manual_cleanup', array( __CLASS__, 'manual_cleanup_ajax' ) );
    }

    /**
     * Schedule cleanup events
     *
     * @since    1.0.0
     */
    public static function schedule_cleanup_events() {
        // Load options helper
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $auto_cleanup_enabled = UNPLUG_Options::get_setting( 'auto_cleanup_enabled', true );
        
        if ( $auto_cleanup_enabled ) {
            // Schedule daily cleanup if not already scheduled
            if ( ! wp_next_scheduled( self::CLEANUP_HOOK ) ) {
                wp_schedule_event( time(), 'daily', self::CLEANUP_HOOK );
            }
            
            // Schedule hourly queue cleanup if not already scheduled
            if ( ! wp_next_scheduled( self::QUEUE_CLEANUP_HOOK ) ) {
                wp_schedule_event( time(), 'hourly', self::QUEUE_CLEANUP_HOOK );
            }
        }
    }

    /**
     * Unschedule cleanup events
     *
     * @since    1.0.0
     */
    public static function unschedule_cleanup_events() {
        wp_clear_scheduled_hook( self::CLEANUP_HOOK );
        wp_clear_scheduled_hook( self::QUEUE_CLEANUP_HOOK );
    }

    /**
     * Run scheduled cleanup
     *
     * @since    1.0.0
     */
    public static function run_scheduled_cleanup() {
        // Load required classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $start_time = microtime( true );
        $cleanup_results = array();
        
        // Update last cleanup run timestamp
        UNPLUG_Options::update_setting( 'last_cleanup_run', time() );
        
        // Run all cleanup procedures
        $cleanup_results['scan_results'] = self::cleanup_old_scan_results();
        $cleanup_results['queue_items'] = self::cleanup_expired_queue_items();
        $cleanup_results['orphaned_locations'] = self::cleanup_orphaned_locations();
        $cleanup_results['old_transients'] = self::cleanup_old_transients();
        
        $end_time = microtime( true );
        $cleanup_results['execution_time'] = round( ( $end_time - $start_time ), 2 );
        $cleanup_results['timestamp'] = current_time( 'mysql' );
        
        // Log cleanup results if logging is enabled
        if ( UNPLUG_Options::get_setting( 'enable_logging', false ) ) {
            self::log_cleanup_results( $cleanup_results );
        }
        
        return $cleanup_results;
    }

    /**
     * Cleanup old scan results
     *
     * @since    1.0.0
     * @return   int    Number of deleted records
     */
    public static function cleanup_old_scan_results() {
        global $wpdb;
        
        // Load options helper
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        
        $retention_days = UNPLUG_Options::get_setting( 'cleanup_retention_days', 30 );
        $table_names = UNPLUG_Database::get_table_names();
        
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );
        
        // This direct query is safe: admin-only, on-demand cleanup. Caching not needed.
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific cleanup operation
        $deleted = $wpdb->query( $wpdb->prepare(
            "DELETE FROM `%s` WHERE `scan_timestamp` < %s",
            $table_names['scan_results'], $cutoff_date
        ) );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        return (int) $deleted;
    }

    /**
     * Cleanup expired queue items
     *
     * @since    1.0.0
     * @return   int    Number of deleted records
     */
    public static function cleanup_expired_queue_items() {
        global $wpdb;
        
        // Load required classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        
        $table_names = UNPLUG_Database::get_table_names();
        
        // Delete completed/failed items older than 7 days
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( '-7 days' ) );
        
        // This direct query is safe: admin-only, on-demand cleanup. Caching not needed.
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup operation
        $deleted = $wpdb->query( $wpdb->prepare(
            "DELETE FROM `%s` 
             WHERE `status` IN ('completed', 'failed', 'cancelled') 
             AND `completed_at` < %s",
            $table_names['queue'], $cutoff_date
        ) );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        // Reset stuck jobs (running for more than 1 hour)
        $stuck_cutoff = gmdate( 'Y-m-d H:i:s', strtotime( '-1 hour' ) );
        
        // This direct query is safe: admin-only, on-demand maintenance. Caching not needed.
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup operation
        $reset_count = $wpdb->query( $wpdb->prepare(
            "UPDATE `%s` 
             SET `status` = 'pending', `started_at` = NULL, `error_message` = 'Reset due to timeout'
             WHERE `status` = 'in-progress' 
             AND `started_at` < %s",
            $table_names['queue'], $stuck_cutoff
        ) );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        return (int) $deleted + (int) $reset_count;
    }

    /**
     * Cleanup orphaned location records
     *
     * @since    1.0.0
     * @return   int    Number of deleted records
     */
    public static function cleanup_orphaned_locations() {
        global $wpdb;
        
        // Load required classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        
        $table_names = UNPLUG_Database::get_table_names();
        
        // Delete location records that don't have corresponding scan results
        // This direct query is safe: admin-only, on-demand cleanup. Caching not needed.
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup operation
        $deleted = $wpdb->query( $wpdb->prepare(
            "DELETE l FROM `%s` l
             LEFT JOIN `%s` s ON l.scan_result_id = s.id
             WHERE s.id IS NULL",
            $table_names['locations'], $table_names['scan_results']
        ) );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        return (int) $deleted;
    }

    /**
     * Cleanup old transients
     *
     * @since    1.0.0
     * @return   int    Number of deleted transients
     */
    public static function cleanup_old_transients() {
        global $wpdb;
        
        $count = 0;
        
        // Get all unplug transients
        // This direct query is safe: admin-only, on-demand cleanup. Caching not needed.
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup operation
        $transients = $wpdb->get_results(
            "SELECT option_name FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_unplug_%' 
             OR option_name LIKE '_transient_timeout_unplug_%'"
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        foreach ( $transients as $transient ) {
            $transient_name = str_replace( array( '_transient_', '_transient_timeout_' ), '', $transient->option_name );
            
            // Check if transient has expired
            if ( false === get_transient( $transient_name ) ) {
                delete_transient( $transient_name );
                $count++;
            }
        }
        
        return $count;
    }

    /**
     * Manual cleanup via AJAX
     *
     * @since    1.0.0
     */
    public static function manual_cleanup_ajax() {
        // Load security class
        require_once plugin_dir_path( __FILE__ ) . 'class-unplug-security.php';
        
        // Use centralized security system
        UNPLUG_Security::verify_ajax_request( 'admin', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'cleanup', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_manual_cleanup', 'Manual cleanup requested' );
        
        // Verify nonce and sanitize input
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'unplug_cleanup' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'unplug' ) ) );
            return;
        }
        
        if ( ! isset( $_POST['cleanup_type'] ) ) {
            wp_send_json_error( array( 'message' => __( 'Cleanup type is required.', 'unplug' ) ) );
            return;
        }
        
        $cleanup_type = sanitize_text_field( wp_unslash( $_POST['cleanup_type'] ) );
        $result = 0;
        
        switch ( $cleanup_type ) {
            case 'scan_results':
                $result = self::cleanup_old_scan_results();
                break;
            case 'queue':
                $result = self::cleanup_expired_queue_items();
                break;
            case 'locations':
                $result = self::cleanup_orphaned_locations();
                break;
            case 'transients':
                $result = self::cleanup_old_transients();
                break;
            case 'all':
                $results = self::run_scheduled_cleanup();
                $result = array_sum( array_filter( $results, 'is_numeric' ) );
                break;
            default:
                wp_die( esc_html( __( 'Invalid cleanup type', 'unplug' ) ) );
        }
        
        wp_send_json_success( array(
            'deleted_count' => $result,
            // translators: %d: number of records processed
                            'message' => sprintf( 
                    /* translators: %d: number of records processed during cleanup */
                    __( 'Cleanup completed. %d records processed.', 'unplug' ), 
                    $result 
                )
        ) );
    }

    /**
     * Get cleanup statistics
     *
     * @since    1.0.0
     * @return   array    Array of cleanup statistics
     */
    public static function get_cleanup_stats() {
        // Check cache first for admin stats
        $cache_key = 'unplug_cleanup_stats';
        $cached_stats = wp_cache_get( $cache_key );
        if ( false !== $cached_stats ) {
            return $cached_stats;
        }
        
        global $wpdb;
        
        // Load required classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $table_names = UNPLUG_Database::get_table_names();
        $retention_days = UNPLUG_Options::get_setting( 'scan_retention_days', 30 );
        $cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( "-{$retention_days} days" ) );
        
        $stats = array();
        
        // Count old scan results with caching
        $scan_cache_key = 'unplug_old_scan_results_' . md5( $cutoff_date );
        $stats['old_scan_results'] = wp_cache_get( $scan_cache_key );
        if ( false === $stats['old_scan_results'] ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup analysis, results are cached
            $stats['old_scan_results'] = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM `%s` WHERE `scan_timestamp` < %s",
                $table_names['scan_results'], $cutoff_date
            ) );
            wp_cache_set( $scan_cache_key, $stats['old_scan_results'], '', 300 ); // Cache for 5 minutes
        }
        
        // Count expired queue items with caching
        $queue_cutoff = gmdate( 'Y-m-d H:i:s', strtotime( '-7 days' ) );
        $queue_cache_key = 'unplug_expired_queue_items_' . md5( $queue_cutoff );
        $stats['expired_queue_items'] = wp_cache_get( $queue_cache_key );
        if ( false === $stats['expired_queue_items'] ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup analysis, results are cached
            $stats['expired_queue_items'] = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM `%s` 
                 WHERE `status` IN ('completed', 'failed', 'cancelled') 
                 AND `completed_at` < %s",
                $table_names['queue'], $queue_cutoff
            ) );
            wp_cache_set( $queue_cache_key, $stats['expired_queue_items'], '', 300 ); // Cache for 5 minutes
        }
        
        // Count orphaned locations with caching
        $locations_cache_key = 'unplug_orphaned_locations';
        $stats['orphaned_locations'] = wp_cache_get( $locations_cache_key );
        if ( false === $stats['orphaned_locations'] ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific cleanup analysis, results are cached
            $stats['orphaned_locations'] = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM `%s` l
                 LEFT JOIN `%s` s ON l.scan_result_id = s.id
                 WHERE s.id IS NULL",
                $table_names['locations'], $table_names['scan_results']
            ) );
            wp_cache_set( $locations_cache_key, $stats['orphaned_locations'], '', 300 ); // Cache for 5 minutes
        }
        
        // Count old transients with caching
        $transients_cache_key = 'unplug_old_transients';
        $stats['old_transients'] = wp_cache_get( $transients_cache_key );
        if ( false === $stats['old_transients'] ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin-specific transient cleanup analysis, results are cached
            $stats['old_transients'] = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_unplug_%' 
                 OR option_name LIKE '_transient_timeout_unplug_%'"
            );
            wp_cache_set( $transients_cache_key, $stats['old_transients'], '', 300 ); // Cache for 5 minutes
        }
        
        // Last cleanup run
        $stats['last_cleanup_run'] = UNPLUG_Options::get_setting( 'last_cleanup_run', 0 );
        $stats['last_cleanup_formatted'] = $stats['last_cleanup_run'] ? 
            gmdate( 'Y-m-d H:i:s', $stats['last_cleanup_run'] ) : 
            __( 'Never', 'unplug' );
        
        // Next scheduled cleanup
        $next_cleanup = wp_next_scheduled( self::CLEANUP_HOOK );
        $stats['next_cleanup'] = $next_cleanup ? gmdate( 'Y-m-d H:i:s', $next_cleanup ) : __( 'Not scheduled', 'unplug' );
        
        // Cache the complete stats
        wp_cache_set( $cache_key, $stats, '', 300 ); // Cache for 5 minutes
        
        return $stats;
    }

    /**
     * Log cleanup results
     *
     * @since    1.0.0
     * @param    array    $results    Cleanup results
     */
    private static function log_cleanup_results( $results ) {
        $log_message = sprintf(
            'Unplug Cleanup: Scan Results: %d, Queue Items: %d, Locations: %d, Transients: %d, Time: %s seconds',
            $results['scan_results'],
            $results['queue_items'],
            $results['orphaned_locations'],
            $results['old_transients'],
            $results['execution_time']
        );
        
        // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log( $log_message );
        // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
    }

    /**
     * Force cleanup all data (for debugging/testing)
     *
     * @since    1.0.0
     * @return   array    Cleanup results
     */
    public static function force_cleanup_all() {
        global $wpdb;
        
        // Load required classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
        
        $table_names = UNPLUG_Database::get_table_names();
        $results = array();
        
        // Truncate all tables (removes all data but keeps structure)
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        foreach ( $table_names as $key => $table_name ) {
            $wpdb->query( $wpdb->prepare( "TRUNCATE TABLE `%s`", $table_name ) );
            $results[$key] = 'truncated';
        }
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        // Clear all transients
        $results['transients'] = self::cleanup_old_transients();
        
        return $results;
    }

    /**
     * Get cleanup settings for admin interface
     *
     * @since    1.0.0
     * @return   array    Cleanup settings
     */
    public static function get_cleanup_settings() {
        // Load options helper
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        return array(
            'auto_cleanup_enabled' => UNPLUG_Options::get_setting( 'auto_cleanup_enabled', true ),
            'cleanup_retention_days' => UNPLUG_Options::get_setting( 'cleanup_retention_days', 30 ),
            'enable_logging' => UNPLUG_Options::get_setting( 'enable_logging', false ),
            'last_cleanup_run' => UNPLUG_Options::get_setting( 'last_cleanup_run', 0 )
        );
    }

    /**
     * Update cleanup settings
     *
     * @since    1.0.0
     * @param    array    $settings    New cleanup settings
     * @return   boolean               True if successful, false otherwise
     */
    public static function update_cleanup_settings( $settings ) {
        // Load options helper
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $success = true;
        
        if ( isset( $settings['auto_cleanup_enabled'] ) ) {
            $success &= UNPLUG_Options::update_setting( 'auto_cleanup_enabled', (bool) $settings['auto_cleanup_enabled'] );
            
            // Update scheduled events based on setting
            if ( $settings['auto_cleanup_enabled'] ) {
                self::schedule_cleanup_events();
            } else {
                self::unschedule_cleanup_events();
            }
        }
        
        if ( isset( $settings['cleanup_retention_days'] ) ) {
            $retention_days = absint( $settings['cleanup_retention_days'] );
            $retention_days = max( 1, min( 365, $retention_days ) ); // Between 1 and 365 days
            $success &= UNPLUG_Options::update_setting( 'cleanup_retention_days', $retention_days );
        }
        
        if ( isset( $settings['enable_logging'] ) ) {
            $success &= UNPLUG_Options::update_setting( 'enable_logging', (bool) $settings['enable_logging'] );
        }
        
        return $success;
    }
} 