<?php
/**
 * Unplug Asset Detector
 * Detects and catalogs all WordPress assets (CSS, JS, images) loaded on pages
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Asset_Detector {
    
    /**
     * Collected assets data
     *
     * @var array
     */
    private static $assets = array(
        'scripts' => array(),
        'styles' => array(),
        'images' => array(),
        'fonts' => array()
    );
    
    /**
     * Asset detection hooks initialized
     *
     * @var bool
     */
    private static $hooks_initialized = false;
    
    /**
     * Initialize asset detection hooks
     */
    public static function init() {
        if ( self::$hooks_initialized ) {
            return;
        }
        
        // Hook into WordPress enqueue system
        add_action( 'wp_enqueue_scripts', array( __CLASS__, 'capture_enqueued_assets' ), 999 );
        add_action( 'admin_enqueue_scripts', array( __CLASS__, 'capture_enqueued_assets' ), 999 );
        
        // Hook into output buffer for HTML parsing
        add_action( 'template_redirect', array( __CLASS__, 'start_output_buffer' ), 1 );
        add_action( 'admin_init', array( __CLASS__, 'start_output_buffer' ), 1 );
        
        self::$hooks_initialized = true;
    }
    
    /**
     * Start output buffer to capture HTML for asset parsing
     */
    public static function start_output_buffer() {
        if ( ! self::should_detect_assets() ) {
            return;
        }
        
        ob_start( array( __CLASS__, 'parse_html_assets' ) );
    }
    
    /**
     * Check if asset detection should run
     *
     * @return bool
     */
    private static function should_detect_assets() {
        // Skip if not admin or not doing asset scan
        if ( ! is_admin() && ! isset( $_GET['unplug_asset_scan'] ) ) {
            return false;
        }
        
        // Verify nonce for asset scan requests
        if ( isset( $_GET['unplug_asset_scan'] ) && ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'unplug_asset_scan' ) ) ) {
            return false;
        }
        
        // Skip AJAX requests
        if ( wp_doing_ajax() ) {
            return false;
        }
        
        // Skip REST API requests
        if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Capture assets from WordPress enqueue system
     */
    public static function capture_enqueued_assets() {
        global $wp_scripts, $wp_styles;
        
        // Capture enqueued scripts
        if ( $wp_scripts && ! empty( $wp_scripts->queue ) ) {
            foreach ( $wp_scripts->queue as $handle ) {
                if ( isset( $wp_scripts->registered[ $handle ] ) ) {
                    $script = $wp_scripts->registered[ $handle ];
                    
                    self::$assets['scripts'][ $handle ] = array(
                        'handle' => $handle,
                        'src' => $script->src,
                        'deps' => $script->deps,
                        'ver' => $script->ver,
                        'in_footer' => isset( $script->extra['group'] ) && $script->extra['group'] === 1,
                        'source' => self::determine_asset_source( $script->src ),
                        'type' => 'enqueued',
                        'size' => self::get_asset_size( $script->src ),
                        'async' => isset( $script->extra['async'] ) && $script->extra['async'],
                        'defer' => isset( $script->extra['defer'] ) && $script->extra['defer']
                    );
                }
            }
        }
        
        // Capture enqueued styles
        if ( $wp_styles && ! empty( $wp_styles->queue ) ) {
            foreach ( $wp_styles->queue as $handle ) {
                if ( isset( $wp_styles->registered[ $handle ] ) ) {
                    $style = $wp_styles->registered[ $handle ];
                    
                    self::$assets['styles'][ $handle ] = array(
                        'handle' => $handle,
                        'src' => $style->src,
                        'deps' => $style->deps,
                        'ver' => $style->ver,
                        'media' => $style->args,
                        'source' => self::determine_asset_source( $style->src ),
                        'type' => 'enqueued',
                        'size' => self::get_asset_size( $style->src ),
                        'critical' => self::is_critical_css( $style->src, $handle )
                    );
                }
            }
        }
    }
    
    /**
     * Parse HTML output for additional assets
     *
     * @param string $html HTML content
     * @return string Modified HTML content
     */
    public static function parse_html_assets( $html ) {
        // Parse script tags
        preg_match_all( '/<script[^>]*src=["\']([^"\']*)["\'][^>]*>/i', $html, $script_matches );
        if ( ! empty( $script_matches[1] ) ) {
            foreach ( $script_matches[1] as $index => $src ) {
                $full_tag = $script_matches[0][ $index ];
                $handle = 'inline-' . md5( $src );
                
                if ( ! isset( self::$assets['scripts'][ $handle ] ) ) {
                    self::$assets['scripts'][ $handle ] = array(
                        'handle' => $handle,
                        'src' => $src,
                        'deps' => array(),
                        'ver' => null,
                        'in_footer' => strpos( $html, $full_tag ) > strpos( $html, '</head>' ),
                        'source' => self::determine_asset_source( $src ),
                        'type' => 'inline',
                        'size' => self::get_asset_size( $src ),
                        'async' => strpos( $full_tag, 'async' ) !== false,
                        'defer' => strpos( $full_tag, 'defer' ) !== false
                    );
                }
            }
        }
        
        // Parse link tags (CSS)
        preg_match_all( '/<link[^>]*href=["\']([^"\']*)["\'][^>]*rel=["\']stylesheet["\'][^>]*>/i', $html, $style_matches );
        if ( ! empty( $style_matches[1] ) ) {
            foreach ( $style_matches[1] as $index => $href ) {
                $full_tag = $style_matches[0][ $index ];
                $handle = 'inline-' . md5( $href );
                
                if ( ! isset( self::$assets['styles'][ $handle ] ) ) {
                    preg_match( '/media=["\']([^"\']*)["\']/', $full_tag, $media_match );
                    $media = ! empty( $media_match[1] ) ? $media_match[1] : 'all';
                    
                    self::$assets['styles'][ $handle ] = array(
                        'handle' => $handle,
                        'src' => $href,
                        'deps' => array(),
                        'ver' => null,
                        'media' => $media,
                        'source' => self::determine_asset_source( $href ),
                        'type' => 'inline',
                        'size' => self::get_asset_size( $href ),
                        'critical' => self::is_critical_css( $href, $handle )
                    );
                }
            }
        }
        
        // Parse image tags
        preg_match_all( '/<img[^>]*src=["\']([^"\']*)["\'][^>]*>/i', $html, $image_matches );
        if ( ! empty( $image_matches[1] ) ) {
            foreach ( $image_matches[1] as $index => $src ) {
                $full_tag = $image_matches[0][ $index ];
                $handle = 'img-' . md5( $src );
                
                if ( ! isset( self::$assets['images'][ $handle ] ) ) {
                    self::$assets['images'][ $handle ] = array(
                        'handle' => $handle,
                        'src' => $src,
                        'source' => self::determine_asset_source( $src ),
                        'type' => 'inline',
                        'size' => self::get_asset_size( $src ),
                        'lazy' => strpos( $full_tag, 'loading="lazy"' ) !== false,
                        'alt' => self::extract_attribute( $full_tag, 'alt' ),
                        'width' => self::extract_attribute( $full_tag, 'width' ),
                        'height' => self::extract_attribute( $full_tag, 'height' )
                    );
                }
            }
        }
        
        // Parse font references
        preg_match_all( '/@font-face[^}]*src:[^}]*url\(["\']?([^"\']*)["\']?\)/i', $html, $font_matches );
        if ( ! empty( $font_matches[1] ) ) {
            foreach ( $font_matches[1] as $src ) {
                $handle = 'font-' . md5( $src );
                
                if ( ! isset( self::$assets['fonts'][ $handle ] ) ) {
                    self::$assets['fonts'][ $handle ] = array(
                        'handle' => $handle,
                        'src' => $src,
                        'source' => self::determine_asset_source( $src ),
                        'type' => 'font',
                        'size' => self::get_asset_size( $src )
                    );
                }
            }
        }
        
        return $html;
    }
    
    /**
     * Determine the source of an asset (plugin, theme, core, external)
     *
     * @param string $src Asset source URL
     * @return string Asset source type
     */
    private static function determine_asset_source( $src ) {
        if ( empty( $src ) ) {
            return 'unknown';
        }
        
        // Convert relative URLs to absolute
        if ( strpos( $src, '//' ) === 0 ) {
            $src = 'https:' . $src;
        } elseif ( strpos( $src, '/' ) === 0 ) {
            $src = home_url( $src );
        }
        
        $site_url = site_url();
        
        // External asset
        if ( strpos( $src, $site_url ) !== 0 ) {
            return 'external';
        }
        
        // Remove site URL to get relative path
        $relative_path = str_replace( $site_url, '', $src );
        
        // Check for plugin assets
        if ( strpos( $relative_path, '/wp-content/plugins/' ) !== false ) {
            preg_match( '/\/wp-content\/plugins\/([^\/]+)/', $relative_path, $matches );
            return ! empty( $matches[1] ) ? 'plugin:' . $matches[1] : 'plugin';
        }
        
        // Check for theme assets
        if ( strpos( $relative_path, '/wp-content/themes/' ) !== false ) {
            preg_match( '/\/wp-content\/themes\/([^\/]+)/', $relative_path, $matches );
            return ! empty( $matches[1] ) ? 'theme:' . $matches[1] : 'theme';
        }
        
        // Check for WordPress core assets
        if ( strpos( $relative_path, '/wp-includes/' ) !== false || strpos( $relative_path, '/wp-admin/' ) !== false ) {
            return 'core';
        }
        
        // Check for uploads
        if ( strpos( $relative_path, '/wp-content/uploads/' ) !== false ) {
            return 'uploads';
        }
        
        return 'unknown';
    }
    
    /**
     * Get asset file size
     *
     * @param string $src Asset source URL
     * @return int File size in bytes, 0 if cannot determine
     */
    private static function get_asset_size( $src ) {
        if ( empty( $src ) ) {
            return 0;
        }
        
        // Convert to absolute path if possible
        $file_path = self::url_to_path( $src );
        
        if ( $file_path && file_exists( $file_path ) ) {
            return filesize( $file_path );
        }
        
        return 0;
    }
    
    /**
     * Convert URL to local file path
     *
     * @param string $url Asset URL
     * @return string|false Local file path or false if cannot convert
     */
    private static function url_to_path( $url ) {
        $site_url = site_url();
        
        if ( strpos( $url, $site_url ) !== 0 ) {
            return false;
        }
        
        $relative_path = str_replace( $site_url, '', $url );
        $file_path = untrailingslashit( ABSPATH ) . '/' . ltrim( $relative_path, '/' );
        
        return $file_path;
    }
    
    /**
     * Check if CSS is critical for above-the-fold rendering
     *
     * @param string $src CSS source URL
     * @param string $handle CSS handle
     * @return bool True if critical CSS
     */
    private static function is_critical_css( $src, $handle ) {
        // Common critical CSS patterns
        $critical_patterns = array(
            'style.css',
            'main.css',
            'theme.css',
            'critical.css',
            'above-fold.css'
        );
        
        // Check handle for critical indicators
        $critical_handles = array(
            'style',
            'main',
            'theme',
            'critical',
            'bootstrap',
            'foundation'
        );
        
        foreach ( $critical_patterns as $pattern ) {
            if ( strpos( $src, $pattern ) !== false ) {
                return true;
            }
        }
        
        foreach ( $critical_handles as $critical_handle ) {
            if ( strpos( $handle, $critical_handle ) !== false ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Extract attribute value from HTML tag
     *
     * @param string $tag HTML tag
     * @param string $attribute Attribute name
     * @return string Attribute value or empty string
     */
    private static function extract_attribute( $tag, $attribute ) {
        preg_match( '/' . $attribute . '=["\']([^"\']*)["\']/', $tag, $matches );
        return ! empty( $matches[1] ) ? $matches[1] : '';
    }
    
    /**
     * Get all detected assets
     *
     * @return array All detected assets
     */
    public static function get_assets() {
        return self::$assets;
    }
    
    /**
     * Get assets by type
     *
     * @param string $type Asset type (scripts, styles, images, fonts)
     * @return array Assets of specified type
     */
    public static function get_assets_by_type( $type ) {
        return isset( self::$assets[ $type ] ) ? self::$assets[ $type ] : array();
    }
    
    /**
     * Get assets by source
     *
     * @param string $source Asset source (plugin, theme, core, external)
     * @return array Assets from specified source
     */
    public static function get_assets_by_source( $source ) {
        $filtered_assets = array();
        
        foreach ( self::$assets as $type => $assets ) {
            foreach ( $assets as $handle => $asset ) {
                if ( strpos( $asset['source'], $source ) === 0 ) {
                    $filtered_assets[ $type ][ $handle ] = $asset;
                }
            }
        }
        
        return $filtered_assets;
    }
    
    /**
     * Get total asset size by type
     *
     * @param string $type Asset type
     * @return int Total size in bytes
     */
    public static function get_total_size_by_type( $type ) {
        $total_size = 0;
        
        if ( isset( self::$assets[ $type ] ) ) {
            foreach ( self::$assets[ $type ] as $asset ) {
                $total_size += $asset['size'];
            }
        }
        
        return $total_size;
    }
    
    /**
     * Reset collected assets
     */
    public static function reset_assets() {
        self::$assets = array(
            'scripts' => array(),
            'styles' => array(),
            'images' => array(),
            'fonts' => array()
        );
    }
    
    /**
     * Start asset detection scan
     */
    public static function start_scan() {
        self::reset_assets();
        self::init();
    }
    
    /**
     * Get asset statistics
     *
     * @return array Asset statistics
     */
    public static function get_statistics() {
        $stats = array(
            'total_assets' => 0,
            'total_size' => 0,
            'by_type' => array(),
            'by_source' => array()
        );
        
        foreach ( self::$assets as $type => $assets ) {
            $type_count = count( $assets );
            $type_size = self::get_total_size_by_type( $type );
            
            $stats['total_assets'] += $type_count;
            $stats['total_size'] += $type_size;
            
            $stats['by_type'][ $type ] = array(
                'count' => $type_count,
                'size' => $type_size
            );
            
            // Count by source
            foreach ( $assets as $asset ) {
                $source = $asset['source'];
                if ( ! isset( $stats['by_source'][ $source ] ) ) {
                    $stats['by_source'][ $source ] = array(
                        'count' => 0,
                        'size' => 0
                    );
                }
                $stats['by_source'][ $source ]['count']++;
                $stats['by_source'][ $source ]['size'] += $asset['size'];
            }
        }
        
        return $stats;
    }
} 