<?php
/**
 * Unplug Asset Attribution
 * Maps detected assets back to their source plugins or themes
 *
 * @since 1.0.0
 * @package Unplug
 * @subpackage Unplug/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class UNPLUG_Asset_Attribution {
    
    /**
     * Plugin asset cache
     *
     * @var array
     */
    private static $plugin_assets_cache = array();
    
    /**
     * Theme asset cache
     *
     * @var array
     */
    private static $theme_assets_cache = array();
    
    /**
     * Asset attribution cache
     *
     * @var array
     */
    private static $attribution_cache = array();
    
    /**
     * Attribute assets to their source plugins/themes
     *
     * @param array $assets Assets from UNPLUG_Asset_Detector
     * @return array Assets with detailed attribution
     */
    public static function attribute_assets( $assets ) {
        $attributed_assets = array();
        
        // Get plugin and theme information
        $plugins = self::get_plugin_info();
        $themes = self::get_theme_info();
        
        foreach ( $assets as $asset_type => $asset_list ) {
            $attributed_assets[ $asset_type ] = array();
            
            foreach ( $asset_list as $handle => $asset ) {
                $attribution = self::determine_detailed_attribution( $asset, $plugins, $themes );
                $attributed_assets[ $asset_type ][ $handle ] = array_merge( $asset, $attribution );
            }
        }
        
        return $attributed_assets;
    }
    
    /**
     * Get detailed plugin information
     *
     * @return array Plugin information
     */
    private static function get_plugin_info() {
        if ( ! empty( self::$plugin_assets_cache ) ) {
            return self::$plugin_assets_cache;
        }
        
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $plugins = get_plugins();
        $plugin_info = array();
        
        foreach ( $plugins as $plugin_file => $plugin_data ) {
            $plugin_slug = dirname( $plugin_file );
            if ( $plugin_slug === '.' ) {
                $plugin_slug = basename( $plugin_file, '.php' );
            }
            
            $plugin_info[ $plugin_slug ] = array(
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'author' => $plugin_data['Author'],
                'description' => $plugin_data['Description'],
                'plugin_file' => $plugin_file,
                'plugin_dir' => plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file ),
                'plugin_url' => plugins_url( '', $plugin_file ),
                'is_active' => is_plugin_active( $plugin_file ),
                'is_network_active' => is_plugin_active_for_network( $plugin_file ),
                'text_domain' => $plugin_data['TextDomain'],
                'domain_path' => $plugin_data['DomainPath']
            );
        }
        
        self::$plugin_assets_cache = $plugin_info;
        return $plugin_info;
    }
    
    /**
     * Get detailed theme information
     *
     * @return array Theme information
     */
    private static function get_theme_info() {
        if ( ! empty( self::$theme_assets_cache ) ) {
            return self::$theme_assets_cache;
        }
        
        $themes = wp_get_themes();
        $theme_info = array();
        
        foreach ( $themes as $theme_slug => $theme_obj ) {
            $theme_info[ $theme_slug ] = array(
                'name' => $theme_obj->get( 'Name' ),
                'version' => $theme_obj->get( 'Version' ),
                'author' => $theme_obj->get( 'Author' ),
                'description' => $theme_obj->get( 'Description' ),
                'theme_dir' => $theme_obj->get_stylesheet_directory(),
                'theme_url' => $theme_obj->get_stylesheet_directory_uri(),
                'is_active' => ( get_stylesheet() === $theme_slug ),
                'is_parent' => ( get_template() === $theme_slug ),
                'parent_theme' => $theme_obj->get( 'Template' ),
                'text_domain' => $theme_obj->get( 'TextDomain' )
            );
        }
        
        self::$theme_assets_cache = $theme_info;
        return $theme_info;
    }
    
    /**
     * Determine detailed attribution for an asset
     *
     * @param array $asset Asset data
     * @param array $plugins Plugin information
     * @param array $themes Theme information
     * @return array Attribution data
     */
    private static function determine_detailed_attribution( $asset, $plugins, $themes ) {
        $attribution = array(
            'attributed_to' => 'unknown',
            'attribution_type' => 'unknown',
            'attribution_name' => 'Unknown',
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 0,
            'attribution_method' => 'unknown',
            'attribution_details' => array()
        );
        
        // Check cache first
        $cache_key = md5( $asset['src'] . $asset['handle'] );
        if ( isset( self::$attribution_cache[ $cache_key ] ) ) {
            return self::$attribution_cache[ $cache_key ];
        }
        
        // Parse asset source for attribution
        $source_analysis = self::analyze_asset_source( $asset['src'] );
        
        if ( $source_analysis['type'] === 'plugin' ) {
            $attribution = self::attribute_to_plugin( $asset, $source_analysis, $plugins );
        } elseif ( $source_analysis['type'] === 'theme' ) {
            $attribution = self::attribute_to_theme( $asset, $source_analysis, $themes );
        } elseif ( $source_analysis['type'] === 'core' ) {
            $attribution = self::attribute_to_core( $asset, $source_analysis );
        } elseif ( $source_analysis['type'] === 'external' ) {
            $attribution = self::attribute_to_external( $asset, $source_analysis );
        }
        
        // Try additional attribution methods if confidence is low
        if ( $attribution['attribution_confidence'] < 70 ) {
            $additional_attribution = self::try_additional_attribution_methods( $asset, $plugins, $themes );
            if ( $additional_attribution['attribution_confidence'] > $attribution['attribution_confidence'] ) {
                $attribution = $additional_attribution;
            }
        }
        
        // Cache the result
        self::$attribution_cache[ $cache_key ] = $attribution;
        
        return $attribution;
    }
    
    /**
     * Analyze asset source URL for attribution clues
     *
     * @param string $src Asset source URL
     * @return array Source analysis
     */
    private static function analyze_asset_source( $src ) {
        $analysis = array(
            'type' => 'unknown',
            'identifier' => '',
            'path' => '',
            'filename' => ''
        );
        
        if ( empty( $src ) ) {
            return $analysis;
        }
        
        // Normalize URL
        $src = self::normalize_url( $src );
        $site_url = site_url();
        
        // External asset
        if ( strpos( $src, $site_url ) !== 0 ) {
            $analysis['type'] = 'external';
            $analysis['identifier'] = wp_parse_url( $src, PHP_URL_HOST );
            return $analysis;
        }
        
        // Get relative path
        $relative_path = str_replace( $site_url, '', $src );
        $analysis['path'] = $relative_path;
        $analysis['filename'] = basename( $relative_path );
        
        // Plugin asset
        if ( preg_match( '/\/wp-content\/plugins\/([^\/]+)/', $relative_path, $matches ) ) {
            $analysis['type'] = 'plugin';
            $analysis['identifier'] = $matches[1];
            return $analysis;
        }
        
        // Theme asset
        if ( preg_match( '/\/wp-content\/themes\/([^\/]+)/', $relative_path, $matches ) ) {
            $analysis['type'] = 'theme';
            $analysis['identifier'] = $matches[1];
            return $analysis;
        }
        
        // Core asset
        if ( strpos( $relative_path, '/wp-includes/' ) !== false || strpos( $relative_path, '/wp-admin/' ) !== false ) {
            $analysis['type'] = 'core';
            $analysis['identifier'] = 'wordpress';
            return $analysis;
        }
        
        // Upload asset
        if ( strpos( $relative_path, '/wp-content/uploads/' ) !== false ) {
            $analysis['type'] = 'uploads';
            $analysis['identifier'] = 'uploads';
            return $analysis;
        }
        
        return $analysis;
    }
    
    /**
     * Attribute asset to plugin
     *
     * @param array $asset Asset data
     * @param array $source_analysis Source analysis
     * @param array $plugins Plugin information
     * @return array Attribution data
     */
    private static function attribute_to_plugin( $asset, $source_analysis, $plugins ) {
        $plugin_slug = $source_analysis['identifier'];
        
        if ( isset( $plugins[ $plugin_slug ] ) ) {
            $plugin = $plugins[ $plugin_slug ];
            
            return array(
                'attributed_to' => $plugin_slug,
                'attribution_type' => 'plugin',
                'attribution_name' => $plugin['name'],
                'attribution_version' => $plugin['version'],
                'attribution_author' => $plugin['author'],
                'attribution_confidence' => 95,
                'attribution_method' => 'path_analysis',
                'attribution_details' => array(
                    'plugin_file' => $plugin['plugin_file'],
                    'is_active' => $plugin['is_active'],
                    'is_network_active' => $plugin['is_network_active'],
                    'text_domain' => $plugin['text_domain']
                )
            );
        }
        
        // Plugin directory exists but no plugin data
        return array(
            'attributed_to' => $plugin_slug,
            'attribution_type' => 'plugin',
            'attribution_name' => ucwords( str_replace( array( '-', '_' ), ' ', $plugin_slug ) ),
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 80,
            'attribution_method' => 'directory_analysis',
            'attribution_details' => array(
                'note' => 'Plugin directory found but no plugin data available'
            )
        );
    }
    
    /**
     * Attribute asset to theme
     *
     * @param array $asset Asset data
     * @param array $source_analysis Source analysis
     * @param array $themes Theme information
     * @return array Attribution data
     */
    private static function attribute_to_theme( $asset, $source_analysis, $themes ) {
        $theme_slug = $source_analysis['identifier'];
        
        if ( isset( $themes[ $theme_slug ] ) ) {
            $theme = $themes[ $theme_slug ];
            
            return array(
                'attributed_to' => $theme_slug,
                'attribution_type' => 'theme',
                'attribution_name' => $theme['name'],
                'attribution_version' => $theme['version'],
                'attribution_author' => $theme['author'],
                'attribution_confidence' => 95,
                'attribution_method' => 'path_analysis',
                'attribution_details' => array(
                    'is_active' => $theme['is_active'],
                    'is_parent' => $theme['is_parent'],
                    'parent_theme' => $theme['parent_theme'],
                    'text_domain' => $theme['text_domain']
                )
            );
        }
        
        // Theme directory exists but no theme data
        return array(
            'attributed_to' => $theme_slug,
            'attribution_type' => 'theme',
            'attribution_name' => ucwords( str_replace( array( '-', '_' ), ' ', $theme_slug ) ),
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 80,
            'attribution_method' => 'directory_analysis',
            'attribution_details' => array(
                'note' => 'Theme directory found but no theme data available'
            )
        );
    }
    
    /**
     * Attribute asset to WordPress core
     *
     * @param array $asset Asset data
     * @param array $source_analysis Source analysis
     * @return array Attribution data
     */
    private static function attribute_to_core( $asset, $source_analysis ) {
        global $wp_version;
        
        return array(
            'attributed_to' => 'wordpress',
            'attribution_type' => 'core',
            'attribution_name' => 'WordPress',
            'attribution_version' => $wp_version,
            'attribution_author' => 'WordPress.org',
            'attribution_confidence' => 100,
            'attribution_method' => 'path_analysis',
            'attribution_details' => array(
                'core_component' => strpos( $source_analysis['path'], '/wp-admin/' ) !== false ? 'admin' : 'includes'
            )
        );
    }
    
    /**
     * Attribute asset to external source
     *
     * @param array $asset Asset data
     * @param array $source_analysis Source analysis
     * @return array Attribution data
     */
    private static function attribute_to_external( $asset, $source_analysis ) {
        $host = $source_analysis['identifier'];
        
        // Try to identify common external services
        $external_services = array(
            'fonts.googleapis.com' => 'Google Fonts',
            'fonts.gstatic.com' => 'Google Fonts',
            'cdnjs.cloudflare.com' => 'Cloudflare CDN',
            'ajax.googleapis.com' => 'Google AJAX Libraries',
            'code.jquery.com' => 'jQuery CDN',
            'maxcdn.bootstrapcdn.com' => 'Bootstrap CDN',
            'use.fontawesome.com' => 'Font Awesome',
            'cdn.jsdelivr.net' => 'jsDelivr CDN',
            'unpkg.com' => 'UNPKG CDN'
        );
        
        $service_name = isset( $external_services[ $host ] ) ? $external_services[ $host ] : $host;
        
        return array(
            'attributed_to' => $host,
            'attribution_type' => 'external',
            'attribution_name' => $service_name,
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 90,
            'attribution_method' => 'domain_analysis',
            'attribution_details' => array(
                'host' => $host,
                'full_url' => $asset['src']
            )
        );
    }
    
    /**
     * Try additional attribution methods
     *
     * @param array $asset Asset data
     * @param array $plugins Plugin information
     * @param array $themes Theme information
     * @return array Attribution data
     */
    private static function try_additional_attribution_methods( $asset, $plugins, $themes ) {
        // Method 1: Handle-based attribution
        $handle_attribution = self::attribute_by_handle( $asset, $plugins, $themes );
        if ( $handle_attribution['attribution_confidence'] > 60 ) {
            return $handle_attribution;
        }
        
        // Method 2: Enqueue hook analysis
        $hook_attribution = self::attribute_by_enqueue_hook( $asset );
        if ( $hook_attribution['attribution_confidence'] > 50 ) {
            return $hook_attribution;
        }
        
        // Method 3: File content analysis
        $content_attribution = self::attribute_by_file_content( $asset, $plugins, $themes );
        if ( $content_attribution['attribution_confidence'] > 40 ) {
            return $content_attribution;
        }
        
        // Return default unknown attribution
        return array(
            'attributed_to' => 'unknown',
            'attribution_type' => 'unknown',
            'attribution_name' => 'Unknown',
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 0,
            'attribution_method' => 'no_method_successful',
            'attribution_details' => array()
        );
    }
    
    /**
     * Attribute asset by handle name
     *
     * @param array $asset Asset data
     * @param array $plugins Plugin information
     * @param array $themes Theme information
     * @return array Attribution data
     */
    private static function attribute_by_handle( $asset, $plugins, $themes ) {
        $handle = $asset['handle'];
        
        // Check for plugin patterns in handle
        foreach ( $plugins as $plugin_slug => $plugin_data ) {
            if ( strpos( $handle, $plugin_slug ) !== false || strpos( $handle, str_replace( '-', '_', $plugin_slug ) ) !== false ) {
                return array(
                    'attributed_to' => $plugin_slug,
                    'attribution_type' => 'plugin',
                    'attribution_name' => $plugin_data['name'],
                    'attribution_version' => $plugin_data['version'],
                    'attribution_author' => $plugin_data['author'],
                    'attribution_confidence' => 70,
                    'attribution_method' => 'handle_analysis',
                    'attribution_details' => array(
                        'handle_match' => $handle
                    )
                );
            }
        }
        
        // Check for theme patterns in handle
        foreach ( $themes as $theme_slug => $theme_data ) {
            if ( strpos( $handle, $theme_slug ) !== false || strpos( $handle, str_replace( '-', '_', $theme_slug ) ) !== false ) {
                return array(
                    'attributed_to' => $theme_slug,
                    'attribution_type' => 'theme',
                    'attribution_name' => $theme_data['name'],
                    'attribution_version' => $theme_data['version'],
                    'attribution_author' => $theme_data['author'],
                    'attribution_confidence' => 70,
                    'attribution_method' => 'handle_analysis',
                    'attribution_details' => array(
                        'handle_match' => $handle
                    )
                );
            }
        }
        
        return array(
            'attributed_to' => 'unknown',
            'attribution_type' => 'unknown',
            'attribution_name' => 'Unknown',
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 0,
            'attribution_method' => 'handle_analysis_failed',
            'attribution_details' => array()
        );
    }
    
    /**
     * Attribute asset by enqueue hook analysis
     *
     * @param array $asset Asset data
     * @return array Attribution data
     */
    private static function attribute_by_enqueue_hook( $asset ) {
        // This would require runtime hook tracking, which is complex
        // For now, return low confidence unknown
        return array(
            'attributed_to' => 'unknown',
            'attribution_type' => 'unknown',
            'attribution_name' => 'Unknown',
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 0,
            'attribution_method' => 'hook_analysis_not_implemented',
            'attribution_details' => array()
        );
    }
    
    /**
     * Attribute asset by file content analysis
     *
     * @param array $asset Asset data
     * @param array $plugins Plugin information
     * @param array $themes Theme information
     * @return array Attribution data
     */
    private static function attribute_by_file_content( $asset, $plugins, $themes ) {
        $file_path = self::url_to_path( $asset['src'] );
        
        if ( ! $file_path || ! file_exists( $file_path ) ) {
            return array(
                'attributed_to' => 'unknown',
                'attribution_type' => 'unknown',
                'attribution_name' => 'Unknown',
                'attribution_version' => '',
                'attribution_author' => '',
                'attribution_confidence' => 0,
                'attribution_method' => 'file_not_found',
                'attribution_details' => array()
            );
        }
        
        // Read first 1KB of file for analysis
        $content = file_get_contents( $file_path, false, null, 0, 1024 );
        
        if ( ! $content ) {
            return array(
                'attributed_to' => 'unknown',
                'attribution_type' => 'unknown',
                'attribution_name' => 'Unknown',
                'attribution_version' => '',
                'attribution_author' => '',
                'attribution_confidence' => 0,
                'attribution_method' => 'file_read_failed',
                'attribution_details' => array()
            );
        }
        
        // Look for plugin/theme identifiers in content
        foreach ( $plugins as $plugin_slug => $plugin_data ) {
            if ( strpos( $content, $plugin_slug ) !== false || strpos( $content, $plugin_data['name'] ) !== false ) {
                return array(
                    'attributed_to' => $plugin_slug,
                    'attribution_type' => 'plugin',
                    'attribution_name' => $plugin_data['name'],
                    'attribution_version' => $plugin_data['version'],
                    'attribution_author' => $plugin_data['author'],
                    'attribution_confidence' => 60,
                    'attribution_method' => 'content_analysis',
                    'attribution_details' => array(
                        'content_match' => 'Found plugin identifier in file content'
                    )
                );
            }
        }
        
        foreach ( $themes as $theme_slug => $theme_data ) {
            if ( strpos( $content, $theme_slug ) !== false || strpos( $content, $theme_data['name'] ) !== false ) {
                return array(
                    'attributed_to' => $theme_slug,
                    'attribution_type' => 'theme',
                    'attribution_name' => $theme_data['name'],
                    'attribution_version' => $theme_data['version'],
                    'attribution_author' => $theme_data['author'],
                    'attribution_confidence' => 60,
                    'attribution_method' => 'content_analysis',
                    'attribution_details' => array(
                        'content_match' => 'Found theme identifier in file content'
                    )
                );
            }
        }
        
        return array(
            'attributed_to' => 'unknown',
            'attribution_type' => 'unknown',
            'attribution_name' => 'Unknown',
            'attribution_version' => '',
            'attribution_author' => '',
            'attribution_confidence' => 0,
            'attribution_method' => 'content_analysis_no_match',
            'attribution_details' => array()
        );
    }
    
    /**
     * Normalize URL for consistent processing
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private static function normalize_url( $url ) {
        // Convert protocol-relative URLs
        if ( strpos( $url, '//' ) === 0 ) {
            $url = 'https:' . $url;
        }
        
        // Convert relative URLs to absolute
        if ( strpos( $url, '/' ) === 0 ) {
            $url = home_url( $url );
        }
        
        return $url;
    }
    
    /**
     * Convert URL to local file path
     *
     * @param string $url Asset URL
     * @return string|false Local file path or false if cannot convert
     */
    private static function url_to_path( $url ) {
        $url = self::normalize_url( $url );
        $site_url = site_url();
        
        if ( strpos( $url, $site_url ) !== 0 ) {
            return false;
        }
        
        $relative_path = str_replace( $site_url, '', $url );
        $file_path = untrailingslashit( ABSPATH ) . '/' . ltrim( $relative_path, '/' );
        
        return $file_path;
    }
    
    /**
     * Get assets grouped by attribution
     *
     * @param array $attributed_assets Attributed assets
     * @return array Assets grouped by attribution
     */
    public static function group_by_attribution( $attributed_assets ) {
        $grouped = array();
        
        foreach ( $attributed_assets as $asset_type => $assets ) {
            foreach ( $assets as $handle => $asset ) {
                $key = $asset['attribution_type'] . ':' . $asset['attributed_to'];
                
                if ( ! isset( $grouped[ $key ] ) ) {
                    $grouped[ $key ] = array(
                        'attribution_type' => $asset['attribution_type'],
                        'attributed_to' => $asset['attributed_to'],
                        'attribution_name' => $asset['attribution_name'],
                        'attribution_version' => $asset['attribution_version'],
                        'attribution_author' => $asset['attribution_author'],
                        'assets' => array()
                    );
                }
                
                $grouped[ $key ]['assets'][ $asset_type ][ $handle ] = $asset;
            }
        }
        
        return $grouped;
    }
    
    /**
     * Get attribution statistics
     *
     * @param array $attributed_assets Attributed assets
     * @return array Attribution statistics
     */
    public static function get_attribution_statistics( $attributed_assets ) {
        $stats = array(
            'total_assets' => 0,
            'attributed_assets' => 0,
            'unattributed_assets' => 0,
            'by_type' => array(),
            'by_confidence' => array(
                'high' => 0,    // 80-100%
                'medium' => 0,  // 50-79%
                'low' => 0      // 0-49%
            )
        );
        
        foreach ( $attributed_assets as $asset_type => $assets ) {
            if ( ! isset( $stats['by_type'][ $asset_type ] ) ) {
                $stats['by_type'][ $asset_type ] = array(
                    'total' => 0,
                    'attributed' => 0,
                    'unattributed' => 0
                );
            }
            
            foreach ( $assets as $asset ) {
                $stats['total_assets']++;
                $stats['by_type'][ $asset_type ]['total']++;
                
                if ( $asset['attribution_confidence'] > 0 ) {
                    $stats['attributed_assets']++;
                    $stats['by_type'][ $asset_type ]['attributed']++;
                    
                    // Confidence categorization
                    if ( $asset['attribution_confidence'] >= 80 ) {
                        $stats['by_confidence']['high']++;
                    } elseif ( $asset['attribution_confidence'] >= 50 ) {
                        $stats['by_confidence']['medium']++;
                    } else {
                        $stats['by_confidence']['low']++;
                    }
                } else {
                    $stats['unattributed_assets']++;
                    $stats['by_type'][ $asset_type ]['unattributed']++;
                }
            }
        }
        
        return $stats;
    }
    
    /**
     * Clear attribution cache
     */
    public static function clear_cache() {
        self::$plugin_assets_cache = array();
        self::$theme_assets_cache = array();
        self::$attribution_cache = array();
    }
} 