<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
/**
 * Activity Scan Engine - Comprehensive plugin usage detection system
 *
 * This class handles scanning WordPress content to detect actual plugin usage
 * across posts, pages, widgets, menus, and other content types.
 *
 * @link       https://github.com/stnchrch/unplug-wp
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/includes
 */

/**
 * The activity scan engine class.
 *
 * This class provides comprehensive plugin usage detection across WordPress
 * content to determine which plugins are actively being used.
 *
 * @since      1.0.0
 * @package    Unplug
 * @subpackage Unplug/includes
 * @author     Your Name <email@example.com>
 */
class UNPLUG_Activity_Scan_Engine {

    /**
     * The database instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Database    $db    The database instance.
     */
    private $db;

    /**
     * The security instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Security    $security    The security instance.
     */
    private $security;

    /**
     * The queue instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Queue    $queue    The queue instance.
     */
    private $queue;

    /**
     * Cached plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $plugin_inventory    The plugin inventory cache.
     */
    private $plugin_inventory;

    /**
     * Activity scan results cache.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $activity_results    The activity scan results cache.
     */
    private $activity_results;

    /**
     * Confidence level thresholds.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $confidence_thresholds    Confidence level thresholds.
     */
    private $confidence_thresholds = array(
        'used' => 75,           // 75% or higher = "Used"
        'possibly_used' => 25,  // 25-74% = "Possibly Used"
        'no_usage' => 0         // 0-24% = "No Usage Found"
    );

    /**
     * Scoring weights for different detection types.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $scoring_weights    Scoring weights for confidence calculation.
     */
    private $scoring_weights = array(
        'shortcode_usage' => 30,
        'block_usage' => 25,
        'widget_usage' => 20,
        'elementor_usage' => 25,
        'cpt_usage' => 15,
        'taxonomy_usage' => 10,
        'menu_usage' => 10,
        'option_usage' => 10,
        'hook_usage' => 5,
        'api_usage' => 5
    );

    /**
     * The Gutenberg block analyzer instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      UNPLUG_Gutenberg_Block_Analyzer    $block_analyzer    The block analyzer instance.
     */
    private $block_analyzer;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    UNPLUG_Database    $db         The database instance.
     * @param    UNPLUG_Security    $security   The security instance.
     * @param    UNPLUG_Queue       $queue      The queue instance.
     */
    public function __construct( $db, $security, $queue ) {
        $this->db = $db;
        $this->security = $security;
        $this->queue = $queue;
        $this->plugin_inventory = null;
        $this->activity_results = null;
        
        // Initialize the Gutenberg block analyzer
        $this->block_analyzer = new UNPLUG_Gutenberg_Block_Analyzer();
    }

    /**
     * Initialize the activity scan engine.
     *
     * @since    1.0.0
     */
    public static function init() {
        // Hook into content save events to invalidate cache
        add_action( 'save_post', array( __CLASS__, 'invalidate_activity_cache' ) );
        add_action( 'delete_post', array( __CLASS__, 'invalidate_activity_cache' ) );
        add_action( 'activated_plugin', array( __CLASS__, 'invalidate_activity_cache' ) );
        add_action( 'deactivated_plugin', array( __CLASS__, 'invalidate_activity_cache' ) );
        
        // Schedule periodic activity scans
        if ( ! wp_next_scheduled( 'unplug_activity_scan' ) ) {
            wp_schedule_event( time(), 'twicedaily', 'unplug_activity_scan' );
        }
        
        add_action( 'unplug_activity_scan', array( __CLASS__, 'schedule_activity_scan' ) );
    }

    /**
     * Perform comprehensive activity scan.
     *
     * @since    1.0.0
     * @param    bool    $force_refresh    Force refresh of cached data.
     * @return   array                     Complete activity scan results.
     */
    public function perform_activity_scan( $force_refresh = false ) {
        if ( $force_refresh || $this->activity_results === null ) {
            $this->activity_results = $this->build_activity_scan_results();
        }
        
        return $this->activity_results;
    }

    /**
     * Build comprehensive activity scan results.
     *
     * @since    1.0.0
     * @access   private
     * @return   array    Complete activity scan results.
     */
    private function build_activity_scan_results() {
        $start_time = microtime( true );
        
        // Get plugin inventory
        $this->plugin_inventory = UNPLUG_Plugin_Inventory::get_plugin_inventory();
        
        $results = array(
            'plugins' => array(),
            'summary' => array(
                'total_plugins' => 0,
                'used_plugins' => 0,
                'possibly_used_plugins' => 0,
                'unused_plugins' => 0,
                'confidence_levels' => array()
            ),
            'detection_methods' => array(
                'shortcodes' => array(),
                'blocks' => array(),
                'widgets' => array(),
                'elementor' => array(),
                'custom_post_types' => array(),
                'custom_taxonomies' => array(),
                'menus' => array(),
                'options' => array()
            ),
            'metadata' => array(
                'scan_timestamp' => current_time( 'timestamp' ),
                'scan_duration' => 0,
                'content_analyzed' => array()
            )
        );

        // Analyze each plugin
        foreach ( $this->plugin_inventory['plugins'] as $plugin_file => $plugin_data ) {
            $plugin_activity = $this->analyze_plugin_activity( $plugin_file, $plugin_data );
            $results['plugins'][$plugin_file] = $plugin_activity;
            
            // Update summary counts
            $results['summary']['total_plugins']++;
            $confidence_level = $plugin_activity['confidence_level'];
            $results['summary']['confidence_levels'][$confidence_level] = 
                ($results['summary']['confidence_levels'][$confidence_level] ?? 0) + 1;
                
            if ( $confidence_level === 'used' ) {
                $results['summary']['used_plugins']++;
            } elseif ( $confidence_level === 'possibly_used' ) {
                $results['summary']['possibly_used_plugins']++;
            } else {
                $results['summary']['unused_plugins']++;
            }

            // --- NEW: Update location summary option for this plugin ---
            if ( ! class_exists( 'UNPLUG_Location' ) ) {
                require_once plugin_dir_path( __FILE__ ) . 'class-unplug-location.php';
            }
            $plugin_slug = dirname( $plugin_file );
            if ( $plugin_slug === '.' ) {
                $plugin_slug = basename( $plugin_file, '.php' );
            }
            $location_count = UNPLUG_Location::get_location_count(['plugin_slug' => $plugin_slug]);
            $locations = UNPLUG_Location::get_locations(['plugin_slug' => $plugin_slug, 'limit' => 10]);
            $location_details = array();
            foreach ($locations as $loc) {
                $location_details[] = sprintf(
                    '%s in post %d%s',
                    ucfirst($loc['location_type']),
                    $loc['post_id'],
                    isset($loc['element_id']) && $loc['element_id'] ? " (Element ID: {$loc['element_id']})" : ''
                );
            }
            update_option( "unplug_location_result_{$plugin_slug}", array(
                'count' => $location_count,
                'details' => $location_details
            ) );
            // --- END NEW ---

            // --- NEW: Store a human-readable summary for activity details ---
            $activity_details = array();
            if (!empty($plugin_activity['detection_details']) && is_array($plugin_activity['detection_details'])) {
                foreach ($plugin_activity['detection_details'] as $type => $info) {
                    if (!empty($info['count'])) {
                        $activity_details[] = sprintf(
                            '%d %s found%s',
                            $info['count'],
                            ucfirst($type),
                            (!empty($info['high_confidence_count']) ? " ({$info['high_confidence_count']} high confidence)" : '')
                        );
                    }
                }
            }
            if (empty($activity_details)) {
                $activity_details[] = __('No usage detected', 'unplug');
            }
            update_option( "unplug_activity_result_{$plugin_slug}", array(
                'status' => $plugin_activity['confidence_level'],
                'details' => $activity_details,
                'confidence_score' => $plugin_activity['confidence_score'] ?? 0,
                'usage_evidence' => $plugin_activity['usage_evidence'] ?? array(),
                'timestamp' => current_time( 'timestamp' )
            ) );
            // --- END NEW ---
        }

        // Populate detection methods summary
        $results['detection_methods']['shortcodes'] = $this->get_shortcode_summary();
        $results['detection_methods']['blocks'] = $this->get_block_summary();
        $results['detection_methods']['widgets'] = $this->get_widget_summary();
        $results['detection_methods']['elementor'] = $this->get_elementor_summary();
        $results['detection_methods']['custom_post_types'] = $this->get_cpt_summary();
        $results['detection_methods']['custom_taxonomies'] = $this->get_taxonomy_summary();
        $results['detection_methods']['menus'] = $this->get_menu_summary();
        $results['detection_methods']['options'] = $this->get_option_summary();

        // Finalize metadata
        $results['metadata']['scan_duration'] = microtime( true ) - $start_time;
        $results['metadata']['content_analyzed'] = $this->get_content_analysis_summary();

        // Log the scan completion
        $this->security->log_security_event( 'activity_scan_completed', 'Activity scan completed successfully', array(
            'total_plugins' => $results['summary']['total_plugins'],
            'used_plugins' => $results['summary']['used_plugins'],
            'scan_duration' => $results['metadata']['scan_duration']
        ) );

        return $results;
    }

    /**
     * Analyze plugin activity for a single plugin.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Plugin activity analysis.
     */
    private function analyze_plugin_activity( $plugin_file, $plugin_data ) {
        $activity = array(
            'plugin_file' => $plugin_file,
            'plugin_name' => $plugin_data['name'] ?? 'Unknown',
            'is_active' => $plugin_data['active'] ?? false,
            'confidence_score' => 0,
            'confidence_level' => 'no_usage',
            'usage_evidence' => array(),
            'detection_details' => array()
        );

        // Skip inactive plugins for deep analysis unless they have content
        if ( ! $activity['is_active'] ) {
            // Still check for stored content that might reference this plugin
            $stored_evidence = $this->check_stored_content_evidence( $plugin_file, $plugin_data );
            if ( ! empty( $stored_evidence ) ) {
                $activity['usage_evidence'] = $stored_evidence;
                $activity['confidence_score'] = $this->calculate_confidence_score( $stored_evidence );
            }
        } else {
            // Perform comprehensive analysis for active plugins
            $activity['usage_evidence'] = $this->detect_plugin_usage( $plugin_file, $plugin_data );
            $activity['confidence_score'] = $this->calculate_confidence_score( $activity['usage_evidence'] );
        }

        // Determine confidence level
        $activity['confidence_level'] = $this->determine_confidence_level( $activity['confidence_score'] );

        // Add detection details
        $activity['detection_details'] = $this->compile_detection_details( $activity['usage_evidence'] );

        return $activity;
    }

    /**
     * Detect plugin usage across all content types.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Usage evidence array.
     */
    private function detect_plugin_usage( $plugin_file, $plugin_data ) {
        $evidence = array();

        // Detect shortcode usage
        $shortcode_evidence = $this->detect_shortcode_usage( $plugin_file, $plugin_data );
        if ( ! empty( $shortcode_evidence ) ) {
            $evidence['shortcodes'] = $shortcode_evidence;
        }

        // Detect Gutenberg block usage
        $block_evidence = $this->detect_block_usage( $plugin_file, $plugin_data );
        if ( ! empty( $block_evidence ) ) {
            $evidence['blocks'] = $block_evidence;
        }

        // Detect widget usage
        $widget_evidence = $this->detect_widget_usage( $plugin_file, $plugin_data );
        if ( ! empty( $widget_evidence ) ) {
            $evidence['widgets'] = $widget_evidence;
        }

        // Detect Elementor widget usage
        $elementor_evidence = $this->detect_elementor_usage( $plugin_file, $plugin_data );
        if ( ! empty( $elementor_evidence ) ) {
            $evidence['elementor'] = $elementor_evidence;
        }

        // Detect custom post type usage
        $cpt_evidence = $this->detect_cpt_usage( $plugin_file, $plugin_data );
        if ( ! empty( $cpt_evidence ) ) {
            $evidence['custom_post_types'] = $cpt_evidence;
        }

        // Detect custom taxonomy usage
        $taxonomy_evidence = $this->detect_taxonomy_usage( $plugin_file, $plugin_data );
        if ( ! empty( $taxonomy_evidence ) ) {
            $evidence['custom_taxonomies'] = $taxonomy_evidence;
        }

        // Detect menu usage
        $menu_evidence = $this->detect_menu_usage($plugin_file, $plugin_data);
        if ( ! empty( $menu_evidence ) ) {
            $evidence['menus'] = $menu_evidence;
        }

        // Detect option usage
        $option_evidence = $this->detect_option_usage( $plugin_file, $plugin_data );
        if ( ! empty( $option_evidence ) ) {
            $evidence['options'] = $option_evidence;
        }

        return $evidence;
    }

    /**
     * Detect shortcode usage across content.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Shortcode usage evidence.
     */
    private function detect_shortcode_usage( $plugin_file, $plugin_data ) {
        global $wpdb;
        
        $evidence = array();
        
        // Get plugin shortcodes from the plugin inventory
        $plugin_shortcodes = $this->get_plugin_shortcodes( $plugin_file, $plugin_data );
        
        if ( empty( $plugin_shortcodes ) ) {
            return $evidence;
        }

        // Build shortcode search patterns
        $shortcode_patterns = array();
        foreach ( $plugin_shortcodes as $shortcode ) {
            $shortcode_patterns[] = '%[' . $shortcode . '%';
        }

        // Search in posts, pages, custom post types, and reusable blocks
        $content_types = array( 'post', 'page', 'wp_block' );
        // Include custom post types
        $custom_post_types = get_post_types( array( 'public' => true, '_builtin' => false ) );
        $content_types = array_merge( $content_types, $custom_post_types );

        $scanned_posts = array(); // To avoid duplicate evidence
        foreach ( $content_types as $post_type ) {
            foreach ( $shortcode_patterns as $pattern ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific content analysis
                $results = $wpdb->get_results( $wpdb->prepare( "
                    SELECT ID, post_title, post_content 
                    FROM {$wpdb->posts} 
                    WHERE post_type = %s 
                    AND post_status = 'publish' 
                    AND post_content LIKE %s
                    LIMIT 50
                ", $post_type, $pattern ) );
                
                if ( $results ) {
                    foreach ( $results as $result ) {
                        $key = $result->ID . '|' . $post_type;
                        if (!isset($scanned_posts[$key])) {
                            $scanned_posts[$key] = true;
                            $shortcode_matches = $this->extract_shortcode_from_content( $result->post_content, $plugin_shortcodes );
                            if (!empty($shortcode_matches)) {
                                $evidence[] = array(
                                    'type' => 'shortcode',
                                    'post_id' => $result->ID,
                                    'post_title' => $result->post_title,
                                    'post_type' => $post_type,
                                    'shortcode' => $shortcode_matches,
                                    'confidence' => 90 // High confidence for shortcode usage
                                );
                            }
                        }
                    }
                }
            }
        }

        // --- Advanced: Scan all published posts/pages/custom types and reusable blocks for shortcodes in blocks/attributes ---
        $all_types = array_unique($content_types);
        foreach ($all_types as $post_type) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific content analysis
            $posts = $wpdb->get_results( $wpdb->prepare( "
                SELECT ID, post_title, post_content 
                FROM {$wpdb->posts} 
                WHERE post_type = %s 
                AND post_status = 'publish' 
                LIMIT 100
            ", $post_type ) );
            if ($posts) {
                foreach ($posts as $post) {
                    $key = $post->ID . '|' . $post_type . '|blocks';
                    if (!isset($scanned_posts[$key])) {
                        $scanned_posts[$key] = true;
                        $blocks = function_exists('parse_blocks') ? parse_blocks($post->post_content) : array();
                        if (!empty($blocks)) {
                            $block_evidence = $this->extract_shortcodes_from_blocks($blocks, $plugin_shortcodes, $post->ID, $post->post_title, $post_type);
                            if (!empty($block_evidence)) {
                                $evidence = array_merge($evidence, $block_evidence);
                            }
                        }
                    }
                }
            }
        }

        // Search in widgets
        $widget_evidence = $this->search_shortcodes_in_widgets( $plugin_shortcodes );
        $evidence = array_merge( $evidence, $widget_evidence );

        // --- Elementor: Scan _elementor_data for shortcodes in widget settings ---
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific Elementor data analysis
        $elementor_posts = $wpdb->get_results( $wpdb->prepare( "
            SELECT post_id, meta_value 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = %s 
            AND meta_value != ''
            LIMIT 500
        ", '_elementor_data' ) );
        if ($elementor_posts) {
            foreach ($elementor_posts as $row) {
                $post_id = $row->post_id;
                $post = get_post($post_id);
                if (!$post) continue;
                $post_title = $post->post_title;
                $post_type = $post->post_type;
                $data = json_decode($row->meta_value, true);
                if (is_array($data)) {
                    $elementor_evidence = $this->extract_shortcodes_from_elementor($data, $plugin_shortcodes, $post_id, $post_title, $post_type);
                    if (!empty($elementor_evidence)) {
                        $evidence = array_merge($evidence, $elementor_evidence);
                    }
                }
            }
        }

        return $evidence;
    }

    /**
     * Recursively extract shortcodes from Gutenberg blocks and their attributes.
     *
     * @param array $blocks Parsed blocks from parse_blocks().
     * @param array $shortcodes List of shortcodes to search for.
     * @param int $post_id The post ID.
     * @param string $post_title The post title.
     * @param string $post_type The post type.
     * @return array Evidence array for found shortcodes.
     */
    private function extract_shortcodes_from_blocks($blocks, $shortcodes, $post_id, $post_title, $post_type) {
        $evidence = array();
        foreach ($blocks as $block) {
            // Check innerHTML/content for shortcodes
            if (isset($block['innerHTML'])) {
                $matches = $this->extract_shortcode_from_content($block['innerHTML'], $shortcodes);
                if (!empty($matches)) {
                    $evidence[] = array(
                        'type' => 'shortcode_block',
                        'post_id' => $post_id,
                        'post_title' => $post_title,
                        'post_type' => $post_type,
                        'shortcode' => $matches,
                        'block_name' => $block['blockName'] ?? '',
                        'confidence' => 90
                    );
                }
            }
            // Check block attributes (recursively for arrays)
            if (isset($block['attrs']) && !empty($block['attrs'])) {
                $attr_matches = $this->extract_shortcodes_from_any($block['attrs'], $shortcodes);
                if (!empty($attr_matches)) {
                    $evidence[] = array(
                        'type' => 'shortcode_block_attr',
                        'post_id' => $post_id,
                        'post_title' => $post_title,
                        'post_type' => $post_type,
                        'shortcode' => $attr_matches,
                        'block_name' => $block['blockName'] ?? '',
                        'confidence' => 85
                    );
                }
            }
            // Recursively check inner blocks
            if (isset($block['innerBlocks']) && is_array($block['innerBlocks']) && !empty($block['innerBlocks'])) {
                $inner_evidence = $this->extract_shortcodes_from_blocks($block['innerBlocks'], $shortcodes, $post_id, $post_title, $post_type);
                if (!empty($inner_evidence)) {
                    $evidence = array_merge($evidence, $inner_evidence);
                }
            }
        }
        return $evidence;
    }

    /**
     * Recursively extract shortcodes from any string or array (for block attributes).
     *
     * @param mixed $data String or array to search.
     * @param array $shortcodes List of shortcodes to search for.
     * @return array Array of found shortcodes.
     */
    private function extract_shortcodes_from_any($data, $shortcodes) {
        $found = array();
        if (is_string($data)) {
            $matches = $this->extract_shortcode_from_content($data, $shortcodes);
            if (!empty($matches)) {
                $found = array_merge($found, $matches);
            }
        } elseif (is_array($data)) {
            foreach ($data as $item) {
                $item_matches = $this->extract_shortcodes_from_any($item, $shortcodes);
                if (!empty($item_matches)) {
                    $found = array_merge($found, $item_matches);
                }
            }
        }
        return $found;
    }

    /**
     * Detect Gutenberg block usage using comprehensive block analysis.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Block usage evidence.
     */
    private function detect_block_usage( $plugin_file, $plugin_data ) {
        global $wpdb;
        
        $evidence = array();
        
        // Get plugin blocks from the plugin inventory
        $plugin_blocks = $this->get_plugin_blocks( $plugin_file, $plugin_data );
        
        if ( empty( $plugin_blocks ) ) {
            return $evidence;
        }

        // Use comprehensive block analysis for better detection
        $site_analysis = $this->block_analyzer->analyze_site_blocks( array(
            'posts_per_page' => 50,
            'include_custom_post_types' => true,
            'include_reusable_blocks' => true
        ) );

        // Check if any plugin blocks are found in the site analysis
        foreach ( $plugin_blocks as $plugin_block ) {
            if ( isset( $site_analysis['blocks_found'][$plugin_block] ) ) {
                $block_usage = $site_analysis['blocks_found'][$plugin_block];
                
                // Create evidence for each post using this block
                foreach ( $block_usage['posts'] as $post_id ) {
                    $post = get_post( $post_id );
                    if ( $post ) {
                        // Get detailed block analysis for this post
                        $post_analysis = $this->block_analyzer->analyze_post_content( $post_id );
                        
                        // Find the specific block instances
                        foreach ( $post_analysis['blocks'] as $block_id => $block_info ) {
                            if ( $block_info['name'] === $plugin_block ) {
                                $evidence[] = array(
                                    'type' => 'block',
                                    'post_id' => $post_id,
                                    'post_title' => $post->post_title,
                                    'post_type' => $post->post_type,
                                    'block_type' => $block_info['name'],
                                    'block_id' => $block_id,
                                    'block_attributes' => $block_info['attributes'],
                                    'block_depth' => $block_info['depth'],
                                    'is_custom' => $block_info['is_custom'],
                                    'source' => $block_info['source'],
                                    'confidence' => $this->calculate_block_confidence( $block_info ),
                                    'analysis_method' => 'comprehensive'
                                );
                            }
                        }
                    }
                }
            }
        }

        return $evidence;
    }

    /**
     * Detect widget usage.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Widget usage evidence.
     */
    private function detect_widget_usage( $plugin_file, $plugin_data ) {
        $evidence = array();
        $plugin_widgets = $this->get_plugin_widgets($plugin_file, $plugin_data);
        if (empty($plugin_widgets)) {
            return $evidence;
        }
        $widget_plugin_map = $this->build_widget_plugin_map();
        $widget_instances = $this->get_all_widget_instances();
        foreach ($widget_instances as $sidebar_id => $widgets) {
            foreach ($widgets as $widget_id => $widget_settings) {
                $widget_type = $this->get_widget_type_from_id($widget_id);
                $plugin_source = isset($widget_plugin_map[$widget_type]) ? $widget_plugin_map[$widget_type] : 'unknown';
                $confidence = in_array($widget_type, $plugin_widgets) ? 85 : 60;
                if (in_array($widget_type, $plugin_widgets)) {
                    $evidence[] = array(
                        'type' => 'widget',
                        'widget_type' => $widget_type,
                        'widget_id' => $widget_id,
                        'sidebar_id' => $sidebar_id,
                        'widget_settings' => $widget_settings,
                        'plugin_source' => $plugin_source,
                        'confidence' => $confidence
                    );
                }
            }
        }
        return $evidence;
    }

    /**
     * Detect Elementor widget usage.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Elementor usage evidence.
     */
    private function detect_elementor_usage( $plugin_file, $plugin_data ) {
        global $wpdb;
        
        $evidence = array();
        
        // Check if Elementor is active
        if ( ! is_plugin_active( 'elementor/elementor.php' ) ) {
            return $evidence;
        }

        // Get plugin Elementor widgets
        $plugin_elementor_widgets = $this->get_plugin_elementor_widgets( $plugin_file, $plugin_data );
        
        if ( empty( $plugin_elementor_widgets ) ) {
            return $evidence;
        }

        // Search for Elementor data in post meta
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific Elementor data analysis
        $elementor_posts = $wpdb->get_results( "
            SELECT post_id, meta_value 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_elementor_data' 
            AND meta_value != ''
            LIMIT 100
        " );

        foreach ( $elementor_posts as $elementor_post ) {
            $elementor_data = json_decode( $elementor_post->meta_value, true );
            
            if ( $elementor_data ) {
                $widget_usage = array();
                $this->scan_elementor_widgets($elementor_data, $plugin_elementor_widgets, array(), $elementor_post->post_id, false, $widget_usage);
                
                if ( ! empty( $widget_usage ) ) {
                    $evidence[] = array(
                        'type' => 'elementor',
                        'post_id' => $elementor_post->post_id,
                        'widgets_found' => $widget_usage,
                        'confidence' => 85 // High confidence for Elementor widget usage
                    );
                }
            }
        }

        return $evidence;
    }

    /**
     * Calculate block confidence based on comprehensive analysis.
     *
     * @since    1.0.0
     * @access   private
     * @param    array  $block_info  Block information from comprehensive analysis.
     * @return   int                 Confidence score (0-100).
     */
    private function calculate_block_confidence( $block_info ) {
        $confidence = 70; // Base confidence for block detection
        
        // Higher confidence for custom blocks
        if ( $block_info['is_custom'] ) {
            $confidence += 15;
        }
        
        // Higher confidence if source is identified
        if ( isset( $block_info['source'] ) && $block_info['source']['confidence'] === 'high' ) {
            $confidence += 10;
        }
        
        // Higher confidence for blocks with attributes
        if ( ! empty( $block_info['attributes'] ) ) {
            $confidence += 5;
        }
        
        // Cap at 100
        return min( 100, $confidence );
    }

    /**
     * Enhanced CPT detection: Analyze fields, taxonomies, relationships, and plugin attribution.
     *
     * @param string $plugin_file
     * @param array $plugin_data
     * @return array
     */
    public function detect_cpt_usage( $plugin_file, $plugin_data ) {
        $evidence = array();
        $plugin_cpts = $this->get_plugin_custom_post_types( $plugin_file, $plugin_data );
        
        if ( empty( $plugin_cpts ) ) {
            return $evidence;
        }
        $cpt_plugin_map = $this->build_cpt_plugin_map();
        foreach ( $plugin_cpts as $cpt ) {
            // Count posts of this type with caching
            $cache_key = 'unplug_cpt_post_count_' . $cpt;
            $post_count = wp_cache_get( $cache_key );
            if ( false === $post_count ) {
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific CPT analysis
                $post_count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = %s AND post_status != 'auto-draft'", $cpt ) );
                wp_cache_set( $cache_key, $post_count, '', 3600 ); // Cache for 1 hour
            }
            
            if ( $post_count > 0 ) {
                // Enumerate fields (meta keys) with caching
                $meta_cache_key = 'unplug_cpt_meta_keys_' . $cpt;
                $meta_keys = wp_cache_get( $meta_cache_key );
                if ( false === $meta_keys ) {
                    global $wpdb;
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific CPT analysis
                    $meta_keys = $wpdb->get_col( $wpdb->prepare( "SELECT DISTINCT meta_key FROM {$wpdb->postmeta} pm JOIN {$wpdb->posts} p ON pm.post_id = p.ID WHERE p.post_type = %s LIMIT 50", $cpt ) );
                    wp_cache_set( $meta_cache_key, $meta_keys, '', 3600 ); // Cache for 1 hour
                }
                
                // Detect ACF fields (if ACF is present)
                $acf_fields = function_exists('acf_get_field_groups') ? $this->get_acf_fields_for_cpt($cpt) : array();
                // List associated taxonomies
                $taxonomies = get_object_taxonomies( $cpt );
                // Detect relationships (parent/child, post-to-post)
                $relationships = $this->get_cpt_relationships($cpt);
                // Attribute CPT to plugin/theme
                $source = isset($cpt_plugin_map[$cpt]) ? $cpt_plugin_map[$cpt] : 'unknown';
                $evidence[] = array(
                    'type' => 'custom_post_type',
                    'cpt_name' => $cpt,
                    'post_count' => $post_count,
                    'meta_keys' => $meta_keys,
                    'acf_fields' => $acf_fields,
                    'taxonomies' => $taxonomies,
                    'relationships' => $relationships,
                    'source' => $source,
                    'confidence' => 90
                );
            }
        }
        return $evidence;
    }

    /**
     * Build a mapping of CPT to plugin/theme source.
     * @return array
     */
    private function build_cpt_plugin_map() {
        $map = array();
        if (!isset($this->plugin_inventory['plugins'])) return $map;
        foreach ($this->plugin_inventory['plugins'] as $plugin_slug => $plugin_data) {
            if (isset($plugin_data['custom_post_types'])) {
                foreach ($plugin_data['custom_post_types'] as $cpt) {
                    $map[$cpt] = $plugin_slug;
                }
            }
        }
        // TODO: Add theme attribution if needed
        return $map;
    }

    /**
     * Get ACF fields for a CPT (if ACF is present).
     * @param string $cpt
     * @return array
     */
    private function get_acf_fields_for_cpt($cpt) {
        $fields = array();
        $groups = acf_get_field_groups(array('post_type' => $cpt));
        foreach ($groups as $group) {
            $acf_fields = acf_get_fields($group['key']);
            if ($acf_fields) {
                foreach ($acf_fields as $field) {
                    $fields[] = $field['name'];
                }
            }
        }
        return $fields;
    }

    /**
     * Detect CPT relationships (parent/child, post-to-post).
     * @param string $cpt
     * @return array
     */
    private function get_cpt_relationships($cpt) {
        $relationships = array();
        // Parent/child (hierarchical CPTs)
        $cpt_obj = get_post_type_object($cpt);
        if ($cpt_obj && $cpt_obj->hierarchical) {
            $relationships[] = 'parent/child';
        }
        // Post-to-post (look for meta keys referencing other post IDs)
        // (Simple heuristic: meta keys ending with _id or _ids)
        $meta_cache_key = 'unplug_cpt_relationships_' . $cpt;
        $meta_keys = wp_cache_get( $meta_cache_key );
        if ( false === $meta_keys ) {
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom post type meta key analysis requires direct query, results are cached
            $meta_keys = $wpdb->get_col( $wpdb->prepare( "SELECT DISTINCT meta_key FROM {$wpdb->postmeta} pm JOIN {$wpdb->posts} p ON pm.post_id = p.ID WHERE p.post_type = %s LIMIT 50", $cpt ) );
            wp_cache_set( $meta_cache_key, $meta_keys, '', 3600 ); // Cache for 1 hour
        }
        foreach ($meta_keys as $key) {
            if (preg_match('/(_id|_ids)$/', $key)) {
                $relationships[] = 'meta:' . $key;
            }
        }
        return $relationships;
    }

    /**
     * Detect custom taxonomy usage.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Taxonomy usage evidence.
     */
    private function detect_taxonomy_usage( $plugin_file, $plugin_data ) {
        global $wpdb;
        
        $evidence = array();
        
        // Get plugin custom taxonomies
        $plugin_taxonomies = $this->get_plugin_custom_taxonomies( $plugin_file, $plugin_data );
        
        if ( empty( $plugin_taxonomies ) ) {
            return $evidence;
        }

        foreach ( $plugin_taxonomies as $taxonomy ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific taxonomy analysis requires direct query for custom taxonomy counts
            $term_count = $wpdb->get_var( $wpdb->prepare( "
                SELECT COUNT(*) 
                FROM {$wpdb->term_taxonomy} 
                WHERE taxonomy = %s
            ", $taxonomy ) );
            
            if ( $term_count > 0 ) {
                $evidence[] = array(
                    'type' => 'custom_taxonomy',
                    'taxonomy_name' => $taxonomy,
                    'term_count' => $term_count,
                    'confidence' => 85 // High confidence for taxonomy with terms
                );
            }
        }

        return $evidence;
    }

    /**
     * Enhanced menu item analysis: Enumerate, attribute, and record evidence for all menu items.
     *
     * @param string $plugin_file
     * @param array $plugin_data
     * @return array
     */
    public function detect_menu_usage($plugin_file, $plugin_data) {
        $evidence = array();
        $menus = wp_get_nav_menus();
        if (empty($menus)) {
            return $evidence;
        }
        $plugin_menu_items = isset($plugin_data['menu_items']) ? $plugin_data['menu_items'] : array();
        foreach ($menus as $menu) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin-specific menu analysis
            $menu_items = wp_get_nav_menu_items($menu->term_id);
            if (empty($menu_items)) {
                continue;
            }
            // Build a map of item ID to item for parent/child relationships
            $item_map = array();
            foreach ($menu_items as $item) {
                $item_map[$item->ID] = $item;
            }
            foreach ($menu_items as $item) {
                $path = $this->get_menu_item_path($item, $item_map);
                $source = 'unknown';
                $confidence = 60;
                // Custom menu item
                if ($item->type === 'custom') {
                    $source = 'user';
                    $confidence = 70;
                }
                // Plugin attribution
                $plugin_match = $this->check_menu_item_against_plugin($item, $plugin_menu_items);
                if ($plugin_match) {
                    $source = 'plugin';
                    $confidence = 90;
                }
                // TODO: Add theme attribution and file scanning if needed
                $evidence[] = array(
                    'type' => 'menu_item',
                    'menu_id' => $menu->term_id,
                    'menu_name' => $menu->name,
                    'item_id' => $item->ID,
                    'title' => $item->title,
                    'url' => $item->url,
                    'item_type' => $item->type,
                    'parent' => $item->menu_item_parent,
                    'order' => $item->menu_order,
                    'path' => $path,
                    'source' => $source,
                    'confidence' => $confidence
                );
            }
        }
        return $evidence;
    }

    /**
     * Build the full path from root to a menu item.
     * @param object $item
     * @param array $item_map
     * @return array
     */
    private function get_menu_item_path($item, $item_map) {
        $path = array($item->title);
        $parent_id = $item->menu_item_parent;
        while ($parent_id && isset($item_map[$parent_id])) {
            $parent = $item_map[$parent_id];
            array_unshift($path, $parent->title);
            $parent_id = $parent->menu_item_parent;
        }
        return $path;
    }

    /**
     * Detect option usage.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Option usage evidence.
     */
    private function detect_option_usage( $plugin_file, $plugin_data ) {
        global $wpdb;
        
        $evidence = array();
        
        // Get plugin options
        $plugin_options = $this->get_plugin_options( $plugin_file, $plugin_data );
        
        if ( empty( $plugin_options ) ) {
            return $evidence;
        }

        foreach ( $plugin_options as $option_name ) {
            $option_value = get_option( $option_name, false );
            
            if ( $option_value !== false ) {
                $evidence[] = array(
                    'type' => 'option',
                    'option_name' => $option_name,
                    'has_value' => ! empty( $option_value ),
                    'value_type' => gettype( $option_value ),
                    'confidence' => 60 // Medium confidence for option existence
                );
            }
        }

        return $evidence;
    }

    /**
     * Calculate confidence score based on usage evidence.
     *
     * @since    1.0.0
     * @access   private
     * @param    array   $evidence  Usage evidence array.
     * @return   int                Confidence score (0-100).
     */
    private function calculate_confidence_score( $evidence ) {
        if ( empty( $evidence ) ) {
            return 0;
        }

        $total_score = 0;
        $max_possible_score = 0;

        foreach ( $evidence as $evidence_type => $evidence_items ) {
            if ( ! isset( $this->scoring_weights[$evidence_type . '_usage'] ) ) {
                continue;
            }

            $weight = $this->scoring_weights[$evidence_type . '_usage'];
            $max_possible_score += $weight;

            if ( ! empty( $evidence_items ) ) {
                // Calculate score based on evidence strength and quantity
                $evidence_score = 0;
                $evidence_count = count( $evidence_items );

                foreach ( $evidence_items as $item ) {
                    $item_confidence = $item['confidence'] ?? 50;
                    $evidence_score += $item_confidence;
                }

                // Average confidence, but give bonus for multiple instances
                $average_confidence = $evidence_score / $evidence_count;
                $quantity_bonus = min( $evidence_count * 5, 20 ); // Max 20% bonus for quantity
                $final_evidence_score = min( $average_confidence + $quantity_bonus, 100 );

                $total_score += ( $final_evidence_score / 100 ) * $weight;
            }
        }

        // Calculate final percentage
        $final_score = $max_possible_score > 0 ? ( $total_score / $max_possible_score ) * 100 : 0;
        
        return min( round( $final_score ), 100 );
    }

    /**
     * Determine confidence level based on score.
     *
     * @since    1.0.0
     * @access   private
     * @param    int     $score  Confidence score.
     * @return   string          Confidence level.
     */
    private function determine_confidence_level( $score ) {
        if ( $score >= $this->confidence_thresholds['used'] ) {
            return 'used';
        } elseif ( $score >= $this->confidence_thresholds['possibly_used'] ) {
            return 'possibly_used';
        } else {
            return 'no_usage';
        }
    }

    /**
     * Compile detection details from evidence.
     *
     * @since    1.0.0
     * @access   private
     * @param    array   $evidence  Usage evidence array.
     * @return   array              Detection details.
     */
    private function compile_detection_details( $evidence ) {
        $details = array();

        foreach ( $evidence as $evidence_type => $evidence_items ) {
            if ( ! empty( $evidence_items ) ) {
                $details[$evidence_type] = array(
                    'count' => count( $evidence_items ),
                    'high_confidence_count' => count( array_filter( $evidence_items, function( $item ) {
                        return ( $item['confidence'] ?? 0 ) >= 80;
                    } ) ),
                    'sample_items' => array_slice( $evidence_items, 0, 3 ) // First 3 items as samples
                );
            }
        }

        return $details;
    }

    // Helper methods for extracting plugin-specific data

    /**
     * Get plugin shortcodes from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of shortcodes.
     */
    private function get_plugin_shortcodes( $plugin_file, $plugin_data ) {
        // This would integrate with the plugin scanner to get detected shortcodes
        // For now, we'll use a basic approach
        $shortcodes = array();
        
        // Check if plugin inventory has shortcode data
        if ( isset( $plugin_data['shortcodes'] ) ) {
            return $plugin_data['shortcodes'];
        }
        
        // Fallback: scan plugin files for shortcode registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $shortcodes = $this->scan_plugin_files_for_shortcodes( $plugin_path );
        
        return $shortcodes;
    }

    /**
     * Get plugin blocks from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of block types.
     */
    private function get_plugin_blocks( $plugin_file, $plugin_data ) {
        $blocks = array();
        
        // Check if plugin inventory has block data
        if ( isset( $plugin_data['blocks'] ) ) {
            return $plugin_data['blocks'];
        }
        
        // Fallback: scan plugin files for block registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $blocks = $this->scan_plugin_files_for_blocks( $plugin_path );
        
        return $blocks;
    }

    /**
     * Get plugin widgets from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of widget types.
     */
    private function get_plugin_widgets( $plugin_file, $plugin_data ) {
        $widgets = array();
        
        // Check if plugin inventory has widget data
        if ( isset( $plugin_data['widgets'] ) ) {
            return $plugin_data['widgets'];
        }
        
        // Fallback: scan plugin files for widget registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $widgets = $this->scan_plugin_files_for_widgets( $plugin_path );
        
        return $widgets;
    }

    /**
     * Get plugin Elementor widgets from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of Elementor widget types.
     */
    private function get_plugin_elementor_widgets( $plugin_file, $plugin_data ) {
        $widgets = array();
        
        // Check if plugin inventory has Elementor widget data
        if ( isset( $plugin_data['elementor_widgets'] ) ) {
            return $plugin_data['elementor_widgets'];
        }
        
        // Fallback: scan plugin files for Elementor widget registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $widgets = $this->scan_plugin_files_for_elementor_widgets( $plugin_path );
        
        return $widgets;
    }

    /**
     * Get plugin custom post types from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of custom post types.
     */
    private function get_plugin_custom_post_types( $plugin_file, $plugin_data ) {
        $cpts = array();
        
        // Check if plugin inventory has CPT data
        if ( isset( $plugin_data['custom_post_types'] ) ) {
            return $plugin_data['custom_post_types'];
        }
        
        // Fallback: scan plugin files for CPT registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $cpts = $this->scan_plugin_files_for_cpts( $plugin_path );
        
        return $cpts;
    }

    /**
     * Get plugin custom taxonomies from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of custom taxonomies.
     */
    private function get_plugin_custom_taxonomies( $plugin_file, $plugin_data ) {
        $taxonomies = array();
        
        // Check if plugin inventory has taxonomy data
        if ( isset( $plugin_data['custom_taxonomies'] ) ) {
            return $plugin_data['custom_taxonomies'];
        }
        
        // Fallback: scan plugin files for taxonomy registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $taxonomies = $this->scan_plugin_files_for_taxonomies( $plugin_path );
        
        return $taxonomies;
    }

    /**
     * Get plugin menu items from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of menu item patterns.
     */
    private function get_plugin_menu_items( $plugin_file, $plugin_data ) {
        $menu_items = array();
        
        // Check if plugin inventory has menu data
        if ( isset( $plugin_data['menu_items'] ) ) {
            return $plugin_data['menu_items'];
        }
        
        // Fallback: scan plugin files for menu registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $menu_items = $this->scan_plugin_files_for_menu_items( $plugin_path );
        
        return $menu_items;
    }

    /**
     * Get plugin options from plugin inventory.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file path.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Array of option names.
     */
    private function get_plugin_options( $plugin_file, $plugin_data ) {
        $options = array();
        
        // Check if plugin inventory has option data
        if ( isset( $plugin_data['options'] ) ) {
            return $plugin_data['options'];
        }
        
        // Fallback: scan plugin files for option registrations
        $plugin_path = plugin_dir_path( WP_PLUGIN_DIR . '/' . $plugin_file );
        $options = $this->scan_plugin_files_for_options( $plugin_path );
        
        return $options;
    }

    // Additional helper methods would continue here...
    // (The file continues with more implementation methods)

    /**
     * Invalidate activity cache.
     *
     * @since    1.0.0
     */
    public static function invalidate_activity_cache() {
        // Clear cached activity results
        wp_cache_delete( 'unplug_activity_results' );
        
        // Clear any stored transients
        delete_transient( 'unplug_activity_scan_results' );
    }

    /**
     * Schedule activity scan.
     *
     * @since    1.0.0
     */
    public static function schedule_activity_scan() {
        // Add to queue for background processing
        $queue = new UNPLUG_Queue( 
            new UNPLUG_Database(), 
            new UNPLUG_Security(), 
            new UNPLUG_Queue( new UNPLUG_Database(), new UNPLUG_Security(), null )
        );
        
        $queue->add_task( 'activity_scan', array(
            'force_refresh' => true,
            'scheduled' => true,
            'timestamp' => current_time( 'timestamp' )
        ), 5 );
    }

    /**
     * Get activity scan results.
     *
     * @since    1.0.0
     * @param    bool    $force_refresh    Force refresh of cached data.
     * @return   array                     Activity scan results.
     */
    public function get_activity_results( $force_refresh = false ) {
        return $this->perform_activity_scan( $force_refresh );
    }

    /**
     * Get plugin activity by confidence level.
     *
     * @since    1.0.0
     * @param    string  $confidence_level  The confidence level to filter by.
     * @return   array                      Filtered plugin activities.
     */
    public function get_plugins_by_confidence_level( $confidence_level ) {
        $results = $this->get_activity_results();
        $filtered = array();
        
        foreach ( $results['plugins'] as $plugin_file => $plugin_activity ) {
            if ( $plugin_activity['confidence_level'] === $confidence_level ) {
                $filtered[$plugin_file] = $plugin_activity;
            }
        }
        
        return $filtered;
    }

    /**
     * Get activity scan summary.
     *
     * @since    1.0.0
     * @return   array    Activity scan summary.
     */
    public function get_activity_summary() {
        $results = $this->get_activity_results();
        return $results['summary'];
    }

    /**
     * Export activity results.
     *
     * @since    1.0.0
     * @param    string  $format  Export format ('json', 'csv', 'xml').
     * @return   string           Exported data.
     */
    public function export_activity_results( $format = 'json' ) {
        $results = $this->get_activity_results();
        
        switch ( $format ) {
            case 'csv':
                return $this->export_activity_results_csv( $results );
            case 'xml':
                return $this->export_activity_results_xml( $results );
            default:
                return wp_json_encode( $results, JSON_PRETTY_PRINT );
        }
    }

    // More helper methods would continue here...
    // The implementation continues with all the scanning and helper methods

    /**
     * Scan plugin files for shortcode registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of shortcodes found.
     */
    private function scan_plugin_files_for_shortcodes( $plugin_path ) {
        $shortcodes = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $shortcodes;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for add_shortcode calls
                if ( preg_match_all( '/add_shortcode\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $shortcodes = array_merge( $shortcodes, $matches[1] );
                }
                
                // Look for shortcode_atts calls (indicates shortcode usage)
                if ( preg_match_all( '/shortcode_atts\s*\(\s*[^,]+,\s*\$atts,\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $shortcodes = array_merge( $shortcodes, $matches[1] );
                }
            }
        }
        
        return array_unique( $shortcodes );
    }

    /**
     * Scan plugin files for block registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of blocks found.
     */
    private function scan_plugin_files_for_blocks( $plugin_path ) {
        $blocks = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $blocks;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( in_array( $file->getExtension(), array( 'php', 'js' ) ) ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for register_block_type calls
                if ( preg_match_all( '/register_block_type\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $blocks = array_merge( $blocks, $matches[1] );
                }
                
                // Look for registerBlockType calls (JavaScript)
                if ( preg_match_all( '/registerBlockType\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $blocks = array_merge( $blocks, $matches[1] );
                }
            }
        }
        
        return array_unique( $blocks );
    }

    /**
     * Scan plugin files for widget registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of widgets found.
     */
    private function scan_plugin_files_for_widgets( $plugin_path ) {
        $widgets = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $widgets;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for register_widget calls
                if ( preg_match_all( '/register_widget\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $widgets = array_merge( $widgets, $matches[1] );
                }
                
                // Look for class extensions of WP_Widget
                if ( preg_match_all( '/class\s+(\w+)\s+extends\s+WP_Widget/', $content, $matches ) ) {
                    $widgets = array_merge( $widgets, $matches[1] );
                }
            }
        }
        
        return array_unique( $widgets );
    }

    /**
     * Scan plugin files for Elementor widget registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of Elementor widgets found.
     */
    private function scan_plugin_files_for_elementor_widgets( $plugin_path ) {
        $widgets = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $widgets;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for Elementor widget registrations
                if ( preg_match_all( '/extends\s+\\\\?Elementor\\\\Widget_Base/', $content, $matches ) ) {
                    // Extract class names
                    if ( preg_match_all( '/class\s+(\w+)\s+extends\s+\\\\?Elementor\\\\Widget_Base/', $content, $class_matches ) ) {
                        $widgets = array_merge( $widgets, $class_matches[1] );
                    }
                }
                
                // Look for get_name() method returns in Elementor widgets
                if ( preg_match_all( '/function\s+get_name\s*\(\s*\)\s*{[^}]*return\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $widgets = array_merge( $widgets, $matches[1] );
                }
            }
        }
        
        return array_unique( $widgets );
    }

    /**
     * Scan plugin files for custom post type registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of custom post types found.
     */
    private function scan_plugin_files_for_cpts( $plugin_path ) {
        $cpts = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $cpts;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for register_post_type calls
                if ( preg_match_all( '/register_post_type\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $cpts = array_merge( $cpts, $matches[1] );
                }
            }
        }
        
        return array_unique( $cpts );
    }

    /**
     * Scan plugin files for custom taxonomy registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of custom taxonomies found.
     */
    private function scan_plugin_files_for_taxonomies( $plugin_path ) {
        $taxonomies = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $taxonomies;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for register_taxonomy calls
                if ( preg_match_all( '/register_taxonomy\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $taxonomies = array_merge( $taxonomies, $matches[1] );
                }
            }
        }
        
        return array_unique( $taxonomies );
    }

    /**
     * Scan plugin files for menu item registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of menu items found.
     */
    private function scan_plugin_files_for_menu_items( $plugin_path ) {
        $menu_items = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $menu_items;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for add_menu_page calls
                if ( preg_match_all( '/add_menu_page\s*\([^,]+,\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $menu_items = array_merge( $menu_items, $matches[1] );
                }
                
                // Look for add_submenu_page calls
                if ( preg_match_all( '/add_submenu_page\s*\([^,]+,\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $menu_items = array_merge( $menu_items, $matches[1] );
                }
            }
        }
        
        return array_unique( $menu_items );
    }

    /**
     * Scan plugin files for option registrations.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_path  The plugin directory path.
     * @return   array                 Array of options found.
     */
    private function scan_plugin_files_for_options( $plugin_path ) {
        $options = array();
        
        if ( ! is_dir( $plugin_path ) ) {
            return $options;
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $plugin_path ),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ( $iterator as $file ) {
            if ( $file->getExtension() === 'php' ) {
                $content = file_get_contents( $file->getPathname() );
                
                // Look for add_option calls
                if ( preg_match_all( '/add_option\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $options = array_merge( $options, $matches[1] );
                }
                
                // Look for update_option calls
                if ( preg_match_all( '/update_option\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $options = array_merge( $options, $matches[1] );
                }
                
                // Look for get_option calls
                if ( preg_match_all( '/get_option\s*\(\s*[\'"]([^\'"]+)[\'"]/', $content, $matches ) ) {
                    $options = array_merge( $options, $matches[1] );
                }
            }
        }
        
        return array_unique( $options );
    }

    /**
     * Extract shortcode from content.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $content    The content to search.
     * @param    array   $shortcodes Array of shortcodes to find.
     * @return   array               Found shortcodes.
     */
    private function extract_shortcode_from_content( $content, $shortcodes ) {
        $found_shortcodes = array();
        
        foreach ( $shortcodes as $shortcode ) {
            if ( preg_match_all( '/\[' . preg_quote( $shortcode, '/' ) . '([^\]]*)\]/i', $content, $matches ) ) {
                $found_shortcodes[] = array(
                    'shortcode' => $shortcode,
                    'matches' => $matches[0],
                    'attributes' => $matches[1]
                );
            }
        }
        
        return $found_shortcodes;
    }

    /**
     * Extract block information from content.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $content The content to search.
     * @param    array   $blocks  Array of blocks to find.
     * @return   array            Found block information.
     */
    private function extract_block_info_from_content( $content, $blocks ) {
        $found_blocks = array();
        
        foreach ( $blocks as $block ) {
            $pattern = '/<!-- wp:' . preg_quote( $block, '/' ) . '(\s+[^>]*)?(?:-->.*?<!-- \/wp:' . preg_quote( $block, '/' ) . ' -->|\s*\/-->)/s';
            if ( preg_match_all( $pattern, $content, $matches ) ) {
                $found_blocks[] = array(
                    'block_type' => $block,
                    'matches' => $matches[0],
                    'attributes' => isset( $matches[1] ) ? $matches[1] : array()
                );
            }
        }
        
        return $found_blocks;
    }

    /**
     * Get all widget instances.
     *
     * @since    1.0.0
     * @access   private
     * @return   array    All widget instances.
     */
    private function get_all_widget_instances() {
        $widget_instances = array();
        $sidebars = wp_get_sidebars_widgets();
        
        foreach ( $sidebars as $sidebar_id => $widgets ) {
            if ( ! is_array( $widgets ) ) {
                continue;
            }
            
            foreach ( $widgets as $widget_id ) {
                $widget_type = $this->get_widget_type_from_id( $widget_id );
                $widget_data = $this->get_widget_data( $widget_id );
                
                if ( $widget_type && $widget_data ) {
                    $widget_instances[$sidebar_id][$widget_id] = $widget_data;
                }
            }
        }
        
        return $widget_instances;
    }

    /**
     * Get widget type from widget ID.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $widget_id  The widget ID.
     * @return   string              The widget type.
     */
    private function get_widget_type_from_id( $widget_id ) {
        if ( preg_match( '/^(.+?)-\d+$/', $widget_id, $matches ) ) {
            return $matches[1];
        }
        return $widget_id;
    }

    /**
     * Get widget data.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $widget_id  The widget ID.
     * @return   array               Widget data.
     */
    private function get_widget_data( $widget_id ) {
        $widget_type = $this->get_widget_type_from_id( $widget_id );
        
        if ( preg_match( '/^(.+?)-(\d+)$/', $widget_id, $matches ) ) {
            $base_id = $matches[1];
            $instance_id = intval( $matches[2] );
            
            $widget_instances = get_option( 'widget_' . $base_id, array() );
            
            if ( isset( $widget_instances[$instance_id] ) ) {
                return $widget_instances[$instance_id];
            }
        }
        
        return array();
    }

    /**
     * Search shortcodes in widgets.
     *
     * @since    1.0.0
     * @access   private
     * @param    array   $shortcodes  Array of shortcodes to search for.
     * @return   array                Widget evidence.
     */
    private function search_shortcodes_in_widgets( $shortcodes ) {
        $evidence = array();
        $widget_instances = $this->get_all_widget_instances();
        
        foreach ( $widget_instances as $sidebar_id => $widgets ) {
            foreach ( $widgets as $widget_id => $widget_data ) {
                $widget_content = wp_json_encode( $widget_data );
                
                foreach ( $shortcodes as $shortcode ) {
                    if ( strpos( $widget_content, '[' . $shortcode ) !== false ) {
                        $evidence[] = array(
                            'type' => 'widget_shortcode',
                            'widget_id' => $widget_id,
                            'sidebar_id' => $sidebar_id,
                            'shortcode' => $shortcode,
                            'confidence' => 85
                        );
                    }
                }
            }
        }
        
        return $evidence;
    }

    /**
     * Recursively scan Elementor widgets in _elementor_data, capturing full context and resolving global widgets.
     *
     * @param array $elements The elements array from _elementor_data.
     * @param array $widget_source_map Mapping of widget types/prefixes to plugin/theme sources.
     * @param array $path Current hierarchy path (for context).
     * @param int $post_id The post ID being scanned.
     * @param bool $is_global Whether this scan is for a global widget template.
     * @param array $results Accumulator for found widgets.
     * @return array Detailed widget evidence.
     */
    private function scan_elementor_widgets($elements, $widget_source_map, $path = array(), $post_id = 0, $is_global = false, &$results = array()) {
        if (!is_array($elements)) {
            return $results;
        }
        foreach ($elements as $index => $element) {
            $current_path = array_merge($path, [$index]);
            $el_type = $element['elType'] ?? 'unknown';
            if ($el_type === 'widget') {
                $widget_type = $element['widgetType'] ?? 'unknown';
                $settings = $element['settings'] ?? array();
                $is_global_widget = isset($settings['__globals__']) || isset($settings['_global_widget_template_id']);
                $template_id = $settings['_global_widget_template_id'] ?? '';
                $source_plugin = $this->identify_widget_source($widget_type, $widget_source_map);
                $widget_evidence = array(
                    'post_id' => $post_id,
                    'widget_type' => $widget_type,
                    'widget_id' => $element['id'] ?? '',
                    'settings' => $settings,
                    'path' => $current_path,
                    'is_global' => $is_global_widget,
                    'source_plugin' => $source_plugin,
                    'confidence' => $is_global_widget ? 90 : 80
                );
                $results[] = $widget_evidence;
                // If global, resolve and scan the template
                if ($is_global_widget && $template_id) {
                    $template_data = get_post_meta($template_id, '_elementor_data', true);
                    if ($template_data) {
                        $template_elements = json_decode($template_data, true);
                        if (is_array($template_elements)) {
                            $this->scan_elementor_widgets($template_elements, $widget_source_map, $current_path, $template_id, true, $results);
                        }
                    }
                }
            } else {
                // Optionally record section/column context
                // $results[] = array('element_id' => $element['id'] ?? '', 'element_type' => $el_type, 'path' => $current_path);
            }
            // Recursively process nested elements
            if (isset($element['elements']) && is_array($element['elements'])) {
                $this->scan_elementor_widgets($element['elements'], $widget_source_map, $current_path, $post_id, $is_global, $results);
            }
        }
        return $results;
    }

    /**
     * Identify the source plugin/theme for a widget type using a mapping table.
     *
     * @param string $widget_type
     * @param array $widget_source_map
     * @return string Plugin or theme slug, or 'unknown'.
     */
    private function identify_widget_source($widget_type, $widget_source_map) {
        // Exact match
        if (isset($widget_source_map[$widget_type])) {
            return $widget_source_map[$widget_type];
        }
        // Prefix match
        foreach ($widget_source_map as $prefix => $source) {
            if (substr($prefix, -1) === '-' && strpos($widget_type, $prefix) === 0) {
                return $source;
            }
        }
        return 'unknown';
    }

    /**
     * Build a widget source mapping table from known core/pro widgets and active plugins.
     *
     * @return array Mapping of widget types/prefixes to plugin/theme slugs.
     */
    private function build_widget_source_map() {
        $map = array(
            // Core Elementor widgets
            'button' => 'elementor',
            'image' => 'elementor',
            // ... add more core widgets ...
            // Pro widgets
            'posts' => 'elementor-pro',
            'form' => 'elementor-pro',
            // Third-party (prefixes)
            'premium-' => 'premium-addons-for-elementor',
            'eael-' => 'essential-addons-for-elementor-lite',
            // ... extend as needed ...
        );
        // Optionally, dynamically discover from active plugins
        // ...
        return $map;
    }

    /**
     * Check menu item against plugin patterns.
     *
     * @since    1.0.0
     * @access   private
     * @param    object  $menu_item     The menu item.
     * @param    array   $plugin_items  Plugin menu items.
     * @return   array|false            Match information or false.
     */
    private function check_menu_item_against_plugin( $menu_item, $plugin_items ) {
        foreach ( $plugin_items as $plugin_item ) {
            if ( strpos( $menu_item->title, $plugin_item ) !== false ) {
                return array(
                    'type' => 'title_match',
                    'pattern' => $plugin_item
                );
            }
            
            if ( strpos( $menu_item->url, $plugin_item ) !== false ) {
                return array(
                    'type' => 'url_match',
                    'pattern' => $plugin_item
                );
            }
        }
        
        return false;
    }

    /**
     * Check stored content evidence for inactive plugins.
     *
     * @since    1.0.0
     * @access   private
     * @param    string  $plugin_file  The plugin file.
     * @param    array   $plugin_data  The plugin data.
     * @return   array                 Evidence array.
     */
    private function check_stored_content_evidence( $plugin_file, $plugin_data ) {
        $evidence = array();
        
        // Check if plugin has stored content signatures
        // This would check the database for any references to this plugin
        // Even though it's inactive, there might be stored content
        
        return $evidence;
    }

    /**
     * Get detection method summaries.
     */
    private function get_shortcode_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_block_summary() {
        // Use comprehensive block analyzer for summary
        $site_analysis = $this->block_analyzer->analyze_site_blocks( array(
            'posts_per_page' => 100,
            'include_custom_post_types' => true,
            'include_reusable_blocks' => true
        ) );
        
        $custom_blocks = 0;
        $plugins_using = 0;
        
        foreach ( $site_analysis['blocks_found'] as $block_name => $block_data ) {
            if ( strpos( $block_name, 'core/' ) !== 0 ) {
                $custom_blocks++;
            }
        }
        
        // Count unique plugins providing blocks
        $plugin_sources = array();
        foreach ( $site_analysis['sources'] as $source_key => $source_data ) {
            if ( $source_data['type'] === 'plugin' ) {
                $plugin_sources[$source_data['name']] = true;
            }
        }
        
        $plugins_using = count( $plugin_sources );
        
        return array( 
            'total_detected' => count( $site_analysis['blocks_found'] ),
            'custom_blocks' => $custom_blocks,
            'plugins_using' => $plugins_using,
            'core_blocks' => count( $site_analysis['blocks_found'] ) - $custom_blocks,
            'posts_with_blocks' => $site_analysis['summary']['posts_with_blocks']
        );
    }

    private function get_widget_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_elementor_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_cpt_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_taxonomy_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_menu_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_option_summary() {
        return array( 'total_detected' => 0, 'plugins_using' => 0 );
    }

    private function get_content_analysis_summary() {
        return array(
            'posts_scanned' => 0,
            'widgets_scanned' => 0,
            'menus_scanned' => 0,
            'cpts_analyzed' => 0
        );
    }

    /**
     * Export activity results to CSV.
     *
     * @since    1.0.0
     * @access   private
     * @param    array   $results  The results to export.
     * @return   string            CSV data.
     */
    private function export_activity_results_csv( $results ) {
        $csv = "Plugin,Status,Confidence Level,Confidence Score,Evidence Types\n";
        
        foreach ( $results['plugins'] as $plugin_file => $plugin_data ) {
            $evidence_types = implode( ';', array_keys( $plugin_data['usage_evidence'] ) );
            $csv .= sprintf( 
                "%s,%s,%s,%d,%s\n",
                $plugin_data['plugin_name'],
                $plugin_data['is_active'] ? 'Active' : 'Inactive',
                $plugin_data['confidence_level'],
                $plugin_data['confidence_score'],
                $evidence_types
            );
        }
        
        return $csv;
    }

    /**
     * Export activity results to XML.
     *
     * @since    1.0.0
     * @access   private
     * @param    array   $results  The results to export.
     * @return   string            XML data.
     */
    private function export_activity_results_xml( $results ) {
        $xml = "<?xml version='1.0' encoding='UTF-8'?>\n<activity_scan>\n";
        
        foreach ( $results['plugins'] as $plugin_file => $plugin_data ) {
            $xml .= "  <plugin>\n";
            $xml .= "    <name>" . htmlspecialchars( $plugin_data['plugin_name'] ) . "</name>\n";
            $xml .= "    <file>" . htmlspecialchars( $plugin_file ) . "</file>\n";
            $xml .= "    <status>" . ( $plugin_data['is_active'] ? 'Active' : 'Inactive' ) . "</status>\n";
            $xml .= "    <confidence_level>" . $plugin_data['confidence_level'] . "</confidence_level>\n";
            $xml .= "    <confidence_score>" . $plugin_data['confidence_score'] . "</confidence_score>\n";
            $xml .= "  </plugin>\n";
        }
        
        $xml .= "</activity_scan>\n";
        
        return $xml;
    }

    /**
     * Build a mapping of widget class to plugin slug for attribution.
     *
     * @return array Mapping of widget class => plugin slug
     */
    private function build_widget_plugin_map() {
        $map = array();
        if (!isset($this->plugin_inventory['plugins'])) return $map;
        foreach ($this->plugin_inventory['plugins'] as $plugin_file => $plugin_data) {
            if (isset($plugin_data['widgets']) && is_array($plugin_data['widgets'])) {
                foreach ($plugin_data['widgets'] as $widget_class) {
                    $map[$widget_class] = $plugin_file;
                }
            }
        }
        // Optionally, scan all active plugins for WP_Widget subclasses
        // ...
        return $map;
    }

    /**
     * Set the plugin inventory for scan context (for testing or runtime injection)
     * @param array $inventory
     */
    public function set_plugin_inventory($inventory) {
        $this->plugin_inventory = $inventory;
    }

    /**
     * Recursively extract shortcodes from Elementor widget data.
     *
     * @param array $elements Elementor elements array (from _elementor_data JSON).
     * @param array $shortcodes List of shortcodes to search for.
     * @param int $post_id The post ID.
     * @param string $post_title The post title.
     * @param string $post_type The post type.
     * @param array $path Widget hierarchy path (for context).
     * @return array Evidence array for found shortcodes.
     */
    private function extract_shortcodes_from_elementor($elements, $shortcodes, $post_id, $post_title, $post_type, $path = array()) {
        $evidence = array();
        if (!is_array($elements)) return $evidence;
        foreach ($elements as $index => $element) {
            $current_path = array_merge($path, [$index]);
            // Check widget settings for shortcodes
            if (isset($element['elType']) && $element['elType'] === 'widget') {
                $widget_type = $element['widgetType'] ?? '';
                $settings = $element['settings'] ?? array();
                $matches = $this->extract_shortcodes_from_any($settings, $shortcodes);
                if (!empty($matches)) {
                    $evidence[] = array(
                        'type' => 'shortcode_elementor',
                        'post_id' => $post_id,
                        'post_title' => $post_title,
                        'post_type' => $post_type,
                        'shortcode' => $matches,
                        'widget_type' => $widget_type,
                        'widget_path' => $current_path,
                        'confidence' => 90
                    );
                }
            }
            // Recursively check nested elements
            if (isset($element['elements']) && is_array($element['elements']) && !empty($element['elements'])) {
                $inner_evidence = $this->extract_shortcodes_from_elementor($element['elements'], $shortcodes, $post_id, $post_title, $post_type, $current_path);
                if (!empty($inner_evidence)) {
                    $evidence = array_merge($evidence, $inner_evidence);
                }
            }
        }
        return $evidence;
    }
} 