<?php

/**
 * Provide a admin area view for the plugin settings
 *
 * This file is used to markup the admin-facing aspects of the plugin settings.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/admin/partials
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Double-check user capabilities
if ( ! current_user_can( 'manage_options' ) ) {
    wp_die( esc_html( __( 'You do not have sufficient permissions to access this page.', 'unplug' ) ) );
}

// Load helper classes
require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-unplug-options.php';

// Get current settings
$user_tier = UNPLUG_Options::get_user_tier();
$api_token = UNPLUG_Options::get_api_token();
$license_status = get_option('unplug_license_status', array());

?>

<div class="wrap">
    <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
    
    <?php
    // Show success message if settings were saved
    // phpcs:disable WordPress.Security.NonceVerification.Recommended
    if ( isset( $_GET['settings-updated'] ) && $_GET['settings-updated'] == 'true' ) {
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e( 'Settings saved successfully!', 'unplug' ); ?></p>
        </div>
        <?php
    }
    
    // Show API validation status
    if ( ! empty( $api_token ) ) {
        ?>
        <div class="notice notice-info">
            <p>
                <strong><?php esc_html_e( 'API Token Status:', 'unplug' ); ?></strong>
                <span class="api-status-badge api-status-<?php echo esc_attr( strtolower( $user_tier ) ); ?>">
                    <?php echo esc_html( ucfirst( $user_tier ) ); ?>
                </span>
            </p>
            <p>
                <?php 
                if ( $user_tier === 'free' ) {
                    esc_html_e( 'Your API token is valid but you\'re on the free tier. Upgrade to unlock Pro features.', 'unplug' );
                } else {
                    esc_html_e( 'Your API token is valid and active.', 'unplug' );
                }
                ?>
            </p>
        </div>
        <?php
    }

    // Show license status and last checked time
    if ( ! empty( $api_token ) ) {
        echo '<div class="notice notice-info unplug-license-status">';
        if ( isset( $license_status['status'] ) ) {
            if ( $license_status['status'] === 'valid' ) {
                echo '<p><span class="dashicons dashicons-yes" style="color:green;"></span> ' . esc_html( __( 'License is valid.', 'unplug' ) ) . '</p>';
            } elseif ( $license_status['status'] === 'expired' ) {
                echo '<p><span class="dashicons dashicons-warning" style="color:orange;"></span> ' . esc_html( __( 'License expired.', 'unplug' ) ) . '</p>';
            } elseif ( $license_status['status'] === 'invalid' ) {
                echo '<p><span class="dashicons dashicons-no" style="color:red;"></span> ' . esc_html( __( 'License is invalid.', 'unplug' ) ) . '</p>';
            }
        }
        if ( isset( $license_status['degraded'] ) && $license_status['degraded'] ) {
            echo '<p><span class="dashicons dashicons-info"></span> ' . esc_html( __( 'Using last known valid license state. The licensing server could not be reached.', 'unplug' ) ) . '</p>';
        }
        if ( isset( $license_status['error'] ) ) {
            echo '<p style="color:red;"><span class="dashicons dashicons-warning"></span> ' . esc_html( $license_status['error'] ) . '</p>';
        }
        if ( isset( $license_status['checked_at'] ) ) {
            echo '<p><em>' . sprintf( 
                /* translators: %s: formatted date and time of last license check */
                esc_html( __( 'Last checked: %s', 'unplug' ) ), 
                esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), intval( $license_status['checked_at'] ) ) ) 
            ) . '</em></p>';
        }
        echo '</div>';
    }
    ?>
    
    <form method="post" action="options.php">
        <?php
        settings_fields( 'unplug_settings' );
        
        // Add security nonce field
        require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-unplug-security.php';
        UNPLUG_Security::nonce_field( 'settings', 'unplug_settings_nonce' );
        ?>
        
        <h2><?php esc_html_e( 'General Settings', 'unplug' ); ?></h2>
        <table class="form-table">
            <tbody>
                <tr>
                    <th scope="row">
                        <label for="settings_api_token"><?php esc_html_e( 'API Token', 'unplug' ); ?></label>
                    </th>
                    <td>
                        <input type="password" 
                               id="settings_api_token" 
                               name="unplug_settings[api_token]" 
                               value="<?php echo esc_attr( $api_token ); ?>" 
                               class="regular-text" 
                               autocomplete="new-password"
                               placeholder="<?php esc_attr_e( 'Enter your API token...', 'unplug' ); ?>" />
                        <button type="button" id="toggle-token-visibility" class="button button-secondary">
                            <span class="dashicons dashicons-visibility"></span>
                            <?php esc_html_e( 'Show', 'unplug' ); ?>
                        </button>
                        <p class="description">
                            <?php esc_html_e( 'Enter your Unplug API token to access Pro features. Your token is stored securely and encrypted.', 'unplug' ); ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <?php esc_html_e( 'Current Tier', 'unplug' ); ?>
                    </th>
                    <td>
                        <div class="tier-display">
                            <span class="tier-badge tier-<?php echo esc_attr( strtolower( $user_tier ) ); ?>">
                                <?php echo esc_html( ucfirst( $user_tier ) ); ?>
                            </span>
                            
                            <?php if ( $user_tier === 'free' ) : ?>
                                <div class="tier-features">
                                    <h4><?php esc_html_e( 'Free Tier Features:', 'unplug' ); ?></h4>
                                    <ul>
                                        <li><?php esc_html_e( 'Basic Activity Scan', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Plugin Usage Detection', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Location Count (summary)', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'CSV Export', 'unplug' ); ?></li>
                                    </ul>
                                </div>
                            <?php elseif ( $user_tier === 'pro' ) : ?>
                                <div class="tier-features">
                                    <h4><?php esc_html_e( 'Pro Tier Features:', 'unplug' ); ?></h4>
                                    <ul>
                                        <li><?php esc_html_e( 'Advanced Activity Scan', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Deactivation Testing', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Detailed Location Analysis', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Speed Impact Analysis', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Enhanced Export Options', 'unplug' ); ?></li>
                                    </ul>
                                </div>
                            <?php elseif ( $user_tier === 'agency' ) : ?>
                                <div class="tier-features">
                                    <h4><?php esc_html_e( 'Agency Tier Features:', 'unplug' ); ?></h4>
                                    <ul>
                                        <li><?php esc_html_e( 'All Pro Features', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'White-label Reports', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Advanced Analytics', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Priority Support', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Multi-site Management', 'unplug' ); ?></li>
                                    </ul>
                                </div>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            </tbody>
        </table>
        
        <h2><?php esc_html_e( 'Advanced Settings', 'unplug' ); ?></h2>
        
        <table class="form-table">
            <tbody>
                <tr>
                    <th scope="row">
                        <label for="scan_timeout"><?php esc_html_e( 'Scan Timeout', 'unplug' ); ?></label>
                    </th>
                    <td>
                        <input type="number" 
                               id="scan_timeout" 
                               name="unplug_settings[scan_timeout]" 
                               value="<?php echo esc_attr( UNPLUG_Options::get_scan_timeout() ); ?>" 
                               class="small-text" 
                               min="30" 
                               max="600" />
                        <span class="description"><?php esc_html_e( 'seconds (30-600)', 'unplug' ); ?></span>
                        <p class="description">
                            <?php esc_html_e( 'Maximum time allowed for activity scans. Increase if you have a large site.', 'unplug' ); ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="cleanup_retention"><?php esc_html_e( 'Data Retention', 'unplug' ); ?></label>
                    </th>
                    <td>
                        <select id="cleanup_retention" name="unplug_settings[cleanup_retention]">
                            <option value="7" <?php selected( UNPLUG_Options::get_cleanup_retention_days(), 7 ); ?>>
                                <?php esc_html_e( '7 days', 'unplug' ); ?>
                            </option>
                            <option value="30" <?php selected( UNPLUG_Options::get_cleanup_retention_days(), 30 ); ?>>
                                <?php esc_html_e( '30 days', 'unplug' ); ?>
                            </option>
                            <option value="90" <?php selected( UNPLUG_Options::get_cleanup_retention_days(), 90 ); ?>>
                                <?php esc_html_e( '90 days', 'unplug' ); ?>
                            </option>
                            <option value="365" <?php selected( UNPLUG_Options::get_cleanup_retention_days(), 365 ); ?>>
                                <?php esc_html_e( '1 year', 'unplug' ); ?>
                            </option>
                        </select>
                        <p class="description">
                            <?php esc_html_e( 'How long to keep scan results and queue data before automatic cleanup.', 'unplug' ); ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="enable_logging"><?php esc_html_e( 'Debug Logging', 'unplug' ); ?></label>
                    </th>
                    <td>
                        <fieldset>
                            <legend class="screen-reader-text">
                                <span><?php esc_html_e( 'Debug Logging', 'unplug' ); ?></span>
                            </legend>
                            <label for="enable_logging">
                                <input type="checkbox" 
                                       id="enable_logging" 
                                       name="unplug_settings[enable_logging]" 
                                       value="1" 
                                       <?php checked( UNPLUG_Options::get_enable_logging() ); ?> />
                                <?php esc_html_e( 'Enable debug logging', 'unplug' ); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e( 'Enable detailed logging for troubleshooting. Logs are stored in the WordPress debug.log file.', 'unplug' ); ?>
                            </p>
                        </fieldset>
                    </td>
                </tr>
            </tbody>
        </table>
        
        <?php submit_button(); ?>
    </form>
    
    <div class="unplug-help-section">
        <h2><?php esc_html_e( 'Getting Started', 'unplug' ); ?></h2>
        <div class="help-cards">
            <div class="help-card">
                <h3><?php esc_html_e( 'Need an API Token?', 'unplug' ); ?></h3>
                <p><?php esc_html_e( 'Visit our website to sign up for a free account and get your API token.', 'unplug' ); ?></p>
                <a href="https://mulberrytech.ca/unplug/signup" target="_blank" class="button button-primary">
                    <?php esc_html_e( 'Get API Token', 'unplug' ); ?>
                </a>
            </div>
            
            <div class="help-card">
                <h3><?php esc_html_e( 'Documentation', 'unplug' ); ?></h3>
                <p><?php esc_html_e( 'Learn how to use Unplug to identify and safely remove plugin bloat.', 'unplug' ); ?></p>
                <a href="https://mulberrytech.ca/unplug/docs" target="_blank" class="button button-secondary">
                    <?php esc_html_e( 'View Documentation', 'unplug' ); ?>
                </a>
            </div>
            
            <div class="help-card">
                <h3><?php esc_html_e( 'Support', 'unplug' ); ?></h3>
                <p><?php esc_html_e( 'Need help? Our support team is here to assist you.', 'unplug' ); ?></p>
                <a href="https://mulberrytech.ca/unplug/support" target="_blank" class="button button-secondary">
                    <?php esc_html_e( 'Get Support', 'unplug' ); ?>
                </a>
            </div>
        </div>
    </div>
</div>

 