<?php

/**
 * Provide a admin area view for the plugin
 *
 * This file is used to markup the admin-facing aspects of the plugin.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/admin/partials
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Double-check user capabilities
if ( ! current_user_can( 'manage_options' ) ) {
    wp_die( esc_html( __( 'You do not have sufficient permissions to access this page.', 'unplug' ) ) );
}

// Load helper classes
require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-unplug-options.php';

// Ensure we have access to plugin functions
if ( ! function_exists( 'get_plugins' ) ) {
    require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
}

// Get current user tier
$user_tier = UNPLUG_Options::get_user_tier();
$api_token = UNPLUG_Options::get_api_token();

// Initialize stats variables
$total_plugins = 0;
$active_plugins = 0;
$inactive_plugins = 0;
$last_scan = null;

// Get plugin statistics
$all_plugins = get_plugins();
$active_plugins_list = get_option( 'active_plugins', array() );

$total_plugins = count( $all_plugins );
$active_plugins = count( $active_plugins_list );
$inactive_plugins = $total_plugins - $active_plugins;

// Get last scan time
$last_scan = get_option( 'unplug_last_scan_time', null );

?>

<?php
// Enqueue admin styles inline
wp_add_inline_style( 'unplug-admin', '
.unplug-summary-table {
    margin: 1em auto;
    width: 100%;
    border-collapse: collapse;
    text-align: center;
}
.unplug-summary-table thead tr {
    text-align: center;
}
.unplug-summary-table th, .unplug-summary-table td {
    text-align: center;
    vertical-align: middle;
    padding: 8px 12px;
    border: 1px solid #e1e1e1;
}
.unplug-summary-table th {
    background: #f7f7f7;
    font-weight: bold;
}

/* Center headers for all columns on the Activity Scan table */
.unplug-plugins-table thead th {
    text-align: center !important;
}
' );
?>

<div class="wrap">
    <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
    
    <!-- Tab Navigation -->
    <div class="unplug-tabs">
                <button class="unplug-tab-link active" data-tab="home"><?php esc_html_e( 'Home', 'unplug' ); ?></button>
                <button class="unplug-tab-link" data-tab="main"><?php esc_html_e( 'Activity Scan', 'unplug' ); ?></button>
                <?php // Temporarily hidden: Advanced Tools tab ?>
                <?php // <button class="unplug-tab-link" data-tab="advanced">Advanced Tools</button> ?>
                <button class="unplug-tab-link" data-tab="settings"><?php esc_html_e( 'Settings', 'unplug' ); ?></button>
    </div>

        <!-- Home Tab Content -->
        <div class="unplug-tab-content" id="unplug-tab-home">
            <div class="unplug-intro">
                <div class="unplug-hero-section">
                    <div class="unplug-hero-content">
                        <div class="unplug-hero-text">
                            <h2><?php esc_html_e( 'Optimize Your WordPress Performance', 'unplug' ); ?></h2>
                            <p class="unplug-hero-description">
                                <?php esc_html_e( 'Unplug intelligently analyzes your WordPress plugins to identify unused code, measure performance impact, and safely remove plugin bloat. Perfect for inherited websites, client audits, and performance optimization.', 'unplug' ); ?>
                            </p>
                            <div class="unplug-hero-features">
                                <div class="unplug-feature">
                                    <span class="dashicons dashicons-search"></span>
                                    <span><?php esc_html_e( 'Smart Plugin Analysis', 'unplug' ); ?></span>
                                </div>
                                <div class="unplug-feature">
                                    <span class="dashicons dashicons-performance"></span>
                                    <span><?php esc_html_e( 'Performance Impact Testing', 'unplug' ); ?></span>
                                </div>
                                <div class="unplug-feature">
                                    <span class="dashicons dashicons-shield"></span>
                                    <span><?php esc_html_e( 'Safe Deactivation Testing', 'unplug' ); ?></span>
                                </div>
                            </div>
                        </div>
                        <div class="unplug-hero-visual">
                            <div class="unplug-dashboard-preview">
                                <div class="unplug-preview-header">
                                    <span class="unplug-preview-dot"></span>
                                    <span class="unplug-preview-dot"></span>
                                    <span class="unplug-preview-dot"></span>
                                </div>
                                <div class="unplug-preview-content">
                                    <div class="unplug-preview-row">
                                        <div class="unplug-preview-plugin">Plugin Name</div>
                                        <div class="unplug-preview-status active">Active</div>
                                        <div class="unplug-preview-result used">✅ Used</div>
                                    </div>
                                    <div class="unplug-preview-row">
                                        <div class="unplug-preview-plugin">Another Plugin</div>
                                        <div class="unplug-preview-status inactive">Inactive</div>
                                        <div class="unplug-preview-result unused">❌ Unused</div>
                                    </div>
                                    <div class="unplug-preview-row">
                                        <div class="unplug-preview-plugin">Third Plugin</div>
                                        <div class="unplug-preview-status active">Active</div>
                                        <div class="unplug-preview-result maybe">⚠️ Maybe</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="unplug-benefits-section">
                    <h3><?php esc_html_e( 'Why Choose Unplug?', 'unplug' ); ?></h3>
                    <div class="unplug-benefits-grid">
                        <div class="unplug-benefit">
                            <div class="unplug-benefit-icon">
                                <span class="dashicons dashicons-clock"></span>
                            </div>
                            <h4><?php esc_html_e( 'Save Time', 'unplug' ); ?></h4>
                            <p><?php esc_html_e( 'Instantly identify which plugins are actually being used instead of manually checking each one.', 'unplug' ); ?></p>
                        </div>
                        <div class="unplug-benefit">
                            <div class="unplug-benefit-icon">
                                <span class="dashicons dashicons-dashboard"></span>
                            </div>
                            <h4><?php esc_html_e( 'Boost Performance', 'unplug' ); ?></h4>
                            <p><?php esc_html_e( 'Remove unused plugins to reduce server load, improve page speed, and enhance user experience.', 'unplug' ); ?></p>
                        </div>
                        <div class="unplug-benefit">
                            <div class="unplug-benefit-icon">
                                <span class="dashicons dashicons-lock"></span>
                            </div>
                            <h4><?php esc_html_e( 'Enhance Security', 'unplug' ); ?></h4>
                            <p><?php esc_html_e( 'Reduce your attack surface by removing unnecessary plugins that could become security vulnerabilities.', 'unplug' ); ?></p>
                        </div>
                        <div class="unplug-benefit">
                            <div class="unplug-benefit-icon">
                                <span class="dashicons dashicons-admin-tools"></span>
                            </div>
                            <h4><?php esc_html_e( 'Easy Maintenance', 'unplug' ); ?></h4>
                            <p><?php esc_html_e( 'Simplify your WordPress maintenance by keeping only the plugins you actually need.', 'unplug' ); ?></p>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="unplug-help-section">
                <h3><?php esc_html_e( 'Getting Started', 'unplug' ); ?></h3>
                <div class="help-cards">
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Need an API Token?', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Visit our website to sign up for a free account and get your API token.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug" target="_blank" class="button button-primary">
                            <?php esc_html_e( 'Get API Token', 'unplug' ); ?>
                        </a>
                    </div>
                    
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Documentation', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Learn how to use Unplug to identify and safely remove plugin bloat.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug/docs" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'View Documentation', 'unplug' ); ?>
                        </a>
                    </div>
                    
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Support', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Need help? Our support team is here to assist you.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug/support" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'Get Support', 'unplug' ); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Tab Content -->
        <div class="unplug-tab-content" id="unplug-tab-main" style="display: none;">
            <?php
            // Show API token setup message if no token is configured
            if ( empty( $api_token ) ) {
                ?>
                <div class="notice notice-warning">
                    <p>
                        <strong><?php esc_html_e( 'API Token Required', 'unplug' ); ?></strong>
                    </p>
                    <p>
                        <?php esc_html_e( 'To unlock Pro features, please configure your API token in the settings tab.', 'unplug' ); ?>
                        <button class="button button-primary unplug-tab-link" data-tab="settings">
                            <?php esc_html_e( 'Configure API Token', 'unplug' ); ?>
                        </button>
                    </p>
                </div>
                <?php
            }
            ?>
            <!-- Queue Progress Bar (Hidden by default) -->
            <div id="unplug-progress-status" style="display: none;"></div>
            <progress id="unplug-progress-bar" value="0" max="100" style="display: none;"></progress>
            <span id="unplug-progress-percent" style="display: none;">0%</span>
            <pre id="unplug-progress-log" style="display: none;"></pre>
            <!-- End Queue Progress Bar -->

            <div class="unplug-main-content">
                <div class="unplug-scan-controls">
                    <button id="start-scan" class="button button-primary">
                        <span class="dashicons dashicons-update"></span>
                        <span class="button-text"><?php esc_html_e( 'Start Activity Scan', 'unplug' ); ?></span>
                    </button>
                    <button id="refresh-plugins" class="button button-secondary">
                        <span class="dashicons dashicons-update"></span>
                        <span class="button-text"><?php esc_html_e( 'Refresh Plugin List', 'unplug' ); ?></span>
                    </button>
                </div>

                <!-- Progress Bar -->
                <div id="unplug-scan-progress" class="unplug-progress-container" style="display: none;">
                    <div class="unplug-progress-bar">
                        <div class="unplug-progress-fill" style="width: 0%;"></div>
                    </div>
                    <span id="progress-percentage">0%</span>
                </div>

                <!-- Results Section -->
                <div id="unplug-results-content">
                    <?php
                    // Get all plugins for the analysis table
                    $all_plugins = get_plugins();
                    $active_plugins_list = get_option( 'active_plugins', array() );
                    if ( ! empty( $all_plugins ) ) :
                    ?>
                        <div class="unplug-plugins-table-container">
                            <table class="wp-list-table widefat fixed striped unplug-plugins-table">
                                <thead>
                                    <tr>
                                        <th class="column-plugin-name"><?php esc_html_e( 'Plugin Name', 'unplug' ); ?></th>
                                        <th class="column-status"><?php esc_html_e( 'Status', 'unplug' ); ?></th>
                                        <th class="column-activity"><?php esc_html_e( 'Activity Scan', 'unplug' ); ?></th>
                                        <th class="column-locations"><?php esc_html_e( 'Locations', 'unplug' ); ?></th>
                                        <?php if ( $user_tier !== 'free' ) : ?>
                                            <th class="column-speed"><?php esc_html_e( 'Speed Impact', 'unplug' ); ?></th>
                                            <th class="column-deactivation"><?php esc_html_e( 'Deactivation Test', 'unplug' ); ?></th>
                                        <?php endif; ?>
                                        <th class="column-actions"><?php esc_html_e( 'Actions', 'unplug' ); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php
                                    foreach ( $all_plugins as $plugin_file => $plugin_data ) :
                                        $is_active = in_array( $plugin_file, $active_plugins_list );
                                        $plugin_slug = dirname( $plugin_file );
                                        if ( $plugin_slug === '.' ) {
                                            $plugin_slug = basename( $plugin_file, '.php' );
                                        }
                                        // Get test results (placeholder for now)
                                        $activity_result = get_option( "unplug_activity_result_{$plugin_slug}", null );
                                        $location_result = get_option( "unplug_location_result_{$plugin_slug}", null );
                                        $speed_result = get_option( "unplug_speed_result_{$plugin_slug}", null );
                                        $deactivation_result = get_option( "unplug_deactivation_result_{$plugin_slug}", null );
                                    ?>
                                    <tr data-plugin-slug="<?php echo esc_attr( $plugin_slug ); ?>" data-plugin-file="<?php echo esc_attr( $plugin_file ); ?>">
                                        <td class="plugin-name">
                                            <strong><?php echo esc_html( $plugin_data['Name'] ); ?></strong>
                                            <?php if ( ! empty( $plugin_data['Version'] ) ) : ?>
                                                <span class="plugin-version">v<?php echo esc_html( $plugin_data['Version'] ); ?></span>
                                            <?php endif; ?>
                                            <?php if ( ! empty( $plugin_data['Description'] ) ) : ?>
                                                <div class="plugin-description"><?php echo esc_html( wp_trim_words( $plugin_data['Description'], 15 ) ); ?></div>
                                            <?php endif; ?>
                                        </td>
                                        <td class="plugin-status">
                                            <span class="status-badge status-<?php echo $is_active ? 'active' : 'inactive'; ?>">
                                                <?php echo $is_active ? esc_html( __( 'Active', 'unplug' ) ) : esc_html( __( 'Inactive', 'unplug' ) ); ?>
                                            </span>
                                        </td>
                                        
                                        <td class="plugin-activity">
                                            <?php
                                            // Show 'Used' if any Usage Detected (high_confidence_count > 0 for any type)
                                            $usage_detected = false;
                                            if ( is_array( $activity_result['details'] ) ) {
                                                foreach (array('shortcodes','blocks','widgets','elementor','custom_post_types','custom_taxonomies','menus','options') as $type) {
                                                    if (!empty($activity_result['details'][$type]['high_confidence_count']) && intval($activity_result['details'][$type]['high_confidence_count']) > 0) {
                                                        $usage_detected = true;
                                                        break;
                                                    }
                                                }
                                            }
                                            ?>
                                            <span class="test-result result-<?php echo esc_attr( $usage_detected ? 'used' : $activity_result['status'] ); ?>">
                                                <?php
                                                if ($usage_detected) {
                                                    echo '✅ ' . esc_html( __( 'Usage Detected', 'unplug' ) );
                                                } else {
                                                    switch ( $activity_result['status'] ) {
                                                        case 'used':
                                                            echo '✅ ' . esc_html( __( 'Usage Detected', 'unplug' ) );
                                                            break;
                                                        case 'possibly_used':
                                                            echo '⚠️ ' . esc_html( __( 'Possibly Used', 'unplug' ) );
                                                            break;
                                                        case 'no_usage':
                                                            echo '❌ ' . esc_html( __( 'No Usage Found', 'unplug' ) );
                                                            break;
                                                        default:
                                                            echo '❓ ' . esc_html( __( 'Unknown', 'unplug' ) );
                                                    }
                                                }
                                                ?>
                                            </span>
                                        </td>
                                        
                                        <td class="plugin-locations">
                                            <?php
                                            // Show count of Usage Detected Locations (sum of all high confidence locations)
                                            // Now also include high-confidence, non-post (global/site-wide) evidence
                                            $usage_detected_locations_count = 0; // post/page-bound
                                            $global_usage_detected_count   = 0; // non-post/site-wide
                                            if ( is_array( $activity_result['details'] ) ) {
                                                foreach ( array( 'shortcodes','blocks','widgets','elementor','custom_post_types','custom_taxonomies','menus','options' ) as $type ) {
                                                    if ( ! empty( $activity_result['details'][ $type ]['sample_items'] ) ) {
                                                        foreach ( $activity_result['details'][ $type ]['sample_items'] as $item ) {
                                                            if ( isset( $item['confidence'] ) && intval( $item['confidence'] ) >= 80 ) {
                                                                if ( isset( $item['post_id'] ) && $item['post_id'] ) {
                                                                    $usage_detected_locations_count++;
                                                                } else {
                                                                    $global_usage_detected_count++;
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                            }
                                            ?>
                                            <span class="location-count">
                                                <?php echo esc_html( $usage_detected_locations_count ); ?> <?php esc_html_e( 'locations', 'unplug' ); ?>
                                            </span>
                                            <?php if ( $global_usage_detected_count > 0 ) : ?>
                                                <span class="location-count" title="High-confidence usage detected that is not tied to a specific post/page (e.g., global hooks, header/footer injections)">
                                                    + <?php echo esc_html( $global_usage_detected_count ); ?> <?php esc_html_e( 'site-wide', 'unplug' ); ?>
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        
                                        <?php if ( $user_tier !== 'free' ) : ?>
                                            <td class="plugin-speed">
                                                <?php if ( $speed_result ) : ?>
                                                    <span class="speed-impact impact-<?php echo esc_attr( $speed_result['impact'] ); ?>">
                                                        <?php
                                                        switch ( $speed_result['impact'] ) {
                                                            case 'high':
                                                                echo '🔴 ' . esc_html( __( 'High', 'unplug' ) );
                                                                break;
                                                            case 'medium':
                                                                echo '🟡 ' . esc_html( __( 'Medium', 'unplug' ) );
                                                                break;
                                                            case 'low':
                                                                echo '🟢 ' . esc_html( __( 'Low', 'unplug' ) );
                                                                break;
                                                            default:
                                                                echo '⚪ ' . esc_html( __( 'Unknown', 'unplug' ) );
                                                        }
                                                        ?>
                                                    </span>
                                                <?php else : ?>
                                                    <span class="test-result result-untested">
                                                        <?php esc_html_e( 'Not Tested', 'unplug' ); ?>
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            
                                            <td class="plugin-deactivation">
                                                <?php if ( $deactivation_result ) : ?>
                                                    <span class="deactivation-result result-<?php echo esc_attr( $deactivation_result['status'] ); ?>">
                                                        <?php
                                                        switch ( $deactivation_result['status'] ) {
                                                            case 'safe':
                                                                echo '✅ ' . esc_html( __( 'Safe', 'unplug' ) );
                                                                break;
                                                            case 'warning':
                                                                echo '⚠️ ' . esc_html( __( 'Warning', 'unplug' ) );
                                                                break;
                                                            case 'critical':
                                                                echo '❌ ' . esc_html( __( 'Critical', 'unplug' ) );
                                                                break;
                                                            default:
                                                                echo '❓ ' . esc_html( __( 'Unknown', 'unplug' ) );
                                                        }
                                                        ?>
                                                    </span>
                                                <?php else : ?>
                                                    <span class="test-result result-untested">
                                                        <?php esc_html_e( 'Not Tested', 'unplug' ); ?>
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                        <?php endif; ?>
                                        
                                        <td class="plugin-actions">
                                            <div class="action-buttons">
                                                <?php if ( ! $activity_result ) : ?>
                                                                                                        <button class="button button-secondary run-activity-test" data-plugin="<?php echo esc_attr( $plugin_slug ); ?>">
                                                    <?php esc_html_e( 'Run Activity Scan', 'unplug' ); ?>
                                                </button>
                                                <?php endif; ?>
                                                
                                                <?php if ( $user_tier !== 'free' ) : ?>
                                                    <?php if ( ! $speed_result ) : ?>
                                                        <button class="button button-secondary run-speed-test" data-plugin="<?php echo esc_attr( $plugin_slug ); ?>">
                                                            <?php esc_html_e( 'Run Speed', 'unplug' ); ?>
                                                        </button>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ( ! $deactivation_result && $is_active ) : ?>
                                                        <button class="button button-secondary run-deactivation-test" data-plugin="<?php echo esc_attr( $plugin_slug ); ?>">
                                                            <?php esc_html_e( 'Test Deactivation', 'unplug' ); ?>
                                                        </button>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                                
                                                <?php
                                                // Require valid token for viewing detailed locations table
                                                require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-unplug-security.php';
                                                $has_valid_token = UNPLUG_Security::has_valid_api_token();
                                                ?>
                                                <button class="button button-link toggle-plugin-details <?php echo $has_valid_token ? '' : 'disabled'; ?>" data-plugin="<?php echo esc_attr( $plugin_slug ); ?>" <?php echo $has_valid_token ? '' : 'disabled title="' . esc_attr__( 'This feature requires a valid API token. Upgrade to Pro to unlock detailed locations.', 'unplug' ) . '"'; ?>>
                                                    <?php esc_html_e( 'Details', 'unplug' ); ?>
                                                    <?php if ( ! $has_valid_token ) : ?><span class="tier-badge tier-pro">Pro</span><?php endif; ?>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    
                                    <!-- Plugin Details Row (Hidden by default) -->
                                    <tr class="plugin-details-row" id="details-<?php echo esc_attr( $plugin_slug ); ?>" style="display: none;">
                                        <td colspan="<?php echo $user_tier !== 'free' ? '7' : '5'; ?>">
                                            <div class="plugin-details-content">
                                                <div class="plugin-info">
                                                    <h4><?php esc_html_e( 'Plugin Information', 'unplug' ); ?></h4>
                                                    <p><strong><?php esc_html_e( 'File:', 'unplug' ); ?></strong> <?php echo esc_html( $plugin_file ); ?></p>
                                                    <?php if ( ! empty( $plugin_data['Author'] ) ) : ?>
                                                        <p><strong><?php esc_html_e( 'Author:', 'unplug' ); ?></strong> <?php echo esc_html( $plugin_data['Author'] ); ?></p>
                                                    <?php endif; ?>
                                                    <?php if ( ! empty( $plugin_data['PluginURI'] ) ) : ?>
                                                        <p><strong><?php esc_html_e( 'Plugin URI:', 'unplug' ); ?></strong> <a href="<?php echo esc_url( $plugin_data['PluginURI'] ); ?>" target="_blank"><?php echo esc_html( $plugin_data['PluginURI'] ); ?></a></p>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <div class="test-results-details">
                                                    <h4><?php esc_html_e( 'Test Results Details', 'unplug' ); ?></h4>
                                                    <div class="test-details-grid">
                                                        <?php if ( $activity_result ) : ?>
                                                            <div class="test-detail">
                                                                <strong><?php esc_html_e( 'Activity Scan Results:', 'unplug' ); ?></strong>
                                                                <?php
                                                                // Define all possible evidence types
                                                                $evidence_types = array('shortcodes','blocks','widgets','elementor','custom_post_types','custom_taxonomies','menus','options');
                                                                // No longer show high confidence locations here
                                                                if ( is_array( $activity_result['details'] ) ) :
                                                                    echo '<table class="unplug-summary-table"><thead><tr>';
                                                                    echo '<th>' . esc_html( __( 'Type', 'unplug' ) ) . '</th>';
                                                                    echo '<th>' . esc_html( __( 'Matching Results', 'unplug' ) ) . '</th>';
                                                                    echo '<th>' . esc_html( __( 'Usage Detected', 'unplug' ) ) . '</th>';
                                                                    echo '</tr></thead><tbody>';
                                                                    foreach ( $evidence_types as $type ) {
                                                                        $count = isset($activity_result['details'][$type]['count']) ? intval($activity_result['details'][$type]['count']) : 0;
                                                                        $high_conf = isset($activity_result['details'][$type]['high_confidence_count']) ? intval($activity_result['details'][$type]['high_confidence_count']) : 0;
                                                                        // Collect high confidence sample_items with post_id for locations
                                                                        if (!empty($activity_result['details'][$type]['sample_items'])) {
                                                                            foreach ($activity_result['details'][$type]['sample_items'] as $item) {
                                                                                if ((isset($item['confidence']) && $item['confidence'] >= 80) && isset($item['post_id'])) {
                                                                                    $high_confidence_locations[$type][] = $item;
                                                                                }
                                                                            }
                                                                        }
                                                                        echo '<tr>';
                                                                        echo '<td>' . esc_html(ucwords(str_replace('_', ' ', $type))) . '</td>';
                                                                        echo '<td>' . esc_html($count) . '</td>';
                                                                        echo '<td>' . esc_html($high_conf) . '</td>';
                                                                        echo '</tr>';
                                                                    }
                                                                    echo '</tbody></table>';
                                                                else :
                                                                    echo '<p>' . esc_html( $activity_result['details'] ?? esc_html( __( 'No details available', 'unplug' ) ) ) . '</p>';
                                                                endif;
                                                                ?>
                                                            </div>
                                                        <?php endif; ?>
                                                        
                                                        <?php if ( $location_result ) : ?>
                                                            <div class="test-detail">
                                                                <strong><?php esc_html_e( 'Pages Where Plugin Is Used', 'unplug' ); ?></strong>
                                                                <?php
                                                                // Show high confidence locations here
                                                                $high_confidence_locations = array();
                                                                if ( is_array( $activity_result['details'] ) ) {
                                                                    foreach ( $evidence_types as $type ) {
                                                                        if (!empty($activity_result['details'][$type]['sample_items'])) {
                                                                            foreach ($activity_result['details'][$type]['sample_items'] as $item) {
                                                                                if ((isset($item['confidence']) && $item['confidence'] >= 80) && isset($item['post_id'])) {
                                                                                    $high_confidence_locations[$type][] = $item;
                                                                                }
                                                                            }
                                                                        }
                                                                    }
                                                                }
                                                                if ( is_array( $location_result['details'] ) ) :
                                                                    echo '<ul>';
                                                                    foreach ( $location_result['details'] as $detail ) {
                                                                        if (is_array($detail) || is_object($detail)) {
                                                                            echo '<li><pre>' . esc_html( wp_json_encode($detail, JSON_PRETTY_PRINT) ) . '</pre></li>';
                                                                        } else {
                                                                            echo '<li>' . esc_html($detail) . '</li>';
                                                                        }
                                                                    }
                                                                    echo '</ul>';
                                                                else :
                                                                    echo '<p>' . esc_html( $location_result['details'] ?? __( 'No details available', 'unplug' ) ) . '</p>';
                                                                endif;
                                                                // Show high confidence locations as URLs under Location Test
                                                                foreach ($evidence_types as $type) {
                                                                    if (!empty($high_confidence_locations[$type])) {
                                                                        echo '<div class="unplug-high-confidence-locations">';
                                                                        echo '<strong>' . esc_html(ucwords(str_replace('_', ' ', $type))) . ' ' . esc_html( __('Usage Detected', 'unplug') ) . ':</strong>';
                                                                        echo '<ul>';
                                                                        foreach ($high_confidence_locations[$type] as $item) {
                                                                            $url = get_permalink($item['post_id']);
                                                                            $title = isset($item['post_title']) ? $item['post_title'] : ('Post ID ' . $item['post_id']);
                                                                            echo '<li><a href="' . esc_url($url) . '" target="_blank">' . esc_html($title) . '</a></li>';
                                                                        }
                                                                        echo '</ul>';
                                                                        echo '</div>';
                                                                    }
                                                                }
                                                                ?>
                                                            </div>
                                                        <?php endif; ?>
                                                        
                                                        <?php if ( $user_tier !== 'free' && $speed_result ) : ?>
                                                            <div class="test-detail">
                                                                <strong><?php esc_html_e( 'Speed Test:', 'unplug' ); ?></strong>
                                                                <p><?php echo esc_html( $speed_result['details'] ?? __( 'No details available', 'unplug' ) ); ?></p>
                                                            </div>
                                                        <?php endif; ?>
                                                        
                                                        <?php if ( $user_tier !== 'free' && $deactivation_result ) : ?>
                                                            <div class="test-detail">
                                                                <strong><?php esc_html_e( 'Deactivation Test:', 'unplug' ); ?></strong>
                                                                <p><?php echo esc_html( $deactivation_result['details'] ?? __( 'No details available', 'unplug' ) ); ?></p>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else : ?>
                        <p><?php esc_html_e( 'No plugins found. Please check your WordPress installation.', 'unplug' ); ?></p>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="unplug-help-section">
                <h3><?php esc_html_e( 'Getting Started', 'unplug' ); ?></h3>
                <div class="help-cards">
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Need an API Token?', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Visit our website to sign up for a free account and get your API token.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug" target="_blank" class="button button-primary">
                            <?php esc_html_e( 'Get API Token', 'unplug' ); ?>
                        </a>
                    </div>
                    
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Documentation', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Learn how to use Unplug to identify and safely remove plugin bloat.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug/docs" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'View Documentation', 'unplug' ); ?>
                        </a>
                    </div>
                    
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Support', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Need help? Our support team is here to assist you.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug/support" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'Get Support', 'unplug' ); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <?php // Temporarily hidden: Advanced Tab Content ?>
        <?php /* <div class="unplug-tab-content" id="unplug-tab-advanced" style="display: none;">
            <div class="unplug-advanced-content">
                <div class="unplug-advanced-intro">
                    <h3><?php esc_html_e( 'Advanced Plugin Analysis Tools', 'unplug' ); ?></h3>
                    <p><?php esc_html_e( 'Access powerful diagnostic tools for comprehensive plugin analysis, conflict detection, and system optimization.', 'unplug' ); ?></p>
                </div>
                
                <div class="unplug-advanced-sections">
                    <div class="unplug-advanced-section">
                        <h4><?php esc_html_e( 'Data Management', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Manage your plugin analysis data and export results for reporting.', 'unplug' ); ?></p>
                        <div class="context-actions">
                            <button type="button" class="button button-secondary" id="clear-location-data">
                                <span class="dashicons dashicons-trash"></span>
                                <?php esc_html_e( 'Clear Location Data', 'unplug' ); ?>
                            </button>
                            <button type="button" class="button button-secondary" id="export-location-data">
                                <span class="dashicons dashicons-download"></span>
                                <?php esc_html_e( 'Export Data', 'unplug' ); ?>
                            </button>
                        </div>
                    </div>
                    
                    <div class="unplug-advanced-section">
                        <h4><?php esc_html_e( 'Safe Deactivation Test', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Test if plugins can be safely deactivated without breaking your site functionality.', 'unplug' ); ?></p>
                        <div class="unplug-safe-mode-controls">
                            <button id="unplug-safe-mode-status" class="button button-primary <?php echo $user_tier === 'free' ? 'disabled' : ''; ?>" <?php echo $user_tier === 'free' ? 'disabled title="' . esc_attr__( 'This feature requires a Pro subscription. Upgrade to access safe mode protection during plugin testing.', 'unplug' ) . '"' : ''; ?>>
                                <span class="dashicons dashicons-shield"></span>
                                <?php esc_html_e( 'Run Safe Deactivation Test', 'unplug' ); ?>
                                <?php if ( $user_tier === 'free' ) : ?>
                                    <span class="tier-badge tier-pro">Pro</span>
                                <?php endif; ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="unplug-help-section">
                <h3><?php esc_html_e( 'Getting Started', 'unplug' ); ?></h3>
                <div class="help-cards">
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Need an API Token?', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Visit our website to sign up for a free account and get your API token.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug" target="_blank" class="button button-primary">
                            <?php esc_html_e( 'Get API Token', 'unplug' ); ?>
                        </a>
                    </div>
                    
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Documentation', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Learn how to use Unplug to identify and safely remove plugin bloat.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug/docs" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'View Documentation', 'unplug' ); ?>
                        </a>
                    </div>
                    
                    <div class="help-card">
                        <h4><?php esc_html_e( 'Support', 'unplug' ); ?></h4>
                        <p><?php esc_html_e( 'Need help? Our support team is here to assist you.', 'unplug' ); ?></p>
                        <a href="https://mulberrytech.ca/unplug/support" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'Get Support', 'unplug' ); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div> */ ?>
    </div>

    <!-- Results containers for advanced features -->
    <div id="unplug-inventory-results" class="unplug-results-container" style="display: none;"></div>
    <div id="unplug-conflict-results" class="unplug-results-container" style="display: none;"></div>
    <div id="unplug-category-results" class="unplug-results-container" style="display: none;"></div>
    <div id="unplug-advanced-conflict-results" class="unplug-results-container" style="display: none;"></div>
    <div id="unplug-warning-results" class="unplug-results-container" style="display: none;"></div>
    <div id="unplug-safe-mode-results" class="unplug-results-container" style="display: none;"></div>
</div>

<!-- Settings Tab Content -->
<div class="unplug-tab-content" id="unplug-tab-settings" style="display: none;">
    <?php
    // Get current settings
    $license_status = get_option('unplug_license_status', array());
    
    // Show success message if settings were updated
    if ( isset( $_GET['settings-updated'] ) && $_GET['settings-updated'] == 'true' && 
         isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'unplug_settings_nonce' ) ) {
        ?>
        <div class="notice notice-success">
            <p><?php esc_html_e( 'Settings saved successfully!', 'unplug' ); ?></p>
        </div>
        <?php
    }
    
    // Show API validation status
    if ( ! empty( $api_token ) ) {
        ?>
        <div class="notice notice-info">
            <p>
                <strong><?php esc_html_e( 'API Token Status:', 'unplug' ); ?></strong>
                <span class="api-status-badge api-status-<?php echo esc_attr( strtolower( $user_tier ) ); ?>">
                    <?php echo esc_html( ucfirst( $user_tier ) ); ?>
                </span>
            </p>
            <p>
                <?php 
                if ( $user_tier === 'free' ) {
                    esc_html_e( 'Your API token is valid but you\'re on the free tier. Upgrade to unlock Pro features.', 'unplug' );
                } else {
                    esc_html_e( 'Your API token is valid and active.', 'unplug' );
                }
                ?>
            </p>
        </div>
        <?php
    }

    // Show license status and last checked time
    if ( ! empty( $api_token ) ) {
        echo '<div class="notice notice-info unplug-license-status">';
        if ( isset( $license_status['status'] ) ) {
            if ( $license_status['status'] === 'valid' ) {
                echo '<p><span class="dashicons dashicons-yes" style="color:green;"></span> ' . esc_html( __( 'License is valid.', 'unplug' ) ) . '</p>';
            } elseif ( $license_status['status'] === 'expired' ) {
                echo '<p><span class="dashicons dashicons-warning" style="color:orange;"></span> ' . esc_html( __( 'License expired.', 'unplug' ) ) . '</p>';
            } elseif ( $license_status['status'] === 'invalid' ) {
                echo '<p><span class="dashicons dashicons-no" style="color:red;"></span> ' . esc_html( __( 'License is invalid.', 'unplug' ) ) . '</p>';
            }
        }
        if ( isset( $license_status['degraded'] ) && $license_status['degraded'] ) {
            echo '<p><span class="dashicons dashicons-info"></span> ' . esc_html( __( 'Using last known valid license state. The licensing server could not be reached.', 'unplug' ) ) . '</p>';
        }
        if ( isset( $license_status['error'] ) ) {
            echo '<p style="color:red;"><span class="dashicons dashicons-warning"></span> ' . esc_html( $license_status['error'] ) . '</p>';
        }
        if ( isset( $license_status['checked_at'] ) ) {
            // translators: %s: formatted date and time of last license check
            echo '<p><em>' . esc_html( sprintf( 
                /* translators: %s: formatted date and time of last license check */
                __( 'Last checked: %s', 'unplug' ), 
                esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), intval( $license_status['checked_at'] ) ) ) 
            ) ) . '</em></p>';
        }
        echo '</div>';
    }
    ?>
    
    <form method="post" action="options.php">
        <?php
        settings_fields( 'unplug_settings' );
        
        // Add security nonce field
        require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-unplug-security.php';
        UNPLUG_Security::nonce_field( 'settings', 'unplug_settings_nonce' );
        ?>
        
        <h2><?php esc_html_e( 'General Settings', 'unplug' ); ?></h2>
        
        <table class="form-table">
            <tbody>
                <tr>
                    <th scope="row">
                        <label for="api_token"><?php esc_html_e( 'API Token', 'unplug' ); ?></label>
                    </th>
                    <td>
                        <input type="password" 
                               id="main_api_token" 
                               name="unplug_settings[api_token]" 
                               value="<?php echo esc_attr( $api_token ); ?>" 
                               class="regular-text" 
                               autocomplete="new-password"
                               placeholder="<?php esc_html_e( 'Enter your API token...', 'unplug' ); ?>" />
                        <button type="button" id="main-toggle-token-visibility" class="button button-secondary">
                            <span class="dashicons dashicons-visibility"></span>
                            <?php esc_html_e( 'Show', 'unplug' ); ?>
                        </button>
                        <p class="description">
                            <?php esc_html_e( 'Enter your Unplug API token to access Pro features. Your token is stored securely and encrypted.', 'unplug' ); ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <?php esc_html_e( 'Current Tier', 'unplug' ); ?>
                    </th>
                    <td>
                        <div class="tier-display">
                            <span class="tier-badge tier-<?php echo esc_attr( strtolower( $user_tier ) ); ?>">
                                <?php echo esc_html( ucfirst( $user_tier ) ); ?>
                            </span>
                            
                            <?php if ( $user_tier === 'free' ) : ?>
                                <div class="tier-features">
                                    <h4><?php esc_html_e( 'Free Tier Features:', 'unplug' ); ?></h4>
                                    <ul>
                                        <li><?php esc_html_e( 'Basic Activity Scan', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Plugin Usage Detection', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Location Count (summary)', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'CSV Export', 'unplug' ); ?></li>
                                    </ul>
                                </div>
                            <?php elseif ( $user_tier === 'pro' ) : ?>
                                <div class="tier-features">
                                    <h4><?php esc_html_e( 'Pro Tier Features:', 'unplug' ); ?></h4>
                                    <ul>
                                        <li><?php esc_html_e( 'Advanced Activity Scan', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Deactivation Testing', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Detailed Location Analysis', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Speed Impact Analysis', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Enhanced Export Options', 'unplug' ); ?></li>
                                    </ul>
                                </div>
                            <?php elseif ( $user_tier === 'agency' ) : ?>
                                <div class="tier-features">
                                    <h4><?php esc_html_e( 'Agency Tier Features:', 'unplug' ); ?></h4>
                                    <ul>
                                        <li><?php esc_html_e( 'All Pro Features', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'White-label Reports', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Advanced Analytics', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Priority Support', 'unplug' ); ?></li>
                                        <li><?php esc_html_e( 'Multi-site Management', 'unplug' ); ?></li>
                                    </ul>
                                </div>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            </tbody>
        </table>
        
        <h2><?php esc_html_e( 'Advanced Settings', 'unplug' ); ?></h2>
        
        <?php
        // Render only the advanced settings section specifically
        global $wp_settings_sections, $wp_settings_fields;
        
        if ( isset( $wp_settings_sections['unplug_settings']['unplug_advanced'] ) ) {
            $section = $wp_settings_sections['unplug_settings']['unplug_advanced'];
            
            if ( $section['callback'] ) {
                call_user_func( $section['callback'], $section );
            }
            
            if ( ! isset( $wp_settings_fields ) || ! isset( $wp_settings_fields['unplug_settings'] ) || ! isset( $wp_settings_fields['unplug_settings']['unplug_advanced'] ) ) {
                return;
            }
            
            echo '<table class="form-table" role="presentation">';
            do_settings_fields( 'unplug_settings', 'unplug_advanced' );
            echo '</table>';
        }
        ?>
        
        <?php submit_button(); ?>
    </form>
    
    <div class="unplug-help-section">
        <h3><?php esc_html_e( 'Getting Started', 'unplug' ); ?></h3>
        <div class="help-cards">
            <div class="help-card">
                <h4><?php esc_html_e( 'Need an API Token?', 'unplug' ); ?></h4>
                <p><?php esc_html_e( 'Visit our website to sign up for a free account and get your API token.', 'unplug' ); ?></p>
                <a href="https://mulberrytech.ca/unplug" target="_blank" class="button button-primary">
                    <?php esc_html_e( 'Get API Token', 'unplug' ); ?>
                </a>
            </div>
            
            <div class="help-card">
                <h4><?php esc_html_e( 'Documentation', 'unplug' ); ?></h4>
                <p><?php esc_html_e( 'Learn how to use Unplug to identify and safely remove plugin bloat.', 'unplug' ); ?></p>
                <a href="https://mulberrytech.ca/unplug/docs" target="_blank" class="button button-secondary">
                    <?php esc_html_e( 'View Documentation', 'unplug' ); ?>
                </a>
            </div>
            
            <div class="help-card">
                <h4><?php esc_html_e( 'Support', 'unplug' ); ?></h4>
                <p><?php esc_html_e( 'Need help? Our support team is here to assist you.', 'unplug' ); ?></p>
                <a href="https://mulberrytech.ca/unplug/support" target="_blank" class="button button-secondary">
                    <?php esc_html_e( 'Get Support', 'unplug' ); ?>
                </a>
            </div>
        </div>
    </div>
</div>
<!-- End Settings Tab -->

<?php
// Enqueue admin JavaScript inline
$admin_js = "
jQuery(document).ready(function($) {
    // Initialize dashboard functionality
    
    // Plugin table sorting
    $('.wp-list-table th.sortable a').on('click', function(e) {
        e.preventDefault();
        var \$this = $(this);
        var \$th = \$this.closest('th');
        var column = \$th.attr('class').match(/column-(\\w+)/)[1];
        
        // Toggle sort direction
        if (\$th.hasClass('sorted')) {
            \$th.toggleClass('asc desc');
        } else {
            $('.wp-list-table th').removeClass('sorted asc desc');
            \$th.addClass('sorted asc');
        }
        
        // TODO: Implement actual sorting logic
        console.log('Sort by:', column, \$th.hasClass('asc') ? 'asc' : 'desc');
    });
    
    // Refresh plugin list
    $('#refresh-plugins').on('click', function() {
        var \$btn = $(this);
        \$btn.prop('disabled', true);
        
        // TODO: Implement plugin refresh via AJAX
        console.log('Refreshing plugin list...');
        
        setTimeout(function() {
            \$btn.prop('disabled', false);
        }, 2000);
    });
    
    function pollProgress() {
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: { action: 'unplug_get_progress' },
            success: function(response) {
                if (response.success) {
                    $('#unplug-progress-bar').val(response.data.percent);
                    $('#unplug-progress-percent').text(response.data.percent + '%');
                    $('#unplug-progress-status').text(response.data.status);
                    $('#unplug-progress-log').text(response.data.log);
                    if (response.data.percent < 100) {
                        setTimeout(pollProgress, 2000);
                    }
                }
            },
            error: function() {
                $('#unplug-progress-status').text('Error fetching progress');
            }
        });
    }

    // Start polling when scan starts
    $('#start-scan').on('click', function() {
        var \$btn = $(this);
        \$btn.prop('disabled', true);
        
        // Show progress bar
        $('#unplug-scan-progress').show();
        
        // TODO: Implement actual scan via AJAX
        console.log('Starting activity scan...');
        
        // Simulate progress
        var progress = 0;
        var interval = setInterval(function() {
            progress += Math.random() * 10;
            if (progress > 100) {
                progress = 100;
                clearInterval(interval);
                $('#start-scan').prop('disabled', false);
                $('#unplug-scan-progress').hide();
            }
            
            $('.unplug-progress-fill').css('width', progress + '%');
            $('#progress-percentage').text(Math.round(progress) + '%');
        }, 200);
        pollProgress();
    });
    
    // Export functionality
    $('#export-csv').on('click', function(e) {
        e.preventDefault();
        window.open(ajaxurl + '?action=unplug_export_plugins_csv&nonce=' + unplugAdmin.exportNonce, '_blank');
    });
    
    $('#export-report').on('click', function() {
        // TODO: Implement report generation
        console.log('Generating report...');
    });
    
    // Safe mode functionality
    $('#unplug-safe-mode-status').on('click', function() {
        unplugSafeMode.displaySafeModeStatus('unplug-safe-mode-results');
    });
    
    $('#unplug-activate-safe-mode').on('click', function() {
        if (confirm('Are you sure you want to activate Safe Mode? This will disable conflicting plugins.')) {
            unplugSafeMode.activateSafeMode().then(function() {
                unplugSafeMode.displaySafeModeStatus('unplug-safe-mode-results');
            });
        }
    });
    
    $('#unplug-emergency-mode').on('click', function() {
        if (confirm('Are you sure you want to activate Emergency Mode? This will disable most plugins and restrict functionality.')) {
            unplugSafeMode.activateSafeMode({ emergency_mode: true }).then(function() {
                unplugSafeMode.displaySafeModeStatus('unplug-safe-mode-results');
            });
        }
    });

    // Queue Management: Remove task
    $(document).on('click', '.unplug-queue-remove', function() {
        var \$btn = $(this);
        var taskId = \$btn.data('task-id');
        if (!confirm('Are you sure you want to remove this task?')) return;
        $.post(ajaxurl, { 
            action: 'unplug_remove_queue_task', 
            task_id: taskId,
            nonce: unplugAdmin.removeTaskNonce
        }, function(response) {
            if (response.success) {
                \$btn.closest('tr').fadeOut(300, function() { $(this).remove(); });
            } else {
                alert(response.data && response.data.message ? response.data.message : 'Failed to remove task.');
            }
        });
    });

    // Tab switching functionality
    $('.unplug-tab-link').on('click', function(e) {
        e.preventDefault();
        var targetTab = $(this).data('tab');
        
        // Remove active class from all tabs and content
        $('.unplug-tab-link').removeClass('active');
        $('.unplug-tab-content').removeClass('active');
        
        // Add active class to clicked tab and show content
        $(this).addClass('active');
        $('#unplug-tab-' + targetTab).addClass('active');
    });
    
    // Settings tab functionality
    $('#main-toggle-token-visibility').on('click', function() {
        var \$btn = $(this);
        var \$input = $('#main_api_token');
        var \$icon = \$btn.find('.dashicons');
        var \$text = \$btn.contents().filter(function() {
            return this.nodeType === 3; // Text nodes
        });
        
        if (\$input.attr('type') === 'password') {
            \$input.attr('type', 'text');
            \$icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            \$text[0].textContent = '" . esc_js( __( 'Hide', 'unplug' ) ) . "';
        } else {
            \$input.attr('type', 'password');
            \$icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            \$text[0].textContent = '" . esc_js( __( 'Show', 'unplug' ) ) . "';
        }
    });
    
    // API token validation on blur
    $('#main_api_token').on('blur', function() {
        var token = $(this).val();
        if (token.length > 0 && token.length < 32) {
            $(this).css('border-color', '#dc3232');
            if (!$('.main-api-token-warning').length) {
                $(this).after('<p class=\"main-api-token-warning description\" style=\"color: #dc3232;\">" . esc_js( __( 'API token appears to be invalid. Please check your token.', 'unplug' ) ) . "</p>');
            }
        } else {
            $(this).css('border-color', '');
            $('.main-api-token-warning').remove();
        }
    });
});
";

wp_add_inline_script( 'unplug-admin', $admin_js );
?> 