(function( $ ) {
    'use strict';

    /**
     * Unplug Admin JavaScript
     * 
     * This file contains the admin-specific JavaScript functionality
     * for the Unplug plugin dashboard and settings pages.
     */

    let scanInProgress = false;

    function setScanButtonState() {
        // Enable the scan button unless scan is in progress
        $('#start-scan').prop('disabled', scanInProgress);
        
        // Update button text based on state
        if (scanInProgress) {
            $('#start-scan').addClass('loading').find('.button-text').text('Scanning...');
        } else {
            $('#start-scan').removeClass('loading').find('.button-text').text('Start Activity Scan');
        }
    }

    // Initialize when DOM is ready
    $(document).ready(function() {
        
        // Initialize tab navigation
        initializeTabNavigation();
        
        // Initialize dashboard if present
        initializeDashboard();
        
        // Initialize settings if present
        if ($('.unplug-settings').length) {
            initializeSettings();
        }
        
        // Initialize general admin functionality
        initializeAdminCommon();
        
        // Enable scan button on page load
        setScanButtonState();
    });

    /**
     * Initialize tab navigation functionality
     */
    function initializeTabNavigation() {
        $('.unplug-tab-link').on('click', function(e) {
            e.preventDefault();
            const targetTab = $(this).data('tab');
            // Store last active tab in localStorage
            localStorage.setItem('unplugLastTab', targetTab);
            // Remove active class from all tabs and content
            $('.unplug-tab-link').removeClass('active');
            $('.unplug-tab-content').hide();
            // Add active class to clicked tab
            $(this).addClass('active');
            // Show corresponding content
            $('#unplug-tab-' + targetTab).show();
        });
        // On page load, prefer URL param (?unplug_tab=...) then default to Home
        var params = new URLSearchParams(window.location.search);
        var urlTab = params.get('unplug_tab');
        var lastTab = urlTab || 'home';
        $('.unplug-tab-link').removeClass('active');
        $('.unplug-tab-content').hide();
        $(".unplug-tab-link[data-tab='" + lastTab + "']").addClass('active');
        $('#unplug-tab-' + lastTab).show();
    }

    /**
     * Initialize Dashboard functionality
     */
    function initializeDashboard() {
        
        // Plugin table sorting
        $('.wp-list-table th.sortable a').on('click', function(e) {
            e.preventDefault();
            handleTableSort($(this));
        });
        
        // Refresh plugin list
        $('#refresh-plugins').on('click', function() {
            handlePluginRefresh($(this));
        });
        
        // Start scan functionality
        $('#start-scan').on('click', function() {
            handleStartScan($(this));
        });
        
        // Export functionality
        $('#export-csv').on('click', function() {
            handleExportCSV();
        });
        
        $('#export-report').on('click', function() {
            handleExportReport();
        });
        
        // Advanced tab functionality
        $('#clear-location-data').on('click', function() {
            handleClearLocationData($(this));
        });
        
        $('#export-location-data').on('click', function() {
            handleExportLocationData($(this));
        });
        
        // Individual plugin test buttons
        $(document).on('click', '.run-activity-test', function() {
            handleIndividualTest($(this), 'activity');
        });
        
        $(document).on('click', '.run-location-test', function() {
            handleIndividualTest($(this), 'location');
        });
        
        $(document).on('click', '.run-speed-test', function() {
            handleIndividualTest($(this), 'speed');
        });
        
        $(document).on('click', '.run-deactivation-test', function() {
            handleIndividualTest($(this), 'deactivation');
        });
        
        // Plugin details toggle
        $(document).on('click', '.toggle-plugin-details', function(e) {
            // Block when disabled (requires valid token)
            if ($(this).hasClass('disabled') || $(this).is('[disabled]')) {
                e.preventDefault();
                return false;
            }
            togglePluginDetails($(this));
        });
        
        // View location details
        $(document).on('click', '.view-locations', function() {
            viewLocationDetails($(this));
        });
        
        // Check if we should enable scan button
        checkScanButtonState();
    }

    /**
     * Initialize Settings functionality
     */
    function initializeSettings() {
        
        // Token visibility toggle
        $('#toggle-token-visibility').on('click', function() {
            toggleTokenVisibility($(this));
        });
        
        // API token validation
        $('#api_token').on('blur', function() {
            validateApiToken($(this));
        });
        
        // Settings form validation
        $('form[action="options.php"]').on('submit', function(e) {
            if (!validateSettingsForm()) {
                e.preventDefault();
            }
        });
    }

    /**
     * Initialize common admin functionality
     */
    function initializeAdminCommon() {
        
        // Dismiss notices
        $(document).on('click', '.notice-dismiss', function() {
            var $notice = $(this).closest('.notice');
            $notice.fadeOut(300);
        });
        
        // Tooltips (if needed in future)
        if (typeof $.fn.tooltip !== 'undefined') {
            $('[data-tooltip]').tooltip();
        }
    }

    /**
     * Handle table sorting
     */
    function handleTableSort($link) {
        var $th = $link.closest('th');
        var column = $th.attr('class').match(/column-(\w+)/);
        
        if (!column) {
            return;
        }
        
        column = column[1];
        
        // Toggle sort direction
        if ($th.hasClass('sorted')) {
            $th.toggleClass('asc desc');
        } else {
            $('.wp-list-table th').removeClass('sorted asc desc');
            $th.addClass('sorted asc');
        }
        
        var direction = $th.hasClass('asc') ? 'asc' : 'desc';
        
        // TODO: Implement actual sorting logic with AJAX
        // Sort functionality implemented
        
        // For now, just show a loading state
        $th.addClass('loading');
        setTimeout(function() {
            $th.removeClass('loading');
        }, 1000);
    }

    /**
     * Handle plugin list refresh
     */
    function handlePluginRefresh($btn) {
        var originalText = $btn.text();
        
        $btn.prop('disabled', true)
            .addClass('loading')
            .text('Refreshing...');
        
        // Send AJAX request to refresh plugins
        sendAjaxRequest('unplug_refresh_plugins', {}, function(response) {
            $btn.prop('disabled', false)
                .removeClass('loading')
                .text(originalText);
                
            // Show success message
            showNotice(response.message, 'success');
            
            // Refresh the page to show updated plugin table
            location.reload();
        }, function(errorMessage) {
            // Error callback - restore button state
            $btn.prop('disabled', false)
                .removeClass('loading')
                .text(originalText);
        });
    }

    /**
     * Helper function to safely display status text
     */
    function safeStatus(status) {
        if (!status) return 'Processing...';
        return String(status).replace(/[<>&"']/g, function(match) {
            switch(match) {
                case '<': return '&lt;';
                case '>': return '&gt;';
                case '&': return '&amp;';
                case '"': return '&quot;';
                case "'": return '&#x27;';
                default: return match;
            }
        });
    }

    /**
     * Refresh the plugins table without reloading the page
     */
    function refreshPluginsTable() {
        // Send AJAX request to get updated plugin data
        sendAjaxRequest('unplug_refresh_plugins', {}, function(response) {
            if (response.html) {
                // Replace the plugins table with updated HTML
                $('.unplug-plugins-table tbody').html(response.html);
                showNotice('Plugin results updated successfully!', 'success');
            }
        }, function(errorMessage) {
            console.log('Failed to refresh plugins table:', errorMessage);
        });
    }

    /**
     * Handle start scan button click
     */
    function handleStartScan($btn) {
        scanInProgress = true;
        setScanButtonState();
        
        // Update button appearance
        $btn.addClass('loading');
        $btn.find('.button-text').text('Scanning...');
        
        // Show progress bar
        $('#unplug-scan-progress').show();
        
        sendAjaxRequest('unplug_start_scan', {}, function(response) {
            trackScanProgress(response.scan_id);
            showNotice(response.message, 'success');
            
            // Instead of reloading, refresh the plugins table to show updated results
            setTimeout(function() {
                refreshPluginsTable();
            }, 2000);
        }, function(errorMessage) {
            scanInProgress = false;
            setScanButtonState();
            $btn.removeClass('loading');
            $btn.find('.button-text').text('Start Activity Scan');
            $('#unplug-scan-progress').hide();
        });
        
        // Fallback timeout to reset button state
        setTimeout(function() {
            scanInProgress = false;
            setScanButtonState();
            $btn.removeClass('loading');
            $btn.find('.button-text').text('Start Activity Scan');
            $('#unplug-scan-progress').hide();
            $('#export-csv, #export-report').prop('disabled', false);
        }, 30000);
    }

    /**
     * Track scan progress via AJAX
     */
    function trackScanProgress(scanId) {
        var progressInterval = setInterval(function() {
            sendAjaxRequest('unplug_scan_progress', {scan_id: scanId}, function(response) {
                // Update progress bar
                $('.unplug-progress-fill').css('width', response.progress + '%');
                $('#progress-percentage').text(Math.round(response.progress) + '%');
                $('#progress-status').text(safeStatus(response.status));
                
                // Check if scan is complete
                if (response.completed || response.progress >= 100) {
                    clearInterval(progressInterval);
                    
                    // Hide progress bar and restore button
                    setTimeout(function() {
                        $('#unplug-scan-progress').hide();
                        scanInProgress = false;
                        setScanButtonState();
                        
                        // Reset button text
                        $('#start-scan').removeClass('loading');
                        $('#start-scan').find('.button-text').text('Start Activity Scan');
                        
                        // Enable export buttons
                        $('#export-csv, #export-report').prop('disabled', false);
                        
                        // Show completion message
                        showNotice('Activity scan completed successfully!', 'success');
                    }, 1000);
                }
            });
        }, 2000); // Check progress every 2 seconds
    }

    /**
     * Simulate scan progress (fallback for development)
     */
    function simulateScanProgress() {
        var progress = 0;
        var steps = [
            'Analyzing installed plugins...',
            'Scanning post content...',
            'Checking widget areas...',
            'Analyzing theme files...',
            'Calculating confidence scores...',
            'Finalizing results...'
        ];
        var currentStep = 0;
        
        var interval = setInterval(function() {
            progress += Math.random() * 15;
            
            if (progress > 100) {
                progress = 100;
                clearInterval(interval);
            }
            
            // Update progress bar
            $('.unplug-progress-fill').css('width', progress + '%');
            $('#progress-percentage').text(Math.round(progress) + '%');
            
            // Update status text
            if (currentStep < steps.length && progress > (currentStep + 1) * 15) {
                $('#progress-status').text(steps[currentStep]);
                currentStep++;
            }
            
            if (progress >= 100) {
                $('#progress-status').text('Scan complete!');
            }
        }, 300);
    }

    /**
     * Handle CSV export
     */
    function handleExportCSV() {
        var $btn = $('#export-csv');
        var originalText = $btn.text();
        
        $btn.prop('disabled', true)
            .addClass('loading')
            .text('Exporting...');
        
        // Send AJAX request to export CSV
        sendAjaxRequest('unplug_export_csv', {}, function(response) {
            $btn.prop('disabled', false)
                .removeClass('loading')
                .text(originalText);
                
            // Show success message
            showNotice(response.message, 'success');
            
            // Trigger download if URL is provided
            if (response.download_url && response.download_url !== '#') {
                window.location.href = response.download_url;
            }
        }, function(errorMessage) {
            // Error callback - restore button state
            $btn.prop('disabled', false)
                .removeClass('loading')
                .text(originalText);
        });
    }

    /**
     * Handle report export
     */
    function handleExportReport() {
        var $btn = $('#export-report');
        var originalText = $btn.text();
        
        $btn.prop('disabled', true)
            .addClass('loading')
            .text('Generating...');
        
        // Send AJAX request to generate report
        sendAjaxRequest('unplug_generate_report', {}, function(response) {
            $btn.prop('disabled', false)
                .removeClass('loading')
                .text(originalText);
                
            // Show success message
            showNotice(response.message, 'success');
            
            // Trigger download if URL is provided
            if (response.download_url && response.download_url !== '#') {
                window.open(response.download_url, '_blank');
            }
        }, function(errorMessage) {
            // Error callback - restore button state
            $btn.prop('disabled', false)
                .removeClass('loading')
                .text(originalText);
        });
    }

    /**
     * Check if scan button should be enabled
     */
    function checkScanButtonState() {
        // Enable scan button unless scan is in progress
        $('#start-scan').prop('disabled', scanInProgress);
        
        // Update visual state
        if (scanInProgress) {
            $('#start-scan').addClass('loading');
        } else {
            $('#start-scan').removeClass('loading');
        }
    }

    /**
     * Toggle API token visibility
     */
    function toggleTokenVisibility($btn) {
        var $input = $('#api_token');
        var $icon = $btn.find('.dashicons');
        
        if ($input.attr('type') === 'password') {
            $input.attr('type', 'text');
            $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            $btn.find('span:last').text('Hide');
        } else {
            $input.attr('type', 'password');
            $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            $btn.find('span:last').text('Show');
        }
    }

    /**
     * Validate API token
     */
    function validateApiToken($input) {
        var token = $input.val();
        
        // Remove any existing warnings
        $('.api-token-warning').remove();
        $input.css('border-color', '');
        
        if (token.length > 0 && token.length < 32) {
            $input.css('border-color', '#dc3232');
            $input.after('<p class="api-token-warning description" style="color: #dc3232;">API token appears to be invalid. Please check your token.</p>');
            return false;
        }
        
        return true;
    }

    /**
     * Validate settings form
     */
    function validateSettingsForm() {
        var isValid = true;
        
        // Validate API token
        var $apiToken = $('#api_token');
        if ($apiToken.length && !validateApiToken($apiToken)) {
            isValid = false;
        }
        
        // Validate scan timeout
        var $scanTimeout = $('#scan_timeout');
        if ($scanTimeout.length) {
            var timeout = parseInt($scanTimeout.val());
            if (timeout < 30 || timeout > 600) {
                $scanTimeout.css('border-color', '#dc3232');
                isValid = false;
            } else {
                $scanTimeout.css('border-color', '');
            }
        }
        
        return isValid;
    }

    /**
     * Show admin notice
     */
    function showNotice(message, type) {
        type = type || 'info';
        
        var $notice = $('<div class="notice notice-' + type + ' is-dismissible">' +
            '<p>' + message + '</p>' +
            '<button type="button" class="notice-dismiss">' +
                '<span class="screen-reader-text">Dismiss this notice.</span>' +
            '</button>' +
        '</div>');
        
        // Insert after the h1 tag
        $('h1:first').after($notice);
        
        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            $notice.fadeOut(300, function() {
                $(this).remove();
            });
        }, 5000);
    }

    /**
     * AJAX Helper function
     */
    function sendAjaxRequest(action, data, callback, errorCallback) {
        // Check if unplugAdmin is available
        if (typeof unplugAdmin === 'undefined') {
            console.error('unplugAdmin object is not defined');
            if (errorCallback) {
                errorCallback('Configuration error: unplugAdmin object not found');
            }
            return;
        }
        
        // Check if nonces are available
        if (typeof unplugAdmin.nonces === 'undefined') {
            console.error('unplugAdmin.nonces object is not defined');
            if (errorCallback) {
                errorCallback('Configuration error: unplugAdmin.nonces object not found');
            }
            return;
        }
        
        data = data || {};
        data.action = action;
        
        // Use appropriate nonce based on action
        var nonce = unplugAdmin.nonces.admin || ''; // Default
        if (action === 'unplug_start_scan' || action === 'unplug_scan_progress') {
            nonce = unplugAdmin.nonces.scan || '';
        } else if (action === 'unplug_export_csv' || action === 'unplug_generate_report') {
            nonce = unplugAdmin.nonces.export || '';
        } else if (action === 'unplug_run_location_test') {
            nonce = unplugAdmin.nonces.run_location_test || '';
        }
        
        // Check if we have a valid nonce
        if (!nonce) {
            console.error('No valid nonce found for action: ' + action);
            if (errorCallback) {
                errorCallback('Configuration error: No valid nonce found for action: ' + action);
            }
            return;
        }
        
        data.nonce = nonce;
        
        $.ajax({
            url: unplugAdmin.ajaxUrl,
            type: 'POST',
            data: data,
            success: function(response) {
                if (response.success) {
                    if (callback) {
                        callback(response.data);
                    }
                } else {
                    var errorMessage = response.data || 'An error occurred. Please try again.';
                    if (typeof response.data === 'object' && response.data.message) {
                        errorMessage = response.data.message;
                    }
                    showNotice(errorMessage, 'error');
                    if (errorCallback) {
                        errorCallback(errorMessage);
                    }
                }
            },
            error: function(xhr, status, error) {
                var errorMessage = 'Network error. Please check your connection and try again.';
                showNotice(errorMessage, 'error');
                if (errorCallback) {
                    errorCallback(errorMessage);
                }
            }
        });
    }

    /**
     * Confirmation Dialog Management
     */
    function showConfirmationDialog(operation, callback) {
        // First get confirmation token
        $.ajax({
            url: unplugAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'unplug_get_confirmation_token',
                operation: operation,
                nonce: unplugAdmin.nonces.admin
            },
            success: function(response) {
                if (response.success) {
                    // Create and show confirmation dialog
                    var dialogHtml = response.data.dialog_html;
                    $('body').append(dialogHtml);
                    
                    var $dialog = $('#unplug-confirmation-' + operation);
                    
                    // Initialize dialog functionality
                    initConfirmationDialog($dialog, operation, callback);
                } else {
                    showNotice(response.data || 'Failed to create confirmation dialog.', 'error');
                }
            },
            error: function() {
                showNotice('Network error while creating confirmation dialog.', 'error');
            }
        });
    }

    function initConfirmationDialog($dialog, operation, callback) {
        var $proceedBtn = $dialog.find('#unplug-confirmation-proceed');
        var $cancelBtn = $dialog.find('#unplug-confirmation-cancel');
        var $closeBtn = $dialog.find('.unplug-confirmation-close');
        var $checkboxes = $dialog.find('input[type="checkbox"]');
        
        // Update proceed button state when checkboxes change
        $checkboxes.on('change', function() {
            var allChecked = true;
            $checkboxes.each(function() {
                if (!$(this).is(':checked')) {
                    allChecked = false;
                    return false;
                }
            });
            $proceedBtn.prop('disabled', !allChecked);
        });
        
        // Handle proceed button click
        $proceedBtn.on('click', function() {
            var token = $dialog.find('#unplug-confirmation-token').val();
            var operation = $dialog.find('#unplug-confirmation-operation').val();
            
            // Close dialog
            $dialog.remove();
            
            // Execute callback with token
            if (callback) {
                callback(token, operation);
            }
        });
        
        // Handle cancel/close
        $cancelBtn.add($closeBtn).on('click', function() {
            $dialog.remove();
        });
        
        // Handle overlay click
        $dialog.on('click', function(e) {
            if (e.target === $dialog[0]) {
                $dialog.remove();
            }
        });
        
        // Handle escape key
        $(document).on('keydown.unplug-confirmation', function(e) {
            if (e.keyCode === 27) { // Escape key
                $dialog.remove();
                $(document).off('keydown.unplug-confirmation');
            }
        });
    }

    /**
     * Confirmation-protected action wrapper
     */
    function executeWithConfirmation(operation, actionCallback) {
        showConfirmationDialog(operation, function(token, operation) {
            if (actionCallback) {
                actionCallback(token, operation);
            }
        });
    }

    /**
     * Handle confirmed scan operation
     */
    function handleConfirmedScan() {
        // For basic scan operations, just call the original scan handler
        handleStartScan($('#start-scan'));
    }

    /**
     * Initialize confirmation system
     */
    function initConfirmationSystem() {
        // Override dangerous operations to require confirmation
        $('#start-scan').off('click').on('click', function() {
            handleConfirmedScan();
        });
        
        // Add confirmation to any future dangerous operations
        $('.unplug-requires-confirmation').on('click', function(e) {
            e.preventDefault();
            var operation = $(this).data('operation');
            var originalCallback = $(this).data('callback');
            
            if (operation) {
                executeWithConfirmation(operation, function(token, operation) {
                    // Execute original callback with token
                    if (window[originalCallback]) {
                        window[originalCallback](token, operation);
                    }
                });
            }
        });
    }

    // Initialize confirmation system when DOM is ready
    $(document).ready(function() {
        initConfirmationSystem();
    });

    // Export functions for external use - merge with existing localized data
    if (typeof window.unplugAdmin === 'undefined') {
        window.unplugAdmin = {};
    }
    
    // Add functions to existing unplugAdmin object without overwriting localized data
    window.unplugAdmin.showNotice = showNotice;
    window.unplugAdmin.sendAjaxRequest = sendAjaxRequest;
    window.unplugAdmin.showConfirmationDialog = showConfirmationDialog;
    window.unplugAdmin.executeWithConfirmation = executeWithConfirmation;

    // Confirmation dialog functions
    window.unplugConfirmation = {
        showDialog: function(operation, callback) {
            var nonce = unplugSelectNonce(operation);
            
            if (!nonce) {
                console.error('No nonce available for operation:', operation);
                return;
            }
            
            // Get confirmation token
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_get_confirmation_token',
                    operation: operation,
                    nonce: nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Show confirmation dialog
                        var dialog = $('<div id="unplug-confirmation-dialog">' + response.data.dialog_html + '</div>');
                        $('body').append(dialog);
                        
                        // Handle dialog actions
                        dialog.on('click', '.unplug-confirm-proceed', function() {
                            if (callback) {
                                callback(response.data.token);
                            }
                            dialog.remove();
                        });
                        
                        dialog.on('click', '.unplug-confirm-cancel', function() {
                            dialog.remove();
                        });
                        
                        // Show dialog
                        dialog.show();
                    } else {
                        console.error('Failed to get confirmation token:', response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error getting confirmation token:', error);
                }
            });
        }
    };

    // Plugin inventory analysis functions
    window.unplugInventory = {
        
        /**
         * Get comprehensive plugin inventory
         */
        getInventory: function(forceRefresh, callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for inventory scan');
                return;
            }
            
            // Show loading indicator
            this.showLoadingIndicator('Analyzing plugin inventory...');
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_get_plugin_inventory',
                    force_refresh: forceRefresh || false,
                    nonce: nonce
                },
                success: function(response) {
                    unplugInventory.hideLoadingIndicator();
                    
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugInventory.displayInventoryResults(response.data);
                    } else {
                        console.error('Plugin inventory analysis failed:', response.data.message);
                        unplugInventory.showError('Plugin inventory analysis failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    unplugInventory.hideLoadingIndicator();
                    console.error('AJAX error during plugin inventory analysis:', error);
                    unplugInventory.showError('Plugin inventory analysis failed: ' + error);
                }
            });
        },
        
        /**
         * Detect plugin conflicts
         */
        detectConflicts: function(callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for conflict detection');
                return;
            }
            
            // Show loading indicator
            this.showLoadingIndicator('Detecting plugin conflicts...');
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_detect_plugin_conflicts',
                    nonce: nonce
                },
                success: function(response) {
                    unplugInventory.hideLoadingIndicator();
                    
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugInventory.displayConflictResults(response.data);
                    } else {
                        console.error('Plugin conflict detection failed:', response.data.message);
                        unplugInventory.showError('Plugin conflict detection failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    unplugInventory.hideLoadingIndicator();
                    console.error('AJAX error during conflict detection:', error);
                    unplugInventory.showError('Plugin conflict detection failed: ' + error);
                }
            });
        },
        
        /**
         * Get plugins by category
         */
        getPluginsByCategory: function(category, callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for category analysis');
                return;
            }
            
            // Show loading indicator
            this.showLoadingIndicator('Analyzing ' + category + ' plugins...');
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_get_plugins_by_category',
                    category: category,
                    nonce: nonce
                },
                success: function(response) {
                    unplugInventory.hideLoadingIndicator();
                    
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugInventory.displayCategoryResults(response.data);
                    } else {
                        console.error('Plugin category analysis failed:', response.data.message);
                        unplugInventory.showError('Plugin category analysis failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    unplugInventory.hideLoadingIndicator();
                    console.error('AJAX error during category analysis:', error);
                    unplugInventory.showError('Plugin category analysis failed: ' + error);
                }
            });
        },
        
        /**
         * Display inventory results
         */
        displayInventoryResults: function(data) {
            var $container = $('#unplug-inventory-results');
            if (!$container.length) {
                $container = $('<div id="unplug-inventory-results" class="unplug-results-container"></div>');
                $('.unplug-dashboard').append($container);
            }
            
            var html = '<h3>Plugin Inventory Analysis</h3>';
            html += '<div class="unplug-inventory-summary">';
            html += '<div class="unplug-stat-box">';
            html += '<span class="unplug-stat-number">' + data.summary.total_plugins + '</span>';
            html += '<span class="unplug-stat-label">Total Plugins</span>';
            html += '</div>';
            html += '<div class="unplug-stat-box">';
            html += '<span class="unplug-stat-number">' + data.summary.active_plugins + '</span>';
            html += '<span class="unplug-stat-label">Active Plugins</span>';
            html += '</div>';
            html += '<div class="unplug-stat-box">';
            html += '<span class="unplug-stat-number">' + data.summary.conflicts + '</span>';
            html += '<span class="unplug-stat-label">Total Conflicts</span>';
            html += '</div>';
            html += '<div class="unplug-stat-box">';
            html += '<span class="unplug-stat-number">' + data.summary.high_severity_conflicts + '</span>';
            html += '<span class="unplug-stat-label">High Severity</span>';
            html += '</div>';
            html += '</div>';
            
            // Add categories breakdown
            html += '<h4>Plugin Categories</h4>';
            html += '<div class="unplug-categories">';
            for (var category in data.summary.categories) {
                html += '<div class="unplug-category-item">';
                html += '<strong>' + category.charAt(0).toUpperCase() + category.slice(1) + ':</strong> ';
                html += data.summary.categories[category] + ' plugins';
                html += '</div>';
            }
            html += '</div>';
            
            $container.html(html);
            $container.show();
        },
        
        /**
         * Display conflict results
         */
        displayConflictResults: function(data) {
            var $container = $('#unplug-conflict-results');
            if (!$container.length) {
                $container = $('<div id="unplug-conflict-results" class="unplug-results-container"></div>');
                $('.unplug-dashboard').append($container);
            }
            
            var html = '<h3>Plugin Conflict Detection</h3>';
            
            if (data.total_conflicts === 0) {
                html += '<div class="unplug-no-conflicts">✅ No plugin conflicts detected!</div>';
            } else {
                html += '<div class="unplug-conflicts-summary">';
                html += '<p><strong>' + data.total_conflicts + '</strong> potential conflicts detected:</p>';
                html += '<ul>';
                html += '<li>High severity: ' + Object.keys(data.high_severity).length + '</li>';
                html += '<li>Medium severity: ' + Object.keys(data.medium_severity).length + '</li>';
                html += '<li>Low severity: ' + Object.keys(data.low_severity).length + '</li>';
                html += '</ul>';
                html += '</div>';
                
                // Display conflicts by severity
                html += this.formatConflictsBySeverity('High', data.high_severity, 'high');
                html += this.formatConflictsBySeverity('Medium', data.medium_severity, 'medium');
                html += this.formatConflictsBySeverity('Low', data.low_severity, 'low');
            }
            
            $container.html(html);
            $container.show();
        },
        
        /**
         * Format conflicts by severity
         */
        formatConflictsBySeverity: function(severityLabel, conflicts, severityClass) {
            if (Object.keys(conflicts).length === 0) {
                return '';
            }
            
            var html = '<div class="unplug-conflicts-section unplug-conflicts-' + severityClass + '">';
            html += '<h4>' + severityLabel + ' Severity Conflicts</h4>';
            
            for (var conflictId in conflicts) {
                var conflict = conflicts[conflictId];
                html += '<div class="unplug-conflict-item">';
                html += '<div class="unplug-conflict-title">' + conflict.description + '</div>';
                html += '<div class="unplug-conflict-plugins">Affected plugins: ' + conflict.plugins.join(', ') + '</div>';
                html += '<div class="unplug-conflict-recommendation">' + conflict.recommendation + '</div>';
                html += '</div>';
            }
            
            html += '</div>';
            return html;
        },
        
        /**
         * Display category results
         */
        displayCategoryResults: function(data) {
            var $container = $('#unplug-category-results');
            if (!$container.length) {
                $container = $('<div id="unplug-category-results" class="unplug-results-container"></div>');
                $('.unplug-dashboard').append($container);
            }
            
            var html = '<h3>' + data.category.charAt(0).toUpperCase() + data.category.slice(1) + ' Plugins</h3>';
            html += '<div class="unplug-category-summary">';
            html += '<p><strong>' + data.total_plugins + '</strong> ' + data.category + ' plugins found:</p>';
            html += '<ul>';
            html += '<li>Active: ' + data.active_count + '</li>';
            html += '<li>Inactive: ' + data.inactive_count + '</li>';
            html += '</ul>';
            html += '</div>';
            
            // List plugins
            html += '<div class="unplug-plugin-list">';
            for (var pluginFile in data.plugins) {
                var plugin = data.plugins[pluginFile];
                html += '<div class="unplug-plugin-item ' + (plugin.active ? 'active' : 'inactive') + '">';
                html += '<div class="unplug-plugin-name">' + plugin.name + '</div>';
                html += '<div class="unplug-plugin-version">Version: ' + plugin.version + '</div>';
                html += '<div class="unplug-plugin-status">' + (plugin.active ? 'Active' : 'Inactive') + '</div>';
                html += '<div class="unplug-plugin-performance">Performance Impact: ' + plugin.performance_impact + '</div>';
                html += '</div>';
            }
            html += '</div>';
            
            $container.html(html);
            $container.show();
        },
        
        /**
         * Show loading indicator
         */
        showLoadingIndicator: function(message) {
            var $indicator = $('#unplug-loading-indicator');
            if (!$indicator.length) {
                $indicator = $('<div id="unplug-loading-indicator" class="unplug-loading">' +
                    '<div class="unplug-loading-spinner"></div>' +
                    '<div class="unplug-loading-message"></div>' +
                    '</div>');
                $('body').append($indicator);
            }
            
            $indicator.find('.unplug-loading-message').text(message);
            $indicator.show();
        },
        
        /**
         * Hide loading indicator
         */
        hideLoadingIndicator: function() {
            $('#unplug-loading-indicator').hide();
        },
        
        /**
         * Show error message
         */
        showError: function(message) {
            var $errorDiv = $('<div class="unplug-error notice notice-error">' +
                '<p>' + message + '</p>' +
                '</div>');
            $('.unplug-dashboard').prepend($errorDiv);
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                $errorDiv.fadeOut(function() {
                    $errorDiv.remove();
                });
            }, 5000);
        }
    };

    // Add event handlers for plugin inventory functionality
    $(document).on('click', '#unplug-analyze-inventory', function(e) {
        e.preventDefault();
        var forceRefresh = $(this).data('force-refresh') || false;
        unplugInventory.getInventory(forceRefresh);
    });
    
    $(document).on('click', '#unplug-detect-conflicts', function(e) {
        e.preventDefault();
        unplugInventory.detectConflicts();
    });
    
    $(document).on('click', '.unplug-analyze-category', function(e) {
        e.preventDefault();
        var category = $(this).data('category');
        unplugInventory.getPluginsByCategory(category);
    });

    // Advanced conflict pattern analysis functionality
    window.unplugConflictEngine = {
        
        /**
         * Run advanced conflict pattern analysis
         */
        analyzeConflictPatterns: function(callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for conflict pattern analysis');
                return;
            }
            
            // Show loading indicator
            this.showLoadingIndicator('Running advanced conflict pattern analysis...');
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_analyze_conflict_patterns',
                    nonce: nonce
                },
                success: function(response) {
                    unplugConflictEngine.hideLoadingIndicator();
                    
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugConflictEngine.displayAdvancedConflictResults(response.data);
                    } else {
                        console.error('Advanced conflict analysis failed:', response.data.message);
                        unplugConflictEngine.showError('Advanced conflict analysis failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    unplugConflictEngine.hideLoadingIndicator();
                    console.error('AJAX error during advanced conflict analysis:', error);
                    unplugConflictEngine.showError('Advanced conflict analysis failed: ' + error);
                }
            });
        },
        
        /**
         * Export conflict analysis in specified format
         */
        exportConflictAnalysis: function(format) {
            var nonce = unplugSelectNonce('export');
            
            if (!nonce) {
                console.error('No nonce available for conflict analysis export');
                return;
            }
            
            format = format || 'json';
            
            // Show loading indicator
            this.showLoadingIndicator('Exporting conflict analysis as ' + format.toUpperCase() + '...');
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_export_conflict_analysis',
                    format: format,
                    nonce: nonce
                },
                success: function(response) {
                    unplugConflictEngine.hideLoadingIndicator();
                    
                    if (response.success) {
                        // Create download link
                        unplugConflictEngine.downloadData(
                            response.data.data,
                            response.data.filename,
                            response.data.content_type
                        );
                    } else {
                        console.error('Conflict analysis export failed:', response.data.message);
                        unplugConflictEngine.showError('Export failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    unplugConflictEngine.hideLoadingIndicator();
                    console.error('AJAX error during export:', error);
                    unplugConflictEngine.showError('Export failed: ' + error);
                }
            });
        },
        
        /**
         * Display advanced conflict analysis results
         */
        displayAdvancedConflictResults: function(data) {
            var $container = $('#unplug-advanced-conflict-results');
            if (!$container.length) {
                $container = $('<div id="unplug-advanced-conflict-results" class="unplug-results-container"></div>');
                $('.unplug-dashboard').append($container);
            }
            
            var analysis = data.analysis;
            var html = '<h3>Advanced Conflict Pattern Analysis</h3>';
            
            // Summary section
            html += '<div class="unplug-advanced-summary">';
            html += '<div class="unplug-summary-grid">';
            html += '<div class="unplug-summary-item critical">';
            html += '<span class="number">' + analysis.summary.critical_conflicts + '</span>';
            html += '<span class="label">Critical</span>';
            html += '</div>';
            html += '<div class="unplug-summary-item high">';
            html += '<span class="number">' + analysis.summary.high_conflicts + '</span>';
            html += '<span class="label">High</span>';
            html += '</div>';
            html += '<div class="unplug-summary-item medium">';
            html += '<span class="number">' + analysis.summary.medium_conflicts + '</span>';
            html += '<span class="label">Medium</span>';
            html += '</div>';
            html += '<div class="unplug-summary-item low">';
            html += '<span class="number">' + analysis.summary.low_conflicts + '</span>';
            html += '<span class="label">Low</span>';
            html += '</div>';
            html += '</div>';
            html += '</div>';
            
            // Detailed analysis overview
            if (analysis.detailed_analysis && analysis.detailed_analysis.overview) {
                html += '<div class="unplug-overview-section">';
                html += '<h4>Overall Risk Assessment: <span class="risk-' + analysis.detailed_analysis.overview.risk_assessment + '">' + 
                        analysis.detailed_analysis.overview.risk_assessment.toUpperCase() + '</span></h4>';
                
                if (analysis.detailed_analysis.overview.primary_concerns && analysis.detailed_analysis.overview.primary_concerns.length > 0) {
                    html += '<div class="unplug-primary-concerns">';
                    html += '<h5>Primary Concerns:</h5>';
                    html += '<ul>';
                    analysis.detailed_analysis.overview.primary_concerns.forEach(function(concern) {
                        html += '<li>' + concern + '</li>';
                    });
                    html += '</ul>';
                    html += '</div>';
                }
                html += '</div>';
            }
            
            // Conflict breakdown by type
            html += '<div class="unplug-conflict-breakdown">';
            html += '<h4>Conflict Analysis by Type</h4>';
            
            var conflictTypes = ['javascript', 'css', 'admin', 'database', 'api', 'resource', 'known_patterns'];
            
            conflictTypes.forEach(function(type) {
                if (analysis.by_type[type] && Object.keys(analysis.by_type[type]).length > 0) {
                    html += '<div class="unplug-conflict-type-section" data-type="' + type + '">';
                    html += '<h5>' + type.charAt(0).toUpperCase() + type.slice(1).replace('_', ' ') + ' Conflicts</h5>';
                    html += '<div class="unplug-conflict-items">';
                    html += unplugConflictEngine.formatConflictsByType(type, analysis.by_type[type]);
                    html += '</div>';
                    html += '</div>';
                }
            });
            
            html += '</div>';
            
            // Recommendations
            if (analysis.recommendations && analysis.recommendations.length > 0) {
                html += '<div class="unplug-recommendations">';
                html += '<h4>Recommendations</h4>';
                analysis.recommendations.forEach(function(rec) {
                    html += '<div class="unplug-recommendation priority-' + rec.priority + '">';
                    html += '<h5>' + rec.title + '</h5>';
                    html += '<p>' + rec.description + '</p>';
                    if (rec.actions && rec.actions.length > 0) {
                        html += '<ul class="unplug-action-list">';
                        rec.actions.forEach(function(action) {
                            html += '<li>' + action + '</li>';
                        });
                        html += '</ul>';
                    }
                    html += '</div>';
                });
                html += '</div>';
            }
            
            // Export options
            html += '<div class="unplug-export-options">';
            html += '<h4>Export Analysis</h4>';
            html += '<div class="unplug-export-buttons">';
            html += '<button class="unplug-export-btn" data-format="json">Export as JSON</button>';
            html += '<button class="unplug-export-btn" data-format="csv">Export as CSV</button>';
            html += '<button class="unplug-export-btn" data-format="xml">Export as XML</button>';
            html += '</div>';
            html += '</div>';
            
            $container.html(html);
            $container.show();
        },
        
        /**
         * Format conflicts by type for display
         */
        formatConflictsByType: function(type, conflicts) {
            var html = '';
            
            if (Array.isArray(conflicts)) {
                conflicts.forEach(function(conflict) {
                    html += '<div class="unplug-conflict-detail">';
                    html += '<div class="unplug-conflict-severity severity-' + (conflict.severity || 5) + '">' + (conflict.severity || 5) + '</div>';
                    html += '<div class="unplug-conflict-info">';
                    html += '<div class="unplug-conflict-desc">' + (conflict.description || conflict.type || 'Conflict detected') + '</div>';
                    if (conflict.plugins && conflict.plugins.length > 0) {
                        html += '<div class="unplug-conflict-plugins">Affected plugins: ' + conflict.plugins.join(', ') + '</div>';
                    }
                    if (conflict.resolution) {
                        html += '<div class="unplug-conflict-resolution">' + conflict.resolution + '</div>';
                    }
                    html += '</div>';
                    html += '</div>';
                });
            } else {
                // Handle object-based conflicts
                Object.keys(conflicts).forEach(function(key) {
                    var conflict = conflicts[key];
                    html += '<div class="unplug-conflict-detail">';
                    html += '<div class="unplug-conflict-severity severity-' + (conflict.severity || 5) + '">' + (conflict.severity || 5) + '</div>';
                    html += '<div class="unplug-conflict-info">';
                    html += '<div class="unplug-conflict-desc">' + (conflict.description || key) + '</div>';
                    if (conflict.plugins && conflict.plugins.length > 0) {
                        html += '<div class="unplug-conflict-plugins">Affected plugins: ' + conflict.plugins.join(', ') + '</div>';
                    }
                    if (conflict.resolution) {
                        html += '<div class="unplug-conflict-resolution">' + conflict.resolution + '</div>';
                    }
                    html += '</div>';
                    html += '</div>';
                });
            }
            
            return html;
        },
        
        /**
         * Download data as file
         */
        downloadData: function(data, filename, contentType) {
            var blob = new Blob([data], { type: contentType });
            var url = window.URL.createObjectURL(blob);
            var a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        },
        
        /**
         * Show loading indicator
         */
        showLoadingIndicator: function(message) {
            unplugInventory.showLoadingIndicator(message);
        },
        
        /**
         * Hide loading indicator
         */
        hideLoadingIndicator: function() {
            unplugInventory.hideLoadingIndicator();
        },
        
        /**
         * Show error message
         */
        showError: function(message) {
            unplugInventory.showError(message);
        }
    };

    // Add event handlers for advanced conflict analysis
    $(document).on('click', '#unplug-analyze-advanced-conflicts', function(e) {
        e.preventDefault();
        unplugConflictEngine.analyzeConflictPatterns();
    });
    
    $(document).on('click', '.unplug-export-btn', function(e) {
        e.preventDefault();
        var format = $(this).data('format');
        unplugConflictEngine.exportConflictAnalysis(format);
    });

    // Warning system functionality
    window.unplugWarningSystem = {
        
        /**
         * Get warnings
         */
        getWarnings: function(filters, callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for warning retrieval');
                return;
            }
            
            var data = {
                action: 'unplug_get_warnings',
                nonce: nonce
            };
            
            // Add filters if provided
            if (filters) {
                if (filters.type) data.type = filters.type;
                if (filters.severity) data.severity = filters.severity;
                if (filters.status) data.status = filters.status;
                if (filters.active_only) data.active_only = filters.active_only;
            }
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: data,
                success: function(response) {
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugWarningSystem.displayWarnings(response.data.warnings);
                    } else {
                        console.error('Warning retrieval failed:', response.data.message);
                        unplugWarningSystem.showError('Warning retrieval failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error during warning retrieval:', error);
                    unplugWarningSystem.showError('Warning retrieval failed: ' + error);
                }
            });
        },
        
        /**
         * Display warnings in the UI
         */
        displayWarnings: function(warnings) {
            var $container = $('#unplug-warnings-container');
            if (!$container.length) {
                $container = $('<div id="unplug-warnings-container" class="unplug-warnings-container"></div>');
                $('.unplug-dashboard').prepend($container);
            }
            
            var html = '';
            
            if (Object.keys(warnings).length === 0) {
                html = '<div class="unplug-no-warnings">✅ No active warnings</div>';
            } else {
                html = '<h3>Active Warnings</h3>';
                html += '<div class="unplug-warnings-list">';
                
                for (var warningId in warnings) {
                    var warning = warnings[warningId];
                    html += unplugWarningSystem.renderWarning(warning);
                }
                
                html += '</div>';
            }
            
            $container.html(html);
            $container.show();
        },
        
        /**
         * Render a single warning
         */
        renderWarning: function(warning) {
            var html = '<div class="unplug-warning unplug-warning-' + warning.type + ' unplug-warning-severity-' + warning.severity + '" data-warning-id="' + warning.id + '">';
            
            // Warning header
            html += '<div class="unplug-warning-header">';
            html += '<div class="unplug-warning-icon">';
            html += unplugWarningSystem.getWarningIcon(warning.type, warning.severity);
            html += '</div>';
            html += '<div class="unplug-warning-title">';
            html += '<h4>' + warning.title + '</h4>';
            html += '<span class="unplug-warning-meta">' + warning.severity.toUpperCase() + ' • ' + warning.type.toUpperCase() + '</span>';
            html += '</div>';
            html += '<div class="unplug-warning-actions">';
            
            if (warning.dismissible) {
                html += '<button class="unplug-warning-dismiss" data-warning-id="' + warning.id + '" title="Dismiss warning">';
                html += '<span class="dashicons dashicons-dismiss"></span>';
                html += '</button>';
            }
            
            html += '</div>';
            html += '</div>';
            
            // Warning content
            html += '<div class="unplug-warning-content">';
            html += '<p class="unplug-warning-message">' + warning.message + '</p>';
            
            if (warning.details) {
                html += '<div class="unplug-warning-details">';
                html += '<p>' + warning.details + '</p>';
                html += '</div>';
            }
            
            // Related plugins
            if (warning.related_plugins && warning.related_plugins.length > 0) {
                html += '<div class="unplug-warning-plugins">';
                html += '<strong>Affected plugins:</strong> ';
                html += warning.related_plugins.join(', ');
                html += '</div>';
            }
            
            // Warning actions
            if (warning.actions && warning.actions.length > 0) {
                html += '<div class="unplug-warning-action-buttons">';
                for (var i = 0; i < warning.actions.length; i++) {
                    var action = warning.actions[i];
                    html += '<button class="unplug-warning-action-btn" data-action="' + action.action + '" data-warning-id="' + warning.id + '">';
                    html += action.label;
                    html += '</button>';
                }
                html += '</div>';
            }
            
            html += '</div>';
            html += '</div>';
            
            return html;
        },
        
        /**
         * Get warning icon based on type and severity
         */
        getWarningIcon: function(type, severity) {
            var iconClass = 'dashicons-info';
            
            switch (type) {
                case 'blocking':
                    iconClass = 'dashicons-lock';
                    break;
                case 'advisory':
                    iconClass = 'dashicons-warning';
                    break;
                case 'informational':
                    iconClass = 'dashicons-info';
                    break;
            }
            
            if (severity === 'critical') {
                iconClass = 'dashicons-dismiss';
            }
            
            return '<span class="dashicons ' + iconClass + '"></span>';
        },
        
        /**
         * Update warning status
         */
        updateWarningStatus: function(warningId, status, callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for warning status update');
                return;
            }
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_update_warning_status',
                    nonce: nonce,
                    warning_id: warningId,
                    status: status
                },
                success: function(response) {
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugWarningSystem.showSuccess('Warning status updated successfully');
                    } else {
                        console.error('Warning status update failed:', response.data.message);
                        unplugWarningSystem.showError('Warning status update failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error during warning status update:', error);
                    unplugWarningSystem.showError('Warning status update failed: ' + error);
                }
            });
        },
        
        /**
         * Dismiss warning
         */
        dismissWarning: function(warningId, callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for warning dismissal');
                return;
            }
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_dismiss_warning',
                    nonce: nonce,
                    warning_id: warningId
                },
                success: function(response) {
                    if (response.success) {
                        // Remove warning from UI
                        $('[data-warning-id="' + warningId + '"]').fadeOut(function() {
                            $(this).remove();
                        });
                        
                        if (callback) {
                            callback(response.data);
                        }
                        unplugWarningSystem.showSuccess('Warning dismissed successfully');
                    } else {
                        console.error('Warning dismissal failed:', response.data.message);
                        unplugWarningSystem.showError('Warning dismissal failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error during warning dismissal:', error);
                    unplugWarningSystem.showError('Warning dismissal failed: ' + error);
                }
            });
        },
        
        /**
         * Generate warnings from conflict analysis
         */
        generateWarnings: function(callback) {
            var nonce = unplugSelectNonce('scan');
            
            if (!nonce) {
                console.error('No nonce available for warning generation');
                return;
            }
            
            // Show loading indicator
            unplugWarningSystem.showLoadingIndicator('Generating warnings from conflict analysis...');
            
            $.ajax({
                url: unplug_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'unplug_generate_warnings',
                    nonce: nonce
                },
                success: function(response) {
                    unplugWarningSystem.hideLoadingIndicator();
                    
                    if (response.success) {
                        if (callback) {
                            callback(response.data);
                        }
                        unplugWarningSystem.showSuccess(response.data.message);
                        
                        // Refresh warnings display
                        unplugWarningSystem.getWarnings({ active_only: true });
                    } else {
                        console.error('Warning generation failed:', response.data.message);
                        unplugWarningSystem.showError('Warning generation failed: ' + response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    unplugWarningSystem.hideLoadingIndicator();
                    console.error('AJAX error during warning generation:', error);
                    unplugWarningSystem.showError('Warning generation failed: ' + error);
                }
            });
        },
        
        /**
         * Show loading indicator
         */
        showLoadingIndicator: function(message) {
            unplugInventory.showLoadingIndicator(message);
        },
        
        /**
         * Hide loading indicator
         */
        hideLoadingIndicator: function() {
            unplugInventory.hideLoadingIndicator();
        },
        
        /**
         * Show error message
         */
        showError: function(message) {
            unplugInventory.showError(message);
        },
        
        /**
         * Show success message
         */
        showSuccess: function(message) {
            var $successDiv = $('<div class="unplug-success notice notice-success">' +
                '<p>' + message + '</p>' +
                '</div>');
            $('.unplug-dashboard').prepend($successDiv);
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                $successDiv.fadeOut(function() {
                    $successDiv.remove();
                });
            }, 5000);
        }
    };

    // Add event handlers for warning system
    $(document).on('click', '.unplug-warning-dismiss', function(e) {
        e.preventDefault();
        var warningId = $(this).data('warning-id');
        unplugWarningSystem.dismissWarning(warningId);
    });
    
    $(document).on('click', '.unplug-warning-action-btn', function(e) {
        e.preventDefault();
        var action = $(this).data('action');
        var warningId = $(this).data('warning-id');
        
        // Handle different warning actions
        switch(action) {
            case 'view_conflict_details':
                // Trigger conflict analysis view
                unplugConflictEngine.analyzeConflictPatterns();
                break;
            case 'view_pattern_details':
            case 'view_resource_details':
                // Show detailed conflict analysis
                unplugConflictEngine.analyzeConflictPatterns();
                break;
            case 'acknowledge':
                unplugWarningSystem.updateWarningStatus(warningId, 'acknowledged');
                break;
            default:
                // Unknown warning action handled
        }
    });
    
    $(document).on('click', '#unplug-generate-warnings', function(e) {
        e.preventDefault();
        unplugWarningSystem.generateWarnings();
    });
    
    $(document).on('click', '#unplug-show-warnings', function(e) {
        e.preventDefault();
        unplugWarningSystem.getWarnings({ active_only: true });
    });

    // Safe Mode Management System
    window.unplugSafeMode = {
        // Get safe mode status
        getStatus: function() {
            return new Promise(function(resolve, reject) {
                unplugInventory.showLoadingIndicator('Getting safe mode status...');
                
                var nonce = unplugSelectNonce('scan');
                if (!nonce) {
                    unplugInventory.hideLoadingIndicator();
                    reject(new Error('No nonce available for safe mode status request'));
                    return;
                }
                
                $.ajax({
                    url: unplug_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'unplug_get_safe_mode_status',
                        nonce: nonce
                    },
                    success: function(response) {
                        unplugInventory.hideLoadingIndicator();
                        
                        if (response.success) {
                            resolve(response.data);
                        } else {
                            reject(new Error(response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        unplugInventory.hideLoadingIndicator();
                        reject(new Error('Request failed: ' + error));
                    }
                });
            });
        },

        // Activate safe mode
        activateSafeMode: function(options) {
            options = options || {};
            
            return new Promise(function(resolve, reject) {
                unplugInventory.showLoadingIndicator('Activating safe mode...');
                
                var nonce = unplugSelectNonce('scan');
                if (!nonce) {
                    unplugInventory.hideLoadingIndicator();
                    reject(new Error('No nonce available for safe mode activation'));
                    return;
                }
                
                $.ajax({
                    url: unplug_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'unplug_activate_safe_mode',
                        nonce: nonce,
                        trigger: options.trigger || 'user_activated',
                        conflicting_plugins: options.conflicting_plugins || [],
                        emergency_mode: options.emergency_mode || false
                    },
                    success: function(response) {
                        unplugInventory.hideLoadingIndicator();
                        
                        if (response.success) {
                            unplugInventory.showMessage('success', response.data.message);
                            resolve(response.data);
                        } else {
                            unplugInventory.showMessage('error', response.data.message || 'Safe mode activation failed');
                            reject(new Error(response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        unplugInventory.hideLoadingIndicator();
                        unplugInventory.showMessage('error', 'Request failed: ' + error);
                        reject(new Error('Request failed: ' + error));
                    }
                });
            });
        },

        // Deactivate safe mode
        deactivateSafeMode: function(restorePlugins) {
            restorePlugins = restorePlugins || false;
            
            return new Promise(function(resolve, reject) {
                unplugInventory.showLoadingIndicator('Deactivating safe mode...');
                
                var nonce = unplugSelectNonce('scan');
                if (!nonce) {
                    unplugInventory.hideLoadingIndicator();
                    reject(new Error('No nonce available for safe mode deactivation'));
                    return;
                }
                
                $.ajax({
                    url: unplug_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'unplug_deactivate_safe_mode',
                        nonce: nonce,
                        restore_plugins: restorePlugins
                    },
                    success: function(response) {
                        unplugInventory.hideLoadingIndicator();
                        
                        if (response.success) {
                            unplugInventory.showMessage('success', response.data.message);
                            resolve(response.data);
                        } else {
                            unplugInventory.showMessage('error', response.data.message || 'Safe mode deactivation failed');
                            reject(new Error(response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        unplugInventory.hideLoadingIndicator();
                        unplugInventory.showMessage('error', 'Request failed: ' + error);
                        reject(new Error('Request failed: ' + error));
                    }
                });
            });
        },

        // Display safe mode dashboard
        displaySafeModeStatus: function(containerId) {
            var container = $('#' + containerId);
            if (!container.length) return;

            this.getStatus().then(function(data) {
                var dashboardData = data.dashboard_data;
                var html = '<div class="unplug-safe-mode-dashboard">';
                
                // Safe mode status
                html += '<div class="unplug-safe-mode-status">';
                html += '<h3>Safe Mode Status</h3>';
                
                if (dashboardData.is_active) {
                    var modeText = dashboardData.is_emergency ? 'Emergency Mode' : 'Safe Mode';
                    var modeClass = dashboardData.is_emergency ? 'emergency' : 'safe';
                    html += '<div class="unplug-safe-mode-indicator ' + modeClass + '">';
                    html += '<span class="dashicons dashicons-shield-alt"></span>';
                    html += '<strong>' + modeText + ' is Active</strong>';
                    html += '</div>';
                    
                    // Show quarantined plugins
                    if (dashboardData.status.quarantined_plugins.length > 0) {
                        html += '<div class="unplug-quarantined-plugins">';
                        html += '<h4>Quarantined Plugins:</h4>';
                        html += '<ul>';
                        dashboardData.status.quarantined_plugins.forEach(function(plugin) {
                            html += '<li>' + plugin + '</li>';
                        });
                        html += '</ul>';
                        html += '</div>';
                    }
                    
                    // Safe mode controls
                    html += '<div class="unplug-safe-mode-controls">';
                    html += '<button class="button button-primary" onclick="unplugSafeMode.deactivateSafeMode(true).then(() => location.reload())">Deactivate Safe Mode & Restore Plugins</button>';
                    html += '<button class="button" onclick="unplugSafeMode.deactivateSafeMode(false).then(() => location.reload())">Deactivate Safe Mode Only</button>';
                    html += '</div>';
                } else {
                    html += '<div class="unplug-safe-mode-indicator normal">';
                    html += '<span class="dashicons dashicons-yes-alt"></span>';
                    html += '<strong>Normal Mode</strong>';
                    html += '</div>';
                    
                    // Safe mode controls
                    html += '<div class="unplug-safe-mode-controls">';
                    html += '<button class="button" onclick="unplugSafeMode.activateSafeMode().then(() => location.reload())">Activate Safe Mode</button>';
                    html += '<button class="button" onclick="unplugSafeMode.activateSafeMode({emergency_mode: true}).then(() => location.reload())">Activate Emergency Mode</button>';
                    html += '</div>';
                }
                
                html += '</div>';
                
                // Recent backup states
                if (dashboardData.backups && dashboardData.backups.length > 0) {
                    html += '<div class="unplug-backup-states">';
                    html += '<h3>Recent Backup States</h3>';
                    html += '<ul>';
                    dashboardData.backups.forEach(function(backup) {
                        var date = new Date(backup.created * 1000);
                        html += '<li>';
                        html += '<strong>' + backup.id + '</strong> - ' + date.toLocaleString();
                        html += ' <button class="button button-small" onclick="unplugSafeMode.restoreFromBackup(\'' + backup.id + '\').then(() => location.reload())">Restore</button>';
                        html += '</li>';
                    });
                    html += '</ul>';
                    html += '</div>';
                }
                
                html += '</div>';
                
                container.html(html);
            }).catch(function(error) {
                container.html('<div class="error"><p>Error loading safe mode status: ' + error.message + '</p></div>');
            });
        },

        // Restore from backup
        restoreFromBackup: function(backupId) {
            return new Promise(function(resolve, reject) {
                unplugInventory.showLoadingIndicator('Restoring from backup...');
                
                var nonce = unplugSelectNonce('scan');
                if (!nonce) {
                    unplugInventory.hideLoadingIndicator();
                    reject(new Error('No nonce available for backup restoration'));
                    return;
                }
                
                $.ajax({
                    url: unplug_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'unplug_restore_from_backup',
                        nonce: nonce,
                        backup_id: backupId
                    },
                    success: function(response) {
                        unplugInventory.hideLoadingIndicator();
                        
                        if (response.success) {
                            unplugInventory.showMessage('success', response.data.message);
                            resolve(response.data);
                        } else {
                            unplugInventory.showMessage('error', response.data.message || 'Backup restoration failed');
                            reject(new Error(response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        unplugInventory.hideLoadingIndicator();
                        unplugInventory.showMessage('error', 'Request failed: ' + error);
                        reject(new Error('Request failed: ' + error));
                    }
                });
            });
        }
    };

    // Add event handlers for safe mode system
    $(document).on('click', '#unplug-safe-mode-status', function(e) {
        e.preventDefault();
        unplugSafeMode.displaySafeModeStatus('unplug-safe-mode-results');
    });

    // Context control handlers
    $('#clear-location-data').on('click', function() {
        if (confirm('Are you sure you want to clear all location data? This action cannot be undone.')) {
            $.post(ajaxurl, {
                action: 'unplug_clear_location_data',
                nonce: unplug_admin.nonce
            }, function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert('Error clearing location data: ' + response.data);
                }
            });
        }
    });

    $('#export-location-data').on('click', function() {
        const button = $(this);
        const originalText = button.html();
        
        button.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Exporting...');
        
        $.post(ajaxurl, {
            action: 'unplug_export_location_data',
            nonce: unplug_admin.nonce
        }, function(response) {
            if (response.success) {
                // Create download link
                const blob = new Blob([response.data.csv], { type: 'text/csv' });
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = 'unplug-location-data-' + new Date().toISOString().split('T')[0] + '.csv';
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
            } else {
                alert('Error exporting location data: ' + response.data);
            }
        }).always(function() {
            button.prop('disabled', false).html(originalText);
        });
    });

    // Add spinning animation for export button
    $('<style>').text(`
        .spin {
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    `).appendTo('head');

    /**
     * Handle individual plugin test execution
     */
    function handleIndividualTest($btn, testType) {
        const pluginSlug = $btn.data('plugin');
        const $row = $btn.closest('tr');
        
        // Disable button and show loading state
        $btn.prop('disabled', true).addClass('loading');
        const originalText = $btn.text();
        $btn.text('Running...');
        
        // Determine the AJAX action based on test type
        let ajaxAction = '';
        switch (testType) {
            case 'activity':
                ajaxAction = 'unplug_run_activity_test';
                break;
            case 'location':
                ajaxAction = 'unplug_run_location_test';
                break;
            case 'speed':
                ajaxAction = 'unplug_run_speed_test';
                break;
            case 'deactivation':
                ajaxAction = 'unplug_run_deactivation_test';
                break;
        }
        
        // Send AJAX request
        sendAjaxRequest(ajaxAction, {
            plugin_slug: pluginSlug
        }, function(response) {
            // Success - update the UI with results
            updatePluginTestResult($row, testType, response);
            
            // Remove the button since test is complete
            $btn.remove();
            
            showNotice(`${testType} test completed for ${pluginSlug}`, 'success');
        }, function(errorMessage) {
            // Error - restore button state
            $btn.prop('disabled', false).removeClass('loading');
            $btn.text(originalText);
            
            showNotice(`Failed to run ${testType} test: ${errorMessage}`, 'error');
        });
    }
    
    /**
     * Update plugin test result in the table
     */
    function updatePluginTestResult($row, testType, result) {
        let $cell;
        let resultHtml = '';
        
        switch (testType) {
            case 'activity':
                $cell = $row.find('.plugin-activity');
                switch (result.status) {
                    case 'used':
                        resultHtml = '<span class="test-result result-used">✅ Used</span>';
                        break;
                    case 'possibly_used':
                        resultHtml = '<span class="test-result result-possibly_used">⚠️ Possibly Used</span>';
                        break;
                    case 'no_usage':
                        resultHtml = '<span class="test-result result-no_usage">❌ No Usage Found</span>';
                        break;
                    default:
                        resultHtml = '<span class="test-result result-untested">❓ Unknown</span>';
                }
                break;
                
            case 'location':
                $cell = $row.find('.plugin-locations');
                resultHtml = `<span class="location-count">${result.count} locations</span>`;
                if (result.details && result.details.length > 0) {
                    resultHtml += ` <button class="button button-link view-locations" data-plugin="${$row.data('plugin-slug')}">View Details</button>`;
                }
                break;
                
            case 'speed':
                $cell = $row.find('.plugin-speed');
                let impactClass = 'impact-' + result.impact;
                let impactIcon = '';
                switch (result.impact) {
                    case 'high':
                        impactIcon = '🔴';
                        break;
                    case 'medium':
                        impactIcon = '🟡';
                        break;
                    case 'low':
                        impactIcon = '🟢';
                        break;
                    default:
                        impactIcon = '⚪';
                }
                resultHtml = `<span class="speed-impact ${impactClass}">${impactIcon} ${result.impact.charAt(0).toUpperCase() + result.impact.slice(1)}</span>`;
                break;
                
            case 'deactivation':
                $cell = $row.find('.plugin-deactivation');
                switch (result.status) {
                    case 'safe':
                        resultHtml = '<span class="deactivation-result result-safe">✅ Safe</span>';
                        break;
                    case 'warning':
                        resultHtml = '<span class="deactivation-result result-warning">⚠️ Warning</span>';
                        break;
                    case 'critical':
                        resultHtml = '<span class="deactivation-result result-critical">❌ Critical</span>';
                        break;
                    default:
                        resultHtml = '<span class="deactivation-result result-untested">❓ Unknown</span>';
                }
                break;
        }
        
        if ($cell.length && resultHtml) {
            $cell.html(resultHtml);
        }
    }
    
    /**
     * Toggle plugin details row
     */
    function togglePluginDetails($btn) {
        const pluginSlug = $btn.data('plugin');
        const $detailsRow = $('#details-' + pluginSlug);
        
        if ($detailsRow.is(':visible')) {
            $detailsRow.hide();
            $btn.text('Details');
        } else {
            // Hide all other details rows first
            $('.plugin-details-row').hide();
            $('.toggle-plugin-details').text('Details');
            
            // Show this details row
            $detailsRow.show();
            $btn.text('Hide Details');
        }
    }
    
    /**
     * View location details for a plugin
     */
    function viewLocationDetails($btn) {
        const pluginSlug = $btn.data('plugin');
        
        // Create a modal or popup to show location details
        showLocationModal(pluginSlug);
    }
    
    /**
     * Show location details modal
     */
    function showLocationModal(pluginSlug) {
        // For now, just show an alert - this would be replaced with a proper modal
        sendAjaxRequest('unplug_get_location_details', {
            plugin_slug: pluginSlug
        }, function(response) {
            if (response.details && response.details.length > 0) {
                let locationList = response.details.map(function(location) {
                    return `• ${location.type}: ${location.title} (${location.url})`;
                }).join('\n');
                
                alert(`Location details for ${pluginSlug}:\n\n${locationList}`);
            } else {
                alert(`No location details found for ${pluginSlug}`);
            }
        }, function(errorMessage) {
            alert(`Failed to load location details: ${errorMessage}`);
        });
    }
    
    /**
     * Handle Clear Location Data button
     */
    function handleClearLocationData($btn) {
        if (!confirm('Are you sure you want to clear all location data? This action cannot be undone.')) {
            return;
        }
        
        $btn.prop('disabled', true);
        $btn.find('span').text('Clearing...');
        
        sendAjaxRequest('unplug_clear_location_data', {}, function(response) {
            if (response.success) {
                alert('Location data cleared successfully.');
                location.reload();
            } else {
                alert('Error clearing location data: ' + (response.message || 'Unknown error'));
            }
        }, function() {
            alert('Error clearing location data. Please try again.');
        }, function() {
            $btn.prop('disabled', false);
            $btn.find('span').text('Clear Location Data');
        });
    }
    
    /**
     * Handle Export Data button
     */
    function handleExportLocationData($btn) {
        $btn.prop('disabled', true);
        $btn.find('span').text('Exporting...');
        
        sendAjaxRequest('unplug_export_location_data', {}, function(response) {
            if (response.success && response.download_url) {
                window.open(response.download_url, '_blank');
            } else {
                alert('Error exporting data: ' + (response.message || 'Unknown error'));
            }
        }, function() {
            alert('Error exporting data. Please try again.');
        }, function() {
            $btn.prop('disabled', false);
            $btn.find('span').text('Export Data');
        });
    }

})(jQuery); 