<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://mulberrytech.ca/unplug
 * @since      1.0.0
 *
 * @package    Unplug
 * @subpackage Unplug/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Unplug
 * @subpackage Unplug/admin
 * @author     Mulberry <support@mulberrytech.ca>
 */
class UNPLUG_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        add_action( 'wp_ajax_unplug_run_location_test', array( $this, 'ajax_run_location_test' ) );
        add_action( 'wp_ajax_nopriv_unplug_run_location_test', array( $this, 'ajax_run_location_test' ) );
        add_action( 'wp_ajax_unplug_refresh_plugins', array( $this, 'ajax_refresh_plugins' ) );
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in UNPLUG_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The UNPLUG_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        // Only load on Unplug admin pages for better performance
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'unplug') === false) {
            return;
        }

        wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/unplug-admin.css', array(), $this->version, 'all' );
        
        // Add inline CSS for settings page styling
        $settings_css = "
        .tier-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 3px;
            font-weight: bold;
            text-transform: uppercase;
            font-size: 12px;
        }
        
        .tier-free {
            background-color: #e1e1e1;
            color: #666;
        }
        
        .tier-pro {
            background-color: #0073aa;
            color: #fff;
        }
        
        .tier-agency {
            background-color: #d63638;
            color: #fff;
        }
        
        .tier-features {
            margin-top: 15px;
        }
        
        .tier-features ul {
            list-style: disc;
            margin-left: 20px;
        }
        
        .tier-features li {
            margin-bottom: 5px;
        }
        
        .help-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .help-card {
            background: #fff;
            border: 1px solid #c3c4c7;
            border-radius: 4px;
            padding: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
        }
        
        .help-card h3 {
            margin-top: 0;
            margin-bottom: 10px;
            color: #1d2327;
        }
        
        .help-card p {
            color: #646970;
            line-height: 1.5;
        }
        
        .help-card .button {
            margin-top: 15px;
        }
        
        .api-status-badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
        }
        
        .api-status-free {
            background-color: #e1e1e1;
            color: #666;
        }
        
        .api-status-pro {
            background-color: #00a32a;
            color: #fff;
        }
        
        .api-status-agency {
            background-color: #d63638;
            color: #fff;
        }";
        wp_add_inline_style( $this->plugin_name, $settings_css );
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in UNPLUG_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The UNPLUG_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        // Only load on Unplug admin pages for better performance
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'unplug') === false) {
            return;
        }

        wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/unplug-admin.js', array( 'jquery' ), $this->version, false );
        
        // Add inline JavaScript for settings page functionality
        $settings_js = "
        jQuery(document).ready(function($) {
            // Toggle API token visibility
            $('#toggle-token-visibility').on('click', function() {
                var \$btn = $(this);
                var \$input = $('#settings_api_token');
                var \$icon = \$btn.find('.dashicons');
                var \$text = \$btn.contents().filter(function() {
                    return this.nodeType === 3; // Text nodes
                });
                
                if (\$input.attr('type') === 'password') {
                    \$input.attr('type', 'text');
                    \$icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
                    \$text[0].textContent = '" . esc_js( __( 'Hide', 'unplug' ) ) . "';
                } else {
                    \$input.attr('type', 'password');
                    \$icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
                    \$text[0].textContent = '" . esc_js( __( 'Show', 'unplug' ) ) . "';
                }
            });
            
            // API token validation on blur
            $('#settings_api_token').on('blur', function() {
                var token = $(this).val();
                if (token.length > 0 && token.length < 32) {
                    $(this).css('border-color', '#dc3232');
                    if (!$('.api-token-warning').length) {
                        $(this).after('<p class=\"api-token-warning description\" style=\"color: #dc3232;\">" . esc_js( __( 'API token appears to be invalid. Please check your token.', 'unplug' ) ) . "</p>');
                    }
                } else {
                    $(this).css('border-color', '');
                    $('.api-token-warning').remove();
                }
            });
        });";
        wp_add_inline_script( $this->plugin_name, $settings_js );
        
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Create nonces using Security class
        $nonces = array(
            'admin' => UNPLUG_Security::create_nonce( 'admin' ),
            'scan' => UNPLUG_Security::create_nonce( 'scan' ),
            'export' => UNPLUG_Security::create_nonce( 'export' ),
            'ajax' => UNPLUG_Security::create_nonce( 'ajax' ),
            'run_location_test' => wp_create_nonce( 'unplug_admin_action' )
        );
        
        // Prepare localization data
        $localization_data = array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonces' => $nonces,
            'strings' => array(
                'confirmScan' => esc_html( __( 'Are you sure you want to start an activity scan? This may take several minutes.', 'unplug' ) ),
                'scanProgress' => esc_html( __( 'Scanning in progress...', 'unplug' ) ),
                'scanComplete' => esc_html( __( 'Activity scan completed!', 'unplug' ) ),
                'scanFailed' => esc_html( __( 'Activity scan failed. Please try again.', 'unplug' ) ),
                'refreshSuccess' => esc_html( __( 'Plugin list refreshed successfully!', 'unplug' ) ),
                'refreshFailed' => esc_html( __( 'Failed to refresh plugin list. Please try again.', 'unplug' ) ),
                'networkError' => esc_html( __( 'Network error. Please check your connection and try again.', 'unplug' ) ),
                'unknownError' => esc_html( __( 'An unexpected error occurred. Please try again.', 'unplug' ) )
            )
        );
        
        // Localize script with AJAX data
        wp_localize_script( $this->plugin_name, 'unplugAdmin', $localization_data );

        // Additional scripts and styles can be enqueued here if needed
    }
    


    /**
     * Add the plugin admin menu.
     *
     * @since    1.0.0
     */
    public function add_admin_menu() {
        add_menu_page(
            esc_html( __( 'Unplug', 'unplug' ) ),
            esc_html( __( 'Unplug', 'unplug' ) ),
            'manage_options',
            'unplug',
            array( $this, 'display_admin_page' ),
            'dashicons-admin-plugins',
            30
        );
    }

    /**
     * Display the admin page.
     *
     * @since    1.0.0
     */
    public function display_admin_page() {
        include_once plugin_dir_path( __FILE__ ) . 'partials/unplug-admin-display.php';
    }

    /**
     * Initialize the plugin settings.
     *
     * @since    1.0.0
     */
    public function init_settings() {
        register_setting(
            'unplug_settings',
            'unplug_settings',
            array( $this, 'validate_settings' )
        );

        add_settings_section(
            'unplug_general',
            esc_html( __( 'General Settings', 'unplug' ) ),
            array( $this, 'settings_section_callback' ),
            'unplug_settings'
        );

        // API Token field is rendered directly in the settings template to avoid duplication.

        // Add advanced settings section
        add_settings_section(
            'unplug_advanced',
            esc_html( __( 'Advanced Settings', 'unplug' ) ),
            array( $this, 'advanced_settings_section_callback' ),
            'unplug_settings'
        );

        add_settings_field(
            'scan_timeout',
            esc_html( __( 'Scan Timeout', 'unplug' ) ),
            array( $this, 'scan_timeout_callback' ),
            'unplug_settings',
            'unplug_advanced'
        );

        add_settings_field(
            'cleanup_retention',
            esc_html( __( 'Data Retention', 'unplug' ) ),
            array( $this, 'cleanup_retention_callback' ),
            'unplug_settings',
            'unplug_advanced'
        );

        add_settings_field(
            'enable_logging',
            esc_html( __( 'Debug Logging', 'unplug' ) ),
            array( $this, 'enable_logging_callback' ),
            'unplug_settings',
            'unplug_advanced'
        );
    }

    /**
     * Settings section callback.
     *
     * @since    1.0.0
     */
    public function settings_section_callback() {
        echo '<p>' . esc_html( __( 'Configure your Unplug settings below.', 'unplug' ) ) . '</p>';
    }

    /**
     * API token field callback.
     *
     * @since    1.0.0
     */
    public function api_token_callback() {
        // Load options helper class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $api_token = UNPLUG_Options::get_api_token();
        $user_tier = UNPLUG_Options::get_user_tier();
        
        echo '<input type="password" id="settings_api_token" name="unplug_settings[api_token]" value="' . esc_attr( $api_token ) . '" class="regular-text" autocomplete="new-password" />';
        echo '<p class="description">' . esc_html( __( 'Enter your Unplug API token to unlock Pro features.', 'unplug' ) ) . '</p>';
        // translators: %s: user tier (e.g., Free, Pro)
                    echo '<p><strong>' . esc_html( sprintf( 
                /* translators: %s: user tier level (free, pro, agency) */
                esc_html( __( 'Current Tier: %s', 'unplug' ) ), 
                esc_html( ucfirst( $user_tier ) )
            ) ) . '</strong></p>';
        echo '<p><a href="https://mulberrytech.ca/unplug" target="_blank">' . esc_html( __( 'Get your API token', 'unplug' ) ) . '</a></p>';
    }

    /**
     * Advanced settings section callback.
     *
     * @since    1.0.0
     */
    public function advanced_settings_section_callback() {
        echo '<p>' . esc_html( __( 'Configure advanced settings for scan behavior and data management.', 'unplug' ) ) . '</p>';
    }

    /**
     * Scan timeout field callback.
     *
     * @since    1.0.0
     */
    public function scan_timeout_callback() {
        // Load options helper class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $scan_timeout = UNPLUG_Options::get_scan_timeout();
        
        echo '<input type="number" id="scan_timeout" name="unplug_settings[scan_timeout]" value="' . esc_attr( $scan_timeout ) . '" class="small-text" min="30" max="600" />';
        echo '<span class="description">' . esc_html( __( 'seconds (30-600)', 'unplug' ) ) . '</span>';
        echo '<p class="description">' . esc_html( __( 'Maximum time allowed for activity scans. Increase if you have a large site.', 'unplug' ) ) . '</p>';
    }

    /**
     * Cleanup retention field callback.
     *
     * @since    1.0.0
     */
    public function cleanup_retention_callback() {
        // Load options helper class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $cleanup_retention = UNPLUG_Options::get_cleanup_retention_days();
        
        echo '<select id="cleanup_retention" name="unplug_settings[cleanup_retention]">';
        echo '<option value="7" ' . selected( $cleanup_retention, 7, false ) . '>' . esc_html( __( '7 days', 'unplug' ) ) . '</option>';
        echo '<option value="30" ' . selected( $cleanup_retention, 30, false ) . '>' . esc_html( __( '30 days', 'unplug' ) ) . '</option>';
        echo '<option value="90" ' . selected( $cleanup_retention, 90, false ) . '>' . esc_html( __( '90 days', 'unplug' ) ) . '</option>';
        echo '<option value="365" ' . selected( $cleanup_retention, 365, false ) . '>' . esc_html( __( '1 year', 'unplug' ) ) . '</option>';
        echo '</select>';
        echo '<p class="description">' . esc_html( __( 'How long to keep scan results and queue data before automatic cleanup.', 'unplug' ) ) . '</p>';
    }

    /**
     * Enable logging field callback.
     *
     * @since    1.0.0
     */
    public function enable_logging_callback() {
        // Load options helper class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        
        $enable_logging = UNPLUG_Options::get_enable_logging();
        
        echo '<fieldset>';
        echo '<legend class="screen-reader-text"><span>' . esc_html( __( 'Debug Logging', 'unplug' ) ) . '</span></legend>';
        echo '<label for="enable_logging">';
        echo '<input type="checkbox" id="enable_logging" name="unplug_settings[enable_logging]" value="1" ' . checked( 1, get_option( 'unplug_settings[enable_logging]', 0 ), false ) . ' />';
        echo esc_html( __( 'Enable debug logging', 'unplug' ) );
        echo '</label>';
        echo '<p class="description">' . esc_html( __( 'Enable detailed logging for troubleshooting. Logs are stored in the WordPress debug.log file.', 'unplug' ) ) . '</p>';
        echo '</fieldset>';
    }

    /**
     * Handle AJAX request for refreshing plugin list
     *
     * @since    1.0.0
     */
    public function ajax_refresh_plugins() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'admin', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_refresh_plugins', 'Plugin list refresh requested' );

        // Load helper classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        try {
            // Get fresh plugin list
            $plugins = get_plugins();
            $active_plugins = get_option( 'active_plugins' );

            $plugin_data = array();
            foreach ( $plugins as $plugin_path => $plugin_info ) {
                $plugin_data[] = array(
                    'name' => UNPLUG_Security::sanitize_input( $plugin_info['Name'], 'text' ),
                    'version' => UNPLUG_Security::sanitize_input( $plugin_info['Version'], 'text' ),
                    'status' => in_array( $plugin_path, $active_plugins ) ? 'active' : 'inactive',
                    'path' => UNPLUG_Security::sanitize_input( $plugin_path, 'text' ),
                    'description' => UNPLUG_Security::sanitize_input( $plugin_info['Description'], 'textarea' )
                );
            }

            wp_send_json_success( array(
                'plugins' => $plugin_data,
                'message' => esc_html( __( 'Plugin list refreshed successfully!', 'unplug' ) )
            ) );

        } catch ( Exception $e ) {
            UNPLUG_Security::log_security_event( 'ajax_refresh_plugins_error', 'Failed to refresh plugin list', array( 'error' => $e->getMessage() ) );
            wp_send_json_error( esc_html( __( 'Failed to refresh plugin list.', 'unplug' ) ) );
        }
    }

    /**
     * Handle AJAX request for starting activity scan
     *
     * @since    1.0.0
     */
    public function ajax_start_scan() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Check for confirmation token (required for dangerous operations)
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $confirmation_token = sanitize_text_field( wp_unslash( $_POST['confirmation_token'] ?? '' ) );
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $operation = sanitize_text_field( wp_unslash( $_POST['operation'] ?? '' ) );
        
                    if ( ! empty( $confirmation_token ) && ! empty( $operation ) ) {
                // This is a confirmed dangerous operation
                if ( ! UNPLUG_Security::validate_confirmation_request( $operation, $confirmation_token, true ) ) {
                    wp_send_json_error( esc_html( __( 'Confirmation validation failed.', 'unplug' ) ) );
                }
            
            // Log confirmed operation
            UNPLUG_Security::log_security_event( 'ajax_start_scan_confirmed', 'Activity scan started with confirmation', array( 'operation' => $operation ) );
        } else {
            // Regular scan without confirmation
            UNPLUG_Security::log_security_event( 'ajax_start_scan', 'Activity scan requested' );
        }

        // Load helper classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        try {
            // Get activity scan engine
            global $unplug_activity_scan_engine;
            
            if ( ! $unplug_activity_scan_engine ) {
                // Initialize the activity scan engine if not already done
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-queue.php';
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-activity-scan-engine.php';
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-queue-worker.php';
                
                $database = new UNPLUG_Database();
                $security = new UNPLUG_Security();
                $queue = new UNPLUG_Queue( $database );
                $unplug_activity_scan_engine = new UNPLUG_Activity_Scan_Engine( $database, $security, $queue );
                $GLOBALS['unplug_activity_scan_engine'] = $unplug_activity_scan_engine;
            }
            
            // Perform the activity scan
            $scan_results = $unplug_activity_scan_engine->perform_activity_scan( true );
            
            // Store results in options for retrieval
            update_option( 'unplug_activity_scan_results', $scan_results );
            
            // Store individual plugin results for table display
            if ( isset( $scan_results['plugins'] ) && is_array( $scan_results['plugins'] ) ) {
                foreach ( $scan_results['plugins'] as $plugin_file => $plugin_result ) {
                    $plugin_slug = dirname( $plugin_file );
                    if ( $plugin_slug === '.' ) {
                        $plugin_slug = basename( $plugin_file, '.php' );
                    }
                    
                    // Format result for table display
                    $result_data = array(
                        'status' => $plugin_result['confidence_level'],
                        'details' => $plugin_result['detection_details'] ?? 'No details available',
                        'confidence_score' => $plugin_result['confidence_score'] ?? 0,
                        'usage_evidence' => $plugin_result['usage_evidence'] ?? array(),
                        'timestamp' => current_time( 'timestamp' )
                    );
                    
                    update_option( "unplug_activity_result_{$plugin_slug}", $result_data );
                }
            }
            
            // Return scan ID for progress tracking
            $scan_id = wp_generate_uuid4();
            
            wp_send_json_success( array(
                'scan_id' => $scan_id,
                'message' => esc_html( __( 'Activity scan completed successfully!', 'unplug' ) ),
                'results' => $scan_results
            ) );

        } catch ( Exception $e ) {
            UNPLUG_Security::log_security_event( 'ajax_start_scan_error', 'Failed to start activity scan', array( 'error' => $e->getMessage() ) );
            wp_send_json_error( esc_html( __( 'Failed to start activity scan.', 'unplug' ) ) );
        }
    }

    /**
     * Handle AJAX request for scan progress
     *
     * @since    1.0.0
     */
    public function ajax_scan_progress() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $scan_id = sanitize_text_field( wp_unslash( $_POST['scan_id'] ?? '' ) );
        
        if ( empty( $scan_id ) ) {
            wp_send_json_error( esc_html( __( 'Invalid scan ID.', 'unplug' ) ) );
        }

        try {
            // TODO: Implement actual progress tracking
            // For now, return mock progress data
            
            wp_send_json_success( array(
                'progress' => 75,
                'status' => esc_html( __( 'Analyzing widget areas...', 'unplug' ) ),
                'completed' => false
            ) );

        } catch ( Exception $e ) {
            UNPLUG_Security::log_security_event( 'ajax_scan_progress_error', 'Failed to get scan progress', array( 'error' => $e->getMessage(), 'scan_id' => $scan_id ) );
            wp_send_json_error( esc_html( __( 'Failed to get scan progress.', 'unplug' ) ) );
        }
    }

    /**
     * Handle AJAX request for individual plugin activity test
     *
     * @since    1.1.0
     */
    public function ajax_run_activity_test() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Security check
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $plugin_slug = sanitize_text_field( wp_unslash( $_POST['plugin_slug'] ?? '' ) );
        
        if ( empty( $plugin_slug ) ) {
            wp_send_json_error( esc_html( __( 'Plugin slug is required.', 'unplug' ) ) );
        }
        
        try {
            // Get activity scan engine
            global $unplug_activity_scan_engine;
            
            if ( ! $unplug_activity_scan_engine ) {
                // Initialize the activity scan engine if not already done
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-database.php';
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-queue.php';
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-activity-scan-engine.php';
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-queue-worker.php';
                
                $database = new UNPLUG_Database();
                $security = new UNPLUG_Security();
                $queue = new UNPLUG_Queue( $database );
                $unplug_activity_scan_engine = new UNPLUG_Activity_Scan_Engine( $database, $security, $queue );
                $GLOBALS['unplug_activity_scan_engine'] = $unplug_activity_scan_engine;
            }
            
            // Get stored scan results
            $scan_results = get_option( 'unplug_activity_scan_results', array() );
            
            // Find results for this specific plugin
            $plugin_result = null;
            if ( isset( $scan_results['plugins'] ) && is_array( $scan_results['plugins'] ) ) {
                foreach ( $scan_results['plugins'] as $plugin_data ) {
                    if ( isset( $plugin_data['slug'] ) && $plugin_data['slug'] === $plugin_slug ) {
                        $plugin_result = $plugin_data;
                        break;
                    }
                }
            }
            
            if ( ! $plugin_result ) {
                // If no cached results, perform individual scan
                $scan_results = $unplug_activity_scan_engine->perform_activity_scan( true );
                update_option( 'unplug_activity_scan_results', $scan_results );
                
                // Try to find results again
                if ( isset( $scan_results['plugins'] ) && is_array( $scan_results['plugins'] ) ) {
                    foreach ( $scan_results['plugins'] as $plugin_data ) {
                        if ( isset( $plugin_data['slug'] ) && $plugin_data['slug'] === $plugin_slug ) {
                            $plugin_result = $plugin_data;
                            break;
                        }
                    }
                }
            }
            
            if ( ! $plugin_result ) {
                throw new Exception( 'No results found for plugin: ' . $plugin_slug );
            }
            
            // Determine activity status
            $activity_status = 'no_usage';
            $evidence_count = 0;
            
            if ( isset( $plugin_result['evidence'] ) && is_array( $plugin_result['evidence'] ) ) {
                $evidence_count = count( $plugin_result['evidence'] );
                
                if ( $evidence_count > 0 ) {
                    // Check evidence strength
                    $strong_evidence = 0;
                    foreach ( $plugin_result['evidence'] as $evidence ) {
                        if ( isset( $evidence['strength'] ) && $evidence['strength'] === 'strong' ) {
                            $strong_evidence++;
                        }
                    }
                    
                    if ( $strong_evidence > 0 ) {
                        $activity_status = 'used';
                    } else {
                        $activity_status = 'possibly_used';
                    }
                }
            }
            
            // Store individual plugin result for display
            $plugin_result_data = array(
                'status' => $activity_status,
                'evidence_count' => $evidence_count,
                'details' => sprintf( 
                    // translators: %1$d: number of evidence pieces, %2$s: status
                    /* translators: %1$d: number of evidence pieces found, %2$s: status of the evidence */
                __( 'Found %1$d pieces of evidence. Status: %2$s', 'unplug' ), 
                    $evidence_count, 
                    esc_html( ucfirst( str_replace( '_', ' ', $activity_status ) ) )
                ),
                'timestamp' => current_time( 'timestamp' )
            );
            
            update_option( "unplug_activity_result_{$plugin_slug}", $plugin_result_data );
            
            wp_send_json_success( array(
                'status' => $activity_status,
                'evidence_count' => $evidence_count,
                'plugin_slug' => $plugin_slug,
                'message' => esc_html( __( 'Activity test completed successfully!', 'unplug' ) )
            ) );
            
        } catch ( Exception $e ) {
            UNPLUG_Security::log_security_event( 'ajax_run_activity_test_error', 'Failed to run activity test', array( 
                'plugin_slug' => $plugin_slug,
                'error' => $e->getMessage() 
            ) );
            wp_send_json_error( esc_html( __( 'Failed to run activity test: ', 'unplug' ) ) . $e->getMessage() );
        }
    }

    /**
     * Handle AJAX request for CSV export
     *
     * @since    1.0.0
     */
    public function ajax_export_csv() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'export', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'export', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_export_csv', 'CSV export requested' );

        try {
            // Get all plugins
            $all_plugins = get_plugins();
            $active_plugins = get_option( 'active_plugins', array() );
            
            // Get scan results
            $scan_results = get_option( 'unplug_activity_scan_results', array() );
            
            // Prepare CSV data
            $csv_data = array();
            $csv_data[] = array( 'Plugin Name', 'Status', 'Activity', 'File Path', 'Version', 'Author' );
            
            foreach ( $all_plugins as $plugin_file => $plugin_data ) {
                $plugin_slug = dirname( $plugin_file );
                if ( $plugin_slug === '.' ) {
                    $plugin_slug = basename( $plugin_file, '.php' );
                }
                
                $is_active = in_array( $plugin_file, $active_plugins );
                
                // Get activity result
                $activity_result = get_option( "unplug_activity_result_{$plugin_slug}", null );
                $activity_status = $activity_result ? esc_html( ucfirst( str_replace( '_', ' ', $activity_result['status'] ) ) ) : esc_html( 'Not Tested' );
                
                $csv_data[] = array(
                    $plugin_data['Name'],
                    $is_active ? 'Active' : 'Inactive',
                    $activity_status,
                    $plugin_file,
                    $plugin_data['Version'],
                    $plugin_data['Author']
                );
            }
            
            // Generate CSV content
            $csv_content = '';
            foreach ( $csv_data as $row ) {
                $csv_content .= implode( ',', array_map( function( $field ) {
                    return '"' . str_replace( '"', '""', $field ) . '"';
                }, $row ) ) . "\n";
            }
            
            // Create temporary file using WordPress file system
            $upload_dir = wp_upload_dir();
            $filename = 'unplug-export-' . gmdate( 'Y-m-d-H-i-s' ) . '.csv';
            $filepath = $upload_dir['path'] . '/' . $filename;
            
            // Initialize WordPress file system
            global $wp_filesystem;
            if ( empty( $wp_filesystem ) ) {
                require_once ABSPATH . '/wp-admin/includes/file.php';
                WP_Filesystem();
            }
            
            $wp_filesystem->put_contents( $filepath, $csv_content );
            
            wp_send_json_success( array(
                'message' => esc_html( __( 'CSV export completed!', 'unplug' ) ),
                'download_url' => $upload_dir['url'] . '/' . $filename,
                'filename' => $filename
            ) );

        } catch ( Exception $e ) {
            UNPLUG_Security::log_security_event( 'ajax_export_csv_error', 'Failed to export CSV', array( 'error' => $e->getMessage() ) );
            wp_send_json_error( esc_html( __( 'Failed to export CSV.', 'unplug' ) ) );
        }
    }

    /**
     * Handle AJAX request for report generation
     *
     * @since    1.0.0
     */
    public function ajax_generate_report() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'export', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'export', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_generate_report', 'Report generation requested' );

        // Load helper classes
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';

        // Check if user has agency tier
        if ( ! UNPLUG_Options::is_feature_allowed('agency') ) {
            UNPLUG_Security::log_security_event( 'ajax_generate_report_denied', 'Report generation denied - insufficient tier', array( 'user_tier' => UNPLUG_Options::get_user_tier() ) );
            wp_send_json_error( esc_html( __( 'Report generation is only available for Agency tier users.', 'unplug' ) ) );
        }

        try {
            // Get all plugins and their data
            $all_plugins = get_plugins();
            $active_plugins = get_option( 'active_plugins', array() );
            $scan_results = get_option( 'unplug_activity_scan_results', array() );
            
            // Generate HTML report
            $html_content = $this->generate_html_report( $all_plugins, $active_plugins, $scan_results );
            
            // Create temporary file using WordPress file system
            $upload_dir = wp_upload_dir();
            $filename = 'unplug-report-' . gmdate( 'Y-m-d-H-i-s' ) . '.html';
            $filepath = $upload_dir['path'] . '/' . $filename;
            
            // Initialize WordPress file system
            global $wp_filesystem;
            if ( empty( $wp_filesystem ) ) {
                require_once ABSPATH . '/wp-admin/includes/file.php';
                WP_Filesystem();
            }
            
            $wp_filesystem->put_contents( $filepath, $html_content );
            
            wp_send_json_success( array(
                'message' => esc_html( __( 'Report generated successfully!', 'unplug' ) ),
                'download_url' => $upload_dir['url'] . '/' . $filename,
                'filename' => $filename
            ) );

        } catch ( Exception $e ) {
            UNPLUG_Security::log_security_event( 'ajax_generate_report_error', 'Failed to generate report', array( 'error' => $e->getMessage() ) );
            wp_send_json_error( esc_html( __( 'Failed to generate report.', 'unplug' ) ) );
        }
    }

    /**
     * Handle AJAX request for getting confirmation token
     *
     * @since    1.0.0
     */
    public function ajax_get_confirmation_token() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Explicit nonce verification for WordPress plugin checker compliance
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'unplug_admin' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid security token' ) );
        }
        
        // Additional security verification
        UNPLUG_Security::verify_ajax_request( 'admin', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'admin', true );
        
        // Get operation from request
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $operation = isset( $_POST['operation'] ) ? sanitize_text_field( wp_unslash( $_POST['operation'] ) ) : '';
        
        if ( empty( $operation ) ) {
            wp_send_json_error( array( 'message' => 'Operation not specified' ) );
        }
        
        // Generate confirmation token
        $token = UNPLUG_Security::generate_confirmation_token( $operation );
        
        if ( $token ) {
            wp_send_json_success( array( 
                'token' => $token,
                'message' => 'Confirmation token generated successfully'
            ) );
        } else {
            wp_send_json_error( array( 'message' => 'Failed to generate confirmation token' ) );
        }
    }

    /**
     * Handle AJAX request for plugin inventory analysis
     *
     * @since    1.0.0
     */
    public function ajax_get_plugin_inventory() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_get_plugin_inventory', 'Plugin inventory analysis requested' );

        // Load plugin inventory class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';

        try {
            // Get force refresh parameter
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $force_refresh = isset( $_POST['force_refresh'] ) && sanitize_text_field( wp_unslash( $_POST['force_refresh'] ) ) === 'true';
            
            // Get inventory data
            $inventory = UNPLUG_Plugin_Inventory::get_plugin_inventory( $force_refresh );
            $summary = UNPLUG_Plugin_Inventory::get_plugin_summary();
            $conflicts = UNPLUG_Plugin_Inventory::get_plugin_conflicts();

            // Format response
            wp_send_json_success( array(
                'inventory' => $inventory,
                'summary' => $summary,
                'conflicts' => $conflicts,
                'message' => 'Plugin inventory analysis completed successfully'
            ) );

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_get_plugin_inventory_error', 'Plugin inventory analysis failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Plugin inventory analysis failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request for plugin conflict detection
     *
     * @since    1.0.0
     */
    public function ajax_detect_plugin_conflicts() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_detect_plugin_conflicts', 'Plugin conflict detection requested' );

        // Load plugin inventory class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';

        try {
            // Get conflicts
            $conflicts = UNPLUG_Plugin_Inventory::get_plugin_conflicts();
            
            // Get high severity conflicts
            $high_severity = array_filter( $conflicts, function( $conflict ) {
                return $conflict['severity'] === 'high';
            });
            
            // Get medium severity conflicts
            $medium_severity = array_filter( $conflicts, function( $conflict ) {
                return $conflict['severity'] === 'medium';
            });
            
            // Get low severity conflicts
            $low_severity = array_filter( $conflicts, function( $conflict ) {
                return $conflict['severity'] === 'low';
            });

            // Format response
            wp_send_json_success( array(
                'conflicts' => $conflicts,
                'high_severity' => $high_severity,
                'medium_severity' => $medium_severity,
                'low_severity' => $low_severity,
                'total_conflicts' => count( $conflicts ),
                'message' => 'Plugin conflict detection completed successfully'
            ) );

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_detect_plugin_conflicts_error', 'Plugin conflict detection failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Plugin conflict detection failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request for plugin analysis by category
     *
     * @since    1.0.0
     */
    public function ajax_get_plugins_by_category() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_get_plugins_by_category', 'Plugin category analysis requested' );

        // Load plugin inventory class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';

        try {
            // Get category from request
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $category = isset( $_POST['category'] ) ? sanitize_text_field( wp_unslash( $_POST['category'] ) ) : '';
            
            if ( empty( $category ) ) {
                wp_send_json_error( array( 'message' => 'Category not specified' ) );
            }
            
            // Get plugins by category
            $plugins = UNPLUG_Plugin_Inventory::get_plugins_by_category( $category );
            
            // Count active vs inactive
            $active_count = count( array_filter( $plugins, function( $plugin ) {
                return $plugin['active'];
            }));
            
            $inactive_count = count( $plugins ) - $active_count;

            // Format response
            wp_send_json_success( array(
                'category' => $category,
                'plugins' => $plugins,
                'total_plugins' => count( $plugins ),
                'active_count' => $active_count,
                'inactive_count' => $inactive_count,
                'message' => 'Plugin category analysis completed successfully'
            ) );

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_get_plugins_by_category_error', 'Plugin category analysis failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Plugin category analysis failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Validate settings.
     *
     * @since    1.0.0
     * @param    array    $input    The input array.
     * @return   array              The validated input array.
     */
    public function validate_settings( $input ) {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        // Load options helper class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-options.php';
        // Load licensing class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-licensing.php';
        
        // Direct nonce verification for WordPress plugin checker compliance
        if ( ! isset( $_POST['unplug_settings_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['unplug_settings_nonce'] ) ), 'unplug_settings_nonce' ) ) {
            wp_die( esc_html( __( 'Security check failed. Please try again.', 'unplug' ) ) );
        }
        
        // Additional security verification (existing wrapper)
        UNPLUG_Security::verify_nonce( sanitize_text_field( wp_unslash( $_POST['unplug_settings_nonce'] ) ), 'settings' );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'settings_validation', 'Settings validation started' );
        
        // Use the security class for validation
        $sanitized_input = UNPLUG_Security::validate_settings( $input );
        
        // Update individual options
        if ( isset( $sanitized_input['api_token'] ) ) {
            $token = trim( (string) $sanitized_input['api_token'] );
            $token_updated = UNPLUG_Options::update_api_token( $token );
            
            if ( $token_updated && ! empty( $sanitized_input['api_token'] ) ) {
                // Development bypass: accept test token without calling licensing server
                if ( $token === 'UNPLUG_TEST_TOKEN_123456' ) {
                    $license_result = array(
                        'status' => 'valid',
                        'tier' => 'pro',
                        'checked_at' => time(),
                        'degraded' => false,
                        'message' => 'Test token accepted (development)'
                    );
                    UNPLUG_Options::update_user_tier( 'pro' );
                    update_option( 'unplug_license_status', $license_result );
                } else {
                    // Validate API token with external licensing service and update user tier
                    $license_result = UNPLUG_Licensing::validate_license( $sanitized_input['api_token'], true );
                    if ( is_wp_error( $license_result ) ) {
                        // Show admin notice for license error
                        add_settings_error(
                            'unplug_settings',
                            'license_error',
                            __( 'License validation failed: ', 'unplug' ) . $license_result->get_error_message(),
                            'error'
                        );
                        // Set user tier to free on error
                        UNPLUG_Options::update_user_tier( 'free' );
                    } else {
                        // Update user tier based on license response
                        $tier = isset( $license_result['tier'] ) ? $license_result['tier'] : 'free';
                        UNPLUG_Options::update_user_tier( $tier );
                        // Optionally store license status/details for admin UI
                        update_option( 'unplug_license_status', $license_result );
                    }
                }
                // Log successful token storage
                UNPLUG_Security::log_security_event( 'api_token_updated', 'API token updated and validated via settings form' );
            } elseif ( ! empty( $sanitized_input['api_token'] ) ) {
                // Log failed token storage
                UNPLUG_Security::log_security_event( 'api_token_update_failed', 'API token update failed via settings form' );
            }
        }
        
        if ( isset( $sanitized_input['scan_timeout'] ) ) {
            UNPLUG_Options::update_setting( 'scan_timeout', $sanitized_input['scan_timeout'] );
        }
        
        if ( isset( $sanitized_input['cleanup_retention'] ) ) {
            UNPLUG_Options::update_setting( 'cleanup_retention_days', $sanitized_input['cleanup_retention'] );
        }
        
        if ( isset( $sanitized_input['enable_logging'] ) ) {
            UNPLUG_Options::update_setting( 'enable_logging', $sanitized_input['enable_logging'] );
        }
        
        // Log successful validation
        UNPLUG_Security::log_security_event( 'settings_validation_complete', 'Settings validation completed successfully' );
        
        return $sanitized_input;
    }

    /**
     * Show migration-related admin notices.
     *
     * @since    1.0.0
     */
    public function show_migration_notices() {
        // Only show to administrators
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Check for migration failure
        if ( get_option( 'unplug_migration_failed' ) ) {
            $error_message = get_option( 'unplug_migration_error', 'Database migration failed. Please check the error logs.' );
            echo '<div class="notice notice-error is-dismissible">';
            echo '<p><strong>' . esc_html( __( 'Unplug Database Migration Failed', 'unplug' ) ) . '</strong></p>';
            echo '<p>' . esc_html( $error_message ) . '</p>';
            echo '<p>' . esc_html( __( 'Please check the error logs or contact support for assistance.', 'unplug' ) ) . '</p>';
            echo '</div>';
        }

        // Check if migration is needed but hasn't been attempted
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-migrator.php';
        
        if ( UNPLUG_Migrator::is_migration_needed() ) {
            echo '<div class="notice notice-warning">';
            echo '<p><strong>' . esc_html( __( 'Unplug Database Update Required', 'unplug' ) ) . '</strong></p>';
            echo '<p>' . esc_html( __( 'A database update is required for Unplug to function properly. The update will run automatically on the next page load.', 'unplug' ) ) . '</p>';
            echo '</div>';
        }
    }

    /**
     * Handle AJAX request for advanced conflict pattern analysis
     *
     * @since    1.0.0
     */
    public function ajax_analyze_conflict_patterns() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_analyze_conflict_patterns', 'Advanced conflict pattern analysis requested' );

        try {
            // Get conflict engine instance
            if ( isset( $GLOBALS['unplug_conflict_engine'] ) ) {
                $conflict_engine = $GLOBALS['unplug_conflict_engine'];
                
                // Get plugin inventory
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';
                $inventory = UNPLUG_Plugin_Inventory::get_plugin_inventory();
                
                // Analyze conflicts using the sophisticated engine
                $conflict_analysis = $conflict_engine->analyze_conflicts( $inventory['plugins'] );
                
                wp_send_json_success( array(
                    'analysis' => $conflict_analysis,
                    'message' => 'Advanced conflict pattern analysis completed successfully'
                ) );
            } else {
                wp_send_json_error( array(
                    'message' => 'Conflict engine not initialized'
                ) );
            }
        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_analyze_conflict_patterns_error', 'Advanced conflict analysis failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Advanced conflict analysis failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request for exporting conflict analysis
     *
     * @since    1.0.0
     */
    public function ajax_export_conflict_analysis() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Explicit nonce verification for WordPress plugin checker compliance
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'unplug_export' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid security token' ) );
        }
        
        // Additional security verification
        UNPLUG_Security::verify_ajax_request( 'export', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'export', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_export_conflict_analysis', 'Conflict analysis export requested' );

        try {
            // Get export format
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $format = isset( $_POST['format'] ) ? sanitize_text_field( wp_unslash( $_POST['format'] ) ) : 'json';
            
            // Validate format
            if ( ! in_array( $format, array( 'json', 'csv', 'xml' ) ) ) {
                wp_send_json_error( array( 'message' => 'Invalid export format specified' ) );
            }
            
            // Get conflict engine instance
            if ( isset( $GLOBALS['unplug_conflict_engine'] ) ) {
                $conflict_engine = $GLOBALS['unplug_conflict_engine'];
                
                // Get plugin inventory
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';
                $inventory = UNPLUG_Plugin_Inventory::get_plugin_inventory();
                
                // Analyze conflicts
                $conflict_analysis = $conflict_engine->analyze_conflicts( $inventory['plugins'] );
                
                // Export analysis
                $exported_data = $conflict_engine->export_conflict_analysis( $conflict_analysis, $format );
                
                // Set appropriate content type
                $content_types = array(
                    'json' => 'application/json',
                    'csv' => 'text/csv',
                    'xml' => 'application/xml'
                );
                
                wp_send_json_success( array(
                    'data' => $exported_data,
                    'format' => $format,
                    'content_type' => $content_types[$format],
                    'filename' => 'unplug-conflict-analysis-' . gmdate( 'Y-m-d-H-i-s' ) . '.' . $format,
                    'message' => 'Conflict analysis exported successfully'
                ) );
            } else {
                wp_send_json_error( array(
                    'message' => 'Conflict engine not initialized'
                ) );
            }
        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_export_conflict_analysis_error', 'Conflict analysis export failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Conflict analysis export failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request to get warnings
     *
     * @since    1.0.0
     */
    public function ajax_get_warnings() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_get_warnings', 'Warning retrieval requested' );

        try {
            // Load warning system class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-warning-system.php';
            
            $filters = array();
            
            // Get filter parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            if ( isset( $_POST['type'] ) && ! empty( $_POST['type'] ) ) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
                $filters['type'] = sanitize_text_field( wp_unslash( $_POST['type'] ) );
            }
            
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            if ( isset( $_POST['severity'] ) && ! empty( $_POST['severity'] ) ) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
                $filters['severity'] = sanitize_text_field( wp_unslash( $_POST['severity'] ) );
            }
            
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            if ( isset( $_POST['status'] ) && ! empty( $_POST['status'] ) ) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
                $filters['status'] = sanitize_text_field( wp_unslash( $_POST['status'] ) );
            }
            
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            if ( isset( $_POST['active_only'] ) && sanitize_text_field( wp_unslash( $_POST['active_only'] ) ) === 'true' ) {
                $filters['active_only'] = true;
            }
            
            // Get warnings for display
            $warnings = UNPLUG_Warning_System::get_warnings_for_display();
            
            // Filter if needed
            if ( ! empty( $filters ) ) {
                $warnings = UNPLUG_Warning_System::get_warnings( $filters );
            }
            
            // Get warning statistics
            $stats = UNPLUG_Warning_System::get_warning_stats();
            
            wp_send_json_success( array(
                'warnings' => $warnings,
                'stats' => $stats,
                'message' => 'Warnings retrieved successfully'
            ) );

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_get_warnings_error', 'Warning retrieval failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Warning retrieval failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request to update warning status
     *
     * @since    1.0.0
     */
    public function ajax_update_warning_status() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_update_warning_status', 'Warning status update requested' );

        try {
            // Load warning system class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-warning-system.php';
            
            // Get and validate parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $warning_id = isset( $_POST['warning_id'] ) ? sanitize_text_field( wp_unslash( $_POST['warning_id'] ) ) : '';
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : '';
            
            if ( empty( $warning_id ) || empty( $status ) ) {
                wp_send_json_error( array(
                    'message' => 'Missing required parameters: warning_id and status'
                ) );
                return;
            }
            
            // Update warning status
            $result = UNPLUG_Warning_System::update_warning_status( $warning_id, $status );
            
            if ( $result ) {
                wp_send_json_success( array(
                    'warning_id' => $warning_id,
                    'status' => $status,
                    'message' => 'Warning status updated successfully'
                ) );
            } else {
                wp_send_json_error( array(
                    'message' => 'Failed to update warning status'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_update_warning_status_error', 'Warning status update failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Warning status update failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request to dismiss warning
     *
     * @since    1.0.0
     */
    public function ajax_dismiss_warning() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Explicit nonce verification for WordPress plugin checker compliance
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'unplug_scan' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid security token' ) );
        }
        
        // Additional security verification
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_dismiss_warning', 'Warning dismissal requested' );

        try {
            // Load warning system class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-warning-system.php';
            
            // Get and validate parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $warning_id = isset( $_POST['warning_id'] ) ? sanitize_text_field( wp_unslash( $_POST['warning_id'] ) ) : '';
            
            if ( empty( $warning_id ) ) {
                wp_send_json_error( array(
                    'message' => 'Missing required parameter: warning_id'
                ) );
                return;
            }
            
            // Dismiss warning
            $result = UNPLUG_Warning_System::update_warning_status( $warning_id, UNPLUG_Warning_System::STATUS_DISMISSED );
            
            if ( $result ) {
                wp_send_json_success( array(
                    'warning_id' => $warning_id,
                    'message' => 'Warning dismissed successfully'
                ) );
            } else {
                wp_send_json_error( array(
                    'message' => 'Failed to dismiss warning'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_dismiss_warning_error', 'Warning dismissal failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Warning dismissal failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request to generate warnings from conflict analysis
     *
     * @since    1.0.0
     */
    public function ajax_generate_warnings() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_generate_warnings', 'Warning generation requested' );

        try {
            // Load warning system class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-warning-system.php';
            
            // Get conflict engine instance
            if ( isset( $GLOBALS['unplug_conflict_engine'] ) ) {
                $conflict_engine = $GLOBALS['unplug_conflict_engine'];
                
                // Get plugin inventory
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-plugin-inventory.php';
                $inventory = UNPLUG_Plugin_Inventory::get_plugin_inventory();
                
                // Analyze conflicts
                $conflict_analysis = $conflict_engine->analyze_conflicts( $inventory['plugins'] );
                
                // Generate warnings from conflict analysis
                $warning_ids = UNPLUG_Warning_System::generate_warnings_from_conflicts( $conflict_analysis );
                
                // Get warning statistics
                $stats = UNPLUG_Warning_System::get_warning_stats();
                
                wp_send_json_success( array(
                    'warning_ids' => $warning_ids,
                    'warnings_generated' => count( $warning_ids ),
                    'stats' => $stats,
                    'message' => sprintf( '%d warnings generated successfully', count( $warning_ids ) )
                ) );
            } else {
                wp_send_json_error( array(
                    'message' => 'Conflict engine not initialized'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_generate_warnings_error', 'Warning generation failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Warning generation failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Get safe mode status
     *
     * @since    1.0.0
     */
    public function ajax_get_safe_mode_status() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_get_safe_mode_status', 'Safe mode status requested' );

        try {
            // Load safe mode class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
            
            // Get safe mode dashboard data
            $dashboard_data = UNPLUG_Safe_Mode::get_dashboard_data();
            
            wp_send_json_success( array(
                'dashboard_data' => $dashboard_data,
                'message' => 'Safe mode status retrieved successfully'
            ) );

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_get_safe_mode_status_error', 'Safe mode status retrieval failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Safe mode status retrieval failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Activate safe mode
     *
     * @since    1.0.0
     */
    public function ajax_activate_safe_mode() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_activate_safe_mode', 'Safe mode activation requested' );

        try {
            // Load safe mode class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
            
            // Get request parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $trigger = sanitize_text_field( wp_unslash( $_POST['trigger'] ?? 'user_activated' ) );
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $conflicting_plugins = isset( $_POST['conflicting_plugins'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['conflicting_plugins'] ) ) : array();
            
            $options = array();
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            if ( isset( $_POST['emergency_mode'] ) && sanitize_text_field( wp_unslash( $_POST['emergency_mode'] ) ) === 'true' ) {
                $trigger = 'emergency';
            }
            
            // Activate safe mode
            $result = UNPLUG_Safe_Mode::activate_safe_mode( $trigger, $conflicting_plugins, $options );
            
            if ( is_wp_error( $result ) ) {
                wp_send_json_error( array(
                    'message' => 'Safe mode activation failed: ' . $result->get_error_message()
                ) );
            } else {
                // Get updated status
                $status = UNPLUG_Safe_Mode::get_safe_mode_status();
                
                wp_send_json_success( array(
                    'status' => $status,
                    'message' => 'Safe mode activated successfully'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_activate_safe_mode_error', 'Safe mode activation failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Safe mode activation failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Deactivate safe mode
     *
     * @since    1.0.0
     */
    public function ajax_deactivate_safe_mode() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_deactivate_safe_mode', 'Safe mode deactivation requested' );

        try {
            // Load safe mode class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
            
            // Get request parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $restore_plugins = isset( $_POST['restore_plugins'] ) && sanitize_text_field( wp_unslash( $_POST['restore_plugins'] ) ) === 'true';
            
            // Deactivate safe mode
            $result = UNPLUG_Safe_Mode::deactivate_safe_mode( $restore_plugins );
            
            if ( is_wp_error( $result ) ) {
                wp_send_json_error( array(
                    'message' => 'Safe mode deactivation failed: ' . $result->get_error_message()
                ) );
            } else {
                // Get updated status
                $status = UNPLUG_Safe_Mode::get_safe_mode_status();
                
                wp_send_json_success( array(
                    'status' => $status,
                    'restored_plugins' => $restore_plugins,
                    'message' => 'Safe mode deactivated successfully'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_deactivate_safe_mode_error', 'Safe mode deactivation failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Safe mode deactivation failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Restore from backup
     *
     * @since    1.0.0
     */
    public function ajax_restore_from_backup() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_restore_from_backup', 'Backup restoration requested' );

        try {
            // Load safe mode class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
            
            // Get request parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $backup_id = sanitize_text_field( wp_unslash( $_POST['backup_id'] ?? '' ) );
            
            if ( empty( $backup_id ) ) {
                wp_send_json_error( array(
                    'message' => 'Backup ID is required'
                ) );
            }
            
            // Restore from backup
            $result = UNPLUG_Safe_Mode::restore_from_backup( $backup_id );
            
            if ( is_wp_error( $result ) ) {
                wp_send_json_error( array(
                    'message' => 'Backup restoration failed: ' . $result->get_error_message()
                ) );
            } else {
                wp_send_json_success( array(
                    'backup_id' => $backup_id,
                    'message' => 'Backup restored successfully'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_restore_from_backup_error', 'Backup restoration failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Backup restoration failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Get backup states
     *
     * @since    1.0.0
     */
    public function ajax_get_backup_states() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_get_backup_states', 'Backup states requested' );

        try {
            // Load safe mode class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
            
            // Get backup states
            $backup_states = UNPLUG_Safe_Mode::get_backup_states();
            
            wp_send_json_success( array(
                'backup_states' => $backup_states,
                'count' => count( $backup_states ),
                'message' => 'Backup states retrieved successfully'
            ) );

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_get_backup_states_error', 'Backup states retrieval failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Backup states retrieval failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Delete backup state
     *
     * @since    1.0.0
     */
    public function ajax_delete_backup_state() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Explicit nonce verification for WordPress plugin checker compliance
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'unplug_manage' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid security token' ) );
        }
        
        // Additional security verification
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_delete_backup_state', 'Backup state deletion requested' );

        try {
            // Load safe mode class
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-safe-mode.php';
            
            // Get request parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $backup_id = sanitize_text_field( wp_unslash( $_POST['backup_id'] ?? '' ) );
            
            if ( empty( $backup_id ) ) {
                wp_send_json_error( array(
                    'message' => 'Backup ID is required'
                ) );
            }
            
            // Delete backup state
            $result = UNPLUG_Safe_Mode::delete_backup_state( $backup_id );
            
            if ( $result ) {
                wp_send_json_success( array(
                    'backup_id' => $backup_id,
                    'message' => 'Backup state deleted successfully'
                ) );
            } else {
                wp_send_json_error( array(
                    'message' => 'Failed to delete backup state'
                ) );
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_delete_backup_state_error', 'Backup state deletion failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Backup state deletion failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request for context control (clear location data and export data)
     *
     * @since    1.0.0
     */
    public function ajax_context_control() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'manage', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'manage', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_context_control', 'Context control requested' );

        try {
            // Get operation from request
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
            $operation = isset( $_POST['operation'] ) ? sanitize_text_field( wp_unslash( $_POST['operation'] ) ) : '';
            
            if ( empty( $operation ) ) {
                wp_send_json_error( array( 'message' => 'Operation not specified' ) );
            }

            // Handle clear location data
            if ( $operation === 'clear_location_data' ) {
                // Load location data class
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-location-data.php';
                
                $result = UNPLUG_Location_Data::clear_location_data();
                
                if ( $result ) {
                    wp_send_json_success( array(
                        'message' => __( 'Location data cleared successfully.', 'unplug' )
                    ) );
                } else {
                    wp_send_json_error( array(
                        'message' => __( 'Failed to clear location data.', 'unplug' )
                    ) );
                }
            }

            // Handle export data
            if ( $operation === 'export_data' ) {
                // Load export class
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-exporter.php';
                
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
                $format = isset( $_POST['format'] ) ? sanitize_text_field( wp_unslash( $_POST['format'] ) ) : 'json';
                
                if ( ! in_array( $format, array( 'json', 'csv', 'xml' ) ) ) {
                    wp_send_json_error( array( 'message' => esc_html( __( 'Invalid export format specified.', 'unplug' ) ) ) );
                }

                $export_data = UNPLUG_Exporter::export_all_data( $format );
                
                if ( $export_data ) {
                    wp_send_json_success( array(
                        'data' => $export_data,
                        'format' => $format,
                        'filename' => 'unplug-export-' . gmdate( 'Y-m-d-H-i-s' ) . '.' . $format,
                        'message' => esc_html( __( 'Data exported successfully!', 'unplug' ) )
                    ) );
                } else {
                    wp_send_json_error( array(
                        'message' => esc_html( __( 'Failed to export data.', 'unplug' ) )
                    ) );
                }
            }

        } catch ( Exception $e ) {
            // Log error
            UNPLUG_Security::log_security_event( 'ajax_context_control_error', 'Context control failed: ' . $e->getMessage() );
            
            wp_send_json_error( array( 
                'message' => 'Context control failed: ' . $e->getMessage()
            ) );
        }
    }

    /**
     * Handle AJAX request to clear location data
     *
     * @since    1.0.0
     */
    public function ajax_clear_location_data() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'settings', 'manage_options', 'nonce' );
        
        try {
            // Get all plugins
            $all_plugins = get_plugins();
            
            // Clear location data for all plugins
            foreach ( $all_plugins as $plugin_file => $plugin_data ) {
                $plugin_slug = dirname( $plugin_file );
                if ( $plugin_slug === '.' ) {
                    $plugin_slug = basename( $plugin_file, '.php' );
                }
                
                delete_option( "unplug_location_result_{$plugin_slug}" );
            }
            
            // Clear general location data
            delete_option( 'unplug_location_scan_results' );
            
            wp_send_json_success( array(
                'message' => esc_html( __( 'Location data cleared successfully.', 'unplug' ) )
            ) );
            
        } catch ( Exception $e ) {
            wp_send_json_error( array(
                'message' => esc_html( __( 'Failed to clear location data.', 'unplug' ) )
            ) );
        }
    }

    /**
     * Handle AJAX request to export location data
     *
     * @since    1.0.0
     */
    public function ajax_export_location_data() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'settings', 'manage_options', 'nonce' );
        
        try {
            // Get all plugins and their location data
            $all_plugins = get_plugins();
            $export_data = array();
            
            foreach ( $all_plugins as $plugin_file => $plugin_data ) {
                $plugin_slug = dirname( $plugin_file );
                if ( $plugin_slug === '.' ) {
                    $plugin_slug = basename( $plugin_file, '.php' );
                }
                
                $location_result = get_option( "unplug_location_result_{$plugin_slug}", null );
                
                if ( $location_result ) {
                    $export_data[] = array(
                        'plugin_name' => $plugin_data['Name'],
                        'plugin_file' => $plugin_file,
                        'location_count' => $location_result['count'] ?? 0,
                        'location_details' => $location_result['details'] ?? array()
                    );
                }
            }
            
            // Create CSV content
            $csv_content = "Plugin Name,Plugin File,Location Count,Location Details\n";
            foreach ( $export_data as $row ) {
                $details = is_array( $row['location_details'] ) ? implode( '; ', $row['location_details'] ) : $row['location_details'];
                $csv_content .= sprintf( '"%s","%s","%s","%s"' . "\n",
                    str_replace( '"', '""', $row['plugin_name'] ),
                    str_replace( '"', '""', $row['plugin_file'] ),
                    $row['location_count'],
                    str_replace( '"', '""', $details )
                );
            }
            
            // Create temporary file using WordPress file system
            $upload_dir = wp_upload_dir();
            $filename = 'unplug-location-data-' . gmdate( 'Y-m-d-H-i-s' ) . '.csv';
            $filepath = $upload_dir['path'] . '/' . $filename;
            
            // Initialize WordPress file system
            global $wp_filesystem;
            if ( empty( $wp_filesystem ) ) {
                require_once ABSPATH . '/wp-admin/includes/file.php';
                WP_Filesystem();
            }
            
            $wp_filesystem->put_contents( $filepath, $csv_content );
            
            wp_send_json_success( array(
                'message' => esc_html( __( 'Location data exported successfully.', 'unplug' ) ),
                'download_url' => $upload_dir['url'] . '/' . $filename
            ) );
            
        } catch ( Exception $e ) {
            wp_send_json_error( array(
                'message' => esc_html( __( 'Failed to export location data.', 'unplug' ) )
            ) );
        }
    }

    /**
     * Generate HTML report content
     *
     * @since    1.1.0
     * @param    array    $all_plugins      All installed plugins
     * @param    array    $active_plugins   Active plugin files
     * @param    array    $scan_results     Scan results data
     * @return   string                     HTML report content
     */
    private function generate_html_report( $all_plugins, $active_plugins, $scan_results ) {
        // Define CSS styles for the HTML report
        $report_styles = '
        body { font-family: Arial, sans-serif; margin: 20px; }
        .header { background: #65626F; color: white; padding: 20px; margin-bottom: 20px; }
        .summary { background: #F7F7F7; padding: 15px; margin-bottom: 20px; border-left: 4px solid #65626F; }
        .plugin-table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
        .plugin-table th, .plugin-table td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        .plugin-table th { background: #65626F; color: white; }
        .status-active { color: #0073aa; font-weight: bold; }
        .status-inactive { color: #666; }
        .activity-used { color: #46b450; }
        .activity-possibly-used { color: #ffb900; }
        .activity-no-usage { color: #dc3232; }
        .activity-not-tested { color: #666; }
        .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; color: #666; }';
        
        $html = '<!DOCTYPE html>
<html>
<head>
    <title>Unplug Plugin Analysis Report</title>
    <style>' . $report_styles . '</style>
</head>
<body>
    <div class="header">
        <h1>Unplug Plugin Analysis Report</h1>
        <p>Generated on ' . gmdate( 'F j, Y \a\t g:i A' ) . '</p>
    </div>
    
    <div class="summary">
        <h2>Summary</h2>';
        
        $total_plugins = count( $all_plugins );
        $active_count = count( $active_plugins );
        $inactive_count = $total_plugins - $active_count;
        
        $html .= '<p><strong>Total Plugins:</strong> ' . $total_plugins . '</p>';
        $html .= '<p><strong>Active Plugins:</strong> ' . $active_count . '</p>';
        $html .= '<p><strong>Inactive Plugins:</strong> ' . $inactive_count . '</p>';
        
        $html .= '
    </div>
    
    <h2>Plugin Details</h2>
    <table class="plugin-table">
        <thead>
            <tr>
                <th>Plugin Name</th>
                <th>Status</th>
                <th>Activity</th>
                <th>Version</th>
                <th>Author</th>
                <th>File</th>
            </tr>
        </thead>
        <tbody>';
        
        foreach ( $all_plugins as $plugin_file => $plugin_data ) {
            $plugin_slug = dirname( $plugin_file );
            if ( $plugin_slug === '.' ) {
                $plugin_slug = basename( $plugin_file, '.php' );
            }
            
            $is_active = in_array( $plugin_file, $active_plugins );
            $activity_result = get_option( "unplug_activity_result_{$plugin_slug}", null );
            
            $status_class = $is_active ? 'status-active' : 'status-inactive';
            $status_text = $is_active ? 'Active' : 'Inactive';
            
            $activity_class = 'activity-not-tested';
            $activity_text = 'Not Tested';
            
            if ( $activity_result ) {
                switch ( $activity_result['status'] ) {
                    case 'used':
                        $activity_class = 'activity-used';
                        $activity_text = 'Used';
                        break;
                    case 'possibly_used':
                        $activity_class = 'activity-possibly-used';
                        $activity_text = 'Possibly Used';
                        break;
                    case 'no_usage':
                        $activity_class = 'activity-no-usage';
                        $activity_text = 'No Usage Found';
                        break;
                }
            }
            
            $html .= '<tr>
                <td>' . esc_html( $plugin_data['Name'] ) . '</td>
                <td class="' . $status_class . '">' . $status_text . '</td>
                <td class="' . $activity_class . '">' . $activity_text . '</td>
                <td>' . esc_html( $plugin_data['Version'] ) . '</td>
                <td>' . esc_html( $plugin_data['Author'] ) . '</td>
                <td><code>' . esc_html( $plugin_file ) . '</code></td>
            </tr>';
        }
        
        $html .= '
        </tbody>
    </table>
    
    <div class="footer">
        <p>Report generated by <strong>Unplug - WordPress Plugin Optimizer</strong></p>
        <p>Visit <a href="https://mulberrytech.ca/unplug">mulberrytech.ca/unplug</a> for more information</p>
    </div>
</body>
</html>';
        
        return $html;
    }

    /**
     * AJAX handler for running a location test for a plugin.
     */
    public function ajax_run_location_test() {
        // Load security class
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-unplug-security.php';
        
        // Nonce verification is handled by UNPLUG_Security::verify_ajax_request() above.
        UNPLUG_Security::verify_ajax_request( 'scan', 'manage_options', 'nonce' );
        
        // Check plugin-specific capability
        UNPLUG_Security::can_perform_action( 'scan', true );
        
        // Log security event
        UNPLUG_Security::log_security_event( 'ajax_run_location_test', 'Location test requested' );

        // Check plugin_slug
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        if ( empty( $_POST['plugin_slug'] ) ) {
            wp_send_json_error( array( 'message' => 'Missing plugin_slug.' ), 400 );
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by UNPLUG_Security::verify_ajax_request()
        $plugin_slug = sanitize_text_field( wp_unslash( $_POST['plugin_slug'] ) );
        
        // Fetch stored location result for this plugin
        $location_result = get_option( "unplug_location_result_{$plugin_slug}", array( 'count' => 0, 'details' => array() ) );
        
        wp_send_json_success( array(
            'count' => $location_result['count'],
            'details' => $location_result['details'],
            'message' => 'Location test completed for ' . $plugin_slug
        ) );
    }
} 

