<?php
/**
 * Plugin Name: Unlock-To-View
 * Description: Unlock videos and other digital content with email codes, log activity, and track invoices.
 * Version: 1.0.0
 * Author: Lise Rasmussen
 * Author URI: https://liserasmussen.se/unlock-to-view/index.html
 * License: GPLv2 or later
 * Text Domain: unlock-to-view
 */

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'UnlockToView' ) ) :

final class UnlockToView {
    
    private static $instance = null;

    public static function instance() : UnlockToView {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
    $this->define_constants();

    add_action( 'init', [ $this, 'register_assets' ] );
    add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend' ] );
    add_action( 'admin_menu', [ $this, 'admin_menu' ] );
    add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );
    add_action( 'admin_init', [ $this, 'admin_init' ] );
    add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_about_styles' ] );

    // Shortcodes
    add_shortcode( 'unlocktoview', [ $this, 'shortcode_unlocktoview' ] );

    // Centralized AJAX registration
    $this->register_ajax_hooks();
    }

    // Called by WordPress when plugin is activated */
    public static function activate() {
        self::instance()->on_activation();
    }

    // Called by WordPress when plugin is deactivated */
    public static function deactivate() {
        self::instance()->on_deactivation();
    }

    private function register_ajax_hooks() {
        
    // Admin-only actions
    add_action( 'wp_ajax_unlocktoview_get_rentals', [ $this, 'ajax_get_rentals' ] );
    add_action( 'wp_ajax_unlocktoview_mark_paid', [ $this, 'ajax_mark_paid' ] );
    add_action( 'wp_ajax_unlocktoview_delete_rental', [ $this, 'ajax_delete_rental' ] );
    add_action( 'wp_ajax_unlocktoview_send_reminder', [ $this, 'ajax_send_reminder' ] );
    add_action('wp_ajax_unlocktoview_save_settings', [ $this, 'ajax_save_settings' ]);
    add_action('wp_ajax_unlocktoview_delete_shortcode', [$this, 'ajax_delete_shortcode']);
    add_action('wp_ajax_unlocktoview_add_shortcode', [$this, 'ajax_add_shortcode']);

    // Frontend + guest
    add_action( 'wp_ajax_unlocktoview_send_code', [ $this, 'ajax_send_code' ] );
    add_action( 'wp_ajax_nopriv_unlocktoview_send_code', [ $this, 'ajax_send_code' ] );
    add_action( 'wp_ajax_unlocktoview_get_video', [ $this, 'ajax_get_video' ] );
    add_action( 'wp_ajax_nopriv_unlocktoview_get_video', [ $this, 'ajax_get_video' ] );
    
    }

   private function define_constants() : void {
    // Main plugin file and directory
    if ( ! defined( 'UNLOCK_TO_VIEW_FILE' ) ) define( 'UNLOCK_TO_VIEW_FILE', __FILE__ );
    if ( ! defined( 'UNLOCK_TO_VIEW_DIR' ) )  define( 'UNLOCK_TO_VIEW_DIR', plugin_dir_path( __FILE__ ) );

    // Plugin URLs
    if ( ! defined( 'UNLOCK_TO_VIEW_URL' ) )        define( 'UNLOCK_TO_VIEW_URL', plugin_dir_url( __FILE__ ) );
    if ( ! defined( 'UNLOCK_TO_VIEW_PLUGIN_URL' ) ) define( 'UNLOCK_TO_VIEW_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

    // Version
    if ( ! defined( 'UNLOCK_TO_VIEW_VERSION' ) ) define( 'UNLOCK_TO_VIEW_VERSION', '1.1.0' );

    // Assets
    if ( ! defined( 'UNLOCK_TO_VIEW_ASSETS' ) )     define( 'UNLOCK_TO_VIEW_ASSETS', trailingslashit( UNLOCK_TO_VIEW_URL . 'assets' ) );
    if ( ! defined( 'UNLOCK_TO_VIEW_ASSETS_DIR' ) ) define( 'UNLOCK_TO_VIEW_ASSETS_DIR', trailingslashit( UNLOCK_TO_VIEW_DIR . 'assets' ) );

    // Uploads & log file
    if ( ! defined( 'UNLOCK_TO_VIEW_UPLOAD_SUBDIR' ) ) define( 'UNLOCK_TO_VIEW_UPLOAD_SUBDIR', 'unlock-to-view' );
    if ( ! defined( 'UNLOCK_TO_VIEW_LOG_FILE' ) ) define(
        'UNLOCK_TO_VIEW_LOG_FILE',
        trailingslashit( wp_upload_dir()['basedir'] ) . UNLOCK_TO_VIEW_UPLOAD_SUBDIR . '/logs/rentals.csv'
    );
    }

    public function on_activation() : void {
        $uploads = wp_upload_dir();
        $base     = trailingslashit( $uploads['basedir'] ) . UNLOCK_TO_VIEW_UPLOAD_SUBDIR;
        $logs_dir = trailingslashit( $base ) . 'logs';
        wp_mkdir_p( $logs_dir );

        if ( ! file_exists( $base . '/index.html' ) ) @file_put_contents( $base . '/index.html', '' );
        if ( ! file_exists( $logs_dir . '/index.html' ) ) @file_put_contents( $logs_dir . '/index.html', '' );
        if ( ! file_exists( $logs_dir . '/.htaccess' ) ) @file_put_contents( $logs_dir . '/.htaccess', "Options -Indexes\nDeny from all\n" );

        $header = ['Date','Email','Video','Code','Expiry','Invoice','IP','ReminderSent','Paid'];

        if ( ! file_exists(UNLOCK_TO_VIEW_LOG_FILE) ) {
            file_put_contents(UNLOCK_TO_VIEW_LOG_FILE, implode(',', $header)."\n");
        }

        add_option( 'unlocktoview_sender_name', get_bloginfo( 'name' ) );
        add_option( 'unlocktoview_sender_email', get_option( 'admin_email' ) );
        add_option( 'unlocktoview_payment_url', '' );
        add_option( 'unlocktoview_price', '3.99' );
        add_option( 'unlocktoview_currency', 'USD' );
        add_option( 'unlocktoview_code_length', 4 );
        add_option( 'unlocktoview_unlock_hours', 48 );
    }

    public function on_deactivation() : void {}

// -----------------
// Admin Menu
// -----------------
public function admin_menu() : void {
    add_menu_page(
        esc_html__( 'Unlock To View', 'unlock-to-view' ), // Page title
        esc_html__( 'Unlock To View', 'unlock-to-view' ), // Menu title
        'manage_options',
        'unlock-to-view',
        [ $this, 'admin_page_shortcodes' ],
        'dashicons-format-video',
        25
    );

    // Rename the first submenu (same slug as parent, but different label)
    add_submenu_page(
        'unlock-to-view',
        esc_html__( 'Shortcode Generator', 'unlock-to-view' ), // Page title
        esc_html__( 'Shortcode Generator', 'unlock-to-view' ), // Menu label
        'manage_options',
        'unlock-to-view',
        [ $this, 'admin_page_shortcodes' ]
    );

    add_submenu_page(
        'unlock-to-view',
        esc_html__( 'Rentals', 'unlock-to-view' ),
        esc_html__( 'Rentals', 'unlock-to-view' ),
        'manage_options',
        'unlock-to-view-rentals',
        [ $this, 'admin_page_rentals' ]
    );

    add_submenu_page(
        'unlock-to-view',
        esc_html__( 'Settings', 'unlock-to-view' ),
        esc_html__( 'Settings', 'unlock-to-view' ),
        'manage_options',
        'unlock-to-view-settings',
        [ $this, 'admin_page_settings' ]
    );
     add_submenu_page(
        'unlock-to-view',
        esc_html__( 'About the Plugin', 'unlock-to-view' ), 
        esc_html__( 'About this Plugin', 'unlock-to-view' ),           
        'manage_options',
        'unlock-to-view-about',
        [ $this, 'admin_page_about' ]                      
    );
}

// -----------------
// Rental Logging
// -----------------
private function get_log_file(): string {
    $log_file = UNLOCK_TO_VIEW_LOG_FILE;
    $log_dir  = dirname($log_file);

    if ( ! file_exists($log_dir) ) {
        wp_mkdir_p($log_dir);
    }

    return $log_file;
}

// -----------------
// About the plugin 
// -----------------
public function admin_page_about() : void {
    include plugin_dir_path( __FILE__ ) . 'admin/about.php';
}

// -----------------
// Generate rental 
// -----------------
public function get_rentals(): array {
    $file   = $this->get_log_file();
    $header = ['Date','Email','Video','Code','Expiry','Invoice','IP','ReminderSent','Paid'];

    // Ensure directory exists
    $log_dir = dirname($file);
    if (!is_dir($log_dir)) {
        wp_mkdir_p($log_dir);
    }

    // If file does not exist or is empty → create with header
    if (!file_exists($file) || filesize($file) === 0) {
        file_put_contents($file, implode(',', $header) . PHP_EOL);
        return [];
    }

    $lines = file($file, FILE_IGNORE_NEW_LINES);

    // Normalize header if wrong
    $first_line = array_map('trim', str_getcsv($lines[0]));
    if ($first_line !== $header) {
        $lines[0] = implode(',', $header);
        file_put_contents($file, implode(PHP_EOL, $lines) . PHP_EOL);
    }

    // Remove header for parsing
    array_shift($lines);
    if (empty($lines)) return [];

    $rentals = [];
    foreach ($lines as $line) {
        if (trim($line) === '') continue;
        $row = str_getcsv($line);
        $row = array_pad($row, count($header), '');
        $rentals[] = array_combine($header, $row);
    }

    // Sort descending by Date
    usort($rentals, function($a, $b) {
        return strtotime($b['Date'] ?? 0) - strtotime($a['Date'] ?? 0);
    });

    return $rentals;
}

// -----------------------------
// Save rentals 
// -----------------------------
public function save_rentals(array $rentals) {
    $file   = $this->get_log_file();
    $header = ['Date','Email','Video','Code','Expiry','Invoice','IP','ReminderSent','Paid'];

    // Ensure directory exists
    $log_dir = dirname($file);
    if (!is_dir($log_dir)) {
        wp_mkdir_p($log_dir);
    }

    $lines_to_write = [];
    $lines_to_write[] = implode(',', $header); // always start with header

    // Write each rental row
    foreach ($rentals as $r) {
        $row = [];
        foreach ($header as $col) {
            // Replace commas inside values to prevent CSV breaking
            $row[] = isset($r[$col]) ? str_replace(',', ' ', $r[$col]) : '';
        }
        $lines_to_write[] = implode(',', $row);
    }

    // OVERWRITE the CSV (no FILE_APPEND!)
    file_put_contents($file, implode(PHP_EOL, $lines_to_write) . PHP_EOL);
}

// -----------------
// Log rentals
// -----------------
public function generate_rental_details(): array {
    $now = current_time('mysql');

    // Retrieve settings array
    $settings = get_option($this->option_name);
    $duration_hours = isset($settings['duration_hours']) ? intval($settings['duration_hours']) : 24;

    // Calculate expiry
    $expiry_timestamp = strtotime($now) + ($duration_hours * HOUR_IN_SECONDS);
    $expiry = date_i18n('Y-m-d H:i:s', $expiry_timestamp);



    // Generate code and invoice
    $code = wp_generate_password(8, false, false);
    $invoice = strtoupper(uniqid('INV'));

    return [
        'Date'    => $now,
        'Expiry'  => $expiry,
        'Code'    => $code,
        'Invoice' => $invoice
    ];
}

// -----------------
// Delete a rental entry from the CSV by email and video
// -----------------
public function delete_rental(string $email, string $video): bool {
    $rentals = $this->get_rentals();
    $deleted = false;

    foreach ($rentals as $key => $row) {
        if (($row['Email'] ?? '') === $email && ($row['Video'] ?? '') === $video) {
            unset($rentals[$key]);
            $deleted = true;
            break;
        }
    }

    if ($deleted) {
        // Reindex array to avoid holes in CSV
        $rentals = array_values($rentals);
        $this->save_rentals($rentals);
    }

    return $deleted;
}

// -----------------
// Shortcode page
// -----------------
    public function admin_page_shortcodes() {
        ?>
        <div class="wrap">
            <div class="unlocktoview-logo">
                <img 
                    src="<?php echo esc_url( UNLOCK_TO_VIEW_PLUGIN_URL . 'assets/img/logo_unlocktoview.svg' ); ?>" 
                    width="300px" 
                    alt="<?php echo esc_attr__( 'Unlock To View Logo', 'unlock-to-view' ); ?>" 
                />
            </div>
            <p>
                <?php
                printf(
                    wp_kses(
                        /* translators: %s: URL to the Unlock To View settings page */
                        __(
                            'Use this page to create shortcodes. Insert a shortcode into a page, and it will display an Unlock button. Configure your options first in the <a href="%s">Settings</a> page.',
                            'unlock-to-view'
                        ),
                        [
                            'a' => [
                                'href'   => [],
                                'target' => [],
                                'rel'    => [],
                            ],
                        ]
                    ),
                    esc_url( admin_url( 'admin.php?page=unlock-to-view-settings' ) )
                );
                ?>
            </p>
    
            <!-- Tabs -->
            <h2 class="unlocktoview-tab-wrapper">
                <a href="#tab-shortcode" class="unlocktoview-tab unlocktoview-tab-active">Shortcode Generator</a>
            </h2>
    
            <!-- Shortcode Generator -->
            <div id="tab-shortcode" class="tab-content active">
                <table class="form-table">
                    <tr>
                        <th>
                            <label for="unlocktoview_video_slug">
                                <?php esc_html_e('Slug', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text"><?php esc_html_e('This is the page slug where your video is embedded. The slug must exactly match your shortcode, otherwise the unlock button will not work.', 'unlock-to-view'); ?></span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <input type="text" id="unlocktoview_video_slug" class="regular-text">
                            <em><?php esc_html_e('Add the page slug where your video is embedded.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>
                    <tr>
                        <th>
                            <label for="unlocktoview_media_player">
                                <?php esc_html_e('Media Player', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text"><?php esc_html_e('Choose the video player you want to use for this video.', 'unlock-to-view'); ?></span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <select id="unlocktoview_media_player" class="regular-text">
                                <option value="">— <?php esc_html_e('Choose a player','unlock-to-view'); ?> —</option>
                                <option value="presto">Presto Player</option>
                                <option value="vimeo">Vimeo</option>
                                <option value="youtube">YouTube</option>
                                <option value="url">Any video URL</option>
                            </select>
                            <em><?php esc_html_e('Select which player should render the video on your page.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>
                    <tr>
                        <th>
                            <label for="unlocktoview_video_id">
                                <?php esc_html_e('Media ID or URL', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text"><?php esc_html_e('Provide the video ID or direct URL depending on the selected player.', 'unlock-to-view'); ?></span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <input type="text" id="unlocktoview_video_id" class="regular-text">
                            <em><?php esc_html_e('For YouTube/Vimeo enter the video URL, for media players enter the video ID.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>
                </table>
    
                <p>
                    <button id="unlocktoview-generate-shortcode" class="button unlocktoview-btn">
                        <?php esc_html_e('Generate Shortcode', 'unlock-to-view'); ?>
                    </button>
                </p>
    
                <div id="unlocktoview-shortcode-wrapper" style="margin-top:20px; display:none;">
                    <textarea id="unlocktoview-shortcode-result" readonly style="width:60%; height:80px;"></textarea>
                    <button id="unlocktoview-copy-shortcode" class="button unlocktoview-btn">
                        <?php esc_html_e('Copy Shortcode', 'unlock-to-view'); ?>
                    </button>
                </div>
            </div>
    
            <?php
            // Read entries from media.csv
            $entries = [];
            $upload_dir = wp_upload_dir();
            $media_file = $upload_dir['basedir'] . '/unlock-to-view/logs/media.csv';
            
            if ( file_exists( $media_file ) ) {
                $lines = file( $media_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES );
                foreach ( $lines as $line ) {
                    $cols = str_getcsv( $line );
                    if ( count( $cols ) >= 2 ) {
                        $entries[] = [
                            'slug'   => $cols[0],
                            'id'     => $cols[1],
                            'player' => $cols[2] ?? ''
                        ];
                    }
                }
            }
            
            // Pagination logic
                $per_page      = 10;
                $total_entries = count( $entries );
                $total_pages   = ceil( $total_entries / $per_page );
                
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $current_page  = isset( $_GET['paged'] ) ? max( 1, intval( $_GET['paged'] ) ) : 1;
                
                $offset        = ( $current_page - 1 ) * $per_page;
                $paged_entries = array_slice( $entries, $offset, $per_page );
            ?>
        
            <!-- --------- All Shortcodes List --------- -->
            <h2><?php esc_html_e('All Shortcodes', 'unlock-to-view'); ?></h2>
            <table class="widefat fixed striped unlocktoview-shortcodes-table">
                <thead>
                    <tr>
                        <th style="width:20%;"><?php esc_html_e('Slug', 'unlock-to-view'); ?></th>
                        <th style="width:70%;"><?php esc_html_e('Shortcode', 'unlock-to-view'); ?></th>
                        <th style="width:10%;"><?php esc_html_e('Actions', 'unlock-to-view'); ?></th>
                    </tr>
                </thead>
                        <tbody>
                <?php if ( ! empty( $paged_entries ) ) : ?>
                    <?php foreach ( $paged_entries as $entry ) :
                        // Ensure all expected keys exist
                        $slug   = isset( $entry['slug'] ) ? $entry['slug'] : '';
                        $id     = isset( $entry['id'] ) ? $entry['id'] : '';
                        $player = isset( $entry['player'] ) ? $entry['player'] : '';
            
                        // Build complete shortcode
                        $shortcode_parts = [
                            'slug="' . esc_attr( $slug ) . '"',
                            'id="' . esc_attr( $id ) . '"'
                        ];
                        if ( ! empty( $player ) ) {
                            $shortcode_parts[] = 'player="' . esc_attr( $player ) . '"';
                        }
                        $shortcode = '[unlocktoview ' . implode( ' ', $shortcode_parts ) . ']';
            
                        // Unique row ID
                        $row_id = 'unlocktoview-' . sanitize_html_class( $slug ) . '-' . sanitize_html_class( $id );
                    ?>
                    <tr id="<?php echo esc_attr( $row_id ); ?>">
                        <td><?php echo esc_html( $slug ); ?></td>
                        <td>
                            <input type="text" class="unlocktoview-shortcode-input" value="<?php echo esc_attr( $shortcode ); ?>" readonly>
                        </td>
                        <td>
                            <button class="button unlocktoview-copy" data-target="#<?php echo esc_attr( $row_id ); ?> .unlocktoview-shortcode-input">
                                <?php esc_html_e( 'Copy', 'unlock-to-view' ); ?>
                            </button>
                            <button class="button unlocktoview-delete-shortcode"
                                data-slug="<?php echo esc_attr( $slug ); ?>"
                                data-id="<?php echo esc_attr( $id ); ?>"
                                data-player="<?php echo esc_attr( $player ); ?>">
                                <?php esc_html_e( 'Delete', 'unlock-to-view' ); ?>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php else : ?>
                    <tr><td colspan="3"><?php esc_html_e( 'No shortcodes found.', 'unlock-to-view' ); ?></td></tr>
                <?php endif; ?>
            </tbody>
            </table>
        
            <!-- Pagination -->
            <div class="unlocktoview-pagination-wrapper">
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <?php
                        $base_url = remove_query_arg( 'paged' );
            
                        if ( $current_page > 1 ) {
                            echo '<a class="unlocktoview-prev-page button" href="' . esc_url( add_query_arg( 'paged', $current_page - 1, $base_url ) ) . '">&laquo; ' . esc_html__( 'Prev', 'unlock-to-view' ) . '</a>';
                        }
            
                        $display_pages = min( $total_pages, 10 );
                        for ( $i = 1; $i <= $display_pages; $i++ ) {
                            $current_class = $i === $current_page ? ' current-page' : '';
                            echo '<a class="unlocktoview-page-number button' . esc_attr( $current_class ) . '" href="' . esc_url( add_query_arg( 'paged', $i, $base_url ) ) . '">' . esc_html( $i ) . '</a>';
                        }
            
                        if ( $current_page < $total_pages ) {
                            echo '<a class="unlocktoview-next-page button" href="' . esc_url( add_query_arg( 'paged', $current_page + 1, $base_url ) ) . '">' . esc_html__( 'Next', 'unlock-to-view' ) . ' &raquo;</a>';
                        } else {
                            echo '<span class="unlocktoview-next-page button disabled">' . esc_html__( 'Next', 'unlock-to-view' ) . ' &raquo;</span>';
                        }
                        ?>
                    </div>
            </div>
            </div>
            <?php
    }
    
// ---------------------------
// AJAX: Add a shortcode
// ---------------------------
public function ajax_add_shortcode() {
    // Verify nonce
    check_ajax_referer( 'unlocktoview_nonce', 'nonce' );
    
    // Sanitize incoming data
    $slug = sanitize_text_field( wp_unslash( $_POST['slug'] ?? '' ) );
    $id   = sanitize_text_field( wp_unslash( $_POST['id'] ?? '' ) );

    // === Detect player value safely ===
    $player = '';

    // List of possible POST keys for player
    $player_keys = ['player', 'unlocktoview_media_player', 'media_player', 'unlocktoview_player', 'player_name'];

    // Check known keys first
    foreach ($player_keys as $key) {
        if (!empty($_POST[$key])) {
            $player = sanitize_text_field(wp_unslash($_POST[$key]));
            break;
        }
    }
 
    // Fallback: any key containing "player"
    if (empty($player)) {
        foreach ($_POST as $k => $v) {
            if (stripos($k, 'player') !== false && !empty($v)) {
                $player = sanitize_text_field(wp_unslash($v));
                break;
            }
        }
    }

    // Last fallback: parse 'shortcode' field if provided
    if (empty($player) && !empty($_POST['shortcode'])) {
    $sc = sanitize_textarea_field( wp_unslash( $_POST['shortcode'] ?? '' ) );
    if (preg_match('/player="([^"]+)"/', $sc, $matches)) {
        $player = sanitize_text_field($matches[1]);
    }
    }

    // Ensure required data
    if (!$slug || !$id) {
        wp_send_json_error(['message' => '❌ Missing Slug or ID']);
    }

    // File location
    $upload_dir = wp_upload_dir();
    $logs_dir   = trailingslashit($upload_dir['basedir']) . 'unlock-to-view/logs/';
    $media_file = $logs_dir . 'media.csv';

    // Ensure directory exists
    if (!file_exists($logs_dir)) {
        wp_mkdir_p($logs_dir);
    }

    // Init WP_Filesystem
    global $wp_filesystem;
    if (!$wp_filesystem) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    // Read existing CSV entries
    $entries = [];
    if ($wp_filesystem->exists($media_file)) {
        $lines = explode("\n", $wp_filesystem->get_contents($media_file));
        foreach ($lines as $line) {
            if (empty(trim($line))) continue;
            $cols = str_getcsv($line);
            while (count($cols) < 3) $cols[] = '';
            $entries[] = $cols;
        }
    }

    // Prepend new entry
    array_unshift($entries, [$slug, $id, $player]);

    // Build CSV content
    $csv_data = '';
    foreach ($entries as $cols) {
        while (count($cols) < 3) $cols[] = '';
        $csv_data .= $this->unlocktoview_csv_line($cols);
    }

    // Write CSV
    $success = $wp_filesystem->put_contents($media_file, $csv_data, FS_CHMOD_FILE);
    if (!$success) {
        wp_send_json_error(['message' => '❌ Failed to write CSV file.']);
    }

    // Build shortcode for JS
    $shortcode_parts = [
        'slug="' . esc_attr($slug) . '"',
        'id="' . esc_attr($id) . '"'
    ];
    if (!empty($player)) {
        $shortcode_parts[] = 'player="' . esc_attr($player) . '"';
    }
    $shortcode = '[unlocktoview ' . implode(' ', $shortcode_parts) . ']';

    // Send JSON response
    wp_send_json_success([
        'message'   => '✅ Shortcode saved successfully.',
        'slug'      => $slug,
        'id'        => $id,
        'player'    => $player,
        'shortcode' => $shortcode
    ]);
}


// ---------------------------
// AJAX: Delete a shortcode
// ---------------------------
public function ajax_delete_shortcode() {
    // Verify nonce
    check_ajax_referer('unlocktoview_nonce', 'nonce');

    // Sanitize inputs
    $slug = isset($_POST['slug']) ? sanitize_text_field(wp_unslash($_POST['slug'])) : '';
    $id   = isset($_POST['id']) ? sanitize_text_field(wp_unslash($_POST['id'])) : '';

    if (!$slug || $id === '') {
        wp_send_json_error(['message' => '❌ Missing identifiers.']);
    }

    // File location
    $upload_dir = wp_upload_dir();
    $media_file = $upload_dir['basedir'] . '/unlock-to-view/logs/media.csv';

    // Init WP_Filesystem if needed
    global $wp_filesystem;
    if (!$wp_filesystem) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    if (!$wp_filesystem->exists($media_file)) {
        wp_send_json_error(['message' => '❌ Media file not found.']);
    }

    // Read and process CSV
    $lines     = explode("\n", $wp_filesystem->get_contents($media_file));
    $found     = false;
    $new_lines = [];

    foreach ($lines as $line) {
        if (trim($line) === '') continue;

        $cols = str_getcsv($line, ',', '"');

        // Pad to 3 columns to preserve structure
        while (count($cols) < 3) {
            $cols[] = '';
        }

        $row_slug = trim($cols[0]);
        $row_id   = trim($cols[1]);

        // Match slug and id
        if ($row_slug === $slug && $row_id === $id) {
            $found = true;
            continue; // Skip this row (delete)
        }

        $new_lines[] = $cols; // Keep the row
    }

    if (!$found) {
        wp_send_json_error(['message' => '❌ Shortcode not found in CSV.']);
    }

    // Rebuild CSV safely
    $csv_data = '';
    foreach ($new_lines as $cols) {
        while (count($cols) < 3) {
            $cols[] = '';
        }
        $csv_data .= '"' . implode('","', array_map(function($v) {
            return str_replace('"', '""', $v);
        }, $cols)) . '"' . "\n";
    }

    if (!$wp_filesystem->put_contents($media_file, $csv_data, FS_CHMOD_FILE)) {
        wp_send_json_error(['message' => '❌ Failed to write CSV.']);
    }

    wp_send_json_success(['message' => '✅ Shortcode deleted.']);
}

// -----------------
// Rental page
// -----------------
public function admin_page_rentals() {

    // Get all rentals
    $rentals = $this->get_rentals();
    $per_page = 20;

    // Sanitize & verify nonce first
    $nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';

    if ( $nonce && wp_verify_nonce( $nonce, 'unlocktoview_pagination_nonce' ) ) {
        // Sanitize page number
        $current_page = isset( $_GET['paged'] ) ? max( 1, intval( wp_unslash( $_GET['paged'] ) ) ) : 1;
    } else {
        $current_page = 1;
    }

    $total_rentals = count( $rentals );
    $paged_rentals = array_slice( $rentals, ( $current_page - 1 ) * $per_page, $per_page );

    // Create nonces
    $ajax_nonce = wp_create_nonce( 'unlocktoview_nonce' );
    $pagination_nonce = wp_create_nonce( 'unlocktoview_pagination_nonce' );

    ?>
    <div class="wrap">

        <!-- Logo -->
        <div class="unlocktoview-logo">
            <img src="<?php echo esc_url( UNLOCK_TO_VIEW_PLUGIN_URL . 'assets/img/logo_unlocktoview.svg' ); ?>" width="300" alt="<?php echo esc_attr__( 'Unlock To View Logo', 'unlock-to-view' ); ?>" />
        </div>

        <p>
            <?php esc_html_e( "You're using Unlock-To-View Lite. Like this plugin? Your support helps keep it free, consider a", 'unlock-to-view' ); ?>
            <a href="https://www.paypal.com/donate/?hosted_button_id=4SMV5H3BFFJKQ" target="_blank"><?php esc_html_e('donation', 'unlock-to-view'); ?></a> ❤️.
        </p>

        <!-- Tabs -->
        <h2 class="unlocktoview-tab-wrapper">
            <a href="#tab-rentals" class="unlocktoview-tab unlocktoview-tab-active"><?php esc_html_e( 'Rental log', 'unlock-to-view' ); ?></a>
        </h2>

        <!-- Rentals Tab -->
        <div id="tab-rentals" class="tab-content active">
            <table class="wp-list-table widefat fixed striped rentals-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Date', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Email', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Video', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Code', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Expiry', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Invoice', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Email Reminder', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Reminder Sent', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'IP', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Mark as Paid', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Paid', 'unlock-to-view' ); ?></th>
                        <th><?php esc_html_e( 'Delete', 'unlock-to-view' ); ?></th>
                    </tr>
                </thead>

                <tbody>
                    <?php if ( ! empty( $paged_rentals ) ) : ?>
                        <?php foreach ( $paged_rentals as $r ) : ?>
                            <tr>
                                <?php foreach ( ['Date','Email','Video','Code','Expiry','Invoice'] as $c ) : ?>
                                    <td><?php echo esc_html( $r[ $c ] ?? '' ); ?></td>
                                <?php endforeach; ?>

                                <!-- Email Reminder -->
                                <td>
                                    <button class="button unlocktoview-reminder"
                                        data-email="<?php echo esc_attr( $r['Email'] ?? '' ); ?>"
                                        data-video="<?php echo esc_attr( $r['Video'] ?? '' ); ?>"
                                        data-nonce="<?php echo esc_attr( $ajax_nonce ); ?>">
                                        <?php esc_html_e( 'Email Reminder', 'unlock-to-view' ); ?>
                                    </button>
                                </td>

                                <!-- Reminder Sent -->
                                <td class="reminder-column"><?php echo esc_html( $r['ReminderSent'] ?? '' ); ?></td>

                                <!-- IP -->
                                <td><?php echo esc_html( $r['IP'] ?? '' ); ?></td>

                                <?php $paid = ( $r['Paid'] ?? '' ) === 'Yes'; ?>

                                <!-- Mark as Paid -->
                                <td>
                                    <button class="button unlocktoview-mark-paid"
                                        data-email="<?php echo esc_attr( $r['Email'] ?? '' ); ?>"
                                        data-video="<?php echo esc_attr( $r['Video'] ?? '' ); ?>"
                                        data-nonce="<?php echo esc_attr( $ajax_nonce ); ?>"
                                        <?php echo $paid ? 'disabled' : ''; ?>>
                                        <?php echo $paid ? esc_html__( 'Paid', 'unlock-to-view' ) : esc_html__( 'Mark as Paid', 'unlock-to-view' ); ?>
                                    </button>
                                </td>
                                <td class="paid-column"><?php echo esc_html( $paid ? 'Yes' : 'No' ); ?></td>

                                <!-- Delete -->
                                <td>
                                    <button class="button unlocktoview-delete"
                                        data-email="<?php echo esc_attr( $r['Email'] ?? '' ); ?>"
                                        data-video="<?php echo esc_attr( $r['Video'] ?? '' ); ?>">
                                        <?php esc_html_e( 'Delete', 'unlock-to-view' ); ?>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr><td colspan="12"><?php esc_html_e( 'No rentals found', 'unlock-to-view' ); ?></td></tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <!-- Export and Pagination -->
            <div style="margin-top:15px; display:flex; justify-content:space-between; align-items:center;">

                <!-- Export to CSV -->
                <div>
                    <a href="<?php echo esc_url( add_query_arg( '_wpnonce', wp_create_nonce('unlocktoview_export_csv'), admin_url('admin-post.php?action=unlocktoview_export_csv') ) ); ?>"
                       class="unlocktoview-btn">
                        <?php esc_html_e( 'Export to CSV', 'unlock-to-view' ); ?>
                    </a>
                </div>

  <div class="unlocktoview-pagination-wrapper">
    <div class="tablenav"><div class="tablenav-pages">

    <?php
    // Base URL
    $base_url = isset( $_SERVER['REQUEST_URI'] )
        ? remove_query_arg( 'paged', esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) )
        : '';
    $total_pages = max( 1, ceil( $total_rentals / $per_page ) );

    // Encode page numbers and nonce
    $prev_page = $current_page - 1;
    $next_page = $current_page + 1;

    // Prev
    if ( $current_page > 1 ) :
        $prev_url = add_query_arg(
            [
                'paged'    => rawurlencode( $prev_page ),
                '_wpnonce' => rawurlencode( $pagination_nonce ),
            ],
            $base_url
        );
    ?>
        <a class="unlocktoview-prev-page button" href="<?php echo esc_url( $prev_url ); ?>">&laquo; <?php esc_html_e( 'Prev', 'unlock-to-view' ); ?></a>
    <?php endif; ?>

    <?php
    // Numbered pages
    $display_pages = min( $total_pages, 20 );
    for ( $i = 1; $i <= $display_pages; $i++ ) :
        $current_class = $i === $current_page ? ' current-page' : '';
        $page_url = add_query_arg(
            [
                'paged'    => rawurlencode( $i ),
                '_wpnonce' => rawurlencode( $pagination_nonce ),
            ],
            $base_url
        );
    ?>
        <a class="unlocktoview-page-number button<?php echo esc_attr( $current_class ); ?>" href="<?php echo esc_url( $page_url ); ?>"><?php echo esc_html( $i ); ?></a>
    <?php endfor; ?>

    <?php
    // Next
    if ( $current_page < $total_pages ) :
        $next_url = add_query_arg(
            [
                'paged'    => rawurlencode( $next_page ),
                '_wpnonce' => rawurlencode( $pagination_nonce ),
            ],
            $base_url
        );
    ?>
        <a class="unlocktoview-next-page button" href="<?php echo esc_url( $next_url ); ?>"><?php esc_html_e( 'Next', 'unlock-to-view' ); ?> &raquo;</a>
    <?php else : ?>
        <span class="unlocktoview-next-page button disabled"><?php esc_html_e( 'Next', 'unlock-to-view' ); ?> &raquo;</span>
    <?php endif; ?>

    </div></div>
</div>

            </div>

        </div>

    </div>
    <?php
}


// -----------------
// Settings page
// -----------------
private $option_name = 'unlocktoview_options';

// -----------------
// Register settings
// -----------------
public function admin_init() {
    // Register settings
    register_setting(
        'unlocktoview_settings_group',
        $this->option_name,
        ['sanitize_callback' => [$this, 'sanitize_settings']]
    );

    // -----------------------
    // General Settings Section
    // -----------------------
    add_settings_section(
        'unlocktoview_main',
        esc_html__('General Settings', 'unlock-to-view'),
        '__return_false',
        'unlock-to-view-settings'
    );

    add_settings_field(
        'price_per_video',
        esc_html__('Price per Video', 'unlock-to-view'),
        [$this, 'field_price_per_video'],
        'unlock-to-view-settings',
        'unlocktoview_main'
    );

    add_settings_field(
        'currency',
        esc_html__('Currency', 'unlock-to-view'),
        [$this, 'field_currency'],
        'unlock-to-view-settings',
        'unlocktoview_main'
    );

    add_settings_field(
        'duration_hours',
        esc_html__('Duration (hours)', 'unlock-to-view'),
        [$this, 'field_duration_hours'],
        'unlock-to-view-settings',
        'unlocktoview_main'
    );

    add_settings_field(
        'notify_admin',
        esc_html__('Send Notification to Admin', 'unlock-to-view'),
        [$this, 'field_notify_admin'],
        'unlock-to-view-settings',
        'unlocktoview_main'
    );

    add_settings_field(
        'confirm_buyer',
        esc_html__('Confirmation mail to Buyer', 'unlock-to-view'),
        [$this, 'field_confirm_buyer'],
        'unlock-to-view-settings',
        'unlocktoview_main'
    );
     add_settings_field(
        'reminder_after_days',                    
        esc_html__('Send Reminder After X Days', 'unlock-to-view'), 
        [$this, 'field_reminder_after_days'],      
        'unlock-to-view-settings',                 
        'unlocktoview_main'                       
    );

    // -----------------------
    // Email Settings Section
    // -----------------------
    add_settings_section(
        'unlocktoview_email',
        esc_html__('Email Settings', 'unlock-to-view'),
        '__return_false',
        'unlock-to-view-settings'
    );

    add_settings_field(
        'sender_name',
        esc_html__('Sender Name', 'unlock-to-view'),
        [$this, 'field_sender_name'],
        'unlock-to-view-settings',
        'unlocktoview_email'
    );

    add_settings_field(
        'sender_email',
        esc_html__('Sender Email', 'unlock-to-view'),
        [$this, 'field_sender_email'],
        'unlock-to-view-settings',
        'unlocktoview_email'
    );

    add_settings_field(
        'email_subject',
        esc_html__('Default Email Subject', 'unlock-to-view'),
        [$this, 'field_email_subject'],
        'unlock-to-view-settings',
        'unlocktoview_email'
    );

    add_settings_field(
        'payment_link',
        esc_html__('Payment Link', 'unlock-to-view'),
        [$this, 'field_payment_link'],
        'unlock-to-view-settings',
        'unlocktoview_email'
    );
    add_settings_field(
    'reminder_email_template',
    esc_html__('Reminder Email Template', 'unlock-to-view'),
    [$this, 'field_reminder_email_template'],
    'unlock-to-view-settings',
    'unlocktoview_email'
    );
    
    add_settings_field(
        'reminder_email_subject',
        esc_html__('Reminder Email Subject', 'unlock-to-view'),
        [$this, 'field_reminder_email_subject'],
        'unlock-to-view-settings',
        'unlocktoview_email'
    );
}

// -----------------
// Sanitize input
// -----------------
public function sanitize_settings($input) {
    $output = [];

    // General
    $output['price_per_video'] = isset($input['price_per_video']) ? floatval($input['price_per_video']) : 0;
    $output['currency']        = sanitize_text_field($input['currency'] ?? 'USD');
    $output['duration_hours']  = isset($input['duration_hours']) ? intval($input['duration_hours']) : 24;
    $output['notify_admin']    = !empty($input['notify_admin']) ? 1 : 0;
    $output['admin_email']     = sanitize_email($input['admin_email'] ?? get_option('admin_email'));

    // Button label (fallback to default if empty)
    $label = sanitize_text_field($input['button_label'] ?? '');
    $output['button_label'] = !empty($label) ? $label : esc_html__('Unlock this video', 'unlock-to-view');

    // Buyer Email
    $output['confirm_buyer']    = !empty($input['confirm_buyer']) ? 1 : 0;
    $output['buyer_subject']    = sanitize_text_field($input['buyer_subject'] ?? '');
    $output['buyer_sender']     = sanitize_text_field($input['buyer_sender'] ?? get_bloginfo('name'));
    $output['buyer_message']    = sanitize_textarea_field($input['buyer_message'] ?? '');

    // Payment
    $output['payment_link']     = esc_url_raw($input['payment_link'] ?? '');

    // Reminder mail
    $output['reminder_email_subject']  = sanitize_text_field($input['reminder_email_subject'] ?? '');
    $output['reminder_email_template'] = sanitize_textarea_field($input['reminder_email_template'] ?? '');

    return $output;
}

// -----------------
// Fields
// -----------------

public function field_price_per_video() {
    $opts = get_option($this->option_name);
    ?>
    <input type="number" step="0.01" name="<?php echo esc_attr( $this->option_name ); ?>[price_per_video]"
           value="<?php echo esc_attr($opts['price_per_video'] ?? '0'); ?>" /> 
    <?php
}

public function field_currency() {
    $opts = get_option($this->option_name);
    ?>
    <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[currency]"
           value="<?php echo esc_attr($opts['currency'] ?? 'USD'); ?>" 
           maxlength="5" size="5" /> 
    <?php
}

public function field_duration_hours() {
    $opts = get_option($this->option_name);
    ?>
    <input type="number" name="<?php echo esc_attr( $this->option_name ); ?>[duration_hours]"
           value="<?php echo esc_attr($opts['duration_hours'] ?? '24'); ?>" /> hours
    <?php
}

public function field_button_label() {
    $opts = get_option($this->option_name);
    ?>
    <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[button_label]"
           value="<?php echo esc_attr($opts['button_label'] ?? esc_html__('Unlock Now', 'unlock-to-view')); ?>"
           placeholder="<?php esc_attr_e('Unlock Now', 'unlock-to-view'); ?>" />
    <p class="description">
        <?php esc_html_e('Text shown on the unlock button (e.g., "Unlock Now", "Watch Video", "Enter Code").', 'unlock-to-view'); ?>
    </p>
    <?php
}

public function field_notify_admin() {
    $opts = get_option($this->option_name);
    ?>
    <label>
        <input type="checkbox" 
               name="<?php echo esc_attr( $this->option_name ); ?>[notify_admin]" 
               value="1" 
               class="toggle-admin" 
               <?php checked( 1, $opts['notify_admin'] ?? 0 ); ?> />
        <?php esc_html_e('Yes', 'unlock-to-view'); ?>
    </label>
    <div class="admin-email-wrap" style="margin-top:6px; <?php echo empty($opts['notify_admin']) ? 'display:none;' : ''; ?>">
        <input type="email" name="<?php echo esc_attr( $this->option_name ); ?>[admin_email]"
               value="<?php echo esc_attr($opts['admin_email'] ?? get_option('admin_email')); ?>"
               placeholder="<?php esc_attr_e('Admin Email', 'unlock-to-view'); ?>" />
    </div>
    <?php
}

public function field_confirm_buyer() {
    $opts = get_option($this->option_name);

    // Default subject
    $default_subject = esc_html__('Unlock video on {site_name}', 'unlock-to-view');
    $buyer_subject = !empty($opts['buyer_subject']) ? $opts['buyer_subject'] : $default_subject;

    // Template with placeholders (always!)
    $buyer_message_template = '
    Hi $email,
    
    Use the code $code to unlock $slug
    The code expires after $duration hours
    Rental price: $price $currency
    
    Our payment options are: 
    $payment_link
    
    Payment Due within 7 days.
    
    If you forget to pay, don’t worry – we will send you a friendly reminder. 😉
    
    Your IP address: $ip
    
    Thank you for renting with us! 🍿
    
    Best regards,
    $site_name
    ';

    // Editable message
    $editable_message = !empty($opts['buyer_message'])
        ? $opts['buyer_message']
        : $buyer_message_template;

    // Build live preview by swapping placeholders with settings values
    $price_val = !empty($opts['price_per_video']) ? $opts['price_per_video'] : '{price}';
    $currency_val = !empty($opts['currency']) ? $opts['currency'] : '{currency}';
    $duration_val = !empty($opts['duration_hours']) ? $opts['duration_hours'] : '{duration}';
    $payment_link_val = !empty($opts['payment_link']) ? $opts['payment_link'] : '{payment_link}';
    $site_name = get_bloginfo('name');

    $preview_message = str_replace(
    ['$price', '$currency', '$payment_link', '$duration', '$site_name'],
    [$price_val, $currency_val, $payment_link_val, $duration_val, $site_name],
    $buyer_message_template
    );
    ?>
    
    <p class="description" style="font-style:italic;">
        <?php esc_html_e('You can customize the email that is sent to the buyer, but it cannot be disabled as it contains the code to unlock a video.', 'unlock-to-view'); ?>
    </p>

    <div class="buyer-fields" style="margin-top:6px; display:block;">
        <!-- Subject -->
        <p>
            <label><?php esc_html_e('Subject:', 'unlock-to-view'); ?></label><br>
           <input type="text" id="buyer-subject" 
          name="<?php echo esc_attr( $this->option_name ); ?>[buyer_subject]" 
           value="<?php echo esc_attr($buyer_subject); ?>"
           data-default="<?php echo esc_attr($default_subject); ?>" />
            <button type="button" id="reset-buyer-subject" class="reset-btn">
                <?php esc_html_e('Reset', 'unlock-to-view'); ?>
            </button><br>
            <p class="description"><em>
                <?php esc_html_e('Subject line for confirmation mail to buyer. Available placeholders: $code, $slug, $duration, $price, $currency, $payment_link, $expiration, $due_date, $ip, $site_name.', 'unlock-to-view'); ?>
            </em></p>
        </p>

        <!-- Sender Name -->
        <p>
            <label><?php esc_html_e('Sender Name:', 'unlock-to-view'); ?></label><br>
            <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[buyer_sender]"
                   value="<?php echo esc_attr($opts['buyer_sender'] ?? get_bloginfo('name')); ?>" />
        </p>

        <!-- Editable Message -->
        <p>
            <label>
                <?php esc_html_e('Message:', 'unlock-to-view'); ?>
                <span class="tooltip">ℹ️
                    <span class="tooltip-text">
                        <?php esc_html_e('List of placeholders: $email, $code, $slug, $duration, $price, $currency, $payment_link, $site_name, $expiration, $due_date, $ip. <br>Do not remove the placeholder $code as it contains the unlock code.', 'unlock-to-view'); ?>
                    </span>
                </span>
            </label><br>
                        <textarea id="buyer-message-textarea"
                      name="<?php echo esc_attr( $this->option_name ); ?>[buyer_message]" 
                      rows="15" 
                      cols="60"
                      style="width:100%;"
                      data-default="<?php echo esc_attr($buyer_message_template); ?>"><?php 
                          echo esc_textarea($editable_message); ?></textarea>
            </p>
                  <p class="description">
    <em><?php esc_html_e('Message sent as confirmation mail to buyer. You may use any of these placeholders in the message:', 'unlock-to-view'); ?></em>
    <em><?php esc_html_e('Available placeholders: $code, $slug, $duration, $price, $currency, $payment_link, $expiration, $due_date, $ip, $site_name.', 'unlock-to-view'); ?></em>
</p>

        <!-- Reset Button -->
        <p>
            <button type="button" id="reset-buyer-message" class="reset-btn">
                <?php esc_html_e('Reset', 'unlock-to-view'); ?>
            </button>
        </p>

        <!-- Live Preview -->
        <div id="buyer-email-preview" class="buyer-email-preview" style="background:#f9f9f9; border:1px solid #ddd; padding:15px; font-family:Arial, sans-serif; font-size:13px; line-height:1.4; margin-top:10px;">
            <p><strong><?php esc_html_e('Preview with current Settings values:', 'unlock-to-view'); ?></strong></p>
        
            <!-- Subject preview -->
            <p><strong><?php esc_html_e('Subject:', 'unlock-to-view'); ?></strong>
                <span id="buyer-preview-subject"><?php echo esc_html($buyer_subject); ?></span>
            </p>
        
            <!-- Message preview -->
            <pre id="buyer-preview-message" style="white-space:pre-wrap;"><?php echo esc_html($preview_message); ?></pre>
        </div>

    </div>
    <?php
}

public function field_sender_name() {
        $opts = get_option($this->option_name);
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[sender_name]"
               value="<?php echo esc_attr($opts['sender_name'] ?? get_bloginfo('name')); ?>" 
               placeholder="<?php esc_attr_e('Sender Name', 'unlock-to-view'); ?>" />
        <?php
    }
    
    public function field_sender_email() {
        $opts = get_option($this->option_name);
        ?>
        <input type="email" name="<?php echo esc_attr( $this->option_name ); ?>[sender_email]"
               value="<?php echo esc_attr($opts['sender_email'] ?? get_option('admin_email')); ?>" 
               placeholder="<?php esc_attr_e('Sender Email', 'unlock-to-view'); ?>" />
        <?php
    }
    
    public function field_email_subject() {
        $opts = get_option($this->option_name);
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[email_subject]"
               value="<?php echo esc_attr($opts['email_subject'] ?? esc_html__('Code to unlock video on $site_name','unlock-to-view')); ?>" 
               placeholder="<?php esc_attr_e('Email Subject', 'unlock-to-view'); ?>" />
        <?php
    }
    
    public function field_payment_link() {
        $opts = get_option($this->option_name);
        ?>
        <textarea 
            name="<?php echo esc_attr( $this->option_name ); ?>[payment_link]" 
            rows="3" 
            cols="50" 
            placeholder="<?php esc_attr_e('Payment information', 'unlock-to-view'); ?>"
            style="width: 400px;"><?php echo esc_textarea($opts['payment_link'] ?? ''); ?></textarea>
        <em><?php esc_html_e('Add payment information like a link to PayPal or Stripe. This information can be included in the confirmation mail to the buyer below.', 'unlock-to-view'); ?></em>
        <?php
    }

     public function field_reminder_email_subject() {
    $opts    = get_option($this->option_name);
    $default = esc_html__('Reminder: {video} rental at {site_name} is due', 'unlock-to-view');
    $value   = $opts['reminder_email_subject'] ?? $default;
    ?>
    <input type="text"
           id="unlocktoview_reminder_subject"
           name="<?php echo esc_attr( $this->option_name ); ?>[reminder_email_subject]"
           value="<?php echo esc_attr($value); ?>"
           data-default="<?php echo esc_attr($default); ?>"
           class="regular-text" />
    <p class="description"><em>
        <?php esc_html_e(
            'Subject line for reminder emails. <br>Available placeholders: $video, $code, $slug, $duration, $price, $currency, $payment_link, $expiration, $due_date, $ip, $site_name.',
            'unlock-to-view'
        ); ?>
    </em></p>
    <?php
}

public function field_reminder_email_template() {
$opts    = get_option($this->option_name);

    // Subject
    $default_subject = esc_html__('Reminder: {video} rental at {site_name} is due', 'unlock-to-view');
    $subject_value   = $opts['reminder_email_subject'] ?? $default_subject;

    // Message
    $default_message = "Hi {email},\n\n" .
        "This is a friendly reminder that your rental for {video} at {site_name} is due.\n\n" .
        "Please pay using the following link: {payment_link}\n\n" .
        "Rental price: {price}\n" .
        "Payment Due Date: {due_date}\n\n" .
        "Your IP address: {ip}\n\n" .
        "Thank you,\n{site_name}";
    $message_value   = $opts['reminder_email_template'] ?? $default_message;

    // Replace placeholders for preview
    $placeholders = ['{email}', '{video}', '{site_name}', '{code}', '{video_url}', '{expiration_date}', '{payment_link}', '{price}', '{ip}', '{due_date}'];
    $ip_address = isset( $_SERVER['REMOTE_ADDR'] ) 
    ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) 
    : '127.0.0.1';
    $replacements = ['example@email.com', 'Video', get_bloginfo('name'), '1234', home_url('/example-video'), '', $opts['payment_link'] ?? '', $opts['price_per_video'] ?? '3.99', $ip_address, date_i18n('Y-m-d', strtotime('+7 days'))];

    $subject_preview = str_replace($placeholders, $replacements, $subject_value);
    $message_preview = str_replace($placeholders, $replacements, $message_value);
    ?>

    <textarea id="unlocktoview_reminder_template"
              name="<?php echo esc_attr( $this->option_name ); ?>[reminder_email_template]"
              rows="10" cols="60"
              style="width:100%;"
              data-default="<?php echo esc_attr($default_message); ?>"><?php 
                  echo esc_textarea($message_value); ?></textarea>

    <p>
        <button type="button" id="reset-reminder-email" class="reset-btn">
            <?php esc_html_e('Reset', 'unlock-to-view'); ?>
        </button>
    </p>

    <p class="description"><em>
        <?php esc_html_e(
            'List of placeholders that are safe to remove: {email}, {video}, {site_name}, {payment_link}, {price}, {due_date}, {ip}.<br><strong>Note: </strong> These placeholders are added from the code and cannot be modified: {email}, {video}, {site_name}, {due_date}, {ip}.',
            'unlock-to-view'
        ); ?>
    </em></p>

    <div style="margin-top:10px;">
        <strong><?php esc_html_e('Preview:', 'unlock-to-view'); ?></strong>
        <div class="buyer-email-preview" style="background:#f9f9f9; border:1px solid #ddd; padding:15px; font-family:Arial, sans-serif; font-size:13px; line-height:1.4; margin-top:5px;">
            <p><strong><?php esc_html_e('Subject:', 'unlock-to-view'); ?></strong>
                <span id="unlocktoview_preview_subject"><?php echo esc_html($subject_preview); ?></span></p>
            <pre id="unlocktoview_preview_message" style="white-space:pre-wrap;"><?php 
                echo esc_html($message_preview);
            ?></pre>
        </div>
    </div>
    <?php
}

// -----------------
// Render Settings Page with Tabs
// -----------------
public function admin_page_settings() {
    $opts = get_option($this->option_name);
    ?>
    <div class="wrap">
        <div class="unlocktoview-logo">
            <img src="<?php echo esc_url(UNLOCK_TO_VIEW_PLUGIN_URL . 'assets/img/logo_unlocktoview.svg'); ?>" width="300px" alt="<?php echo esc_attr__( 'Unlock To View Logo', 'unlock-to-view' ); ?>" />
        </div>
        <p>
        <?php
        printf(
                wp_kses(
                    /* translators: %s: URL to the Unlock To View shortcodes settings page */
                    __(
                        'Use this page to configure settings such as Price, Duration, and Notifications. These settings will be used by the <a href="%s">shortcodes</a>.',
                        'unlock-to-view'
                    ),
                    [
                        'a' => [
                            'href'   => [],
                            'target' => [],
                            'rel'    => [],
                        ],
                    ]
                ),
                esc_url( admin_url( 'admin.php?page=unlock-to-view' ) )
            );
        ?>
        </p>
        <!-- Tabs -->
        <h2 class="unlocktoview-tab-wrapper">
            <a href="#tab-settings" class="unlocktoview-tab unlocktoview-tab-active">Settings</a>
            <a href="#tab-notifications" class="unlocktoview-tab">Notifications</a>
        </h2>
            <form method="post" action="options.php" id="unlocktoview-settings-form-unique">
           <?php settings_fields('unlocktoview_settings_group'); ?>
            <!-- Settings Tab -->
            <div id="tab-settings" class="tab-content">
                <table class="form-table">
                    <tr>
                        <th>
                            <label>
                                <?php esc_html_e('Rental price', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Set the price users must pay to unlock a video. This price will be applied to all rentals. To set individual price per video, consider upgrading to PRO.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_price_per_video(); ?>
                            <em><?php esc_html_e('Add the price a buyer must pay to unlock a video.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>

                    <tr>
                        <th>
                            <label>
                                <?php esc_html_e('Currency', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Add a 3-letter currency like USD, EUR etc.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_currency(); ?>
                            <em><?php esc_html_e('Add a currency like USD, EUR, GBP.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>

                    <tr>
                        <th>
                            <label>
                                <?php esc_html_e('Duration', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Add number of hours the buyer can access the unlocked video. This duration will be applied to all rentals. To set individual duration, consider upgrading to PRO.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_duration_hours(); ?>
                            <em><?php esc_html_e('Add the number of hours a buyer can access the video.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>
                    <tr>
                        <th>
                            <label>
                                <?php esc_html_e('Unlock Button Label', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Text that will appear on the Unlock button for videos.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <input type="text" name="<?php echo esc_attr($this->option_name); ?>[button_label]"
                                   value="<?php echo esc_attr($opts['button_label'] ?? esc_html__('Unlock this video', 'unlock-to-view')); ?>" />
                            <em><?php esc_html_e('Text on the Unlock button.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- Notifications Tab -->
            <div id="tab-notifications" class="tab-content">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="notify_admin">
                                <?php esc_html_e('Send Notification to Admin', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Send an email notification to the admin whenever a video is unlocked. This message is sent automatically if Yes is selected.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_notify_admin(); ?>
                            <em><?php esc_html_e('Keep this enabled to get notifications on new rentals.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>
                    <tr>
                        <th>
                            <label>
                                <?php esc_html_e('Payment Link', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Add your payment options, like a PayPal or Stripe link. This information will be included in the email sent to the buyer.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_payment_link(); ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="confirm_buyer">
                                <?php esc_html_e('Confirmation mail to Buyer', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('The confirmation email with the <strong>code</strong> that is sent to the buyer after unlocking a video. This message will be sent automatically and cannot be disabled.', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_confirm_buyer(); ?>
                            <em><?php esc_html_e('Recommended: Enable this so buyers receive a receipt of purchase.', 'unlock-to-view'); ?></em>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="unlocktoview_reminder_subject">
                                <?php esc_html_e('Reminder Email Subject', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <em><?php esc_html_e('Subject line for reminder emails. You can use placeholders: {email}, {video}, {site_name}, {due_date}', 'unlock-to-view'); ?></em>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_reminder_email_subject(); ?>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="unlocktoview_reminder_template">
                                <?php esc_html_e('Reminder Email Message', 'unlock-to-view'); ?>
                                <span class="tooltip">ℹ️
                                    <span class="tooltip-text">
                                        <?php esc_html_e('Email body sent to the user for reminders. Placeholders: {email}, {video}, {site_name}, {payment_link}, {price}, {due_date}, {ip}', 'unlock-to-view'); ?>
                                    </span>
                                </span>
                            </label>
                        </th>
                        <td>
                            <?php $this->field_reminder_email_template(); ?>
                        </td>
                    </tr>
                </table>
            </div>
            <?php submit_button(esc_html__('Save Changes', 'unlock-to-view')); ?>
        </form>
    </div>
    <?php
}

// -----------------
// Shortcode & Video Loading
// -----------------
public function shortcode_unlocktoview( $atts, $content = null ) : string {
    global $wp_filesystem;

    // Shortcode attributes (now including player)
    $atts = shortcode_atts([
        'slug'   => '',
        'id'     => '', // numeric ID or URL
        'player' => '', // media player type
    ], $atts, 'unlocktoview'); 

    $slug        = sanitize_title( $atts['slug'] );
    $raw_id      = trim( (string) $atts['id'] );
    $player      = sanitize_text_field( $atts['player'] );
    $uid         = wp_rand(1000, 999999);

    // Persist the media descriptor in CSV (slug, id, player)
    if ( $slug && $raw_id ) {
        // Initialize WP Filesystem
        if ( ! function_exists('WP_Filesystem') ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        $upload_dir = wp_upload_dir();
        $log_dir    = $upload_dir['basedir'] . '/unlock-to-view/logs';
        if ( ! $wp_filesystem->is_dir( $log_dir ) ) {
            wp_mkdir_p( $log_dir );
        }
        $media_file = $log_dir . '/media.csv';

        // Load existing CSV entries
        $entries = [];
        if ( $wp_filesystem->exists( $media_file ) ) {
            $lines = $wp_filesystem->get_contents_array( $media_file );
            foreach ( $lines as $line ) {
                $cols = str_getcsv( $line );
                if ( count($cols) < 3 ) $cols = array_pad($cols, 3, '');
                $entries[ $cols[0] ] = $cols; // key = slug
            }
        }

        // Update / add current entry
        $entries[ $slug ] = [$slug, $raw_id, $player];

        // Build CSV content
        $csv_lines = [];
        foreach ( $entries as $cols ) {
            $csv_lines[] = '"' . implode('","', array_map(function($v){
                return str_replace('"','""',$v);
            }, $cols)) . '"';
        }

        $csv_content = implode("\n", $csv_lines);

        // Write CSV
        $wp_filesystem->put_contents( $media_file, $csv_content, FS_CHMOD_FILE );
    }

    // Output unlock widget
    ob_start(); ?>
    <div class="unlocktoview-container" 
         data-uid="<?php echo esc_attr( $uid ); ?>" 
         data-unlocktoview-slug="<?php echo esc_attr( $slug ); ?>">

        <button class="unlocktoview-rent-btn">
            <?php esc_html_e( 'Unlock this video! 🔓', 'unlock-to-view' ); ?>
        </button>

        <div class="unlocktoview-unlock" style="display:none;">
            <input type="email" class="unlocktoview-input" 
                   placeholder="<?php esc_attr_e( 'Your email', 'unlock-to-view' ); ?>" />
            <button class="unlocktoview-send-btn">
                <?php esc_html_e( 'Send code ✉️', 'unlock-to-view' ); ?>
            </button>
            <div class="unlocktoview-message"></div>

            <div class="unlocktoview-verify-wrap" style="display:none;">
                <div class="unlocktoview-already-code-msg unlocktoview-already-code-style" style="display:none;">
                    😎 <?php esc_html_e( 'A code was already sent to this email. Check your inbox!', 'unlock-to-view' ); ?>
                </div>
                <input type="text" class="unlocktoview-input" 
                       maxlength="<?php echo esc_attr( get_option( 'unlocktoview_code_length', 4 ) ); ?>" 
                       placeholder="<?php esc_attr_e( 'Enter code', 'unlock-to-view' ); ?>" />
                <button class="unlocktoview-verify-btn">
                    <?php esc_html_e( 'Unlock! 🎬', 'unlock-to-view' ); ?>
                </button>
                <div class="unlocktoview-code-message"></div>
            </div>
        </div>

       <div class="unlocktoview-video">
        <?php
            // Render video immediately if ID is present
            if ( $raw_id ) {
                // Escape late: allow only safe HTML returned by render_video_by_type()
                echo wp_kses_post( $this->render_video_by_type( $player, $raw_id ) );
            }
            ?>
    </div>
    </div>
    <?php
    return ob_get_clean();
}

// -----------------
// AJAX: Check if user already has a code
// -----------------
public function ajax_check_code_exists() {
    check_ajax_referer('unlocktoview_nonce', 'nonce');

    $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
    $slug  = isset( $_POST['slug'] )  ? sanitize_title( wp_unslash( $_POST['slug'] ) ) : '';

    if (!$email || !$slug) {
        wp_send_json_error(['message' => 'Missing parameters'], 400);
    }

    $upload_dir = wp_upload_dir()['basedir'] . '/unlock-to-view/logs';
    $rental_file = $upload_dir . '/rentals.csv';

    if (!file_exists($rental_file)) {
        wp_send_json_success(['exists' => false]);
    }

    $lines = file($rental_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

    if (empty($lines)) {
        wp_send_json_success(['exists' => false]);
    }

    // -----------------------------
    // Skip the header row
    // -----------------------------
    $header = array_shift($lines);

    // Check most recent rentals first
    $lines = array_reverse($lines);

    foreach ($lines as $line) {
        $cols = str_getcsv($line);

        if (count($cols) < 5) continue; // skip malformed rows

        $log_email = strtolower(trim($cols[1])); // Email
        $log_slug  = trim($cols[2]);             // Video/slug
        $expiry    = trim($cols[4]);             // Expiry date

        // convert expiry to timestamp (assume UTC)
        $expiry_time = strtotime($expiry . ' UTC');

        if ($log_email === strtolower($email) &&
            $log_slug === $slug &&
            $expiry_time > time()
        ) {
            wp_send_json_success(['exists' => true]);
        }
    }

    wp_send_json_success(['exists' => false]);
}

public function ajax_send_code() {
    // Verify AJAX nonce
    check_ajax_referer( 'unlocktoview_nonce', 'nonce' );

    // --- Grab only required POST variables ---
    $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
    $slug  = isset( $_POST['slug'] )  ? sanitize_title( wp_unslash( $_POST['slug'] ) ) : '';

    // --- Fallback: only read required keys from JSON body ---
    if ( empty( $email ) && empty( $slug ) ) {

        // Read raw body
        $raw = wp_unslash( file_get_contents( 'php://input') );

        // Decode JSON
        $json = json_decode( $raw, true );

        if ( is_array( $json ) && json_last_error() === JSON_ERROR_NONE ) {

            // Grab only the keys we care about
            if ( ! empty( $json['email'] ?? $json['Email'] ?? '' ) ) {
                $email = sanitize_email( trim( $json['email'] ?? $json['Email'] ?? '' ) );
            }

            if ( ! empty( $json['slug'] ?? $json['video'] ?? '' ) ) {
                $slug = sanitize_title( trim( $json['slug'] ?? $json['video'] ?? '' ) );
            }
        }
    }

    // --- Validation ---
    if ( empty( $slug ) ) {
        wp_send_json_error( [ 'message' => '❌ Missing video slug (data-slug).' ], 400 );
    }

    if ( empty( $email ) ) {
        wp_send_json_error( [ 'message' => '❌ Please enter a valid email address.' ], 400 );
    }

    if ( ! is_email( $email ) ) {
        wp_send_json_error( [ 'message' => '❌ Invalid email format.' ], 400 );
    }

    // --- Load settings ---
    $opts = get_option($this->option_name, []);
    $price      = $opts['price_per_video'] ?? '{price}';
    $currency   = $opts['currency'] ?? '{currency}';
    $duration   = isset($opts['duration_hours']) ? intval($opts['duration_hours']) : 24;
    $payment    = $opts['payment_link'] ?? '{payment_link}';
    $site_name  = get_bloginfo('name');
    $from_name  = $opts['buyer_sender'] ?? $site_name;
    $from_email = $opts['sender_email'] ?? get_option('admin_email');
    $subject = $opts['buyer_subject'] ?? sprintf(
    /* translators: %s: site name */
    esc_html__( 'Unlock video on %s', 'unlock-to-view' ),
    $site_name
    );
    $template   = $opts['buyer_message'] ?? '';

    // --- CSV logging ---
    $upload_dir = wp_upload_dir()['basedir'] . '/unlock-to-view/logs';
    if (!file_exists($upload_dir)) wp_mkdir_p($upload_dir);
    $log_file = $upload_dir . '/rentals.csv';
    if (!file_exists($log_file)) {
        file_put_contents($log_file, "Date,Email,Video,Code,Expiry,Invoice,IP,ReminderSent,Paid\n");
    }

    // --- Check for existing valid rental ---
    $lines = file($log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $now = time();
    foreach ($lines as $line) {
        if (stripos($line, 'Date,Email,Video') === 0) continue; // skip header
        $cols = str_getcsv($line);
        if (count($cols) < 5) continue;
        $log_email = strtolower(trim($cols[1]));
        $log_slug  = trim($cols[2]);
        $expiry    = strtotime(trim($cols[4]));

        // only consider non-expired rentals
        if ($log_email === strtolower($email) && $log_slug === $slug && $expiry > $now) {
            wp_send_json_success([
                'already_sent' => true,
                'message' => '✅ A code has already been sent to this email. Please check your inbox!'
            ]);
        }
    }

    // --- Generate new code and invoice ---
    $invoice_number = 0;
    foreach ($lines as $line) {
        $cols = str_getcsv($line);
        if (count($cols) >= 6 && preg_match('/INV-(\d+)/', $cols[5] ?? '', $matches)) {
            $num = intval($matches[1]);
            if ($num > $invoice_number) $invoice_number = $num;
        }
    }
    $invoice_number++;
    $invoice_code = 'INV-' . str_pad($invoice_number, 5, '0', STR_PAD_LEFT);

   // WordPress timezone-aware dates
    $rented_at  = date_i18n('Y-m-d H:i:s');
    $hours      = max(1, (int) $duration);
    $expiry_ts  = time() + ($hours * HOUR_IN_SECONDS);
    $expiry     = date_i18n('Y-m-d H:i:s', $expiry_ts);
    
    // Use wp_rand() instead of rand()
    $code       = method_exists($this, 'generate_code') ? $this->generate_code(4) : wp_rand(1000, 9999);
    
    $ip = isset( $_SERVER['REMOTE_ADDR'] ) 
    ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) 
    : 'Unknown';

    // Due date 7 days from now (WP timezone-aware)
    $due_date   = date_i18n('Y-m-d', strtotime('+7 days'));

    $log_line = "{$rented_at},{$email},{$slug},{$code},{$expiry},[{$invoice_code}],{$ip},,\n";
    file_put_contents($log_file, $log_line, FILE_APPEND);

    // --- Prepare email ---
    $post        = get_page_by_path($slug, OBJECT, 'video');
    $video_title = $post ? get_the_title($post->ID) : ucfirst(str_replace('-', ' ', $slug));
    $video_url   = $post ? get_permalink($post->ID) : home_url("/{$slug}");

    $placeholders  = ['$email','$code','$slug','$duration','$price','$currency','$payment_link','$site_name','$expiry','$due_date','$ip'];
    $replacements  = [$email,$code,$slug,$duration,$price,$currency,$payment,$site_name,$expiry,$due_date,$ip];

    $final_subject = str_replace($placeholders, $replacements, $subject);
    $final_message = str_replace($placeholders, $replacements, $template);

    // --- Buyer email ---
    $headers_buyer = ["From: {$from_name} <{$from_email}>", "Bcc: {$from_email}"];
    $sent_buyer = wp_mail($email, $final_subject, $final_message, $headers_buyer);

    // --- Admin notification (optional) ---
    $notify_admin = $opts['notify_admin'] ?? 0; // 1 if checked, 0 if not
    $admin_email  = $opts['admin_email'] ?? get_option('admin_email');
    
    if (intval($notify_admin) === 1 && !empty($admin_email) && is_email($admin_email)) {
        $headers_admin = ["From: {$site_name} <{$from_email}>"];
    
        $admin_subject = "Admin copy: {$final_subject}";
        $admin_message = "Hello Admin,\n\nA new rental has been made.\n\n".
            "Video: {$video_title}\n".
            "User Email: {$email}\n".
            "Code: {$code}\n".
            "Expires: {$expiry}\n".
            "Invoice: {$invoice_code}\n".
            "IP: {$ip}\n\n".
            "This is an automatic notification from {$site_name}.";
    
        wp_mail($admin_email, $admin_subject, $admin_message, $headers_admin);
    }
    
    // --- Only fail if buyer email fails ---
    if (!$sent_buyer) {
        wp_send_json_error(['message' => '❌ Could not send email to buyer. Check server email settings.'], 500);
    }

    wp_send_json_success(['message' => '✅ Code sent! Check your inbox (or Spam folder).']);
}

// -----------------
// AJAX: Get video
// -----------------
public function ajax_get_video() {
    check_ajax_referer('unlocktoview_nonce', 'nonce');

    $slug  = isset($_POST['slug'])  ? sanitize_title(wp_unslash($_POST['slug'])) : '';
    $code  = isset($_POST['code'])  ? sanitize_text_field(wp_unslash($_POST['code'])) : '';
    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';

    if (!$slug || !$code || !$email) {
        wp_send_json_error(['message' => 'Missing parameters.'], 400);
    }

    // Verify rental code
    if (!$this->verify_code($slug, $code, $email)) {
        wp_send_json_error(['message' => '❌ Invalid code or expired.'], 403);
    }

    // Load media.csv
    $media_file = wp_upload_dir()['basedir'] . '/unlock-to-view/logs/media.csv';

    if (!file_exists($media_file) || filesize($media_file) === 0) {
        wp_send_json_error(['message' => 'Media not found.'], 404);
    }

    $lines = file($media_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

    $id     = '';
    $player = '';

    foreach ($lines as $line) {
        $cols = str_getcsv($line);

        // Skip invalid lines
        if (count($cols) < 2) continue;

        if (sanitize_title(trim($cols[0])) === $slug) {
            $id     = trim($cols[1]);
            $player = trim($cols[2]);

            // Treat empty or "any url" as generic URL
            if ($player === '' || strcasecmp($player, 'any url') === 0 || strcasecmp($player, 'any other url') === 0) {
                $player = 'url';
            }

            break;
        }
    }

    if (!$id) {
        wp_send_json_error(['message' => 'Media not found.'], 404);
    }

    // Validate player
    $valid_players = ['youtube', 'vimeo', 'presto', 'presto_player', 'url'];
    if (!in_array($player, $valid_players, true)) {
        wp_send_json_error(['message' => 'Unsupported media player type.'], 400);
    }

    // Render video: player first, then video ID/URL
    $html = $this->render_video_by_type($player, $id);

    wp_send_json_success(['html' => $html]);
}


// -----------------
// Helpers
// -----------------
private function verify_code(string $slug, string $code, string $email) : bool {
    if (!$slug || !$code || !$email) {
        return false;
    }

    $upload_dir  = wp_upload_dir()['basedir'] . '/unlock-to-view/logs';
    $rental_file = $upload_dir . '/rentals.csv';

    if (!file_exists($rental_file) || filesize($rental_file) === 0) {
        return false;
    }

    $lines = file($rental_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $lines = array_reverse($lines); // newest first

    foreach ($lines as $line) {
        // Skip header row
        if (stripos($line, 'Date,Email,Video,Code,Expiry') === 0) continue;

        $cols = array_map('trim', str_getcsv($line));
        if (count($cols) < 5) continue;

        $log_email = $cols[1];
        $log_slug  = $cols[2];
        $log_code  = $cols[3];
        $expiry    = $cols[4];

        if (
            strtolower($log_email) === strtolower($email) &&       // normalize email
            sanitize_title($log_slug) === sanitize_title($slug) && // normalize slug
            strtoupper($log_code) === strtoupper($code)            // normalize code
        ) {
            $expiry_ts = strtotime($expiry);
            if ($expiry_ts !== false && $expiry_ts > time()) {
                return true;
            }
        }
    }

    return false;
}

private function parse_media_descriptor(string $raw): array {
    $raw = trim($raw);
    $value = $raw;
    $type = '';

    // Detect YouTube
    if (preg_match('#(?:youtu\.be/|youtube\.com/(?:watch\?v=|embed/))#', $raw)) {
        $type = 'youtube';
    }
    // Detect Vimeo
    elseif (preg_match('#vimeo\.com/#', $raw)) {
        $type = 'vimeo';
    }
    // Presto Player ID (numeric)
    elseif (is_numeric($raw)) {
        $type = 'presto';
    }
    // Generic URL
    elseif (filter_var($raw, FILTER_VALIDATE_URL)) {
        $type = 'url';
    }

    return [$type, $value];
}

private function render_video_by_type(string $player, string $videoid): string {
    $videoid = trim($videoid);

    // If player is empty, auto-detect type
    if ($player === '' || $player === 'url') {
        [$detected_player, $detected_id] = $this->parse_media_descriptor($videoid);
        $player  = $detected_player;
        $videoid = $detected_id;
    }

    $output = '';

    switch ($player) {
        case 'youtube':
            if (preg_match('/(?:youtu\.be\/|v=|embed\/)([a-zA-Z0-9_-]{11})/', $videoid, $m)) {
                $yt = $m[1];
            } else {
                $yt = $videoid;
            }
            $output = '<iframe width="100%" height="480" src="https://www.youtube.com/embed/' . esc_attr($yt) . '" frameborder="0" allowfullscreen></iframe>';
            break;

        case 'vimeo':
            if (preg_match('/(\d+)/', $videoid, $m)) {
                $output = '<iframe src="https://player.vimeo.com/video/' . esc_attr($m[1]) . '" width="640" height="360" frameborder="0" allowfullscreen></iframe>';
            } else {
                $output = '<p>Invalid Vimeo video ID or URL.</p>';
            }
            break;

        case 'presto':
        case 'presto_player':
            $output = do_shortcode('[presto_player id="' . esc_attr($videoid) . '"]');
            break;

        case 'url':
            // treat as direct video URL (HTML5)
            $output = '<video width="100%" height="auto" controls><source src="' . esc_url($videoid) . '" type="video/mp4"></video>';
            break;

        default:
            $output = '<p>Unsupported media player type.</p>';
            break;
    }

    return $output;
}

// -----------------
// Get rental
// -----------------
public function ajax_get_rentals() {
    check_ajax_referer('unlocktoview_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Not allowed.']);
    }

    $rentals = $this->get_rentals(); // full dataset

    wp_send_json_success([
        'data' => $rentals, // full array for JS
    ]);
}

// -----------------
// Mark as Paid
// -----------------
public function ajax_mark_paid() {
    check_ajax_referer('unlocktoview_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Not allowed.']);
    }

    $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
    $video = isset( $_POST['video'] ) ? sanitize_text_field( wp_unslash( $_POST['video'] ) ) : '';


    if (!$email || !$video) {
        wp_send_json_error(['message' => 'Missing rental identifiers.']);
    }

    $rentals = $this->get_rentals();
    $updated = false;

    foreach ($rentals as &$row) {
        if (($row['Email'] ?? '') === $email && ($row['Video'] ?? '') === $video) {
            $row['Paid'] = 'Yes';
            $updated = true;
            break;
        }
    }
    unset($row);

   if ($updated) {
    $this->save_rentals($rentals);

    wp_send_json_success([
        'message'     => 'Marked as paid.',
        'paid_status' => 'Yes' // include the updated status
    ]);
    } else {
        wp_send_json_error(['message' => 'Rental not found.']);
    }
}

// -----------------
// Delete rental
// -----------------
public function ajax_delete_rental() {
    check_ajax_referer('unlocktoview_nonce', 'nonce'); 

    $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
    $video = isset( $_POST['video'] ) ? sanitize_text_field( wp_unslash( $_POST['video'] ) ) : '';


    if (empty($email) || empty($video)) {
        wp_send_json_error(['message' => 'Missing rental identifiers']);
    }

    $deleted = $this->delete_rental($email, $video);

    if ($deleted) {
        wp_send_json_success(['message' => 'Rental deleted.']);
    } else {
        wp_send_json_error(['message' => 'Rental not found.']);
    }
}

// -----------------
// Send reminder
// -----------------
public function ajax_send_reminder() {
    try {
        check_ajax_referer('unlocktoview_nonce', 'nonce');

        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $video = isset( $_POST['video'] ) ? sanitize_text_field( wp_unslash( $_POST['video'] ) ) : '';

        if (!$email || !$video) {
            wp_send_json_error(['message' => 'Missing rental identifiers']);
        }

        $rentals = $this->get_rentals();
        $found_index = null;

        foreach ($rentals as $i => $r) {
            if (($r['Email'] ?? '') === $email && ($r['Video'] ?? '') === $video) {
                $found_index = $i;
                break;
            }
        }

        if ($found_index === null) {
            wp_send_json_error(['message' => 'Rental not found']);
        }

        // Settings
        $opts = get_option($this->option_name, []);
        $site_name    = get_bloginfo('name');
        $sender_name  = $opts['sender_name'] ?? $site_name;
        $sender_email = $opts['sender_email'] ?? get_option('admin_email');
        $template     = $opts['reminder_email_template'] ?? '';
        $payment_link = $opts['payment_link'] ?? '';
        $price        = $opts['price_per_video'] ?? '';
        $currency     = $opts['currency'] ?? 'USD';
        $ip = isset( $_SERVER['REMOTE_ADDR'] ) 
            ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) 
            : 'Unknown';

        $due_date   = date_i18n('Y-m-d', strtotime('+7 days'));
        $code         = $rentals[$found_index]['Code'] ?? 'N/A';
        $video_url    = get_permalink($rentals[$found_index]['PostID'] ?? 0) ?: '';

        $subject = $opts['reminder_email_subject'] ?? esc_html__('Rental Reminder', 'unlock-to-view');

        $placeholders = [
            '{email}', '{video}', '{site_name}', '{code}', '{video_url}',
            '{expiration_date}', '{payment_link}', '{price}', '{ip}', '{due_date}'
        ];
        $replacements = [
            $email, $video, $site_name, $code, $video_url,
            '', $payment_link, "$price $currency", $ip, $due_date
        ];

        $subject = str_replace($placeholders, $replacements, $subject);
        $message = str_replace($placeholders, $replacements, $template);

        add_filter('wp_mail_from', fn() => $sender_email);
        add_filter('wp_mail_from_name', fn() => $sender_name);

        $headers = ['Content-Type: text/plain; charset=UTF-8'];
        $sent = wp_mail($email, $subject, $message, $headers);

        remove_all_filters('wp_mail_from');
        remove_all_filters('wp_mail_from_name');

        if ($sent) {
            $rentals[$found_index]['ReminderSent'] = current_time('mysql');
            $this->save_rentals($rentals);

            wp_send_json_success(['message' => 'Reminder sent successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to send email']);
        }

    } catch (Exception $e) {
        wp_send_json_error(['message' => 'Server error: ' . $e->getMessage()]);
    }
}
// -----------------
// Scripts & Styles
// -----------------

// Register CSS/JS assets
public function register_assets() {
    $version = defined('UNLOCK_TO_VIEW_VERSION') ? UNLOCK_TO_VIEW_VERSION : '1.0.0';

    // Admin CSS
    $admin_css_path = UNLOCK_TO_VIEW_ASSETS_DIR . 'css/unlock-admin.css';
    $admin_css_ver  = file_exists($admin_css_path) ? filemtime($admin_css_path) : $version;
    wp_register_style('unlocktoview-admin', UNLOCK_TO_VIEW_ASSETS . 'css/unlock-admin.css', [], $admin_css_ver);

    // Frontend CSS
    $frontend_css_path = UNLOCK_TO_VIEW_ASSETS_DIR . 'css/unlock-frontend.css';
    $frontend_css_ver  = file_exists($frontend_css_path) ? filemtime($frontend_css_path) : $version;
    wp_register_style('unlocktoview-frontend', UNLOCK_TO_VIEW_ASSETS . 'css/unlock-frontend.css', [], $frontend_css_ver);

    // Admin JS
    $admin_js_path = UNLOCK_TO_VIEW_ASSETS_DIR . 'js/unlock-admin.js';
    $admin_js_ver  = file_exists($admin_js_path) ? filemtime($admin_js_path) : $version;
    wp_register_script('unlocktoview-admin', UNLOCK_TO_VIEW_ASSETS . 'js/unlock-admin.js', ['jquery'], $admin_js_ver, true);

    // Frontend JS
    $frontend_js_path = UNLOCK_TO_VIEW_ASSETS_DIR . 'js/unlock-frontend.js';
    $frontend_js_ver  = file_exists($frontend_js_path) ? filemtime($frontend_js_path) : $version;
    wp_register_script('unlocktoview-frontend', UNLOCK_TO_VIEW_ASSETS . 'js/unlock-frontend.js', ['jquery'], $frontend_js_ver, true);
}

// Enqueue frontend assets
public function enqueue_frontend() {
    wp_enqueue_style('unlocktoview-frontend');
    wp_enqueue_script('unlocktoview-frontend');

    // Localize AJAX for frontend safely
    wp_localize_script('unlocktoview-frontend', 'unlocktoview_frontend_ajax', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('unlocktoview_nonce'),
    ]);
}

// Enqueue about page style
public function enqueue_admin_about_styles($hook_suffix) {
    // Only load on our About page
    if ( $hook_suffix !== 'toplevel_page_unlock-to-view' && $hook_suffix !== 'unlock-to-view_page_unlock-to-view-about' ) {
        return;
    }

    $css_file = plugin_dir_path( __FILE__ ) . 'assets/css/unlock-about.css';
    
    // Use file modification time as version so it updates when changed
    $version = file_exists( $css_file ) ? filemtime( $css_file ) : '1.0.0';

    wp_enqueue_style(
        'utv-about-style',
        plugin_dir_url( __FILE__ ) . 'assets/css/unlock-about.css',
        [],
        $version
    );
}

// Enqueue admin assets
public function enqueue_admin_assets($hook_suffix) {
    // Only load on plugin admin pages
    if (!isset($hook_suffix) || strpos($hook_suffix, 'unlock-to-view') === false) {
        return;
    }

    // Enqueue registered styles and scripts
    wp_enqueue_style('unlocktoview-admin');
    wp_enqueue_script('unlocktoview-admin');

    // Get plugin options safely
    $opts = get_option($this->option_name, []);

    // Localize AJAX + settings for admin script
    wp_localize_script('unlocktoview-admin', 'unlocktoview_admin_ajax', [
        'ajax_url'         => admin_url('admin-ajax.php'),
        'nonce'            => wp_create_nonce('unlocktoview_nonce'),
        'defaultMessage'   => wp_strip_all_tags($opts['buyer_message'] ?? ''),
        'defaultSubject'   => wp_strip_all_tags($opts['buyer_subject'] ?? esc_html__('Code to unlock video on {site_name}', 'unlock-to-view')),
        'price'            => sanitize_text_field($opts['price_per_video'] ?? ''),
        'currency'         => sanitize_text_field($opts['currency'] ?? ''),
        'duration'         => intval($opts['duration_hours'] ?? 0),
        'payment_link'     => esc_url_raw($opts['payment_link'] ?? ''),
        'site_name'        => get_bloginfo('name'),
        'reminder_subject' => wp_strip_all_tags($opts['reminder_email_subject'] ?? ''),
        'reminder_message' => wp_strip_all_tags($opts['reminder_email_template'] ?? ''),
        'button_label'     => wp_strip_all_tags($opts['button_label'] ?? esc_html__('Unlock', 'unlock-to-view')),
    ]);
}

// ---------------------------
// Utility: Format CSV line
// ---------------------------
private function unlocktoview_csv_line(array $cols): string {
    return '"' . implode('","', array_map('addslashes', $cols)) . '"' . "\n";
}
}
endif; // end class_exists check

// Register activation/deactivation hooks
register_activation_hook( __FILE__, [ 'UnlockToView', 'activate' ] );
register_deactivation_hook( __FILE__, [ 'UnlockToView', 'deactivate' ] );
// Make sure we have the plugin instance
$unlock_to_view = UnlockToView::instance();

// -----------------
// CSV Export Handler
// -----------------
add_action('admin_post_unlocktoview_export_csv', function() use ($unlock_to_view) {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to export rentals.', 'unlock-to-view'));
    }

    $rentals = $unlock_to_view->get_rentals();
    if (empty($rentals)) {
        wp_die(esc_html__('No rentals found to export.', 'unlock-to-view'));
    }

    // Build CSV content in memory
    $csv_lines = [];
    $headers = array_keys(reset($rentals));
    $csv_lines[] = '"' . implode('","', array_map(function($v){ return str_replace('"','""',$v); }, $headers)) . '"';
    foreach ($rentals as $row) {
        $csv_lines[] = '"' . implode('","', array_map(function($v){ return str_replace('"','""',$v); }, $row)) . '"';
    }
    $csv_content = implode("\n", $csv_lines);

    // Send CSV headers
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=rentals.csv');

    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo $csv_content;
    exit;
});