jQuery(document).ready(function ($) {

    $('.unlocktoview-container').each(function () {
        const container = $(this);

        // --- Prevent duplicate initialization ---
        if (container.data('utv-initialized')) return;
        container.data('utv-initialized', true);

        // --- Slug detection ---
        const slug = container.data('unlocktoview-slug') || container.attr('data-slug') || '';

        // Elements
        const rentBtn = container.find('.unlocktoview-rent-btn');
        const unlockSection = container.find('.unlocktoview-unlock');
        const emailInput = unlockSection.find('input[type="email"], input[name="email"]').first();
        const sendBtn = unlockSection.find('.unlocktoview-send-btn');
        const emailMessage = unlockSection.find('.unlocktoview-message');

        const verifyWrap = unlockSection.find('.unlocktoview-verify-wrap');
        const codeInput = verifyWrap.find('input[type="text"]');
        const verifyBtn = verifyWrap.find('.unlocktoview-verify-btn');
        const codeMessage = verifyWrap.find('.unlocktoview-code-message');
        const alreadyMsg = verifyWrap.find('.unlocktoview-already-code-msg');

        const videoContainer = container.find('.unlocktoview-video');

        // --- AJAX context ---
        const ajaxCtx = window.unlocktoview_frontend_ajax || window.unlocktoview_ajax || window.video_unlocker_ajax || {};
        const ajaxUrl = ajaxCtx.ajax_url || '/wp-admin/admin-ajax.php';
        const nonce = ajaxCtx.nonce || '';

        // --- Helper functions ---
        function showMessage(el, text, type) {
            el.removeClass('error success').addClass(type).text(text).show();
        }
        function hideMessage(el) {
            el.removeClass('error success').text('').hide();
        }

        // --- Check if user already has a valid code ---
        function checkExistingCode() {
            const emailVal = (emailInput.val() || '').trim().toLowerCase();
            if (!emailVal || !slug) return;

            $.post(ajaxUrl, {
                action: 'check_code_exists',
                _wpnonce: nonce,
                email: emailVal,
                slug: slug
            }, function (res) {
                if (res.success && res.data?.exists) {
                    unlockSection.addClass('is-visible');
                    verifyWrap.addClass('is-visible');
                    codeInput.show();
                    verifyBtn.show();
                    alreadyMsg.text(res.data?.message || '😎 You already have a code. Enter it below.').show();
                    hideMessage(emailMessage);
                    sendBtn.hide();
                } else {
                    verifyWrap.removeClass('is-visible');
                    codeInput.hide();
                    verifyBtn.hide();
                    alreadyMsg.hide();
                    sendBtn.show();
                    hideMessage(emailMessage);
                }
            });
        }

        emailInput.off('blur', checkExistingCode).on('blur', checkExistingCode);

        // --- Unlock button ---
        rentBtn.off('click').on('click', function () {
            rentBtn.hide();
            unlockSection.addClass('is-visible');
        });

        // --- Send code ---
        sendBtn.off('click').on('click', function () {
            const emailVal = (emailInput.val() || '').trim().toLowerCase();
            hideMessage(emailMessage);
            hideMessage(codeMessage);
            alreadyMsg.hide();

            if (!emailVal) return showMessage(emailMessage, '❌ Please enter a valid email.', 'error');
            if (!slug) return showMessage(emailMessage, '❌ Missing media identifier.', 'error');

            sendBtn.prop('disabled', true).addClass('loading');

            $.ajax({
                url: ajaxUrl,
                method: 'POST',
                dataType: 'json',
                data: { action: 'unlocktoview_send_code', _wpnonce: nonce, email: emailVal, slug: slug },
                success: function (res) {
                    sendBtn.prop('disabled', false).removeClass('loading');
                    unlockSection.addClass('is-visible');
                    verifyWrap.addClass('is-visible');
                    codeInput.show();
                    verifyBtn.show();

                    if (res.success) {
                        if (res.data?.already_sent) {
                            alreadyMsg.text(res.data.message || '✅ You already have a code. Enter it below.').show();
                            hideMessage(emailMessage);
                            sendBtn.hide();
                        } else {
                            alreadyMsg.hide();
                            showMessage(emailMessage, '✅ Code sent! Check your inbox.', 'success');
                        }
                    } else {
                        alreadyMsg.hide();
                        showMessage(emailMessage, res.data?.message || 'Could not send code.', 'error');
                    }
                },
                error: function (xhr) {
                    sendBtn.prop('disabled', false).removeClass('loading');
                    let msg = 'Server error.';
                    if (xhr.status === 403 && xhr.responseText === '-1') msg = '❌ Security check failed (invalid nonce).';
                    else if (xhr.responseJSON?.data?.message) msg = xhr.responseJSON.data.message;
                    else if (xhr.responseJSON?.message) msg = xhr.responseJSON.message;
                    alreadyMsg.hide();
                    showMessage(emailMessage, msg, 'error');
                }
            });
        });

        // --- Verify code ---
        verifyBtn.off('click').on('click', function () {
            const emailVal = (emailInput.val() || '').trim().toLowerCase();
            const codeVal = (codeInput.val() || '').trim();
            const slugVal = slug;

            hideMessage(codeMessage);

            if (!codeVal) return showMessage(codeMessage, '❌ Enter your unlock code.', 'error');

            verifyBtn.prop('disabled', true).addClass('loading');

            $.ajax({
                url: ajaxUrl,
                method: 'POST',
                dataType: 'json',
                data: { 
                    action: 'unlocktoview_get_video', 
                    _wpnonce: nonce, 
                    email: emailVal, 
                    slug: slugVal, 
                    code: codeVal 
                },
                success: function (res) {
                    verifyBtn.prop('disabled', false).removeClass('loading');

                    if (res.success && res.data?.html) {
                        videoContainer.html(res.data.html).fadeIn();
                        unlockSection.removeClass('is-visible');
                        verifyWrap.removeClass('is-visible');
                        alreadyMsg.hide();
                        hideMessage(codeMessage);
                        hideMessage(emailMessage);
                    } else {
                        showMessage(codeMessage, res.data?.message || '❌ Invalid or expired code.', 'error');
                    }
                },
                error: function (xhr) {
                    verifyBtn.prop('disabled', false).removeClass('loading');
                    let msg = '❌ Invalid code or expired. JS AGAIN';
                    if (xhr.status === 403 && xhr.responseText === '-1') msg = '❌ Security check failed (invalid nonce).';
                    else if (xhr.responseJSON?.data?.message) msg = xhr.responseJSON.data.message;
                    else if (xhr.responseJSON?.message) msg = xhr.responseJSON.message;
                    showMessage(codeMessage, msg, 'error');
                }
            });
        });

        // --- Enter key triggers send/verify ---
        emailInput.add(codeInput).off('keypress').on('keypress', function (e) {
            if (e.which === 13) {
                e.preventDefault();
                $(this).is(emailInput) ? sendBtn.click() : verifyBtn.click();
            }
        });
    });
});
