<?php

/**
 * Plugin Name:       Universal Control for TaHoma
 * Description:       Universal Gutenberg block to control Somfy TaHoma devices worldwide via Cloud API.
 * Version:           1.0.4
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Author:            globus2008
 * Text Domain:       universal-control-for-tahoma
 * License: GPL-3.0-or-later
 * License URI: http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package CreateBlock
 */

if (! defined('ABSPATH')) {
	exit;
}


function universal_control_for_tahoma_create_block_block_init()
{
	if (function_exists('wp_register_block_types_from_metadata_collection')) {
		wp_register_block_types_from_metadata_collection(__DIR__ . '/build', __DIR__ . '/build/blocks-manifest.php');
		return;
	}

	if (function_exists('wp_register_block_metadata_collection')) {
		wp_register_block_metadata_collection(__DIR__ . '/build', __DIR__ . '/build/blocks-manifest.php');
	}

	$manifest_path = __DIR__ . '/build/blocks-manifest.php';
	if (file_exists($manifest_path)) {
		$manifest_data = require $manifest_path;
		foreach (array_keys($manifest_data) as $block_type) {
			register_block_type(__DIR__ . "/build/{$block_type}");
		}
	}
	register_block_type(__DIR__ . '/build', array(
		'icon' => 'admin-home',
	));
}
add_action('init', 'universal_control_for_tahoma_create_block_block_init');


add_action('admin_menu', function () {
	add_options_page(
		__('TaHoma Settings', 'universal-control-for-tahoma'),
		__('TaHoma Control', 'universal-control-for-tahoma'),
		'manage_options',
		'tahoma-settings',
		'universal_control_for_tahoma_settings_page_html'
	);
});

function universal_control_for_tahoma_settings_page_html()
{
?>
	<div class="wrap">
		<h1><?php esc_html_e('TaHoma Global Settings', 'universal-control-for-tahoma'); ?></h1>
		<p><?php esc_html_e('The credentials entered here will be used as default for all new blocks.', 'universal-control-for-tahoma'); ?></p>
		<form action="options.php" method="post">
			<?php
			settings_fields('tahoma_settings_group');
			do_settings_sections('tahoma-settings');
			submit_button();
			?>
		</form>
	</div>
<?php
}

add_action('admin_init', function () {
	register_setting('tahoma_settings_group', 'tahoma_global_email', [
		'sanitize_callback' => 'sanitize_email',
	]);

	register_setting('tahoma_settings_group', 'tahoma_global_password', [
		'sanitize_callback' => 'sanitize_text_field',
	]);

	register_setting('tahoma_settings_group', 'tahoma_global_server', [
		'sanitize_callback' => 'sanitize_text_field',
	]);

	add_settings_section(
		'tahoma_main_section',
		__('Cloud Credentials', 'universal-control-for-tahoma'),
		null,
		'tahoma-settings'
	);

	add_settings_field('tahoma_email', 'Email', function () {
		echo '<input type="text" name="tahoma_global_email" value="' . esc_attr(get_option('tahoma_global_email')) . '" class="regular-text">';
	}, 'tahoma-settings', 'tahoma_main_section');

	add_settings_field('tahoma_password', 'Password', function () {
		echo '<input type="password" name="tahoma_global_password" value="' . esc_attr(get_option('tahoma_global_password')) . '" class="regular-text">';
	}, 'tahoma-settings', 'tahoma_main_section');

	add_settings_field('tahoma_server', 'Server', function () {
		$val = get_option('tahoma_global_server', 'www.tahomalink.com');
		echo '<select name="tahoma_global_server">
            <option value="www.tahomalink.com" ' . selected($val, 'www.tahomalink.com', false) . '>TaHomalink</option>
            <option value="ha101-1.overkiz.com" ' . selected($val, 'ha101-1.overkiz.com', false) . '>Overkiz HA1 (Europe)</option>
        </select>';
	}, 'tahoma-settings', 'tahoma_main_section');
});

add_action('enqueue_block_editor_assets', function () {
	wp_localize_script(
		'create-block-universal-control-for-tahoma-editor-script',
		'tahomaGlobal',
		[
			'email'    => get_option('tahoma_global_email'),
			'password' => get_option('tahoma_global_password'),
			'server'   => get_option('tahoma_global_server', 'www.tahomalink.com'),
		]
	);
});


add_action('init', function () {
	register_block_type(__DIR__ . '/build');
});

add_action('rest_api_init', function () {
	register_rest_route('tahoma/v1', '/fetch-devices', [
		'methods' => 'POST',
		'callback' => 'tahoma_api_fetch_devices',
		'permission_callback' => function () {
			return current_user_can('edit_posts');
		}
	]);

	register_rest_route('tahoma/v1', '/execute', [
		'methods' => 'POST',
		'callback' => 'tahoma_api_execute',
		'permission_callback' => function () {
			return is_user_logged_in() || current_user_can('edit_posts');
		}
	]);

	register_rest_route('tahoma/v1', '/status', [
		'methods' => 'POST',
		'callback' => 'tahoma_api_get_device_status',
		'permission_callback' => function () {
			return is_user_logged_in() || current_user_can('edit_posts');
		}
	]);
});


function universal_control_for_tahoma_api_request($server, $endpoint, $payload = null, $session_id = null)
{
	$base_server = !empty($server) ? $server : 'www.tahomalink.com';
	$url = "https://{$base_server}/enduser-mobile-web/enduserAPI/{$endpoint}";

	$args = [
		'method'    => $payload ? 'POST' : 'GET',
		'timeout'   => 30,
		'sslverify' => true,
		'headers'   => ['User-Agent' => 'Mozilla/5.0 WordPress-TaHoma']
	];

	if ($session_id) {
		$args['headers']['Cookie'] = 'JSESSIONID=' . $session_id;
	}

	if ($payload) {
		if ($endpoint === 'login') {
			$args['body'] = ['userId' => $payload['userId'], 'userPassword' => $payload['userPassword']];
		} else {
			$args['headers']['Content-Type'] = 'application/json';
			$args['body'] = json_encode($payload);
		}
	}

	$response = wp_remote_request($url, $args);
	return [
		'body'         => json_decode(wp_remote_retrieve_body($response), true),
		'cookies'      => wp_remote_retrieve_cookies($response),
		'status'       => wp_remote_retrieve_response_code($response),
		'response_raw' => $response
	];
}

function tahoma_get_active_session($server, $email, $password)
{
	$cache_key = 'tahoma_sess_' . md5($email . $server);
	$session = get_transient($cache_key);
	if ($session) return $session;

	$login = universal_control_for_tahoma_api_request($server, 'login', ['userId' => $email, 'userPassword' => $password]);
	if ($login['status'] === 200 && !empty($login['cookies'])) {
		$sid = $login['cookies'][0]->value;
		set_transient($cache_key, $sid, 25 * MINUTE_IN_SECONDS);
		return $sid;
	}
	return false;
}


function tahoma_api_fetch_devices($request)
{
	$p = $request->get_json_params();
	$server = $p['server'] ?? get_option('tahoma_global_server', 'www.tahomalink.com');
	$email = $p['email'] ?? get_option('tahoma_global_email');
	$password = $p['password'] ?? get_option('tahoma_global_password');

	if (empty($email) || empty($password)) {
		return new WP_Error('no_creds', 'Chybí přihlašovací údaje.');
	}

	$cache_key = 'tahoma_dev_list_' . md5($server . $email);
	$cached = get_transient($cache_key);
	if ($cached) return rest_ensure_response(['devices' => $cached]);

	$sid = tahoma_get_active_session($server, $email, $password);
	if (!$sid) return new WP_Error('auth_failed', 'Přihlášení selhalo.', ['status' => 401]);

	$setup = universal_control_for_tahoma_api_request($server, 'setup', null, $sid);
	$devices = $setup['body']['devices'] ?? [];

	if (!empty($devices)) {
		set_transient($cache_key, $devices, HOUR_IN_SECONDS);
	}

	return rest_ensure_response(['devices' => $devices]);
}

function tahoma_api_execute($request)
{
	$p = $request->get_json_params();
	$server   = get_option('tahoma_global_server', 'www.tahomalink.com');
	$email    = get_option('tahoma_global_email');
	$password = get_option('tahoma_global_password');

	$sid = tahoma_get_active_session($server, $email, $password);
	if (!$sid) return new WP_Error('auth_failed', 'Session expired.', ['status' => 401]);

	$final_command = $p['command'];
	$cmd_params = [];


	if (!empty($p['isToggle'])) {
		$device_info = universal_control_for_tahoma_api_request($server, "setup/devices/" . urlencode($p['deviceURL']), null, $sid);
		$states = $device_info['body']['states'] ?? [];
		$is_active = false;
		foreach ($states as $state) {
			if (in_array($state['name'], ['core:OnOffState', 'core:BinaryState', 'core:OpenClosedState'])) {
				if ($state['value'] === 'on' || $state['value'] === 'open' || $state['value'] === 100) {
					$is_active = true;
				}
				break;
			}
		}

		if (strpos($p['uiClass'], 'RollerShutter') !== false || strpos($p['uiClass'], 'Blind') !== false) {
			$final_command = $is_active ? 'close' : 'open';
		} else {
			$final_command = $is_active ? 'off' : 'on';
		}
	} else if ($final_command === 'setClosure') {
		$cmd_params = [(int)$p['value']];
	}

	$action = [
		'deviceURL' => $p['deviceURL'],
		'commands'  => [['name' => $final_command, 'parameters' => $cmd_params]]
	];

	$execution = universal_control_for_tahoma_api_request($server, 'exec/apply', ['actions' => [$action]], $sid);

	if ($execution['status'] === 401) {
		delete_transient('tahoma_sess_' . md5($email . $server));
	}

	return rest_ensure_response($execution['body']);
}

function tahoma_api_get_device_status($request)
{
	$p = $request->get_json_params();

	$server   = get_option('tahoma_global_server', 'www.tahomalink.com');
	$email    = get_option('tahoma_global_email');
	$password = get_option('tahoma_global_password');

	$sid = tahoma_get_active_session($server, $email, $password);
	if (!$sid) return new WP_Error('auth_failed', __('Session expired', 'universal-control-for-tahoma'), ['status' => 401]);

	$res = universal_control_for_tahoma_api_request($server, "setup/devices/" . urlencode($p['deviceURL']), null, $sid);
	$states = $res['body']['states'] ?? [];

	$status_text = __('Unknown status', 'universal-control-for-tahoma');
	$is_on = false;
	$closure = null;
	$found_on_off = false;

	foreach ($states as $state) {
		$n = $state['name'];
		$v = $state['value'];
		if (in_array($n, ['core:OnOffState', 'core:BinaryState', 'core:StatusState', 'core:ActiveControlState'])) {
			$found_on_off = true;
			if ($v === 'on' || $v === 'active' || $v === 1 || $v === true || $v === 'true') $is_on = true;
		}
		if ($n === 'core:ClosureState' || $n === 'core:TargetClosureState') $closure = $v;
		if ($n === 'core:OpenClosedState' && ($v === 'open' || $v === 'partial')) {
			$found_on_off = true;
			$is_on = true;
		}
	}

	if ($found_on_off) {
		if ($closure !== null && $closure > 0 && $closure < 100) {
			// translators: %d is the percentage of device closure.
			$status_text = sprintf(__('Partial: %d%%', 'universal-control-for-tahoma'), $closure);
		} else {
			$status_text = $is_on ? __('On / Open', 'universal-control-for-tahoma') : __('Off / Closed', 'universal-control-for-tahoma');
		}
	} elseif ($closure !== null) {
		$status_text = ($closure == 0) ?
			__('Open', 'universal-control-for-tahoma') : (($closure == 100) ?
				__('Closed', 'universal-control-for-tahoma') :
				// translators: %d is the percentage of device position.
				sprintf(__('Position: %d%%', 'universal-control-for-tahoma'), $closure));
	}

	return rest_ensure_response(['status' => $status_text]);
}
