import { __ } from "@wordpress/i18n";
import { useBlockProps, InspectorControls } from "@wordpress/block-editor";
import {
	PanelBody,
	TextControl,
	SelectControl,
	RangeControl,
	Button,
	Spinner,
	ColorPalette,
	ToggleControl,
} from "@wordpress/components";
import { useState, useEffect } from "@wordpress/element";
import apiFetch from "@wordpress/api-fetch";

export default function Edit({ attributes, setAttributes }) {
	const {
		universal_control_for_tahoma_email,
		universal_control_for_tahoma_password,
		universal_control_for_tahoma_server,
		universal_control_for_tahoma_deviceURL,
		universal_control_for_tahoma_deviceLabel,
		universal_control_for_tahoma_customLabel,
		universal_control_for_tahoma_command,
		universal_control_for_tahoma_value,
		universal_control_for_tahoma_angle,
		universal_control_for_tahoma_bgColor,
		universal_control_for_tahoma_textColor,
		universal_control_for_tahoma_uiClass,
		universal_control_for_tahoma_isToggle,
		universal_control_for_tahoma_width,
		universal_control_for_tahoma_borderWidth,
		universal_control_for_tahoma_borderColor,
		universal_control_for_tahoma_borderRadius,
		universal_control_for_tahoma_fontSize,
	} = attributes;

	const [devices, setDevices] = useState([]);
	const [isLoading, setIsLoading] = useState(false);

	const allowedClasses = [
		"OnOffLight",
		"OnOffPlug",
		"OnOffSmartPlug",
		"OnOffSwitch",
		"SwitchedPlug",
		"PowerOutlet",
		"ToggleLight",
		"DiscreteRSLightingInterface",
		"Light",
		"DimmerLight",
		"RollerShutter",
		"VenetianBlind",
		"GarageDoor",
		"Gate",
	];

	const fetchDevices = async (
		mEmail = universal_control_for_tahoma_email,
		mPass = universal_control_for_tahoma_password,
		mServ = universal_control_for_tahoma_server,
	) => {
		if (!mEmail || !mPass) return;
		setIsLoading(true);
		try {
			const response = await apiFetch({
				path: "/tahoma/v1/fetch-devices",
				method: "POST",
				data: { email: mEmail, password: mPass, server: mServ },
			});
			if (response && response.devices) {
				const filtered = response.devices.filter(
					(d) =>
						(allowedClasses.includes(d.uiClass) ||
							(d.deviceURL &&
								(d.deviceURL.startsWith("io://") ||
									d.deviceURL.startsWith("rts://")))) &&
						d.label,
				);
				setDevices(filtered);
			}
		} catch (error) {
			console.error(error);
		}
		setIsLoading(false);
	};

	useEffect(() => {
		let cEmail = universal_control_for_tahoma_email,
			cPass = universal_control_for_tahoma_password,
			cServ = universal_control_for_tahoma_server;
		if (!universal_control_for_tahoma_email && window.tahomaGlobal?.email) {
			cEmail = window.tahomaGlobal.email;
			cPass = window.tahomaGlobal.password;
			cServ = window.tahomaGlobal.server;
			setAttributes({
				universal_control_for_tahoma_email: cEmail,
				universal_control_for_tahoma_password: cPass,
				universal_control_for_tahoma_server: cServ,
			});
		}
		if (cEmail && cPass) fetchDevices(cEmail, cPass, cServ);
	}, []);

	const getCommandOptions = () => {
		const shutterAndGateClasses = [
			"RollerShutter",
			"VenetianBlind",
			"Awning",
			"ExteriorVenetianBlind",
			"GarageDoor",
			"Gate",
			"SectionalGarageDoor",
		];

		const currentUI = universal_control_for_tahoma_uiClass || "";
		const isShutterOrGate =
			shutterAndGateClasses.includes(currentUI) ||
			currentUI.toLowerCase().includes("shutter") ||
			currentUI.toLowerCase().includes("blind") ||
			currentUI.toLowerCase().includes("door") ||
			currentUI.toLowerCase().includes("gate");

		if (isShutterOrGate) {
			return [
				{ label: __("Open", "universal-control-for-tahoma"), value: "open" },
				{ label: __("Close", "universal-control-for-tahoma"), value: "close" },
				{
					label: __("My Position", "universal-control-for-tahoma"),
					value: "my",
				},
				{ label: __("Stop", "universal-control-for-tahoma"), value: "stop" },
				{
					label: __("Set Position (%)", "universal-control-for-tahoma"),
					value: "setClosure",
				},
			];
		}

		return [
			{ label: __("On", "universal-control-for-tahoma"), value: "on" },
			{ label: __("Off", "universal-control-for-tahoma"), value: "off" },
		];
	};

	return (
		<div
			{...useBlockProps({
				style: {
					backgroundColor: universal_control_for_tahoma_bgColor,
					color: universal_control_for_tahoma_textColor,
					padding: "20px",
					borderRadius: `${universal_control_for_tahoma_borderRadius}px`,
					border: `${universal_control_for_tahoma_borderWidth}px solid ${universal_control_for_tahoma_borderColor}`,
					width: universal_control_for_tahoma_width,
					textAlign: "center",
					fontSize: `${universal_control_for_tahoma_fontSize}px`,
				},
			})}
		>
			<InspectorControls>
				<PanelBody
					title={__("Connection Settings", "universal-control-for-tahoma")}
				>
					<SelectControl
						__nextHasNoMarginBottom
						label={__("Server")}
						value={universal_control_for_tahoma_server}
						options={[
							{ label: "TaHomalink", value: "www.tahomalink.com" },
							{ label: "Overkiz HA1", value: "ha101-1.overkiz.com" },
						]}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_server: val })
						}
					/>
					<TextControl
						__nextHasNoMarginBottom
						label={__("Email")}
						value={universal_control_for_tahoma_email}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_email: val })
						}
					/>
					<TextControl
						__nextHasNoMarginBottom
						label={__("Password")}
						type="password"
						value={universal_control_for_tahoma_password} // OPRAVA: Bylo tu password
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_password: val })
						}
					/>
					<Button
						isPrimary
						onClick={() => fetchDevices()}
						disabled={isLoading}
						style={{
							width: "100%",
							justifyContent: "center",
							marginTop: "10px",
						}}
					>
						{isLoading ? (
							<Spinner />
						) : (
							__("Reload Devices", "universal-control-for-tahoma")
						)}
					</Button>
				</PanelBody>

				<PanelBody title={__("Device Logic", "universal-control-for-tahoma")}>
					<SelectControl
						__nextHasNoMarginBottom
						label={__("Select Device")}
						value={universal_control_for_tahoma_deviceURL}
						options={[
							{ label: __("--- Choose Device ---"), value: "" },
							...devices.map((d) => ({
								label: `${d.label} (${d.uiClass})`,
								value: d.deviceURL,
							})),
						]}
						onChange={(url) => {
							const d = devices.find((dev) => dev.deviceURL === url);
							setAttributes({
								universal_control_for_tahoma_deviceURL: url,
								universal_control_for_tahoma_deviceLabel: d?.label || "",
								universal_control_for_tahoma_uiClass: d?.uiClass || "",
							});
						}}
					/>
					<TextControl
						__nextHasNoMarginBottom
						label={__("Button Label")}
						value={universal_control_for_tahoma_customLabel}
						placeholder={universal_control_for_tahoma_deviceLabel}
						onChange={(val) =>
							setAttributes({
								universal_control_for_tahoma_customLabel: val,
							})
						}
					/>
					<ToggleControl
						__nextHasNoMarginBottom
						label={__("Smart Toggle Mode")}
						checked={universal_control_for_tahoma_isToggle}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_isToggle: val })
						}
					/>

					{!universal_control_for_tahoma_isToggle && (
						<SelectControl
							__nextHasNoMarginBottom
							label={__("Fixed Command")}
							value={universal_control_for_tahoma_command}
							options={getCommandOptions()}
							onChange={(val) =>
								setAttributes({ universal_control_for_tahoma_command: val })
							}
						/>
					)}

					{universal_control_for_tahoma_command === "setClosure" &&
						!universal_control_for_tahoma_isToggle && (
							<RangeControl
								__nextHasNoMarginBottom
								label={__("Position (%)")}
								value={universal_control_for_tahoma_value}
								onChange={(val) =>
									setAttributes({ universal_control_for_tahoma_value: val })
								}
								min={0}
								max={100}
							/>
						)}
				</PanelBody>

				<PanelBody
					title={__("Style Settings", "universal-control-for-tahoma")}
					initialOpen={false}
				>
					<TextControl
						__nextHasNoMarginBottom
						label={__("Width")}
						value={universal_control_for_tahoma_width}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_width: val })
						}
					/>
					<RangeControl
						__nextHasNoMarginBottom
						label={__("Border")}
						value={universal_control_for_tahoma_borderWidth}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_borderWidth: val })
						}
						min={0}
						max={15}
					/>
					<RangeControl
						__nextHasNoMarginBottom
						label={__("Font Size (px)", "universal-control-for-tahoma")}
						value={universal_control_for_tahoma_fontSize}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_fontSize: val })
						}
						min={10}
						max={40}
					/>
					<p>
						<strong>{__("Border Color")}</strong>
					</p>
					<ColorPalette
						value={universal_control_for_tahoma_borderColor}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_borderColor: val })
						}
					/>
					<p>
						<strong>{__("Background")}</strong>
					</p>
					<ColorPalette
						value={universal_control_for_tahoma_bgColor}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_bgColor: val })
						}
					/>
					<p>
						<strong>{__("Text Color")}</strong>
					</p>
					<ColorPalette
						value={universal_control_for_tahoma_textColor}
						onChange={(val) =>
							setAttributes({ universal_control_for_tahoma_textColor: val })
						}
					/>
				</PanelBody>
			</InspectorControls>

			<div style={{ fontWeight: "bold" }}>
				{universal_control_for_tahoma_customLabel ||
					universal_control_for_tahoma_deviceLabel ||
					__("TaHoma Control", "universal-control-for-tahoma")}
			</div>
			{universal_control_for_tahoma_deviceURL && (
				<div style={{ fontSize: "9px", opacity: 0.6, marginTop: "5px" }}>
					{universal_control_for_tahoma_uiClass}
				</div>
			)}
		</div>
	);
}
