<?php
/**
 * Creates a cart with the parameters of the purchase being placed.
 *
 * @package WP_Ultimo
 * @subpackage Order
 * @since 2.0.0
 */

namespace WP_Ultimo\Checkout\Signup_Fields;

use WP_Ultimo\Checkout\Signup_Fields\Base_Signup_Field;

// Exit if accessed directly
defined('ABSPATH') || exit;

/**
 * Creates an cart with the parameters of the purchase being placed.
 *
 * @package WP_Ultimo
 * @subpackage Checkout
 * @since 2.0.0
 */
class Signup_Field_Username extends Base_Signup_Field {

	/**
	 * Returns the type of the field.
	 *
	 * @since 2.0.0
	 * @return string
	 */
	public function get_type() {

		return 'username';
	}

	/**
	 * Returns if this field should be present on the checkout flow or not.
	 *
	 * @since 2.0.0
	 * @return boolean
	 */
	public function is_required() {

		return true;
	}

	/**
	 * Is this a user-related field?
	 *
	 * If this is set to true, this field will be hidden
	 * when the user is already logged in.
	 *
	 * @since 2.0.0
	 * @return boolean
	 */
	public function is_user_field() {

		return true;
	}

	/**
	 * Requires the title of the field/element type.
	 *
	 * This is used on the Field/Element selection screen.
	 *
	 * @since 2.0.0
	 * @return string
	 */
	public function get_title() {

		return __('Username', 'ultimate-multisite');
	}

	/**
	 * Returns the description of the field/element.
	 *
	 * This is used as the title attribute of the selector.
	 *
	 * @since 2.0.0
	 * @return string
	 */
	public function get_description() {

		return __('Adds an username field. This username will be used to create the WordPress user.', 'ultimate-multisite');
	}

	/**
	 * Returns the tooltip of the field/element.
	 *
	 * This is used as the tooltip attribute of the selector.
	 *
	 * @since 2.0.0
	 * @return string
	 */
	public function get_tooltip() {

		return __('Adds an username field. This username will be used to create the WordPress user.', 'ultimate-multisite');
	}

	/**
	 * Returns the icon to be used on the selector.
	 *
	 * Can be either a dashicon class or a wu-dashicon class.
	 *
	 * @since 2.0.0
	 * @return string
	 */
	public function get_icon() {

		return 'dashicons-wu-user1';
	}

	/**
	 * Returns the default values for the field-elements.
	 *
	 * This is passed through a wp_parse_args before we send the values
	 * to the method that returns the actual fields for the checkout form.
	 *
	 * @since 2.0.0
	 * @return array
	 */
	public function defaults() {

		return [
			'auto_generate_username'       => false,
			'enable_inline_login_username' => false,
		];
	}

	/**
	 * List of keys of the default fields we want to display on the builder.
	 *
	 * @since 2.0.0
	 * @return array
	 */
	public function default_fields() {

		return [
			'name',
			'placeholder',
			'tooltip',
		];
	}

	/**
	 * If you want to force a particular attribute to a value, declare it here.
	 *
	 * @since 2.0.0
	 * @return array
	 */
	public function force_attributes() {

		return [
			'id'       => 'username',
			'required' => true,
		];
	}

	/**
	 * Returns the list of additional fields specific to this type.
	 *
	 * @since 2.0.0
	 * @return array
	 */
	public function get_fields() {

		return [
			'auto_generate_username'       => [
				'type'      => 'toggle',
				'title'     => __('Auto-generate', 'ultimate-multisite'),
				'desc'      => __('Check this option to auto-generate this field based on the email address of the customer.', 'ultimate-multisite'),
				'tooltip'   => '',
				'value'     => 0,
				'html_attr' => [
					'v-model' => 'auto_generate_username',
				],
			],
			'enable_inline_login_username' => [
				'type'  => 'toggle',
				'title' => __('Enable Inline Login', 'ultimate-multisite'),
				'desc'  => __('When enabled, users entering an existing username will see an inline login prompt to authenticate with their password without leaving the page.', 'ultimate-multisite'),
				'value' => 1,
			],
		];
	}

	/**
	 * Returns the field/element actual field array to be used on the checkout form.
	 *
	 * @since 2.0.0
	 *
	 * @param array $attributes Attributes saved on the editor form.
	 * @return array An array of fields, not the field itself.
	 */
	public function to_fields_array($attributes) {
		/*
		 * Logged in user, bail.
		 */
		if (is_user_logged_in()) {
			return [];
		}

		if (isset($attributes['auto_generate_username']) && $attributes['auto_generate_username']) {
			return [
				'auto_generate_username' => [
					'type'  => 'hidden',
					'id'    => 'auto_generate_username',
					'value' => 'email',
				],
				'username'               => [
					'type'  => 'hidden',
					'id'    => 'username',
					'value' => uniqid(),
				],
			];
		}

		$fields = [
			'username' => [
				'type'              => 'text',
				'id'                => 'username',
				'name'              => $attributes['name'],
				'placeholder'       => $attributes['placeholder'],
				'tooltip'           => $attributes['tooltip'],
				'wrapper_classes'   => wu_get_isset($attributes, 'wrapper_element_classes', ''),
				'classes'           => wu_get_isset($attributes, 'element_classes', ''),
				'required'          => true,
				'value'             => $this->get_value(),
				'html_attr'         => [
					'v-model'         => 'username',
					'v-init:username' => "'{$this->get_value()}'",
					'autocomplete'    => 'username',
					'@blur'           => "check_user_exists_debounced('username', username)",
				],
				'wrapper_html_attr' => [
					'style' => $this->calculate_style_attr(),
				],
			],
		];

		if (wu_get_isset($attributes, 'enable_inline_login_username', false)) {
			$fields['username_inline_login_prompt'] = [
				'type'              => 'html',
				'id'                => 'username_inline_login_prompt',
				'content'           => [$this, 'render_inline_login_prompt'],
				'wrapper_classes'   => '',
				'wrapper_html_attr' => [
					'v-if'    => "show_login_prompt && login_prompt_field === 'username'",
					'v-cloak' => true,
				],
			];
		}

		return $fields;
	}

	/**
	 * Renders the inline login prompt HTML.
	 *
	 * @since 2.0.20
	 * @return string
	 */
	public function render_inline_login_prompt(): string {
		return wu_get_template_contents('checkout/partials/inline-login-prompt', ['field_type' => 'username']);
	}
}
