<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @since      1.0.0
 * @package    Ukrainian_Cyrillic_To_Latin
 * @subpackage Ukrainian_Cyrillic_To_Latin/public
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for
 * transliterating content in the public-facing side of the site.
 *
 * @package    Ukrainian_Cyrillic_To_Latin
 * @subpackage Ukrainian_Cyrillic_To_Latin/public
 */
class Ukrainian_Cyrillic_To_Latin_Public {

    /**
     * Transliterate Ukrainian text to Latin.
     *
     * @since    1.0.0
     * @param    string    $text    Text to transliterate.
     * @return   string             Transliterated text.
     */
    public function transliterate($text) {
        $ukrainian_chars = array(
            'а' => 'a', 'б' => 'b', 'в' => 'v', 'г' => 'h', 'ґ' => 'g',
            'д' => 'd', 'е' => 'e', 'є' => 'ie', 'ж' => 'zh', 'з' => 'z',
            'и' => 'y', 'і' => 'i', 'ї' => 'yi', 'й' => 'i', 'к' => 'k',
            'л' => 'l', 'м' => 'm', 'н' => 'n', 'о' => 'o', 'п' => 'p',
            'р' => 'r', 'с' => 's', 'т' => 't', 'у' => 'u', 'ф' => 'f',
            'х' => 'kh', 'ц' => 'ts', 'ч' => 'ch', 'ш' => 'sh', 'щ' => 'shch',
            'ь' => '', 'ю' => 'iu', 'я' => 'ia',
            
            // Uppercase
            'А' => 'A', 'Б' => 'B', 'В' => 'V', 'Г' => 'H', 'Ґ' => 'G',
            'Д' => 'D', 'Е' => 'E', 'Є' => 'Ie', 'Ж' => 'Zh', 'З' => 'Z',
            'И' => 'Y', 'І' => 'I', 'Ї' => 'Yi', 'Й' => 'I', 'К' => 'K',
            'Л' => 'L', 'М' => 'M', 'Н' => 'N', 'О' => 'O', 'П' => 'P',
            'Р' => 'R', 'С' => 'S', 'Т' => 'T', 'У' => 'U', 'Ф' => 'F',
            'Х' => 'Kh', 'Ц' => 'Ts', 'Ч' => 'Ch', 'Ш' => 'Sh', 'Щ' => 'Shch',
            'Ь' => '', 'Ю' => 'Iu', 'Я' => 'Ia',
        );
        
        // Get options
        $options = get_option('ukrainian_cyrillic_to_latin_settings', array(
            'remove_apostrophes' => 1,
            'convert_to_lowercase' => 1
        ));
        
        // Replace Ukrainian characters with Latin equivalents
        $text = str_replace(array_keys($ukrainian_chars), array_values($ukrainian_chars), $text);
        
        // Remove apostrophes if option is enabled
        if (!empty($options['remove_apostrophes'])) {
            $text = str_replace("'", '', $text);
        }
        
        // Convert to lowercase if option is enabled
        if (!empty($options['convert_to_lowercase'])) {
            $text = strtolower($text);
        }
        
        return $text;
    }
    
    /**
     * Transliterate post title.
     *
     * @since    1.0.0
     * @param    string    $title    The title to transliterate.
     * @return   string              Transliterated title.
     */
    public function transliterate_title($title) {
        return $this->transliterate($title);
    }
    
    /**
     * Transliterate file name.
     *
     * @since    1.0.0
     * @param    string    $filename    The filename to transliterate.
     * @return   string                 Transliterated filename.
     */
    public function transliterate_filename($filename) {
        return $this->transliterate($filename);
    }
    
    /**
     * Transliterate term slug.
     *
     * @since    1.0.0
     * @param    string    $slug        The term slug to transliterate.
     * @param    string    $taxonomy    The taxonomy slug.
     * @return   string                 Transliterated term slug.
     */
    public function transliterate_term_slug($slug, $taxonomy) {
        return $this->transliterate($slug);
    }
    
    /**
     * Transliterate attachment title.
     *
     * @since    1.0.0
     * @param    array    $data        An array of slashed, sanitized, and processed attachment post data.
     * @param    array    $postarr     An array of slashed and sanitized attachment post data, but not processed.
     * @return   array                 Modified post data with transliterated title.
     */
    public function transliterate_attachment_title($data, $postarr) {
        if (isset($data['post_title'])) {
            $data['post_title'] = $this->transliterate($data['post_title']);
        }
        
        return $data;
    }
}
