<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 * @package    Ukrainian_Cyrillic_To_Latin
 * @subpackage Ukrainian_Cyrillic_To_Latin/admin
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two hooks for
 * enqueuing the admin-specific stylesheet and JavaScript.
 *
 * @package    Ukrainian_Cyrillic_To_Latin
 * @subpackage Ukrainian_Cyrillic_To_Latin/admin
 */
class Ukrainian_Cyrillic_To_Latin_Admin {

    /**
     * Add settings page to WordPress admin.
     *
     * @since    1.0.0
     */
    public function add_settings_page() {
        add_options_page(
            __('Ukrainian Cyrillic to Latin by OSTG', 'ukrainian-cyrillic-to-latin-by-ostg'),
            __('Ukrainian Cyrillic to Latin', 'ukrainian-cyrillic-to-latin-by-ostg'),
            'manage_options',
            'ukrainian-cyrillic-to-latin-by-ostg',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Register plugin settings.
     *
     * @since    1.0.0
     */
    public function register_settings() {
        register_setting(
            'ukrainian_cyrillic_to_latin_by_ostg_group',
            'ukrainian_cyrillic_to_latin_settings',
            array($this, 'sanitize_settings')
        );
        
        add_settings_section(
            'ukrainian_cyrillic_to_latin_by_ostg_section',
            __('Transliteration Settings', 'ukrainian-cyrillic-to-latin-by-ostg'),
            array($this, 'render_section_info'),
            'ukrainian-cyrillic-to-latin-by-ostg'
        );
        
        add_settings_field(
            'remove_apostrophes',
            __('Remove Apostrophes', 'ukrainian-cyrillic-to-latin-by-ostg'),
            array($this, 'render_remove_apostrophes_field'),
            'ukrainian-cyrillic-to-latin-by-ostg',
            'ukrainian_cyrillic_to_latin_by_ostg_section'
        );
        
        add_settings_field(
            'convert_to_lowercase',
            __('Convert to Lowercase', 'ukrainian-cyrillic-to-latin-by-ostg'),
            array($this, 'render_convert_to_lowercase_field'),
            'ukrainian-cyrillic-to-latin-by-ostg',
            'ukrainian_cyrillic_to_latin_by_ostg_section'
        );
    }
    
    /**
     * Sanitize settings.
     *
     * @since    1.0.0
     * @param    array    $input    Settings input.
     * @return   array              Sanitized settings.
     */
    public function sanitize_settings($input) {
        $new_input = array();
        
        if (isset($input['remove_apostrophes'])) {
            $new_input['remove_apostrophes'] = absint($input['remove_apostrophes']);
        }
        
        if (isset($input['convert_to_lowercase'])) {
            $new_input['convert_to_lowercase'] = absint($input['convert_to_lowercase']);
        }
        
        return $new_input;
    }
    
    /**
     * Render settings page.
     *
     * @since    1.0.0
     */
    public function render_settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('ukrainian_cyrillic_to_latin_by_ostg_group');
                do_settings_sections('ukrainian-cyrillic-to-latin-by-ostg');
                submit_button();
                ?>
            </form>
            
            <div class="card">
                <h2><?php esc_html_e('About this Plugin', 'ukrainian-cyrillic-to-latin-by-ostg'); ?></h2>
                <p><?php esc_html_e('This plugin automatically transliterates Ukrainian Cyrillic characters to Latin characters, improving SEO and URL readability.', 'ukrainian-cyrillic-to-latin-by-ostg'); ?></p>
                <p><?php printf(
                    /* translators: %s: OSTG website URL */
                    esc_html__('Plugin developed by %s', 'ukrainian-cyrillic-to-latin-by-ostg'),
                    '<a href="https://ostg.blog" target="_blank">OSTG</a>'
                ); ?></p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render section info.
     *
     * @since    1.0.0
     */
    public function render_section_info() {
        echo '<p>' . esc_html__('Configure how Ukrainian text should be transliterated to Latin characters.', 'ukrainian-cyrillic-to-latin-by-ostg') . '</p>';
    }
    
    /**
     * Render remove apostrophes field.
     *
     * @since    1.0.0
     */
    public function render_remove_apostrophes_field() {
        $options = get_option('ukrainian_cyrillic_to_latin_settings', array('remove_apostrophes' => 1));
        ?>
        <input type="checkbox" id="remove_apostrophes" name="ukrainian_cyrillic_to_latin_settings[remove_apostrophes]" value="1" <?php checked(1, isset($options['remove_apostrophes']) ? $options['remove_apostrophes'] : 0); ?> />
        <label for="remove_apostrophes"><?php echo esc_html__('Remove apostrophes from transliterated text', 'ukrainian-cyrillic-to-latin-by-ostg'); ?></label>
        <?php
    }
    
    /**
     * Render convert to lowercase field.
     *
     * @since    1.0.0
     */
    public function render_convert_to_lowercase_field() {
        $options = get_option('ukrainian_cyrillic_to_latin_settings', array('convert_to_lowercase' => 1));
        ?>
        <input type="checkbox" id="convert_to_lowercase" name="ukrainian_cyrillic_to_latin_settings[convert_to_lowercase]" value="1" <?php checked(1, isset($options['convert_to_lowercase']) ? $options['convert_to_lowercase'] : 0); ?> />
        <label for="convert_to_lowercase"><?php echo esc_html__('Convert transliterated text to lowercase', 'ukrainian-cyrillic-to-latin-by-ostg'); ?></label>
        <?php
    }
    
    /**
     * Display admin notices.
     *
     * @since    1.0.0
     */
    public function admin_notices() {
        // Check if we're on the plugin's settings page
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'ukrainian-cyrillic-to-latin-by-ostg') === false) {
            return;
        }
        
        // Verify nonce before checking for settings-updated
        if (isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true') {
            // Check nonce for security
            if (isset($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'ukrainian_cyrillic_to_latin_by_ostg_group-options')) {
                add_settings_error(
                    'ukrainian_cyrillic_to_latin_messages',
                    'ukrainian_cyrillic_to_latin_message',
                    __('Settings saved successfully.', 'ukrainian-cyrillic-to-latin-by-ostg'),
                    'updated'
                );
            }
        }
        
        settings_errors('ukrainian_cyrillic_to_latin_messages');
    }
}
