<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
class TwoStB_Step_Email extends TwoStB_Step_Provider {
    const TOKEN_META_KEY = '_twoStB_step_factor_email_token';
	const TOKEN_META_KEY_TIMESTAMP = '_twoStB_step_factor_email_token_timestamp';
	const INPUT_NAME_RESEND_CODE = '_twoSt_step_factor_email_code_resend';

    public function __construct() {
		add_action( 'twoStB_factor_user_options_' . __CLASS__, array( $this, 'user_options' ) );
		parent::__construct();
	}
    
    public function get_label() {
		return _x( 'Email', 'Provider Label', 'two-step-factor-by-se' );
	}
	
	public function get_alternative_provider_label() {
		return _x( 'Try Email', 'Provider Label', 'two-step-factor-by-se' );
	}
	
    private function get_token_length() {
		$token_length = (int) apply_filters( 'twoStB_factor_email_token_length', 8 );
		return $token_length;
	}
    
    public function re_generate_email_token( $user ){
        $timestamp = get_user_meta( $user->ID, self::TOKEN_META_KEY_TIMESTAMP, true );
        if ( abs(time() - $timestamp) > 60 ) {
            $this->generate_and_email_token( $user );
            return ['status' => 1, 'msg' => 'Code sent.'];
        }else{
            return ['status' => 0, 'msg' => 'Please wait for 1 minute to resend code.'];
        }
    }
    
    public function generate_and_email_token( $user ) {
		$token = $this->generate_token( $user->ID );

        $subject = wp_strip_all_tags( sprintf( 
            /* translators: %s is the site name */
            __( 'Your login confirmation code for %s', 'two-step-factor-by-se' ),
            wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES ) 
        ) );
        /* translators: %s is the login confirmation token */
		$message = wp_strip_all_tags( sprintf( __( 'Enter %s to log in.', 'two-step-factor-by-se' ), $token ) );
		$subject = apply_filters( 'twoStB_factor_token_email_subject', $subject, $user->ID );
		$message = apply_filters( 'twoStB_factor_token_email_message', $message, $token, $user->ID );
		return wp_mail( $user->user_email, $subject, $message );
	}

    public function generate_token( $user_id ) {
		$token = $this->get_code( $this->get_token_length() );

		update_user_meta( $user_id, self::TOKEN_META_KEY_TIMESTAMP, time() );
		update_user_meta( $user_id, self::TOKEN_META_KEY, wp_hash( $token ) );

		return $token;
	}

    public function user_options( $user ) {
		$email = $user->user_email;
		?>
		<div>
			<?php
			echo esc_html(
				sprintf(
				/* translators: %s: email address */
					__( 'Authentication codes will be sent to %s.', 'two-step-factor-by-se' ),
					$email
				)
			);
			?>
		</div>
		<?php
	}

    public function is_available_for_user( $user ) {
		return true;
	}

    public function validate_token( $user_id, $token ) {
		$hashed_token = $this->get_user_token( $user_id );
		if ( empty( $hashed_token ) || ! hash_equals( wp_hash( $token ), $hashed_token ) ) {
			return ['status' => 0, 'error' => 'Code is invalid.'];
		}
		if ( $this->user_token_has_expired( $user_id ) ) {
			return ['status' => 0, 'error' => 'Code is expired.'];
		}
		$this->delete_token( $user_id );

		return ['status' => 1];
	}
    
    public function delete_token( $user_id ) {
		delete_user_meta( $user_id, self::TOKEN_META_KEY );
	}
	
    public function user_has_token( $user_id ) {
		$hashed_token = $this->get_user_token( $user_id );

		if ( ! empty( $hashed_token ) ) {
			return true;
		}

		return false;
	}

    public function get_user_token( $user_id ) {
		$hashed_token = get_user_meta( $user_id, self::TOKEN_META_KEY, true );
		if ( ! empty( $hashed_token ) && is_string( $hashed_token ) ) {
			return $hashed_token;
		}
		return false;
	}
    
    public function user_token_has_expired( $user_id ) {
		$token_lifetime = $this->user_token_lifetime( $user_id );
		$token_ttl      = $this->user_token_ttl( $user_id );

		if ( is_int( $token_lifetime ) && $token_lifetime <= $token_ttl ) {
			return false;
		}

		return true;
	}

    public function user_token_ttl( $user_id ) {
		$token_ttl = 15 * MINUTE_IN_SECONDS;
		$token_ttl = (int) apply_filters_deprecated( 'twoStB_factor_token_ttl', array( $token_ttl, $user_id ), '0.1', 'twoStB_factor_email_token_ttl' );
		return (int) apply_filters( 'twoStB_factor_email_token_ttl', $token_ttl, $user_id );
	}

    public function user_token_lifetime( $user_id ) {
		$timestamp = intval( get_user_meta( $user_id, self::TOKEN_META_KEY_TIMESTAMP, true ) );
		if ( ! empty( $timestamp ) ) {
			return time() - $timestamp;
		}
		return null;
	}
	
    public function authentication_page( $user ) {
		if ( ! $user ) {
			return;
		}

		if ( ! $this->user_has_token( $user->ID ) || $this->user_token_has_expired( $user->ID ) ) {
			$this->generate_and_email_token( $user );
		}
        
		$token_length = $this->get_token_length();
		$token_placeholder = str_repeat( 'X', $token_length );

		require_once ABSPATH . '/wp-admin/includes/template.php';
		?>
		<p class="two-factor-prompt"><?php esc_html_e( 'A verification code has been sent to the email address associated with your account.', 'two-step-factor-by-se' ); ?></p>
		<p>
			<label for="authcode"><?php esc_html_e( 'Verification Code:', 'two-step-factor-by-se' ); ?></label>
			<input type="text" inputmode="numeric" name="two-provider-auth-code" id="authcode" class="input authcode" value="" size="20" pattern="[0-9 ]*" autocomplete="one-time-code" placeholder="<?php echo esc_attr( $token_placeholder ); ?>" data-digits="<?php echo esc_attr( $token_length ); ?>" />
			<?php submit_button( __( 'Log In', 'two-step-factor-by-se' ) ); ?>
		</p>
        <p class="two-factor-se-email-resend">
            <input type="submit" class="button" name="<?php echo esc_attr( self::INPUT_NAME_RESEND_CODE ); ?>" value="<?php esc_attr_e( 'Resend Code', 'two-step-factor-by-se' ); ?>" />
        </p>
		<script type="text/javascript">
			setTimeout( function(){
				var d;
				try{
					d = document.getElementById('authcode');
					d.value = '';
					d.focus();
				} catch(e){}
			}, 200);
		</script>
		<?php
	}

	public static function sanitize_code_from_request( $field, $length = 0 ) {
		if ( empty( $field ) ) {
			return false;
		}
		$code = wp_unslash( $field );
		$code = preg_replace( '/\s+/', '', $code );

		if ( $length && strlen( $code ) !== $length ) {
			return false;
		}
		return (string) $code;
	}
}
