<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
class TwoStB_Step_Factor_Core {
    const ENABLED_PROVIDERS_USER_META_KEY = '_twoStB_step_factor_enabled_providers';
    const PROVIDER_USER_META_KEY = '_twoStB_step_factor_provider';
	const USER_META_NONCE_KEY = '_twoStB_step_factor_nonce';
	const USER_RATE_LIMIT_KEY = '_twoStB_step_factor_last_login_failure';
	const USER_FAILED_LOGIN_ATTEMPTS_KEY = '_twoStB_step_factor_failed_login_attempts';
	const USER_PASSWORD_WAS_RESET_KEY = '_twoStB_step_factor_password_was_reset';
	const USER_SETTINGS_ACTION_QUERY_VAR = 'twoStB_step_factor_action';
	const USER_SETTINGS_ACTION_NONCE_QUERY_ARG = '_twoStB_step_factor_action_nonce';
	const USER_DUMMY_CODE = '_twoStB_step_factor_dummy_code';
	const USER_BACKUPS_CODE = '_twoStB_step_factor_backup_code';

    private static $password_auth_tokens = array();

    public function __construct() {
        add_action('init', array($this, 'initialize_plugin'));
    }

    public function initialize_plugin(){
        add_action('wp_login', array($this, 'wp_login'), 10, 2);
        add_filter( 'wp_login_errors', array( $this, 'maybe_show_reset_password_notice' ) );
        add_action('show_user_profile', array($this, 'twoStepFactorDisplay'));
        add_action('edit_user_profile', array($this, 'twoStepFactorDisplay'));
        
        add_action('personal_options_update', array($this, 'user_twoStepFactoroptions_update'));
        add_action('edit_user_profile_update', array($this, 'user_twoStepFactoroptions_update'));
        
        add_action( 'login_form_validate_2fa', array($this, 'login_form_validate_2fa' ), 20 );
		add_action( 'login_form_revalidate_2fa_form', array($this, 'login_form_revalidate_2fa' ) );
		
		add_action( 'login_form_alter_validate_2fa', array($this, 'login_form_alter_validate_2fa' ), 20 );
		
		add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_script') );
		add_action('wp_ajax_update_backup_codes', array($this, 'ajax_update_backup_codes') );

    }
    
    function enqueue_admin_script() {
        wp_enqueue_script('backup-codes-js', TWOSTB_STEP_FACTOR_URL . 'assests/js/admin.js', ['jquery'], '1.1.0', true);
        
        // Pass AJAX URL and nonce
        wp_localize_script('backup-codes-js', 'updateBackupCodesNonce', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('update_backup_codes_nonce')
        ]);
    }


    public static function is_user_using_twoStB_factor( $user = null ) {
		$provider = self::get_primary_provider_for_user( $user );
		return ! empty( $provider );
	}

    public static function wp_login( $user_login, $user ) {
        if ( ! self::is_user_using_twoStB_factor( $user->ID ) ) {
            return;
        }
    
        self::destroy_current_session_for_user( $user );
        wp_clear_auth_cookie();
        $login_nonce =  wp_create_nonce('two_stb_factor_login_action');
        self::show_twoStB_factor_login( $user, $login_nonce );
        exit;
    }

    public static function destroy_current_session_for_user( $user ) {
		$session_manager = WP_Session_Tokens::get_instance( $user->ID );

		foreach ( self::$password_auth_tokens as $auth_token ) {
			$session_manager->destroy( $auth_token );
		}
	}

    public static function get_available_providers_for_user( $user = null ) {
		$user = self::fetch_user( $user );
		if ( ! $user ) {
			return array();
		}
		$providers            = self::get_providers();
		$enabled_providers    = self::get_enabled_providers_for_user( $user );
		$configured_providers = array();
		foreach ( $providers as $provider_key => $provider ) {
			if ( in_array( $provider_key, $enabled_providers, true ) && $provider->is_available_for_user( $user ) ) {
				$configured_providers[ $provider_key ] = $provider;
			}
		}
		return $configured_providers;
	}

    protected static function get_user_settings_page_url( $user_id ) {
		if ( defined( 'IS_PROFILE_PAGE' ) && IS_PROFILE_PAGE ) {
			return self_admin_url( 'profile.php' );
		}

		return add_query_arg(
			array(
				'user_id' => intval( $user_id ),
			),
			self_admin_url( 'user-edit.php' )
		);
	}

    public static function get_user_twoStB_factor_revalidate_url( $interim = false ) {
		$args = array(
			'action' => 'revalidate_2fa',
		);
		if ( $interim ) {
			$args['interim-login'] = 1;
		}

		return self::login_url( $args );
	}

    public static function twoStepFactorDisplay( $user ) {
		$notices = [];

		wp_enqueue_style( 'user-edit-2stepfa', plugins_url( 'user-edit.css', __FILE__ ), array(), TWOSTB_STEP_FACTOR_VERSION );

		$enabled_providers = array_keys( self::get_available_providers_for_user( $user ) );
		$primary_provider  = self::get_primary_provider_for_user( $user->ID );
		$primary_provider_key = null;
		if ( ! empty( $primary_provider ) && is_object( $primary_provider ) ) {
			$primary_provider_key = $primary_provider->get_key();
		}

		if ( !is_admin() ) {
			$url = add_query_arg(
				'redirect_to',
				urlencode( self::get_user_settings_page_url( $user->ID ) . '#two-factor-options' ),
				self::get_user_twoStB_factor_revalidate_url()
			);

			$notices['warning two-factor-warning-revalidate-session'] = sprintf(
					esc_html__( 'To update your Two-Factor options, you must first revalidate your session.', 'two-step-factor-by-se' ) .
					' <a class="button" href="%s">' . esc_html__( 'Revalidate now', 'two-step-factor-by-se' ) . '</a>',
					esc_url( $url )
			);
		}

		printf(
			'<fieldset id="two-factor-options" %s>',
			is_admin() ? '' : 'disabled="disabled"'
		);

		// if ( 1 === count( $enabled_providers ) ) {
		// 	$notices['warning two-factor-warning-suggest-backup'] = esc_html__( 'To prevent being locked out of your account, consider enabling a backup method like Recovery Codes in case you lose access to your primary authentication method.', 'two-step-factor-by-se' );
		// }
		?>
		<h2><?php esc_html_e( 'Two-Factor Options', 'two-step-factor-by-se' ); ?></h2>
		<?php foreach ( $notices as $notice_type => $notice ) : ?>
		<div class="<?php echo esc_attr( $notice_type ? 'notice inline notice-' . $notice_type : '' ); ?>">
			<p><?php echo wp_kses_post( $notice ); ?></p>
		</div>
		<?php endforeach; ?>
		<p>
			<?php  esc_html_e( 'Configure a primary two-factor method along with a backup method, such as Recovery Codes, to avoid being locked out if you lose access to your primary method.', 'two-step-factor-by-se' ); ?>
		</p>
		<?php wp_nonce_field( 'user_twoStB_factor_options', '_nonce_user_twoStB_factor_options', false ); ?>
		<input type="hidden" name="<?php echo esc_attr( self::ENABLED_PROVIDERS_USER_META_KEY ); ?>[]" value="<?php /* Dummy input so $_POST value is passed when no providers are enabled. */ ?>" />
		<table class="wp-list-table widefat fixed striped table-view-list two-factor-methods-table">
			<thead>
				<tr>
					<th class="col-enabled" scope="col"><?php esc_html_e( 'Enabled', 'two-step-factor-by-se' ); ?></th>
					<th class="col-primary" scope="col"><?php esc_html_e( 'Primary', 'two-step-factor-by-se' ); ?></th>
					<th class="col-name" scope="col"><?php esc_html_e( 'Type', 'two-step-factor-by-se' ); ?></th>
				</tr>
			</thead>
			<tbody>
			<?php foreach ( self::get_providers() as $provider_key => $object ) : ?>
				<tr>
					<th scope="row"><input id="enabled-<?php echo esc_attr( $provider_key ); ?>" type="checkbox" name="<?php echo esc_attr( self::ENABLED_PROVIDERS_USER_META_KEY ); ?>[]" value="<?php echo esc_attr( $provider_key ); ?>" <?php checked( in_array( $provider_key, $enabled_providers, true ) ); ?> /></th>
					<th scope="row"><input type="radio" name="<?php echo esc_attr( self::PROVIDER_USER_META_KEY ); ?>" value="<?php echo esc_attr( $provider_key ); ?>" <?php checked( $provider_key, $primary_provider_key ); ?> /></th>
					<td>
						<label class="two-factor-method-label" for="enabled-<?php echo esc_attr( $provider_key ); ?>"><?php echo esc_html( $object->get_label() ); ?></label>
						<?php
						do_action_deprecated( 'two-factor-user-options-' . $provider_key, array( $user ), '0.7.0', 'twoStB_factor_user_options_' . $provider_key );
						do_action( 'twoStB_factor_user_options_' . $provider_key, $user );
						?>
					</td>
				</tr>
			<?php endforeach; ?>
			</tbody>
			<tfoot>
				<tr>
					<th class="col-enabled" scope="col"><?php esc_html_e( 'Enabled', 'two-step-factor-by-se' ); ?></th>
					<th class="col-primary" scope="col"><?php esc_html_e( 'Primary', 'two-step-factor-by-se' ); ?></th>
					<th class="col-name" scope="col"><?php esc_html_e( 'Type', 'two-step-factor-by-se' ); ?></th>
				</tr>
			</tfoot>
		</table>
		</fieldset>
		<?php
		do_action( 'twoStB_show_user_security_settings', $user );
	}

    private static function get_default_providers() {
		return array(
			'TwoStB_Step_Email'              => TWOSTB_STEP_FACTOR_DIR . 'providers/class-two-step-factor-email.php',
			'TwoStB_Step_Factor_Dummy'       => TWOSTB_STEP_FACTOR_DIR . 'providers/class-two-step-factor-dummy.php',
			'TwoStB_Step_Factor_BackupCodes' => TWOSTB_STEP_FACTOR_DIR . 'providers/class-two-step-factor-backupcodes.php',
		);
	}

    private static function get_providers() {
		$providers = self::get_default_providers();
		$providers = self::get_providers_classes( $providers );
        foreach ( $providers as $provider_key => $provider_class ) {
			try {
				$providers[ $provider_key ] = call_user_func( array( $provider_class, 'get_instance' ) );
			} catch ( Exception $e ) {
				unset( $providers[ $provider_key ] );
			}
		}
		return $providers;
	}

    private static function get_providers_classes( $providers ) {
		foreach ( $providers as $provider_key => $path ) {
			if ( ! empty( $path ) && is_readable( $path ) ) {
				require_once $path;
			}
			$class = $provider_key;
			$class = apply_filters( "twoStB_factor_provider_classname_{$provider_key}", $class, $path );
// 			if($provider_key != 'TwoStB_Step_Email'){
// 			   prd($class); 
// 			}
			if ( class_exists( $class ) ) {
				$providers[ $provider_key ] = $class;
			} else {
				unset( $providers[ $provider_key ] );
			}
		}
		return $providers;
	}

    public static function fetch_user( $user = null ) {
		if ( null === $user ) {
			$user = wp_get_current_user();
		} elseif ( ! ( $user instanceof WP_User ) ) {
			$user = get_user_by( 'id', $user );
		}
		if ( ! $user || ! $user->exists() ) {
			return false;
		}
		return $user;
	}

    public static function get_primary_provider_for_user( $user = null ) {
		$user = self::fetch_user( $user );
		if ( ! $user ) {
			return null;
		}

		$providers           = self::get_providers();
		$available_providers = self::get_available_providers_for_user( $user );
		if ( empty( $available_providers ) ) {
			return null;
		} elseif ( 1 === count( $available_providers ) ) {
			$provider = key( $available_providers );
		} else {
			$provider = get_user_meta( $user->ID, self::PROVIDER_USER_META_KEY, true );
			if ( ! isset( $available_providers[ $provider ] ) ) {
				$provider = key( $available_providers );
			}
		}
		$provider = apply_filters( 'twoStB_factor_primary_provider_for_user', $provider, $user->ID );
		if ( isset( $providers[ $provider ] ) ) {
			return $providers[ $provider ];
		}
		return null;
	}

    public static function user_twoStepFactoroptions_update( $user_id ) {
		if ( isset( $_POST['_nonce_user_twoStB_factor_options'] ) ) {
			check_admin_referer( 'user_twoStB_factor_options', '_nonce_user_twoStB_factor_options' );
            
            if ( ! current_user_can( 'edit_user', $user_id ) ) {
                return;
            }
            
			if ( ! isset( $_POST[ self::ENABLED_PROVIDERS_USER_META_KEY ] ) ||
					! is_array( $_POST[ self::ENABLED_PROVIDERS_USER_META_KEY ] ) ) {
				return;
			}
			
			$providers          = self::get_providers();
			$enabled_providers = isset($_POST[self::ENABLED_PROVIDERS_USER_META_KEY]) && is_array($_POST[self::ENABLED_PROVIDERS_USER_META_KEY]) 
                    ? array_map('sanitize_text_field', wp_unslash($_POST[self::ENABLED_PROVIDERS_USER_META_KEY])) 
                    : [];
            // $enabled_providers = sanitize_text_field( wp_unslash( $_POST[self::ENABLED_PROVIDERS_USER_META_KEY]) );
			$existing_providers = self::get_enabled_providers_for_user( $user_id );
			$enabled_providers = array_intersect( $enabled_providers, array_keys( $providers ) );
			update_user_meta( $user_id, self::ENABLED_PROVIDERS_USER_META_KEY, $enabled_providers );
			
			$new_provider = isset( $_POST[ self::PROVIDER_USER_META_KEY ] ) ? sanitize_text_field( wp_unslash( $_POST[ self::PROVIDER_USER_META_KEY ] ) ) : '';
            if ( ! empty( $new_provider ) && in_array( $new_provider, $enabled_providers, true ) ) {
				update_user_meta( $user_id, self::PROVIDER_USER_META_KEY, $new_provider );
			}
			
			$user_dummy_code = get_user_meta( $user_id, self::USER_DUMMY_CODE, true );
			
			if( !empty($enabled_providers) && in_array( 'TwoStB_Step_Factor_Dummy', $enabled_providers ) && !$user_dummy_code ){
			    $uniqeCode = str_pad(wp_rand(0, 99999999), 8, '0', STR_PAD_LEFT);
			    update_user_meta( $user_id, self::USER_DUMMY_CODE, $uniqeCode );
			}
			
            $existing_backup_codes = get_user_meta($user_id, self::USER_BACKUPS_CODE, true);
            
            if ( !empty($enabled_providers) && in_array('TwoStB_Step_Factor_BackupCodes', $enabled_providers) && !$existing_backup_codes ) {
                $existing_codes = !empty($existing_backup_codes) ? $existing_backup_codes : [];
                $new_codes = [];
                while (count($new_codes) < 10) {
                    $uniqueCode = str_pad(wp_rand(10000000, 99999999), 8, '0', STR_PAD_LEFT);
                    // Ensure uniqueness across both new and existing codes
                    if (!isset($existing_codes[$uniqueCode]) && !isset($new_codes[$uniqueCode])) {
                        $new_codes[$uniqueCode] = 0; // Assign 0 by default
                    }
                }
                // Save to user meta (WordPress handles serialization automatically)
                update_user_meta($user_id, self::USER_BACKUPS_CODE, $new_codes);
            }
			
			if ( $user_id === get_current_user_id() ) {
			    
				if ( $enabled_providers && ! $existing_providers && ! self::is_current_user_session_twoStB_factor() ) {
					self::update_current_user_session( array(
						'two-factor-provider' => '',
						'two-factor-login'    => time(),
					) );
				} elseif ( $existing_providers && ! $enabled_providers ) {
					self::update_current_user_session( array(
						'two-factor-provider' => null,
						'two-factor-login'    => null,
					) );
				}
			}
			
			if (
				( ! $existing_providers && $enabled_providers ) ||
				( $existing_providers && $enabled_providers && array_diff( $existing_providers, $enabled_providers ) )
			) {
				if ( $user_id === get_current_user_id() ) {
					wp_destroy_other_sessions();
				} else {
					WP_Session_Tokens::get_instance( $user_id )->destroy_all();
				}
			}
		}
	}

    function ajax_update_backup_codes() {
        // Verify nonce for security
        if (!isset($_POST['security']) || !wp_verify_nonce( sanitize_text_field( wp_unslash($_POST['security']) ), 'update_backup_codes_nonce')) {
            wp_send_json_error(['message' => 'Invalid request (nonce failed).']);
            wp_die();
        }
    
        // Check if user is an admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized.']);
            wp_die();
        }
    
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : get_current_user_id();
        $enabled_providers = isset($_POST[self::ENABLED_PROVIDERS_USER_META_KEY]) && is_array($_POST[self::ENABLED_PROVIDERS_USER_META_KEY]) 
                    ? array_map('sanitize_text_field', wp_unslash($_POST[self::ENABLED_PROVIDERS_USER_META_KEY])) 
                    : [];
        $providers          = self::get_providers();
		$enabled_providers  = array_intersect( $enabled_providers, array_keys( $providers ) );
		
        // Call the static function
        $update = self::updateBackupCodes($user_id, $enabled_providers);
        
        if( $update ){
            wp_send_json_success(['message' => 'Backup codes updated successfully.']);
        }else{
            wp_send_json_error(['message' => 'Something went wrong.']);
        }
        wp_die();
    }


    public static function updateBackupCodes( $user_id = null, $enabled_providers = array() ){
        $existing_backup_codes = get_user_meta($user_id, self::USER_BACKUPS_CODE, true);
        $countOfOnes = '';
        if( $existing_backup_codes ){
            $countOfOnes = count(array_filter($existing_backup_codes, function($value) {
                return $value == 1;
            }));
        }else{
            return false;
        }
        
		if ( !empty($enabled_providers) && in_array('TwoStB_Step_Factor_BackupCodes', $enabled_providers) && ( $countOfOnes > 5 ) ) {
            $existing_codes = !empty($existing_backup_codes) ? $existing_backup_codes : [];
            $new_codes = [];
            while (count($new_codes) < 10) {
                $uniqueCode = str_pad(wp_rand(10000000, 99999999), 8, '0', STR_PAD_LEFT);
                // Ensure uniqueness across both new and existing codes
                if (!isset($existing_codes[$uniqueCode]) && !isset($new_codes[$uniqueCode])) {
                    $new_codes[$uniqueCode] = 0; // Assign 0 by default
                }
            }
            // Save to user meta (WordPress handles serialization automatically)
            update_user_meta($user_id, self::USER_BACKUPS_CODE, $new_codes);
            return true;
        }else{
            return false;
        }
    }
    
    public static function is_current_user_session_twoStB_factor() {
		$session = self::get_current_user_session();

		if ( empty( $session['two-factor-login'] ) ) {
			return false;
		}

		return (int) $session['two-factor-login'];
	}
    public static function get_current_user_session() {
		$user_id = get_current_user_id();
		$token   = wp_get_session_token();
		if ( ! $user_id || ! $token ) {
			return false;
		}

		$manager = WP_Session_Tokens::get_instance( $user_id );

		return $manager->get( $token );
	}

    public static function get_enabled_providers_for_user( $user = null ) {
		$user = self::fetch_user( $user );
		if ( ! $user ) {
			return array();
		}

		$providers         = self::get_providers();
		$enabled_providers = get_user_meta( $user->ID, self::ENABLED_PROVIDERS_USER_META_KEY, true );
		if ( empty( $enabled_providers ) ) {
			$enabled_providers = array();
		}
		$enabled_providers = array_intersect( $enabled_providers, array_keys( $providers ) );
		return apply_filters( 'twoStB_factor_enabled_providers_for_user', $enabled_providers, $user->ID );
	}

    public static function update_current_user_session( $data = array() ) {
		$user_id = get_current_user_id();
		$token   = wp_get_session_token();
		if ( ! $user_id || ! $token ) {
			return false;
		}
		$manager = WP_Session_Tokens::get_instance( $user_id );
		$session = $manager->get( $token );
		$session = array_merge( $session, $data );
		foreach ( array_filter( $data, 'is_null' ) as $key => $null ) {
			unset( $session[ $key ] );
		}
		return $manager->update( $token, $session );
	}

    public static function show_twoStB_factor_login( $user, $nonce_req ) {
		if ( ! $user ) {
			$user = wp_get_current_user();
		}

		if ( isset( $nonce_req ) && !empty($nonce_req) ) {
            // Remove slashes and sanitize the nonce value
            $nonce = sanitize_text_field( wp_unslash( $nonce_req ) );
            if ( ! wp_verify_nonce( $nonce, 'two_stb_factor_login_action' ) ) {
                wp_die( esc_html__( 'Nonce verification failed.', 'two-step-factor-by-se' ) );
            }
            
            $login_nonce =  wp_create_nonce('wp-auth-nonce');
            if ( ! $login_nonce ) {
                wp_die( esc_html__( 'Failed to create a login nonce.', 'two-step-factor-by-se' ) );
            }
            
            $redirect_to = isset( $_REQUEST['redirect_to'] ) ? esc_url_raw( wp_unslash($_REQUEST['redirect_to']) ) : admin_url();
            
            self::login_html( $user, $login_nonce, $redirect_to );
        }
	}

	public static function maybe_show_reset_password_notice( $errors ) {
		if ( 'incorrect_password' !== $errors->get_error_code() ) {
			return $errors;
		}
        
        if ( isset( $_REQUEST['_wpnonce'] ) ) {
            // Remove slashes and sanitize the nonce value
            $nonce = sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) );
            if ( ! wp_verify_nonce( $nonce, 'two_stb_factor_login_action' ) ) {
                wp_die( esc_html__( 'Nonce verification failed.', 'two-step-factor-by-se' ) );
            }
            
            if ( ! isset( $_POST['log'] ) ) {
                return $errors;
            }
            
            $user_name      = sanitize_user( wp_unslash( $_POST['log'] ) );
            $attempted_user = get_user_by( 'login', $user_name );
            if ( ! $attempted_user && str_contains( $user_name, '@' ) ) {
                $attempted_user = get_user_by( 'email', $user_name );
            }
            
                if ( ! $attempted_user ) {
            return $errors;
            }
            
            $password_was_reset = get_user_meta( $attempted_user->ID, self::USER_PASSWORD_WAS_RESET_KEY, true );
            
            if ( ! $password_was_reset ) {
                return $errors;
            }
            
            $errors->remove( 'incorrect_password' );
            $errors->add(
                'twoStB_factor_password_reset',
                sprintf(
                    /* translators: %s is the URL for the lost password page */
                    __( 'Your password was reset because of too many failed Two Factor attempts. You will need to <a href="%s">create a new password</a> to regain access. Please check your email for more information.', 'two-step-factor-by-se' ),
                    esc_url( add_query_arg( 'action', 'lostpassword', wp_login_url() ) )
                )
            );
            
            return $errors;
        }else{
            if ( 'incorrect_password' == $errors->get_error_code() ) {
    			return $errors;
    		}
            wp_die( esc_html__( 'Nonce verification failed.', 'two-step-factor-by-se' ) );
        }
	}

    public static function maybe_show_last_login_failure_notice( $user ) {
		$last_failed_twoStB_factor_login = (int) get_user_meta( $user->ID, self::USER_RATE_LIMIT_KEY, true );
		$failed_login_count           = (int) get_user_meta( $user->ID, self::USER_FAILED_LOGIN_ATTEMPTS_KEY, true );

		if ( $last_failed_twoStB_factor_login ) {
			echo '<div id="login_notice" class="message"><strong>';
			/* translators: 1: Number of failed login attempts, 2: Time since last failed login attempt */
            $message = sprintf(
                /* translators: 1: Number of failed login attempts, 2: Time since last failed login attempt */
                _n(
                    'WARNING: Your account has attempted to login %1$s time without providing a valid two-factor token. The last failed login occurred %2$s ago. If this wasn\'t you, you should reset your password.',
                    'WARNING: Your account has attempted to login %1$s times without providing a valid two-factor token. The last failed login occurred %2$s ago. If this wasn\'t you, you should reset your password.',
                    $failed_login_count,
                    'two-step-factor-by-se'
                ),
                number_format_i18n( $failed_login_count ),
                human_time_diff( $last_failed_twoStB_factor_login, time() )
            );
            
            // Escape before output
            printf( '%s', esc_html( $message ) );
			echo '</strong></div>';
		}
	}
    public static function rememberme( $login_nonce ) {
		$rememberme = false;
		
        if ( isset( $login_nonce ) && !empty($login_nonce) ) {
            // Remove slashes and sanitize the nonce value
            $nonce = sanitize_text_field( wp_unslash( $login_nonce ) );
            if ( ! wp_verify_nonce( $nonce, 'wp-auth-nonce' ) ) {
                wp_die( esc_html__( 'Nonce verification failed.', 'two-step-factor-by-se' ) );
            }
            
            if ( ! empty( $_REQUEST['rememberme'] ) ) {
                $rememberme = true;
            }
            return (bool) apply_filters( 'twoStB_factor_rememberme', $rememberme );
        }else{
            wp_die( esc_html__( 'Nonce verification failed.', 'two-step-factor-by-se' ) );
        }
		
	}

    public static function login_html( $user, $login_nonce, $redirect_to, $error_msg = '', $provider = null, $action = 'validate_2fa' ) {
		if($provider != null){
		    $provider = self::get_provider_for_user( $user, $provider );
		    $provider = new $provider;
		}else{
		    $provider = self::get_primary_provider_for_user( $user );
		}
		if ( ! $provider ) {
			wp_die( esc_html__( 'Cheatin&#8217; uh?', 'two-step-factor-by-se' ) );
		}
		$provider_key        = $provider->get_key();
		$available_providers = self::get_available_providers_for_user( $user );
		$backup_providers    = array_diff_key( $available_providers, array( $provider_key => null ) );
		$interim_login       = isset( $_REQUEST['interim-login'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		$rememberme = intval( self::rememberme( $login_nonce ) );

		add_filter( 'login_display_language_dropdown', '__return_false' );
        if( function_exists( 'login_header' ) ){
            login_header();
        }else{ ?>
            <!DOCTYPE html>
            <html <?php language_attributes(); ?>>
            <head>
                <meta charset="<?php bloginfo( 'charset' ); ?>">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title><?php bloginfo( 'name' ); ?> - <?php esc_html_e( 'Login', 'two-step-factor-by-se' ); ?></title>
            
                <?php
                // Enqueue admin styles to make it look like wp-admin
                wp_enqueue_style( 'login', site_url('/wp-admin/css/login.min.css'), array(), filemtime( ABSPATH . 'wp-admin/css/login.min.css') );
                wp_enqueue_emoji_styles();
                wp_print_styles();
                ?>
            </head>
            <body class="login">
            <div id="login">
        <?php }
        
		if ( ! empty( $error_msg ) ) {
			echo '<div id="login_error"><strong>' . esc_html( $error_msg ) . '</strong><br /></div>';
		} elseif ( 'validate_2fa' === $action ) {
			self::maybe_show_last_login_failure_notice( $user );
		}
		?>

		<form name="validate_2fa_form" id="loginform" action="<?php echo esc_url( self::login_url( array( 'action' => $action ), 'login_post' ) ); ?>" method="post" autocomplete="off">
				<input type="hidden" name="provider"      id="provider"      value="<?php echo esc_attr( $provider_key ); ?>" />
				<input type="hidden" name="wp-auth-id"    id="wp-auth-id"    value="<?php echo esc_attr( $user->ID ); ?>" />
				<input type="hidden" name="wp-auth-nonce" id="wp-auth-nonce" value="<?php echo esc_attr($login_nonce); ?>" />
				
				<?php if ( $interim_login ) { ?>
					<input type="hidden" name="interim-login" value="1" />
				<?php } else { ?>
					<input type="hidden" name="redirect_to" value="<?php echo esc_attr( $redirect_to ); ?>" />
				<?php } ?>
				<input type="hidden" name="rememberme"    id="rememberme"    value="<?php echo esc_attr( $rememberme ); ?>" />

				<?php $provider->authentication_page( $user ); ?>
		</form>

		<?php if ( $backup_providers ) :
			$backup_link_args = array(
				'action'        => 'alter_validate_2fa',
				'wp-auth-id'    => $user->ID,
				'wp-auth-nonce' => $login_nonce,
			);
			if ( $rememberme ) {
				$backup_link_args['rememberme'] = $rememberme;
			}
			if ( $redirect_to ) {
				$backup_link_args['redirect_to'] = $redirect_to;
			}
			if ( $interim_login ) {
				$backup_link_args['interim-login'] = 1;
			}
			?>
			<div class="backup-methods-wrap">
				<p>
					<?php esc_html_e( 'Having Problems?', 'two-step-factor-by-se' ); ?>
				</p>
				<ul>
					<?php
					foreach ( $backup_providers as $backup_provider_key => $backup_provider ) :
						$backup_link_args['provider'] = $backup_provider_key;
						?>
						<li>
							<a href="<?php echo esc_url( self::login_url( $backup_link_args ) ); ?>">
								<?php echo esc_html( $backup_provider->get_alternative_provider_label() ); ?>
							</a>
						</li>
					<?php endforeach; ?>
				</ul>
			</div>
		<?php endif; ?>

		<style>
			/* @todo: migrate to an external stylesheet. */
			.backup-methods-wrap {
				margin-top: 16px;
				padding: 0 24px;
			}
			.backup-methods-wrap a {
				text-decoration: none;
			}
			.backup-methods-wrap ul {
				list-style-position: inside;
			}
			/* Prevent Jetpack from hiding our controls, see https://github.com/Automattic/jetpack/issues/3747 */
			.jetpack-sso-form-display #loginform > p,
			.jetpack-sso-form-display #loginform > div {
				display: block;
			}
			#login form p.two-factor-prompt {
				margin-bottom: 1em;
			}
			.input.authcode {
				letter-spacing: .3em;
			}
			.input.authcode::placeholder {
				opacity: 0.5;
			}
			#login_error{
			    border: 1px solid #ff0000;
                padding: 10px;
                color: #ff0000;
                background: #fff;
                text-align: center;
			}
		</style>
		<script>
			(function() {
				// Enforce numeric-only input for numeric inputmode elements.
				const form = document.querySelector( '#loginform' ),
					inputEl = document.querySelector( 'input.authcode[inputmode="numeric"]' ),
					expectedLength = inputEl?.dataset.digits || 0;

				if ( inputEl ) {
					let spaceInserted = false;
					inputEl.addEventListener(
						'input',
						function() {
							let value = this.value.replace(/\s/g, '');
							if ( ! spaceInserted && expectedLength && value.length === Math.floor( expectedLength / 2 ) ) {
								value += ' ';
								spaceInserted = true;
							} else if ( spaceInserted && ! this.value ) {
								spaceInserted = false;
							}

							this.value = value;
							// Auto-submit if it's the expected length.
							if ( expectedLength && value.replace( / /g, '' ).length == expectedLength ) {
								if ( undefined !== form.requestSubmit ) {
									//form.requestSubmit();
									//form.submit.disabled = "disabled";
								}
							}
						}
					);
				}
			})();
		</script>
		<?php
	}
	
	public static function login_url( $params = array(), $scheme = 'login' ) {
		if ( ! is_array( $params ) ) {
			$params = array();
		}
		$params = urlencode_deep( $params );
		if ( isset( $params['action'] ) ) {
			$url = site_url( 'wp-login.php?action=' . $params['action'], $scheme );
		} else {
			$url = site_url( 'wp-login.php', $scheme );
		}
		if ( $params ) {
			$url = add_query_arg( $params, $url );
		}
		return $url;
	}
	
	public static function login_form_validate_2fa($REQUEST) {
	    if (isset($_POST['wp-auth-nonce']) && wp_verify_nonce( sanitize_text_field(wp_unslash($_POST['wp-auth-nonce'])), 'wp-auth-nonce')) {
            // Valid nonce, proceed with action
        } else {
            wp_die( esc_html__( 'Security check failed. Please try again.', 'two-step-factor-by-se' ) );
        }
		$wp_auth_id      = ! empty( $_POST['wp-auth-id'] )    ? absint( $_POST['wp-auth-id'] )        : 0;
		$nonce           = ! empty( $_POST['wp-auth-nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['wp-auth-nonce'] ) ) : '';
		$provider        = ! empty( $_POST['provider'] )      ? sanitize_text_field( wp_unslash( $_POST['provider'] ) ) : '';
		$redirect_to     = ! empty( $_POST['redirect_to'] )   ? sanitize_url( wp_unslash( $_POST['redirect_to'] ) )  : '';
		$code			 = ! empty( $_POST['two-provider-auth-code'] )   ? sanitize_text_field( wp_unslash( $_POST['two-provider-auth-code'] ) )   : '';
		$log			 = ! empty( $_POST['log'] )   ? sanitize_text_field( wp_unslash( $_POST['log'] ) )   : '';
		$username        = ! empty( $_POST['username'] )   ? sanitize_text_field( wp_unslash( $_POST['username'] ) )  : '';
		$pwd			 = ! empty( $_POST['pwd'] )   ? sanitize_text_field( wp_unslash( $_POST['pwd'] ) )  : '';
		$password        = ! empty( $_POST['password'] )   ? sanitize_text_field( wp_unslash( $_POST['password'] ) )  : '';
		
		$is_post_request = (
            isset( $_SERVER['REQUEST_METHOD'] ) &&
            'POST' === strtoupper( sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) )
        );
		$user            = get_user_by( 'id', $wp_auth_id );
        
		if ( ! $wp_auth_id || ! $nonce || ! $user ) {
		    if( isset($_POST['provider']) && $_POST['provider'] == 'TwoStB_Step_Factor_Dummy'){
		        if ( isset( $_POST['wp-auth-id'] ) ) {
                    $user_id = absint( wp_unslash( $_POST['wp-auth-id'] ) );
                    $user    = get_user_by( 'id', $user_id );
                } else {
                    wp_die( esc_html__( 'User ID not provided.', 'two-step-factor-by-se' ) );
                }
                
		        $redirect_to = ! empty( $_POST['redirect_to'] )   ? esc_url_raw( wp_unslash($_POST['redirect_to']) )   : '';
		        $login_nonce = wp_create_nonce('wp-auth-nonce');
    			if ( ! $login_nonce ) {
    				wp_die( esc_html__( 'Failed to create a login nonce.', 'two-step-factor-by-se' ) );
    			}
		        self::login_html( $user, $login_nonce, $redirect_to, $_POST, '', $provider );
		        exit;
		    }else{
		        return;
		    }
		}
        $provider = self::get_provider_for_user( $user, $provider );
        if(!empty($provider)){
            $proivderObj = new $provider;
            if( isset($_POST['_twoSt_step_factor_email_code_resend']) ){
                $re_gen_resp = $proivderObj->re_generate_email_token( $user );
                $login_nonce = wp_create_nonce('wp-auth-nonce');
    			if ( ! $login_nonce ) {
    				wp_die( esc_html__( 'Failed to create a login nonce.', 'two-step-factor-by-se' ) );
    			}
                $error_message = $re_gen_resp['msg'];
    			self::login_html( $user, $login_nonce, $redirect_to, $error_message, $provider );
    			exit;
            }
            $codeStatus = $proivderObj->validate_token($wp_auth_id, $code);
            if( !$codeStatus['status'] ){
    			$login_nonce = wp_create_nonce('wp-auth-nonce');
    			if ( ! $login_nonce ) {
    				wp_die( esc_html__( 'Failed to create a login nonce.', 'two-step-factor-by-se' ) );
    			}
                $error_message = $codeStatus['error'];
    			self::login_html( $user, $login_nonce, $redirect_to, $error_message, $provider );
    			exit;
    		}else{
    	        self::_login_form_validate_2fa( $user, $nonce, $provider, $redirect_to, $is_post_request, $_POST );
    			exit;
    		}
        }else{
            $redirect_to = apply_filters( 'login_redirect', $redirect_to, $redirect_to, $user );
		    wp_safe_redirect( $redirect_to );
        }
	}
	
	public function login_form_alter_validate_2fa(){
	    if (isset($_GET['wp-auth-nonce']) && wp_verify_nonce( sanitize_text_field(wp_unslash($_GET['wp-auth-nonce'])), 'wp-auth-nonce')) {
            // Valid nonce, proceed with action
        } else {
            wp_die( esc_html__( 'Security check failed. Please try again.', 'two-step-factor-by-se' ) );
        }
		$wp_auth_id      = ! empty( $_GET['wp-auth-id'] )    ? absint( $_GET['wp-auth-id'] )        : 0;
		$nonce           = ! empty( $_GET['wp-auth-nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['wp-auth-nonce'] ) ) : '';
		$provider        = ! empty( $_GET['provider'] )      ? sanitize_text_field( wp_unslash( $_GET['provider'] ) ) : '';
		$redirect_to     = ! empty( $_GET['redirect_to'] )   ? sanitize_url( wp_unslash( $_GET['redirect_to'] ) )  : '';
		$user            = get_user_by( 'id', $wp_auth_id );
		
		$login_nonce = wp_create_nonce('wp-auth-nonce');
		if ( ! $login_nonce ) {
			wp_die( esc_html__( 'Failed to create a login nonce.', 'two-step-factor-by-se' ) );
		}
        			
		self::login_html( $user, $login_nonce, $redirect_to, '', $provider );
		exit;
		
	}
	public static function _login_form_validate_2fa( $user, $nonce = '', $provider = '', $redirect_to = '', $is_post_request = false ) {
		// Validate the request.
		if ( !wp_verify_nonce( $nonce, 'wp-auth-nonce') ) {
			wp_safe_redirect( home_url() );
			return;
		}

		$provider = self::get_provider_for_user( $user, $provider );
		if ( ! $provider ) {
			wp_die( esc_html__( 'Cheatin&#8217; uh?', 'two-step-factor-by-se' ) );
		}

		delete_user_meta( $user->ID, self::USER_RATE_LIMIT_KEY );
		delete_user_meta( $user->ID, self::USER_FAILED_LOGIN_ATTEMPTS_KEY );

		$rememberme = false;
		if ( isset( $_REQUEST['rememberme'] ) && sanitize_text_field( wp_unslash( $_REQUEST['rememberme'] ) ) ) {
			$rememberme = true;
		}

		$session_information_callback = static function( $session, $user_id ) use( $provider, $user ) {
			if ( $user->ID === $user_id ) {
				$session['two-factor-login']    = time();
				$session['two-factor-provider'] = $provider->get_key();
			}

			return $session;
		};

		add_filter( 'attach_session_information', $session_information_callback, 10, 2 );

		remove_filter( 'send_auth_cookies', '__return_false', PHP_INT_MAX );
		
		wp_set_auth_cookie( $user->ID, $rememberme );
		
		do_action( 'twoStB_factor_user_authenticated', $user, $provider );
		
		remove_filter( 'attach_session_information', $session_information_callback );
		
		global $interim_login;
		$interim_login = isset( $_REQUEST['interim-login'] );
		
		if ( $interim_login ) {
			$customize_login = isset( $_REQUEST['customize-login'] );
			if ( $customize_login ) {
				wp_enqueue_script( 'customize-base' );
			}
			$message       = '<p class="message">' . __( 'You have logged in successfully.', 'two-step-factor-by-se' ) . '</p>';
			$interim_login = 'success';
			login_header( '', $message );
			?>
			</div>
			<?php
			do_action( 'login_footer' );
			?>
			<?php if ( $customize_login ) : ?>
				<script type="text/javascript">setTimeout( function(){ new wp.customize.Messenger({ url: '<?php echo esc_url( wp_customize_url() ); ?>', channel: 'login' }).send('login') }, 1000 );</script>
			<?php endif; ?>
			</body></html>
			<?php
			return;
		}
		
		$redirect_to = apply_filters( 'login_redirect', $redirect_to, $redirect_to, $user );
		wp_safe_redirect( $redirect_to );
	}

	public static function get_provider_for_user( $user = null, $preferred_provider = null ) {
		$user = self::fetch_user( $user );
		if ( ! $user ) {
			return null;
		}
		if ( $preferred_provider && $preferred_provider instanceof TwoStB_Step_Provider ) {
			$preferred_provider = $preferred_provider->get_key();
		}
		if ( ! $preferred_provider && get_current_user_id() === $user->ID ) {
			$session = self::get_current_user_session();
			if ( ! empty( $session['two-factor-provider'] )	) {
				$preferred_provider = $session['two-factor-provider'];
			}
		}
		if ( is_string( $preferred_provider ) ) {
			$providers = self::get_available_providers_for_user( $user );
			if ( isset( $providers[ $preferred_provider ] ) ) {
				return $providers[ $preferred_provider ];
			}
		}
		return self::get_primary_provider_for_user( $user );
	}
	
}

new TwoStB_Step_Factor_Core();
