<?php
/**
 * Plugin Name: TW Simple Slider
 * Plugin URI: https://technologywisdom.com/
 * Description: Create beautiful sliders with advanced styling options
 * Version: 1.0.0
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Author: technologywisdom
 * Author URI: https://technologywisdom.com/about-us/
 * License: GPL-2.0+
 * License URI: https://opensource.org/licenses/GPL-2.0
 * Text Domain: tw-simple-slider
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define constants with unique prefix
define('TW_SIMPLE_SLIDER_VERSION', '1.0.0');
define('TW_SIMPLE_SLIDER_DIR', plugin_dir_path(__FILE__));
define('TW_SIMPLE_SLIDER_URL', plugin_dir_url(__FILE__));
define('TW_SIMPLE_SLIDER_POST_TYPE', 'twssc_slider');

// Add settings link
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'twssc_simple_slider_action_links');
function twssc_simple_slider_action_links($links) {
    // All Sliders link
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        admin_url('edit.php?post_type=' . TW_SIMPLE_SLIDER_POST_TYPE),
        esc_html__('All Sliders', 'tw-simple-slider')
    );
    
    // Tutorial link
    $tutorial_link = sprintf(
        '<a href="%s" style="color: #2271b1;">%s</a>',
        admin_url('admin.php?page=tw-simple-slider-tutorial'),
        esc_html__('Tutorial', 'tw-simple-slider')
    );
    
    array_unshift($links, $tutorial_link, $settings_link);
    return $links;
}

// Include required files
require_once TW_SIMPLE_SLIDER_DIR . 'includes/class-twssc-slider-post-type.php';
require_once TW_SIMPLE_SLIDER_DIR . 'includes/class-twssc-slider-metabox.php';
require_once TW_SIMPLE_SLIDER_DIR . 'includes/class-twssc-slider-shortcode.php';
require_once TW_SIMPLE_SLIDER_DIR . 'includes/class-twssc-tutorial-page.php';

// Initialize plugin
class Tw_Simple_Slider {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init();
    }
    
    public function init() {
        // Initialize all component classes
        TW_SIMPLE_SLIDER_Post_Type::get_instance();
        TW_SIMPLE_SLIDER_Metabox::get_instance();
        TW_SIMPLE_SLIDER_Shortcode::get_instance();
        TW_SIMPLE_SLIDER_Tutorial::get_instance();
        
        // Enqueue scripts
        add_action('admin_enqueue_scripts', array($this, 'twssc_enqueue_admin_scripts'));
        add_action('wp_enqueue_scripts', array($this, 'twssc_enqueue_frontend_scripts'));
    }
    
    public function twssc_enqueue_admin_scripts($hook) {
        if ('post.php' === $hook || 'post-new.php' === $hook) {
            $screen = get_current_screen();
            if (TW_SIMPLE_SLIDER_POST_TYPE === $screen->post_type) {
                wp_enqueue_media();
                wp_enqueue_style('wp-color-picker');
                
                wp_enqueue_style(
                    'twssc-slider-admin-style',
                    TW_SIMPLE_SLIDER_URL . 'assets/css/admin.css',
                    array(),
                    TW_SIMPLE_SLIDER_VERSION
                );
                
                wp_enqueue_script(
                    'twssc-slider-admin-script',
                    TW_SIMPLE_SLIDER_URL . 'assets/js/admin.js',
                    array('jquery', 'jquery-ui-sortable', 'wp-color-picker'),
                    TW_SIMPLE_SLIDER_VERSION,
                    array(
                        'in_footer' => true,
                        'strategy' => 'defer'
                    )
                );
                
                wp_localize_script('twssc-slider-admin-script', 'twssc_admin', array(
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('twssc_admin_nonce'),
                    'strings' => array(
                        'remove' => __('Remove', 'tw-simple-slider'),
                        'add_images' => __('Add Images', 'tw-simple-slider'),
                        'select_images' => __('Select Images', 'tw-simple-slider'),
                        'add_to_slider' => __('Add to Slider', 'tw-simple-slider'),
                        'media_error' => __('WordPress media library is not available.', 'tw-simple-slider'),
                        'no_images' => __('No images added yet.', 'tw-simple-slider'),
                        'fade_disabled' => __('Fade animation is only available when showing 1 slide', 'tw-simple-slider')
                    )
                ));
            }
        }
    }
    
    public function twssc_enqueue_frontend_scripts() {
        global $post;
        
        // Register styles
        wp_register_style(
            'slick-slider-css',
            TW_SIMPLE_SLIDER_URL . 'assets/css/slick.css',
            array(),
            '1.8.1'
        );
        
        wp_register_style(
            'slick-theme-css',
            TW_SIMPLE_SLIDER_URL . 'assets/css/slick-theme.css',
            array('slick-slider-css'),
            '1.8.1'
        );
        
        wp_register_style(
            'twssc-slider-frontend-style',
            TW_SIMPLE_SLIDER_URL . 'assets/css/frontend.css',
            array('slick-theme-css'),
            TW_SIMPLE_SLIDER_VERSION
        );
        
        // Register scripts
        wp_register_script(
            'slick-slider-js',
            TW_SIMPLE_SLIDER_URL . 'assets/js/slick.min.js',
            array('jquery'),
            '1.8.1',
            array(
                'in_footer' => true,
                'strategy' => 'defer'
            )
        );
        
        wp_register_script(
            'twssc-slider-frontend-script',
            TW_SIMPLE_SLIDER_URL . 'assets/js/frontend.js',
            array('jquery', 'slick-slider-js'),
            TW_SIMPLE_SLIDER_VERSION,
            array(
                'in_footer' => true,
                'strategy' => 'defer'
            )
        );
        
        // Check if we're on a page that might have a slider
        $has_slider = false;
        
        if (is_a($post, 'WP_Post')) {
            $has_slider = has_shortcode($post->post_content, 'twssc_slider');
        }
        
        // Also check if we're on a page that uses widgets
        if (!$has_slider && is_active_widget(false, false, 'text', true)) {
            // We'll enqueue just in case - better safe than sorry
            $has_slider = true;
        }
        
        if ($has_slider) {
            wp_enqueue_style('twssc-slider-frontend-style');
            wp_enqueue_script('twssc-slider-frontend-script');
            
            // Add custom CSS for all published sliders
            $this->twssc_add_inline_slider_styles();
        }
    }
    
    private function twssc_add_inline_slider_styles() {
        $sliders = get_posts(array(
            'post_type' => TW_SIMPLE_SLIDER_POST_TYPE,
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'fields' => 'ids'
        ));
        
        if (empty($sliders)) {
            return;
        }
        
        $custom_css = '';
        
        foreach ($sliders as $slider_id) {
            $custom_style = get_post_meta($slider_id, '_twssc_slider_custom_style', true);
            
            if (!empty($custom_style) && is_array($custom_style)) {
                $escaped_style = array_map('esc_attr', $custom_style);
                
                $custom_css .= "
                #twssc-slider-{$slider_id} .twssc-slider-slide img {
                    object-fit: {$escaped_style['image_fit']};
                    height: {$escaped_style['image_height']}px;
                }
                
                #twssc-slider-{$slider_id} .slick-dots li button {
                    width: {$escaped_style['dots_size']}px !important;
                    height: {$escaped_style['dots_size']}px !important;
                    background-color: {$escaped_style['dots_color']} !important;
                    border-radius: 50% !important;
                }
                
                #twssc-slider-{$slider_id} .slick-dots li.slick-active button {
                    background-color: {$escaped_style['dots_active_color']} !important;
                }
                
                #twssc-slider-{$slider_id} .slick-dots li button:before {
                    display: none !important;
                }
                
                #twssc-slider-{$slider_id} .slick-prev,
                #twssc-slider-{$slider_id} .slick-next {
                    background: {$escaped_style['arrows_bg']};
                    width: {$escaped_style['arrows_size']}px;
                    height: {$escaped_style['arrows_size']}px;
                }
                
                #twssc-slider-{$slider_id} .slick-prev:hover,
                #twssc-slider-{$slider_id} .slick-next:hover {
                    background: {$escaped_style['arrows_hover_bg']};
                }
                
                #twssc-slider-{$slider_id} .slick-prev:before,
                #twssc-slider-{$slider_id} .slick-next:before {
                    background-color: {$escaped_style['arrows_color']};
                    width: {$escaped_style['arrows_icon_size']}px;
                }";
            }
        }
        
        if (!empty($custom_css)) {
            wp_add_inline_style('twssc-slider-frontend-style', $custom_css);
        }
    }
}

// Initialize the plugin
Tw_Simple_Slider::get_instance();

// Activation/Deactivation hooks
register_activation_hook(__FILE__, 'twssc_simple_slider_activate');
function twssc_simple_slider_activate() {
    flush_rewrite_rules();
}

register_deactivation_hook(__FILE__, 'twssc_simple_slider_deactivate');
function twssc_simple_slider_deactivate() {
    flush_rewrite_rules();
}