<?php
/**
 * Plugin Name: TryMyLook Virtual Try-On
 * Plugin URI: https://trymylook.app/wordpress-plugin
 * Description: Add virtual try-on functionality to your WooCommerce products. Let customers see how products look on them before buying.
 * Version:           1.0.3
 * Author:            TryMyLook
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: trymylook-virtual-try-on
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 9.0
 * Requires Plugins: woocommerce
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
// Plugin version
define('TRYMYLOOK_VERSION', '1.0.3');
define('TRYMYLOOK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('TRYMYLOOK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('TRYMYLOOK_API_URL', 'https://trymylook.app/api/v1/wordpress');

// Declare WooCommerce HPOS compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// Check if WooCommerce is active
if (in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    
    /**
     * Main TryMyLook Class
     */
    class TryMyLook_Virtual_TryOn {
        
        /**
         * Instance of this class
         */
        private static $instance = null;
        
        /**
         * Get instance
         */
        public static function get_instance() {
            if (null === self::$instance) {
                self::$instance = new self();
            }
            return self::$instance;
        }
        
        /**
         * Constructor
         */
        private function __construct() {
            // Admin settings
            add_action('admin_menu', array($this, 'add_admin_menu'));
            add_action('admin_init', array($this, 'register_settings'));
            
            // Frontend
            add_action('woocommerce_after_add_to_cart_button', array($this, 'add_tryon_button'));
            add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
            
            // AJAX handlers
            add_action('wp_ajax_trymylook_generate', array($this, 'ajax_generate'));
            add_action('wp_ajax_nopriv_trymylook_generate', array($this, 'ajax_generate'));
            add_action('wp_ajax_trymylook_check_status', array($this, 'ajax_check_status'));
            add_action('wp_ajax_nopriv_trymylook_check_status', array($this, 'ajax_check_status'));
        }
        
        /**
         * Add admin menu
         */
        public function add_admin_menu() {
            add_options_page(
                __('TryMyLook Settings', 'trymylook-virtual-try-on'),
                __('TryMyLook', 'trymylook-virtual-try-on'),
                'manage_options',
                'trymylook-settings',
                array($this, 'settings_page')
            );
        }
        
        /**
         * Register settings
         */
        public function register_settings() {
            register_setting('trymylook_settings', 'trymylook_api_key', array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => ''
            ));
            register_setting('trymylook_settings', 'trymylook_button_text', array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => __('Try On Me', 'trymylook-virtual-try-on')
            ));
            register_setting('trymylook_settings', 'trymylook_generate_button_text', array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => __('Generate', 'trymylook-virtual-try-on')
            ));
            register_setting('trymylook_settings', 'trymylook_enable_download', array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => '1'
            ));
        }
        
        /**
         * Settings page
         */
        public function settings_page() {
            ?>
            <div class="wrap">
                <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
                <form action="options.php" method="post">
                    <?php
                    settings_fields('trymylook_settings');
                    do_settings_sections('trymylook_settings');
                    ?>
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e('API Key', 'trymylook-virtual-try-on'); ?></th>
                            <td>
                                <input type="text" name="trymylook_api_key" 
                                       value="<?php echo esc_attr(get_option('trymylook_api_key')); ?>" 
                                       class="regular-text" />
                                <p class="description">
                                    <?php esc_html_e('Get your API key from', 'trymylook-virtual-try-on'); ?> 
                                    <a href="https://trymylook.app/api_tokens" target="_blank">TryMyLook Dashboard</a>
                                </p>
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e('Try-On Button Text', 'trymylook-virtual-try-on'); ?></th>
                            <td>
                                <input type="text" name="trymylook_button_text" 
                                       value="<?php echo esc_attr(get_option('trymylook_button_text', __('Try On Me', 'trymylook-virtual-try-on'))); ?>" 
                                       class="regular-text" />
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e('Generate Button Text', 'trymylook-virtual-try-on'); ?></th>
                            <td>
                                <input type="text" name="trymylook_generate_button_text" 
                                       value="<?php echo esc_attr(get_option('trymylook_generate_button_text', __('Generate', 'trymylook-virtual-try-on'))); ?>" 
                                       class="regular-text" />
                            </td>
                        </tr>
                        <tr valign="top">
                            <th scope="row"><?php esc_html_e('Enable Image Download', 'trymylook-virtual-try-on'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" name="trymylook_enable_download" 
                                           value="1" <?php checked(get_option('trymylook_enable_download', '1'), '1'); ?> />
                                    <?php esc_html_e('Allow customers to download generated images', 'trymylook-virtual-try-on'); ?>
                                </label>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button(); ?>
                </form>
            </div>
            <?php
        }
        
        /**
         * Add try-on button to product page
         */
        public function add_tryon_button() {
            global $product;
            
            if (!$product) {
                return;
            }
            
            $button_text = get_option('trymylook_button_text', __('Try On Me', 'trymylook-virtual-try-on'));
            
            echo '<button type="button" class="trymylook-button button alt" data-product-id="' . esc_attr($product->get_id()) . '">';
            echo esc_html($button_text);
            echo '</button>';
        }
        
        /**
         * Enqueue scripts and styles
         */
        public function enqueue_scripts() {
            if (!is_product()) {
                return;
            }
            
            wp_enqueue_style(
                'trymylook-styles',
                TRYMYLOOK_PLUGIN_URL . 'assets/css/trymylook.css',
                array(),
                TRYMYLOOK_VERSION
            );
            
            wp_enqueue_script(
                'trymylook-script',
                TRYMYLOOK_PLUGIN_URL . 'assets/js/trymylook.js',
                array('jquery'),
                TRYMYLOOK_VERSION,
                true
            );
            
            wp_localize_script('trymylook-script', 'trymylook_vars', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('trymylook_nonce'),
                'generate_button_text' => get_option('trymylook_generate_button_text', __('Generate', 'trymylook-virtual-try-on')),
                'enable_download' => get_option('trymylook_enable_download', '1'),
                'texts' => array(
                    'upload_your_photo' => __('Upload Your Photo', 'trymylook-virtual-try-on'),
                    'select_product_image' => __('Select Product Image', 'trymylook-virtual-try-on'),
                    'generating' => __('Generating...', 'trymylook-virtual-try-on'),
                    'processing' => __('Processing your image...', 'trymylook-virtual-try-on'),
                    'download' => __('Download Image', 'trymylook-virtual-try-on'),
                    'try_again' => __('Try Again', 'trymylook-virtual-try-on'),
                    'error' => __('Error', 'trymylook-virtual-try-on'),
                    'close' => __('Close', 'trymylook-virtual-try-on'),
                )
            ));
        }
        
        /**
         * AJAX: Generate outfit
         */
        public function ajax_generate() {
            check_ajax_referer('trymylook_nonce', 'nonce');
            
            $api_key = get_option('trymylook_api_key');
            if (empty($api_key)) {
                wp_send_json_error(array('message' => __('API key not configured', 'trymylook-virtual-try-on')));
            }
            
            $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
            
            // Sanitize and validate file upload
            $user_image = null;
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- File array is validated and sanitized below
            if (isset($_FILES['user_image']) && is_array($_FILES['user_image'])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Each field is sanitized individually below
                $file = $_FILES['user_image'];
                // Validate file upload
                if (isset($file['error']) && UPLOAD_ERR_OK === $file['error']) {
                    // Validate that tmp_name is a valid uploaded file
                    $tmp_name = isset($file['tmp_name']) ? $file['tmp_name'] : '';
                    if (!empty($tmp_name) && is_uploaded_file($tmp_name)) {
                        $user_image = array(
                            'name' => isset($file['name']) ? sanitize_file_name($file['name']) : '',
                            'type' => isset($file['type']) ? sanitize_mime_type($file['type']) : '',
                            'tmp_name' => $tmp_name,
                            'size' => isset($file['size']) ? intval($file['size']) : 0
                        );
                    }
                }
            }
            
            $product_image_url = isset($_POST['product_image_url']) ? esc_url_raw(wp_unslash($_POST['product_image_url'])) : '';
            
            if (!$user_image || empty($product_image_url)) {
                wp_send_json_error(array('message' => __('Missing required images', 'trymylook-virtual-try-on')));
            }
            
            // Get product name
            $product = wc_get_product($product_id);
            $product_name = $product ? $product->get_name() : 'Product';
            
            // Prepare API request
            $boundary = wp_generate_password(24);
            $headers = array(
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
                'X-API-Key' => $api_key
            );
            
            // Build multipart body
            $body = $this->build_multipart_body($boundary, array(
                'user_image' => $user_image,
                'product_image' => $product_image_url,
                'product_name' => $product_name
            ));
            
            // Make API request
            $response = wp_remote_post(TRYMYLOOK_API_URL . '/generate', array(
                'headers' => $headers,
                'body' => $body,
                'timeout' => 30
            ));
            
            if (is_wp_error($response)) {
                wp_send_json_error(array('message' => $response->get_error_message()));
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (isset($data['error'])) {
                wp_send_json_error(array('message' => $data['error']));
            }
            
            wp_send_json_success($data);
        }
        
        /**
         * AJAX: Check generation status
         */
        public function ajax_check_status() {
            check_ajax_referer('trymylook_nonce', 'nonce');
            
            $api_key = get_option('trymylook_api_key');
            $outfit_id = isset($_POST['outfit_id']) ? sanitize_text_field(wp_unslash($_POST['outfit_id'])) : '';
            
            if (empty($api_key) || empty($outfit_id)) {
                wp_send_json_error(array('message' => __('Invalid request', 'trymylook-virtual-try-on')));
            }
            
            $response = wp_remote_get(TRYMYLOOK_API_URL . '/status/' . $outfit_id, array(
                'headers' => array('X-API-Key' => $api_key),
                'timeout' => 15
            ));
            
            if (is_wp_error($response)) {
                wp_send_json_error(array('message' => $response->get_error_message()));
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            wp_send_json_success($data);
        }
        
        /**
         * Build multipart form data body
         */
        private function build_multipart_body($boundary, $fields) {
            $body = '';
            
            foreach ($fields as $key => $value) {
                if ($key === 'user_image' && is_array($value)) {
                    // Handle file upload
                    $file_path = $value['tmp_name'];
                    $file_name = $value['name'];
                    $file_content = file_get_contents($file_path);
                    
                    $body .= "--{$boundary}\r\n";
                    $body .= "Content-Disposition: form-data; name=\"{$key}\"; filename=\"{$file_name}\"\r\n";
                    $body .= "Content-Type: " . $value['type'] . "\r\n\r\n";
                    $body .= $file_content . "\r\n";
                } else {
                    // Handle regular field
                    $body .= "--{$boundary}\r\n";
                    $body .= "Content-Disposition: form-data; name=\"{$key}\"\r\n\r\n";
                    $body .= $value . "\r\n";
                }
            }
            
            $body .= "--{$boundary}--\r\n";
            
            return $body;
        }
    }
    
    // Initialize plugin
    function trymylook_init() {
        TryMyLook_Virtual_TryOn::get_instance();
    }
    add_action('plugins_loaded', 'trymylook_init');
    
} else {
    // WooCommerce is not active, show admin notice
    add_action('admin_notices', 'trymylook_woocommerce_missing_notice');
    
    function trymylook_woocommerce_missing_notice() {
        ?>
        <div class="error">
            <p><?php esc_html_e('TryMyLook Virtual Try-On requires WooCommerce to be installed and active.', 'trymylook-virtual-try-on'); ?></p>
        </div>
        <?php
    }
}
