/**
 * TryMyLook Virtual Try-On JavaScript
 * Handles modal UI, image upload, and API communication
 */

(function($) {
    'use strict';

    // Global state
    const TryMyLook = {
        modal: null,
        uploadedImage: null,
        selectedProductImage: null,
        pollingInterval: null,
        outfitId: null
    };

    /**
     * Initialize the plugin
     */
    function init() {
        // Handle "Try On Me" button click
        $(document).on('click', '.trymylook-button', function(e) {
            e.preventDefault();
            openModal();
        });

        // Handle modal close
        $(document).on('click', '.trymylook-modal-close, .trymylook-modal-overlay', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        // Handle file upload
        $(document).on('change', '#trymylook-user-image', function(e) {
            handleFileUpload(e.target.files[0]);
        });

        // Handle drag & drop
        $(document).on('dragover', '.trymylook-upload-area', function(e) {
            e.preventDefault();
            $(this).addClass('dragging');
        });

        $(document).on('dragleave', '.trymylook-upload-area', function(e) {
            e.preventDefault();
            $(this).removeClass('dragging');
        });

        $(document).on('drop', '.trymylook-upload-area', function(e) {
            e.preventDefault();
            $(this).removeClass('dragging');
            const file = e.originalEvent.dataTransfer.files[0];
            if (file && file.type.startsWith('image/')) {
                handleFileUpload(file);
            }
        });

        // Handle product image selection
        $(document).on('click', '.trymylook-product-image', function() {
            $('.trymylook-product-image').removeClass('selected');
            $(this).addClass('selected');
            TryMyLook.selectedProductImage = $(this).data('url');
            updateGenerateButton();
        });

        // Handle generate button
        $(document).on('click', '.trymylook-generate-btn', function() {
            generateOutfit();
        });

        // Handle download button
        $(document).on('click', '.trymylook-download-btn', function() {
            downloadImage();
        });

        // Handle try again button
        $(document).on('click', '.trymylook-try-again-btn', function() {
            resetModal();
        });
    }

    /**
     * Open the modal
     */
    function openModal() {
        if (TryMyLook.modal) {
            TryMyLook.modal.show();
            return;
        }

        // Get product images from WooCommerce gallery
        const productImages = getProductImages();

        // Create modal HTML
        const modalHTML = `
            <div class="trymylook-modal-overlay">
                <div class="trymylook-modal">
                    <button class="trymylook-modal-close">&times;</button>
                    
                    <div class="trymylook-modal-content">
                        <h2>${trymylook_vars.texts.modal_title || 'Virtual Try-On'}</h2>
                        
                        <!-- Step 1: Upload your image -->
                        <div class="trymylook-step trymylook-step-upload active">
                            <h3>Step 1: Upload Your Photo</h3>
                            <div class="trymylook-upload-area">
                                <input type="file" id="trymylook-user-image" accept="image/*" style="display: none;">
                                <div class="trymylook-upload-placeholder">
                                    <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                        <polyline points="17 8 12 3 7 8"></polyline>
                                        <line x1="12" y1="3" x2="12" y2="15"></line>
                                    </svg>
                                    <p>Drag & drop your photo here or click to browse</p>
                                    <button type="button" class="trymylook-browse-btn">Browse Files</button>
                                </div>
                                <div class="trymylook-image-preview" style="display: none;">
                                    <img src="" alt="Your photo">
                                    <button type="button" class="trymylook-remove-image">&times;</button>
                                </div>
                            </div>
                        </div>

                        <!-- Step 2: Select product image -->
                        <div class="trymylook-step trymylook-step-product">
                            <h3>Step 2: Select Product View</h3>
                            <div class="trymylook-product-images">
                                ${productImages.map((img, idx) => `
                                    <div class="trymylook-product-image ${idx === 0 ? 'selected' : ''}" data-url="${img.url}">
                                        <img src="${img.thumb}" alt="${img.alt}">
                                    </div>
                                `).join('')}
                            </div>
                        </div>

                        <!-- Generate button -->
                        <div class="trymylook-actions">
                            <button type="button" class="trymylook-generate-btn" disabled>
                                ${trymylook_vars.texts.generate_button || 'Generate Try-On'}
                            </button>
                        </div>

                        <!-- Loading state -->
                        <div class="trymylook-loading" style="display: none;">
                            <div class="trymylook-spinner"></div>
                            <p>Generating your virtual try-on...</p>
                            <small>This may take 30-60 seconds</small>
                        </div>

                        <!-- Result state -->
                        <div class="trymylook-result" style="display: none;">
                            <h3>Your Virtual Try-On Result</h3>
                            <div class="trymylook-result-image">
                                <img src="" alt="Try-on result">
                            </div>
                            <div class="trymylook-result-actions">
                                ${trymylook_vars.show_download ? '<button type="button" class="trymylook-download-btn">Download Image</button>' : ''}
                                <button type="button" class="trymylook-try-again-btn">Try Another Look</button>
                            </div>
                        </div>

                        <!-- Error state -->
                        <div class="trymylook-error" style="display: none;">
                            <div class="trymylook-error-icon">⚠️</div>
                            <p class="trymylook-error-message"></p>
                            <button type="button" class="trymylook-try-again-btn">Try Again</button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Append to body
        $('body').append(modalHTML);
        TryMyLook.modal = $('.trymylook-modal-overlay');

        // Set first product image as selected
        if (productImages.length > 0) {
            TryMyLook.selectedProductImage = productImages[0].url;
        }

        // Handle browse button click
        $(document).on('click', '.trymylook-browse-btn', function(e) {
            e.stopPropagation(); // Prevent event from bubbling to parent
            $('#trymylook-user-image').click();
        });

        // Handle upload area click
        $(document).on('click', '.trymylook-upload-placeholder', function() {
            $('#trymylook-user-image').click();
        });

        // Handle remove image
        $(document).on('click', '.trymylook-remove-image', function(e) {
            e.stopPropagation();
            removeUploadedImage();
        });
    }

    /**
     * Close the modal
     */
    function closeModal() {
        if (TryMyLook.modal) {
            TryMyLook.modal.fadeOut(300, function() {
                resetModal();
            });
        }
        
        // Clear polling interval
        if (TryMyLook.pollingInterval) {
            clearInterval(TryMyLook.pollingInterval);
            TryMyLook.pollingInterval = null;
        }
    }

    /**
     * Get product images from WooCommerce gallery
     */
    function getProductImages() {
        const images = [];

        // Get main product image
        const mainImage = $('.woocommerce-product-gallery__image img').first();
        if (mainImage.length) {
            // Try to get full-size image from various data attributes
            const fullImageUrl = mainImage.data('large_image') || 
                                mainImage.data('src') || 
                                mainImage.attr('data-large_image') ||
                                mainImage.parent('a').attr('href') ||
                                mainImage.attr('src');
            
            images.push({
                url: fullImageUrl,
                thumb: mainImage.attr('src'),
                alt: mainImage.attr('alt') || 'Product image'
            });
        }

        // Get gallery images
        $('.woocommerce-product-gallery__image').each(function(idx) {
            if (idx === 0) return; // Skip main image (already added)
            
            const img = $(this).find('img');
            if (img.length) {
                // Try to get full-size image from various data attributes
                const fullImageUrl = img.data('large_image') || 
                                    img.data('src') || 
                                    img.attr('data-large_image') ||
                                    $(this).find('a').attr('href') ||
                                    img.attr('src');
                
                images.push({
                    url: fullImageUrl,
                    thumb: img.attr('src'),
                    alt: img.attr('alt') || `Product image ${idx + 1}`
                });
            }
        });

        // Fallback: get from variations if no gallery
        if (images.length === 0) {
            const featuredImage = $('.wp-post-image');
            if (featuredImage.length) {
                // Try to get full-size image
                const fullImageUrl = featuredImage.data('large_image') || 
                                    featuredImage.data('src') || 
                                    featuredImage.attr('data-large_image') ||
                                    featuredImage.parent('a').attr('href') ||
                                    featuredImage.attr('src');
                
                images.push({
                    url: fullImageUrl,
                    thumb: featuredImage.attr('src'),
                    alt: featuredImage.attr('alt') || 'Product image'
                });
            }
        }

        // Log the images for debugging
        if (window.console && window.console.log) {
            console.log('TryMyLook: Found product images:', images);
        }

        return images;
    }

    /**
     * Handle file upload
     */
    function handleFileUpload(file) {
        if (!file || !file.type.startsWith('image/')) {
            showError('Please select a valid image file');
            return;
        }

        // Check file size (max 10MB)
        if (file.size > 10 * 1024 * 1024) {
            showError('Image size must be less than 10MB');
            return;
        }

        // Create preview
        const reader = new FileReader();
        reader.onload = function(e) {
            $('.trymylook-image-preview img').attr('src', e.target.result);
            $('.trymylook-upload-placeholder').hide();
            $('.trymylook-image-preview').show();
            $('.trymylook-step-upload').addClass('completed');
            
            TryMyLook.uploadedImage = file;
            updateGenerateButton();
        };
        reader.readAsDataURL(file);
    }

    /**
     * Remove uploaded image
     */
    function removeUploadedImage() {
        TryMyLook.uploadedImage = null;
        $('.trymylook-image-preview').hide();
        $('.trymylook-upload-placeholder').show();
        $('.trymylook-step-upload').removeClass('completed');
        $('#trymylook-user-image').val('');
        updateGenerateButton();
    }

    /**
     * Update generate button state
     */
    function updateGenerateButton() {
        const canGenerate = TryMyLook.uploadedImage && TryMyLook.selectedProductImage;
        $('.trymylook-generate-btn').prop('disabled', !canGenerate);
    }

    /**
     * Generate outfit via AJAX
     */
    function generateOutfit() {
        if (!TryMyLook.uploadedImage || !TryMyLook.selectedProductImage) {
            showError('Please upload your photo and select a product view');
            return;
        }

        // Hide steps and show loading
        $('.trymylook-step, .trymylook-actions').hide();
        $('.trymylook-loading').show();

        // Prepare form data
        const formData = new FormData();
        formData.append('user_image', TryMyLook.uploadedImage);
        formData.append('product_image_url', TryMyLook.selectedProductImage);
        formData.append('action', 'trymylook_generate');
        formData.append('nonce', trymylook_vars.nonce);

        // Log for debugging
        if (window.console && window.console.log) {
            console.log('TryMyLook: Sending product image URL:', TryMyLook.selectedProductImage);
        }

        // Send AJAX request
        $.ajax({
            url: trymylook_vars.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success && response.data.outfit_id) {
                    TryMyLook.outfitId = response.data.outfit_id;
                    startPolling(response.data.status_url);
                } else {
                    showError(response.data.message || 'Failed to start generation. Please try again.');
                    resetToSteps();
                }
            },
            error: function(xhr) {
                let errorMsg = 'An error occurred. Please try again.';
                if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMsg = xhr.responseJSON.data.message;
                }
                showError(errorMsg);
                resetToSteps();
            }
        });
    }

    /**
     * Start polling for generation status
     */
    function startPolling(statusUrl) {
        // Poll every 3 seconds
        TryMyLook.pollingInterval = setInterval(function() {
            checkStatus();
        }, 3000);

        // Also check immediately
        checkStatus();
    }

    /**
     * Check generation status
     */
    function checkStatus() {
        $.ajax({
            url: trymylook_vars.ajax_url,
            type: 'POST',
            data: {
                action: 'trymylook_check_status',
                nonce: trymylook_vars.nonce,
                outfit_id: TryMyLook.outfitId
            },
            success: function(response) {
                if (response.success) {
                    const status = response.data.status;

                    if (status === 'completed' && response.data.result_image_url) {
                        // Success!
                        clearInterval(TryMyLook.pollingInterval);
                        showResult(response.data.result_image_url);
                    } else if (status === 'failed') {
                        // Failed
                        clearInterval(TryMyLook.pollingInterval);
                        showError(response.data.error || 'Generation failed. Please try again.');
                    }
                    // If still pending or processing, keep polling
                } else {
                    clearInterval(TryMyLook.pollingInterval);
                    showError(response.data.message || 'Failed to check status. Please try again.');
                }
            },
            error: function() {
                clearInterval(TryMyLook.pollingInterval);
                showError('Connection error. Please try again.');
            }
        });
    }

    /**
     * Show result image
     */
    function showResult(imageUrl) {
        $('.trymylook-loading').hide();
        $('.trymylook-result-image img').attr('src', imageUrl);
        $('.trymylook-result').show();
    }

    /**
     * Show error message
     */
    function showError(message) {
        $('.trymylook-step, .trymylook-actions, .trymylook-loading, .trymylook-result').hide();
        $('.trymylook-error-message').text(message);
        $('.trymylook-error').show();
    }

    /**
     * Reset modal to initial state
     */
    function resetModal() {
        // Clear state
        TryMyLook.uploadedImage = null;
        TryMyLook.outfitId = null;
        
        if (TryMyLook.pollingInterval) {
            clearInterval(TryMyLook.pollingInterval);
            TryMyLook.pollingInterval = null;
        }

        // Reset UI
        $('.trymylook-step, .trymylook-actions').show();
        $('.trymylook-loading, .trymylook-result, .trymylook-error').hide();
        $('.trymylook-image-preview').hide();
        $('.trymylook-upload-placeholder').show();
        $('.trymylook-step-upload').removeClass('completed');
        $('#trymylook-user-image').val('');
        
        // Reset product selection to first image
        $('.trymylook-product-image').removeClass('selected').first().addClass('selected');
        const firstImage = $('.trymylook-product-image').first().data('url');
        if (firstImage) {
            TryMyLook.selectedProductImage = firstImage;
        }
        
        updateGenerateButton();
    }

    /**
     * Reset to steps (after error during generation)
     */
    function resetToSteps() {
        $('.trymylook-loading').hide();
        $('.trymylook-step, .trymylook-actions').show();
    }

    /**
     * Download result image
     */
    function downloadImage() {
        const imageUrl = $('.trymylook-result-image img').attr('src');
        if (!imageUrl) return;

        // Create temporary link and trigger download
        const link = document.createElement('a');
        link.href = imageUrl;
        link.download = 'trymylook-result.jpg';
        link.target = '_blank';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Initialize on document ready
    $(document).ready(function() {
        init();
    });

})(jQuery);
