<?php

/**
 * Plugin Name: TrustLens – Abuse & Fraud Prevention for WooCommerce
 * Plugin URI: https://webstepper.io/wordpress/plugins/trustlens
 * Description: Customer Trust Intelligence for WooCommerce. See your customers clearly.
 * Version: 1.1.4
 * Author: Webstepper
 * Author URI: https://webstepper.io
 * License: GPLv3 or later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 * Text Domain: trustlens
 * Domain Path: /languages
 * Requires at least: 6.4
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * WC requires at least: 8.0
 * WC tested up to: 9.5
 * Requires Plugins: woocommerce
 *
 *
 * @package TrustLens
 */
defined( 'ABSPATH' ) || exit;
if ( !function_exists( 'wstl_fs' ) ) {
    // Create a helper function for easy Freemius SDK access. Prefix wstl_ (4+ chars) per WordPress.org naming guidelines.
    function wstl_fs() {
        global $wstl_fs;
        if ( !isset( $wstl_fs ) ) {
            // Include Freemius SDK.
            require_once dirname( __FILE__ ) . '/vendor/freemius/start.php';
            $wstl_fs = fs_dynamic_init( array(
                'id'               => '22495',
                'slug'             => 'trustlens',
                'type'             => 'plugin',
                'public_key'       => 'pk_1758562d770e826016238888879ea',
                'is_premium'       => false,
                'is_premium_only'  => false,
                'has_addons'       => false,
                'has_paid_plans'   => true,
                'is_org_compliant' => true,
                'menu'             => array(
                    'slug'    => 'trustlens',
                    'support' => false,
                ),
                'is_live'          => true,
            ) );
        }
        return $wstl_fs;
    }

    // Init Freemius.
    wstl_fs();
    // Signal that SDK was initiated.
    do_action( 'wstl_fs_loaded' );
    // Hook uninstall cleanup to Freemius (replaces uninstall.php).
    wstl_fs()->add_action( 'after_uninstall', 'trustlens_uninstall' );
}
/**
 * Remove all TrustLens data on uninstall.
 *
 * Hooked to Freemius after_uninstall action. Only runs if the user
 * has enabled data removal in Settings > Data.
 *
 * @since 1.0.0
 */
function trustlens_uninstall() {
    global $wpdb;
    // Check if user wants to remove data on uninstall.
    $remove_data = get_option( 'trustlens_remove_data_on_uninstall', false );
    if ( !$remove_data ) {
        return;
    }
    // Remove custom tables.
    $tables = array(
        $wpdb->prefix . 'trustlens_customers',
        $wpdb->prefix . 'trustlens_events',
        $wpdb->prefix . 'trustlens_signals',
        $wpdb->prefix . 'trustlens_category_stats',
        $wpdb->prefix . 'trustlens_fingerprints',
        $wpdb->prefix . 'trustlens_webhook_logs',
        $wpdb->prefix . 'trustlens_automation_logs'
    );
    foreach ( $tables as $table ) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- DROP TABLE requires direct query, table names from $wpdb->prefix array are safe.
        $wpdb->query( "DROP TABLE IF EXISTS {$table}" );
    }
    // Remove options.
    $options = array(
        // Core settings.
        'trustlens_db_version',
        'trustlens_min_orders',
        'trustlens_enable_blocking',
        'trustlens_enable_dashboard',
        'trustlens_enable_order_warning',
        'trustlens_hold_segments',
        'trustlens_block_message',
        'trustlens_segment_thresholds',
        'trustlens_remove_data_on_uninstall',
        'trustlens_installed_timestamp',
        'trustlens_sync_status',
        'trustlens_api_key',
        'trustlens_category_weights',
        // Module settings.
        'trustlens_module_returns_enabled',
        'trustlens_module_orders_enabled',
        'trustlens_module_coupons_enabled',
        'trustlens_module_categories_enabled',
        'trustlens_module_chargebacks_enabled',
        'trustlens_module_linked_accounts_enabled',
        'trustlens_linked_accounts_penalty',
        'trustlens_returns_high_threshold',
        'trustlens_returns_critical_threshold',
        'trustlens_chargebacks_auto_block',
        'trustlens_coupons_block_linked_abuse',
        'trustlens_coupons_max_first_order',
        'trustlens_enable_linked_accounts',
        // Notification settings.
        'trustlens_enable_notifications',
        'trustlens_notification_email',
        'trustlens_notify_blocked_checkout',
        'trustlens_notify_welcome_summary',
        'trustlens_notify_weekly_summary',
        'trustlens_notify_high_risk_order',
        'trustlens_notify_segment_critical',
        'trustlens_enable_daily_digest',
        'trustlens_notify_high_value_order',
        'trustlens_notify_high_value_threshold',
        'trustlens_notify_high_value_score',
        'trustlens_notify_repeat_refunder',
        'trustlens_notify_repeat_refunder_count',
        'trustlens_notify_repeat_refunder_days',
        'trustlens_notify_velocity',
        'trustlens_notify_velocity_count',
        'trustlens_notify_velocity_hours',
        'trustlens_notify_score_recovery',
        'trustlens_notify_new_customer_risk',
        'trustlens_notify_new_customer_risk_score',
        'trustlens_notify_monthly_report',
        'trustlens_notify_chargeback_filed',
        'trustlens_welcome_summary_sent',
        // Automation settings.
        'trustlens_enable_automation',
        'trustlens_automation_rules',
        // Webhook settings.
        'trustlens_enable_webhooks',
        'trustlens_webhook_endpoints',
        'trustlens_webhook_events',
        // Scheduled reports settings.
        'trustlens_enable_scheduled_reports',
        'trustlens_enable_weekly_reports',
        'trustlens_enable_monthly_reports',
        'trustlens_report_email',
        'trustlens_report_schedules',
        'trustlens_weekly_report_recipients',
        'trustlens_monthly_report_recipients',
        // Payment Method Risk Controls settings.
        'trustlens_enable_payment_method_controls',
        'trustlens_payment_method_control_segments',
        'trustlens_payment_method_control_gateways',
        'trustlens_payment_method_control_min_total',
        'trustlens_payment_method_control_message',
        'trustlens_payment_method_control_velocity_enabled',
        'trustlens_payment_method_control_velocity_count',
        'trustlens_payment_method_control_velocity_hours',
        'trustlens_payment_method_control_linked_risk_enabled',
    );
    foreach ( $options as $option ) {
        delete_option( $option );
    }
    // Remove transients.
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk delete transients, patterns are hardcoded.
    $wpdb->query( "DELETE FROM {$wpdb->options}\n\t\t WHERE option_name LIKE '_transient_wstl\\_%'\n\t\t    OR option_name LIKE '_transient_timeout_wstl\\_%'\n\t\t    OR option_name LIKE '_transient_trustlens\\_%'\n\t\t    OR option_name LIKE '_transient_timeout_trustlens\\_%'" );
    // Remove WordPress cron events.
    $cron_events = array(
        'trustlens/daily_cleanup',
        'trustlens/daily_digest',
        'trustlens/weekly_summary',
        'trustlens/monthly_report',
        'trustlens/welcome_summary'
    );
    foreach ( $cron_events as $event ) {
        $timestamp = wp_next_scheduled( $event );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, $event );
        }
    }
    // Remove scheduled actions if Action Scheduler is available.
    if ( function_exists( 'as_unschedule_all_actions' ) ) {
        as_unschedule_all_actions( 'trustlens/calculate_score' );
        as_unschedule_all_actions( 'trustlens/sync_historical_data' );
    }
}

/**
 * Plugin constants.
 */
define( 'TRUSTLENS_VERSION', '1.1.4' );
define( 'TRUSTLENS_DB_VERSION', '1.0.0' );
define( 'TRUSTLENS_PLUGIN_FILE', __FILE__ );
define( 'TRUSTLENS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'TRUSTLENS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'TRUSTLENS_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
/*
 * Translations: When hosted on WordPress.org, WordPress 4.6+ automatically loads
 * translations for the plugin slug (trustlens) from translate.wordpress.org.
 * We do not call load_plugin_textdomain() so that the directory auto-loading is used.
 * @see https://make.wordpress.org/core/2024/10/21/i18n-improvements-6-7/
 */
/**
 * Check plugin requirements before loading.
 *
 * @since 1.0.0
 * @return bool True if requirements met, false otherwise.
 */
function wstl_check_requirements() : bool {
    $errors = array();
    // Check PHP version.
    if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
        $errors[] = sprintf( 
            /* translators: %s: Required PHP version */
            __( 'TrustLens requires PHP %s or higher.', 'trustlens' ),
            '7.4'
         );
    }
    // Check WordPress version.
    if ( version_compare( get_bloginfo( 'version' ), '6.4', '<' ) ) {
        $errors[] = sprintf( 
            /* translators: %s: Required WordPress version */
            __( 'TrustLens requires WordPress %s or higher.', 'trustlens' ),
            '6.4'
         );
    }
    // Display errors if any.
    if ( !empty( $errors ) ) {
        add_action( 'admin_notices', function () use($errors) {
            echo '<div class="notice notice-error"><p>';
            echo esc_html( implode( ' ', $errors ) );
            echo '</p></div>';
        } );
        return false;
    }
    return true;
}

/**
 * Check if WooCommerce is active.
 *
 * @since 1.0.0
 * @return bool True if WooCommerce is active.
 */
function wstl_is_woocommerce_active() : bool {
    return class_exists( 'WooCommerce' );
}

/**
 * Display WooCommerce required notice.
 *
 * Only shows on WordPress < 6.5 where the Requires Plugins header
 * is not supported. On 6.5+, WordPress handles dependency notices natively.
 *
 * @since 1.0.0
 */
function wstl_woocommerce_required_notice() : void {
    // WordPress 6.5+ handles dependency notices via the Requires Plugins header.
    if ( version_compare( get_bloginfo( 'version' ), '6.5', '>=' ) ) {
        return;
    }
    if ( !current_user_can( 'activate_plugins' ) ) {
        return;
    }
    $install_url = admin_url( 'plugin-install.php?s=woocommerce&tab=search&type=term' );
    echo '<div class="notice notice-warning"><p>';
    printf( 
        /* translators: %1$s: opening link tag, %2$s: closing link tag */
        esc_html__( 'TrustLens requires WooCommerce to be installed and active. %1$sInstall WooCommerce%2$s', 'trustlens' ),
        '<a href="' . esc_url( $install_url ) . '">',
        '</a>'
     );
    echo '</p></div>';
}

/**
 * Initialize the plugin.
 *
 * @since 1.0.0
 */
function wstl_init() : void {
    // Check WooCommerce.
    if ( !wstl_is_woocommerce_active() ) {
        add_action( 'admin_notices', 'wstl_woocommerce_required_notice' );
        return;
    }
    // Load plugin files.
    require_once TRUSTLENS_PLUGIN_DIR . 'includes/functions.php';
    require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-trustlens.php';
    // Initialize main class.
    TrustLens::instance();
}

/**
 * Run on plugin activation.
 *
 * @since 1.0.0
 */
function wstl_activate() : void {
    // Check requirements first.
    if ( !wstl_check_requirements() ) {
        return;
    }
    // Load install class and run activation.
    require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-install.php';
    TrustLens_Install::activate();
}

/**
 * Redirect to TrustLens dashboard immediately after activation.
 *
 * This provides a reliable redirect path on each activation, including
 * cases where transient-based redirects may be skipped.
 *
 * @since 1.1.1
 * @param string $plugin       Activated plugin basename.
 * @param bool   $network_wide Whether activated network-wide.
 */
function wstl_redirect_after_activation(  string $plugin, bool $network_wide  ) : void {
    if ( TRUSTLENS_PLUGIN_BASENAME !== $plugin ) {
        return;
    }
    if ( wp_doing_ajax() || defined( 'WP_CLI' ) && WP_CLI ) {
        return;
    }
    // Do not redirect on multisite network activation or bulk activation.
    // phpcs:disable WordPress.Security.NonceVerification.Recommended
    if ( $network_wide || is_network_admin() || isset( $_GET['activate-multi'] ) ) {
        return;
    }
    // phpcs:enable WordPress.Security.NonceVerification.Recommended
    wp_safe_redirect( admin_url( 'admin.php?page=trustlens' ) );
    exit;
}

/**
 * Run on plugin deactivation.
 *
 * @since 1.0.0
 */
function wstl_deactivate() : void {
    require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-install.php';
    if ( file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-scheduled-reports.php' ) ) {
        require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-scheduled-reports.php';
    }
    TrustLens_Install::deactivate();
}

// Check requirements.
if ( !wstl_check_requirements() ) {
    return;
}
// Register custom cron schedules.
add_filter( 'cron_schedules', 'wstl_cron_schedules' );
/**
 * Add custom cron intervals.
 *
 * @since 1.0.0
 * @param array $schedules Existing cron schedules.
 * @return array Modified schedules.
 */
function wstl_cron_schedules(  array $schedules  ) : array {
    if ( !isset( $schedules['weekly'] ) ) {
        $schedules['weekly'] = array(
            'interval' => WEEK_IN_SECONDS,
            'display'  => __( 'Once Weekly', 'trustlens' ),
        );
    }
    if ( !isset( $schedules['monthly'] ) ) {
        $schedules['monthly'] = array(
            'interval' => MONTH_IN_SECONDS,
            'display'  => __( 'Once Monthly', 'trustlens' ),
        );
    }
    return $schedules;
}

// Register activation/deactivation hooks.
register_activation_hook( __FILE__, 'wstl_activate' );
register_deactivation_hook( __FILE__, 'wstl_deactivate' );
add_action(
    'activated_plugin',
    'wstl_redirect_after_activation',
    20,
    2
);
// Initialize plugin after plugins are loaded.
add_action( 'plugins_loaded', 'wstl_init' );
/**
 * Declare HPOS compatibility.
 *
 * @since 1.0.0
 */
add_action( 'before_woocommerce_init', function () {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );