<?php
/**
 * Public API functions.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Get email hash from email address.
 *
 * @since 1.0.0
 * @param string $email Email address.
 * @return string MD5 hash of lowercase trimmed email.
 */
function wstl_get_email_hash( string $email ): string {
	return md5( strtolower( trim( $email ) ) );
}

/**
 * Get customer key from order.
 *
 * @since 1.0.0
 * @param WC_Order $order WooCommerce order.
 * @return array Customer key data.
 */
function wstl_get_customer_key( WC_Order $order ): array {
	$email = $order->get_billing_email();
	$email_hash = wstl_get_email_hash( $email );
	$customer_id = $order->get_customer_id();

	return array(
		'email_hash'    => $email_hash,
		'customer_id'   => $customer_id > 0 ? $customer_id : null,
		'customer_type' => $customer_id > 0 ? 'user' : 'guest',
		'email'         => $email,
	);
}

/**
 * Get customer by email hash.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @return object|null Customer object or null.
 */
function wstl_get_customer( string $email_hash ): ?object {
	$repo = new TrustLens_Customer_Repository();
	return $repo->get_by_hash( $email_hash );
}

/**
 * Get customer by email address.
 *
 * @since 1.0.0
 * @param string $email Email address.
 * @return object|null Customer object or null.
 */
function wstl_get_customer_by_email( string $email ): ?object {
	return wstl_get_customer( wstl_get_email_hash( $email ) );
}

/**
 * Update customer data.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @param array  $data       Data to update.
 * @return bool True on success.
 */
function wstl_update_customer( string $email_hash, array $data ): bool {
	$repo = new TrustLens_Customer_Repository();
	return $repo->update( $email_hash, $data );
}

/**
 * Create or update customer record.
 *
 * @since 1.0.0
 * @param array $data Customer data.
 * @return bool True on success.
 */
function wstl_upsert_customer( array $data ): bool {
	$repo = new TrustLens_Customer_Repository();
	return $repo->upsert( $data );
}

/**
 * Ensure customer record exists.
 *
 * @since 1.0.0
 * @param array    $customer_key Customer key from wstl_get_customer_key().
 * @param WC_Order $order        Order object.
 * @return bool True on success.
 */
function wstl_ensure_customer_record( array $customer_key, WC_Order $order ): bool {
	$existing = wstl_get_customer( $customer_key['email_hash'] );

	if ( $existing ) {
		return true;
	}

	return wstl_upsert_customer( array(
		'email_hash'     => $customer_key['email_hash'],
		'customer_id'    => $customer_key['customer_id'],
		'customer_email' => $customer_key['email'],
		'customer_type'  => $customer_key['customer_type'],
	) );
}

/**
 * Get trust score for customer.
 *
 * @since 1.0.0
 * @param string $email_or_hash Email address or hash.
 * @return int Trust score (0-100).
 */
function wstl_get_trust_score( string $email_or_hash ): int {
	$hash = strlen( $email_or_hash ) === 32 ? $email_or_hash : wstl_get_email_hash( $email_or_hash );
	$customer = wstl_get_customer( $hash );

	return $customer ? (int) $customer->trust_score : 50;
}

/**
 * Get segment for customer.
 *
 * @since 1.0.0
 * @param string $email_or_hash Email address or hash.
 * @return string Segment name.
 */
function wstl_get_segment( string $email_or_hash ): string {
	$hash = strlen( $email_or_hash ) === 32 ? $email_or_hash : wstl_get_email_hash( $email_or_hash );
	$customer = wstl_get_customer( $hash );

	return $customer ? $customer->segment : 'normal';
}

/**
 * Check if customer is blocked.
 *
 * @since 1.0.0
 * @param string $email_or_hash Email address or hash.
 * @return bool True if blocked.
 */
function wstl_is_blocked( string $email_or_hash ): bool {
	$hash = strlen( $email_or_hash ) === 32 ? $email_or_hash : wstl_get_email_hash( $email_or_hash );
	$customer = wstl_get_customer( $hash );

	return $customer && (bool) $customer->is_blocked;
}

/**
 * Check if customer is allowlisted.
 *
 * @since 1.0.0
 * @param string $email_or_hash Email address or hash.
 * @return bool True if allowlisted.
 */
function wstl_is_allowlisted( string $email_or_hash ): bool {
	$hash = strlen( $email_or_hash ) === 32 ? $email_or_hash : wstl_get_email_hash( $email_or_hash );
	$customer = wstl_get_customer( $hash );

	return $customer && (bool) $customer->is_allowlisted;
}

/**
 * Block a customer.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @param string $reason     Optional reason.
 * @return bool True on success.
 */
function wstl_block_customer( string $email_hash, string $reason = '' ): bool {
	$data = array( 'is_blocked' => 1 );

	if ( $reason ) {
		$customer = wstl_get_customer( $email_hash );
		$notes = $customer ? $customer->admin_notes : '';
		$notes .= "\n[" . current_time( 'mysql' ) . "] Blocked: " . $reason;
		$data['admin_notes'] = trim( $notes );
	}

	$result = wstl_update_customer( $email_hash, $data );

	if ( $result ) {
		do_action( 'trustlens/customer_blocked', $email_hash, $reason );
	}

	return $result;
}

/**
 * Unblock a customer.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @return bool True on success.
 */
function wstl_unblock_customer( string $email_hash ): bool {
	return wstl_update_customer( $email_hash, array( 'is_blocked' => 0 ) );
}

/**
 * Add customer to allowlist.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @return bool True on success.
 */
function wstl_allowlist_customer( string $email_hash ): bool {
	$result = wstl_update_customer( $email_hash, array(
		'is_allowlisted' => 1,
		'trust_score'    => 100,
		'segment'        => 'vip',
	) );

	if ( $result ) {
		do_action( 'trustlens/customer_allowlisted', $email_hash );
	}

	return $result;
}

/**
 * Remove customer from allowlist.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @return bool True on success.
 */
function wstl_remove_from_allowlist( string $email_hash ): bool {
	$result = wstl_update_customer( $email_hash, array( 'is_allowlisted' => 0 ) );

	if ( $result ) {
		// Recalculate score.
		wstl_queue_score_update( $email_hash, 0 );
	}

	return $result;
}

/**
 * Queue customer for score recalculation.
 *
 * @since 1.0.0
 * @param string $email_hash Customer email hash.
 * @param int    $delay      Delay in seconds.
 */
function wstl_queue_score_update( string $email_hash, int $delay = 60 ): void {
	// Check if Action Scheduler is available.
	if ( ! function_exists( 'as_schedule_single_action' ) ) {
		// Fallback: calculate immediately.
		do_action( 'trustlens/calculate_score', $email_hash );
		return;
	}

	// Check for existing pending action.
	$existing = as_get_scheduled_actions( array(
		'hook'   => 'trustlens/calculate_score',
		'args'   => array( $email_hash ),
		'status' => ActionScheduler_Store::STATUS_PENDING,
	), 'ids' );

	if ( ! empty( $existing ) ) {
		return; // Already scheduled.
	}

	as_schedule_single_action(
		time() + $delay,
		'trustlens/calculate_score',
		array( $email_hash ),
		'trustlens'
	);
}

/**
 * Log an event.
 *
 * @since 1.0.0
 * @param string   $email_hash Customer email hash.
 * @param string   $event_type Event type.
 * @param array    $event_data Event data.
 * @param int|null $order_id   Optional order ID.
 * @return bool True on success.
 */
function wstl_log_event( string $email_hash, string $event_type, array $event_data = array(), ?int $order_id = null ): bool {
	global $wpdb;

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
	$result = $wpdb->insert(
		$wpdb->prefix . 'trustlens_events',
		array(
			'email_hash' => $email_hash,
			'event_type' => $event_type,
			'event_data' => wp_json_encode( $event_data ),
			'order_id'   => $order_id,
			'created_at' => current_time( 'mysql' ),
		),
		array( '%s', '%s', '%s', '%d', '%s' )
	);

	if ( $result ) {
		do_action( 'trustlens/event_logged', $email_hash, $event_type, $event_data );
	}

	return (bool) $result;
}

/**
 * Get segment from trust score.
 *
 * @since 1.0.0
 * @param int $score Trust score.
 * @return string Segment name.
 */
function wstl_get_segment_from_score( int $score ): string {
	$thresholds = get_option( 'trustlens_segment_thresholds', array() );

	$defaults = array(
		'vip'      => 90,
		'trusted'  => 70,
		'normal'   => 50,
		'caution'  => 30,
		'risk'     => 10,
		'critical' => 0,
	);

	$thresholds = wp_parse_args( $thresholds, $defaults );
	$thresholds = apply_filters( 'trustlens/segment_thresholds', $thresholds );

	if ( $score >= $thresholds['vip'] ) {
		return 'vip';
	}
	if ( $score >= $thresholds['trusted'] ) {
		return 'trusted';
	}
	if ( $score >= $thresholds['normal'] ) {
		return 'normal';
	}
	if ( $score >= $thresholds['caution'] ) {
		return 'caution';
	}
	if ( $score >= $thresholds['risk'] ) {
		return 'risk';
	}

	return 'critical';
}

/**
 * Get segment display label.
 *
 * @since 1.0.0
 * @param string $segment Segment key.
 * @return string Display label.
 */
function wstl_get_segment_label( string $segment ): string {
	$labels = array(
		'vip'      => __( 'VIP', 'trustlens' ),
		'trusted'  => __( 'Trusted', 'trustlens' ),
		'normal'   => __( 'Normal', 'trustlens' ),
		'caution'  => __( 'Caution', 'trustlens' ),
		'risk'     => __( 'Risk', 'trustlens' ),
		'critical' => __( 'Critical', 'trustlens' ),
	);

	return $labels[ $segment ] ?? ucfirst( $segment );
}

/**
 * Get segment color.
 *
 * @since 1.0.0
 * @param string $segment Segment key.
 * @return string Hex color.
 */
function wstl_get_segment_color( string $segment ): string {
	$colors = array(
		'vip'      => '#28a745',
		'trusted'  => '#17a2b8',
		'normal'   => '#6c757d',
		'caution'  => '#ffc107',
		'risk'     => '#fd7e14',
		'critical' => '#dc3545',
	);

	return $colors[ $segment ] ?? '#6c757d';
}

/**
 * Get active modules.
 *
 * @since 1.0.0
 * @return array Active module instances.
 */
function wstl_get_active_modules(): array {
	return TrustLens::instance()->get_active_modules();
}

/**
 * Calculate days between two dates.
 *
 * @since 1.0.0
 * @param mixed $start Start date.
 * @param mixed $end   End date.
 * @return int Number of days.
 */
function wstl_days_between( $start, $end ): int {
	$start_time = is_numeric( $start ) ? $start : strtotime( $start );
	$end_time = is_numeric( $end ) ? $end : strtotime( $end );

	return (int) floor( abs( $end_time - $start_time ) / DAY_IN_SECONDS );
}

/**
 * Delete a customer and all related data.
 *
 * @since 1.1.0
 * @param string $email_hash Customer email hash.
 * @return bool True on success.
 */
function wstl_delete_customer( string $email_hash ): bool {
	global $wpdb;

	$customers_table = $wpdb->prefix . 'trustlens_customers';
	$events_table = $wpdb->prefix . 'trustlens_events';
	$signals_table = $wpdb->prefix . 'trustlens_signals';
	$fingerprints_table = $wpdb->prefix . 'trustlens_fingerprints';
	$category_stats_table = $wpdb->prefix . 'trustlens_category_stats';

	// Delete customer record.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
	$deleted = $wpdb->delete(
		$customers_table,
		array( 'email_hash' => $email_hash ),
		array( '%s' )
	);

	if ( false === $deleted ) {
		return false;
	}

	// Delete related events.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
	$wpdb->delete( $events_table, array( 'email_hash' => $email_hash ), array( '%s' ) );

	// Delete related signals.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
	$wpdb->delete( $signals_table, array( 'email_hash' => $email_hash ), array( '%s' ) );

	// Delete fingerprints.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
	$wpdb->delete( $fingerprints_table, array( 'email_hash' => $email_hash ), array( '%s' ) );

	// Delete category stats.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
	$wpdb->delete( $category_stats_table, array( 'email_hash' => $email_hash ), array( '%s' ) );

	/**
	 * Fires after a customer is deleted.
	 *
	 * @since 1.1.0
	 * @param string $email_hash Deleted customer's email hash.
	 */
	do_action( 'trustlens/customer_deleted', $email_hash );

	return true;
}

/**
 * Check if Pro license is active.
 *
 * Uses Freemius __premium_only method to strip from free version.
 *
 * @since 1.2.0
 * @return bool True if Pro is active.
 */
function wstl_is_pro(): bool {
	if ( function_exists( 'wstl_fs' ) ) {
		$fs = wstl_fs();
		if ( is_object( $fs ) ) {
			if ( method_exists( $fs, 'can_use_premium_code__premium_only' ) && $fs->can_use_premium_code__premium_only() ) {
				return true;
			}
			if ( method_exists( $fs, 'can_use_premium_code' ) && $fs->can_use_premium_code() ) {
				return true;
			}
		}
	}

	return (bool) apply_filters( 'trustlens/is_pro_active', false );
}

/**
 * Check if Pro features can run on this site.
 *
 * Requires both:
 * - Pro code present in current build.
 * - Active paid license.
 *
 * @since 1.2.0
 * @return bool True if Pro features are available and licensed.
 */
function wstl_can_use_pro(): bool {
	return wstl_has_pro_features() && wstl_is_pro();
}

/**
 * Check if Webhooks feature is present in this build (no license check).
 * Used for WordPress.org: free build has no Pro code; show only promotional upsell when false.
 *
 * @since 1.0.0
 * @return bool True if TrustLens_Webhooks class exists.
 */
function wstl_has_webhooks(): bool {
	return class_exists( 'TrustLens_Webhooks' );
}

/**
 * Check if Automation feature is present in this build (no license check).
 *
 * @since 1.0.0
 * @return bool True if TrustLens_Automation class exists.
 */
function wstl_has_automation(): bool {
	return class_exists( 'TrustLens_Automation' );
}

/**
 * Check if Scheduled Reports feature is present in this build (no license check).
 *
 * @since 1.0.0
 * @return bool True if TrustLens_Scheduled_Reports class exists.
 */
function wstl_has_scheduled_reports(): bool {
	return class_exists( 'TrustLens_Scheduled_Reports' );
}

/**
 * Check if Chargebacks module is present in this build (no license check).
 *
 * @since 1.0.0
 * @return bool True if TrustLens_Module_Chargebacks class exists.
 */
function wstl_has_chargebacks(): bool {
	return class_exists( 'TrustLens_Module_Chargebacks' );
}

/**
 * Check if Payment Method Risk Controls feature is present in this build (no license check).
 *
 * @since 1.0.6
 * @return bool True if TrustLens_Payment_Method_Controls class exists.
 */
function wstl_has_payment_method_controls(): bool {
	return class_exists( 'TrustLens_Payment_Method_Controls' );
}

/**
 * Check if any Pro feature is present in this build (no license check).
 * Used to show Pro UI/options only when Pro code is in the build (WordPress.org compliant).
 *
 * @since 1.0.0
 * @return bool True if any Pro feature class exists.
 */
function wstl_has_pro_features(): bool {
	return wstl_has_webhooks()
		|| wstl_has_automation()
		|| wstl_has_scheduled_reports()
		|| wstl_has_chargebacks()
		|| wstl_has_payment_method_controls();
}

/**
 * Whether to show Pro upsell content (cards, strip, locked lists).
 * True when the build has no Pro code (WordPress.org) OR when the user has a free license.
 * Ensures full-build + free-license (e.g. local dev) sees the same promotions as wp.org users.
 *
 * @since 1.2.0
 * @return bool True if upsells should be shown.
 */
function wstl_show_pro_upsells(): bool {
	return ! wstl_has_pro_features() || wstl_is_free_user();
}

/**
 * Check if user is not paying (for showing upsells).
 *
 * @since 1.2.0
 * @return bool True if user is on free plan.
 */
function wstl_is_free_user(): bool {
	if ( ! function_exists( 'wstl_fs' ) ) {
		return true;
	}

	return wstl_fs()->is_not_paying();
}

/**
 * Get upgrade URL for upsell links.
 *
 * @since 1.2.0
 * @return string Upgrade URL.
 */
function wstl_get_upgrade_url(): string {
	if ( ! function_exists( 'wstl_fs' ) ) {
		return '';
	}

	return wstl_fs()->get_upgrade_url();
}

/**
 * Get fallback URL when Freemius upgrade URL is not available.
 * Used so CTAs are never dead (e.g. on WordPress.org build).
 *
 * @since 1.2.0
 * @return string Fallback URL (plugin/pricing page).
 */
function wstl_get_upgrade_fallback_url(): string {
	return apply_filters( 'trustlens_upgrade_fallback_url', 'https://webstepper.io/wordpress/plugins/trustlens' );
}

/**
 * Get upgrade URL or fallback for use in upsell links.
 * Ensures free users always have a clickable CTA.
 *
 * @since 1.2.0
 * @return string Upgrade or fallback URL.
 */
function wstl_get_upgrade_or_fallback_url(): string {
	$url = wstl_get_upgrade_url();
	return $url ? $url : wstl_get_upgrade_fallback_url();
}

/**
 * Get checkout URL for direct purchase links.
 *
 * @since 1.2.0
 * @param array $params Optional checkout parameters.
 * @return string Checkout URL.
 */
function wstl_get_checkout_url( array $params = array() ): string {
	if ( ! function_exists( 'wstl_fs' ) ) {
		return '';
	}

	return wstl_fs()->checkout_url( '', $params );
}

/**
 * Render Pro badge with optional upsell link.
 *
 * @since 1.2.0
 * @param bool $with_link Include upgrade link for free users.
 * @return string HTML badge.
 */
function wstl_pro_badge( bool $with_link = true ): string {
	$html = '<span class="wstl-badge-pro">Pro</span>';

	if ( $with_link && wstl_is_free_user() ) {
		$url = wstl_get_upgrade_or_fallback_url();
		if ( $url ) {
			$html = sprintf( '<a href="%s" class="wstl-pro-link">%s</a>', esc_url( $url ), $html );
		}
	}

	return $html;
}

/**
 * Get trigger label for automation rules.
 *
 * @since 1.1.0
 * @param string $trigger Trigger key.
 * @return string Label.
 */
function wstl_get_trigger_label( string $trigger ): string {
	$triggers = array(
		'score_updated'     => __( 'Score Updated', 'trustlens' ),
		'segment_changed'   => __( 'Segment Changed', 'trustlens' ),
		'order_completed'   => __( 'Order Completed', 'trustlens' ),
		'refund_processed'  => __( 'Refund Processed', 'trustlens' ),
		'checkout_blocked'  => __( 'Checkout Blocked', 'trustlens' ),
		'score_change'      => __( 'Score Updated', 'trustlens' ),
		'new_order'         => __( 'Order Completed', 'trustlens' ),
		'refund'            => __( 'Refund Processed', 'trustlens' ),
	);
	return $triggers[ $trigger ] ?? ucfirst( str_replace( '_', ' ', $trigger ) );
}

/**
 * Get action label for automation rules.
 *
 * @since 1.1.0
 * @param string $action Action key.
 * @return string Label.
 */
function wstl_get_action_label( string $action ): string {
	$actions = array(
		'send_email'        => __( 'Send Email', 'trustlens' ),
		'add_note'          => __( 'Add Note', 'trustlens' ),
		'block_customer'    => __( 'Block Customer', 'trustlens' ),
		'flag_review'       => __( 'Flag for Review', 'trustlens' ),
		'flag_order'        => __( 'Flag for Review', 'trustlens' ),
		'send_webhook'      => __( 'Send Webhook', 'trustlens' ),
		'send_alert'        => __( 'Send Email', 'trustlens' ),
		'add_to_watch'      => __( 'Flag for Review', 'trustlens' ),
		'hold_order'        => __( 'Hold Order', 'trustlens' ),
		'require_verification' => __( 'Require Verification', 'trustlens' ),
		'add_tag'           => __( 'Add Tag', 'trustlens' ),
	);
	return $actions[ $action ] ?? ucfirst( str_replace( '_', ' ', $action ) );
}

/**
 * Get report type label for scheduled reports.
 *
 * @since 1.1.0
 * @param string $type Report type.
 * @return string Label.
 */
function wstl_get_report_type_label( string $type ): string {
	$types = array(
		'summary'          => __( 'Summary Report', 'trustlens' ),
		'high_risk'        => __( 'High-Risk Customers', 'trustlens' ),
		'segment_changes'  => __( 'Segment Changes', 'trustlens' ),
		'blocked_attempts' => __( 'Blocked Attempts', 'trustlens' ),
		'trends'           => __( 'Trust Trends', 'trustlens' ),
	);
	return $types[ $type ] ?? ucfirst( str_replace( '_', ' ', $type ) );
}

/**
 * Render a Pro upsell panel for gated features.
 *
 * Provides a consistent, in-context upsell pattern with:
 * - Value highlights.
 * - Optional Free vs Pro comparison rows.
 * - Primary and secondary CTAs.
 *
 * @since 1.2.0
 * @param string      $feature_name    Feature name to display.
 * @param string      $description     Feature description.
 * @param array       $features        Value bullets (benefits in Pro).
 * @param array       $comparison_rows Optional comparison rows.
 *                                     Each row format: [ label, free_value, pro_value ].
 * @param string      $kicker          Optional eyebrow text above title.
 * @param string|null $trust_line      Optional trust line under CTA.
 */
function wstl_render_pro_upsell(
	string $feature_name,
	string $description,
	array $features = array(),
	array $comparison_rows = array(),
	string $kicker = '',
	?string $trust_line = null
): void {
	$upgrade_url = wstl_get_upgrade_or_fallback_url();
	$kicker      = $kicker ? $kicker : __( 'Available in Pro', 'trustlens' );
	if ( null === $trust_line ) {
		$trust_line = __( 'Secure checkout powered by Freemius. Cancel anytime.', 'trustlens' );
	}
	?>
	<div class="wstl-card wstl-pro-upsell">
		<div class="wstl-pro-upsell-content">
			<p class="wstl-pro-kicker">
				<?php echo esc_html( $kicker ); ?>
				<span class="wstl-badge-pro"><?php esc_html_e( 'Pro', 'trustlens' ); ?></span>
			</p>
			<h2><?php echo esc_html( $feature_name ); ?></h2>
			<p class="wstl-pro-upsell-description"><?php echo esc_html( $description ); ?></p>

			<?php if ( ! empty( $features ) ) : ?>
				<ul class="wstl-pro-upsell-features" role="list">
					<?php foreach ( $features as $f ) : ?>
						<li>
							<span class="dashicons dashicons-yes-alt" aria-hidden="true"></span>
							<span><?php echo esc_html( $f ); ?></span>
						</li>
					<?php endforeach; ?>
				</ul>
			<?php endif; ?>

			<?php if ( ! empty( $comparison_rows ) ) : ?>
				<div class="wstl-pro-compare" role="table" aria-label="<?php esc_attr_e( 'Free vs Pro comparison', 'trustlens' ); ?>">
					<div class="wstl-pro-compare-head" role="row">
						<div role="columnheader"></div>
						<div role="columnheader"><?php esc_html_e( 'Free', 'trustlens' ); ?></div>
						<div role="columnheader"><?php esc_html_e( 'Pro', 'trustlens' ); ?></div>
					</div>
					<?php foreach ( $comparison_rows as $row ) : ?>
						<?php
						$label = isset( $row[0] ) ? (string) $row[0] : '';
						$free  = isset( $row[1] ) ? (string) $row[1] : '';
						$pro   = isset( $row[2] ) ? (string) $row[2] : '';
						?>
						<div class="wstl-pro-compare-row" role="row">
							<div role="cell"><?php echo esc_html( $label ); ?></div>
							<div role="cell"><?php echo esc_html( $free ); ?></div>
							<div role="cell"><?php echo esc_html( $pro ); ?></div>
						</div>
					<?php endforeach; ?>
				</div>
			<?php endif; ?>

			<?php if ( $upgrade_url ) : ?>
				<div class="wstl-pro-cta-group">
					<a href="<?php echo esc_url( $upgrade_url ); ?>" class="button button-primary button-hero wstl-pro-upsell-button">
						<?php esc_html_e( 'Upgrade to Pro', 'trustlens' ); ?>
					</a>
					<a href="<?php echo esc_url( $upgrade_url ); ?>" class="button wstl-pro-upsell-secondary">
						<?php esc_html_e( 'See Pro Features', 'trustlens' ); ?>
					</a>
				</div>
				<p class="wstl-pro-upsell-trust"><?php echo esc_html( $trust_line ); ?></p>
			<?php endif; ?>
		</div>
	</div>
	<?php
}
