<?php
/**
 * Main TrustLens class.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * TrustLens main class.
 *
 * @since 1.0.0
 */
final class TrustLens {

	/**
	 * Single instance of the class.
	 *
	 * @var TrustLens|null
	 */
	private static ?TrustLens $instance = null;

	/**
	 * Registered modules.
	 *
	 * @var array
	 */
	private array $modules = array();

	/**
	 * Get the single instance.
	 *
	 * @since 1.0.0
	 * @return TrustLens
	 */
	public static function instance(): TrustLens {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Prevent cloning.
	 *
	 * @since 1.0.0
	 */
	private function __clone() {}

	/**
	 * Prevent unserializing.
	 *
	 * @since 1.0.0
	 * @throws \Exception Always throws.
	 */
	public function __wakeup() {
		throw new \Exception( 'Cannot unserialize singleton' );
	}

	/**
	 * Load required files.
	 *
	 * @since 1.0.0
	 */
	private function load_dependencies(): void {
		// Data layer.
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-customer-repository.php';

		// Modules.
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-module.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/modules/class-module-returns.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/modules/class-module-orders.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/modules/class-module-coupons.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/modules/class-module-categories.php';

		// Services.
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-score-calculator.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-historical-sync.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-checkout-blocker.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-notifications.php';
		if ( wstl_is_pro() && file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-notifications-pro.php' ) ) {
			require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-notifications-pro.php';
		}
		if ( wstl_is_pro() && file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-payment-method-controls.php' ) ) {
			require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-payment-method-controls.php';
		}
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-data-export.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-rest-api.php';

		// Phase 3: Advanced features.
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-analytics.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-linked-accounts.php';
		require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-bulk-operations.php';

		// Pro features (files stripped from free version by Freemius deployment).
		if ( wstl_is_pro() ) {
			if ( file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/modules/class-module-chargebacks.php' ) ) {
				require_once TRUSTLENS_PLUGIN_DIR . 'includes/modules/class-module-chargebacks.php';
			}
			if ( file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-automation.php' ) ) {
				require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-automation.php';
			}
			if ( file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-webhooks.php' ) ) {
				require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-webhooks.php';
			}
			if ( file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-scheduled-reports.php' ) ) {
				require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-scheduled-reports.php';
			}
		}

		// Admin.
		if ( is_admin() ) {
			require_once TRUSTLENS_PLUGIN_DIR . 'admin/class-admin.php';

			// Test data seeder (stripped from production builds).
			if ( file_exists( TRUSTLENS_PLUGIN_DIR . 'includes/class-test-data-seeder.php' ) ) {
				require_once TRUSTLENS_PLUGIN_DIR . 'includes/class-test-data-seeder.php';
			}
		}
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks(): void {
		// Initialize components.
		add_action( 'init', array( $this, 'init_modules' ) );

		// Register Action Scheduler hooks.
		add_action( 'trustlens/calculate_score', array( $this, 'process_score_calculation' ) );

		// Initialize historical sync.
		TrustLens_Historical_Sync::instance();

		// Initialize notifications.
		TrustLens_Notifications::instance();
		if ( class_exists( 'TrustLens_Notifications_Pro' ) && wstl_can_use_pro() ) {
			TrustLens_Notifications_Pro::instance();
		}
		if ( class_exists( 'TrustLens_Payment_Method_Controls' ) && wstl_can_use_pro() ) {
			TrustLens_Payment_Method_Controls::instance();
		}

		// Initialize data export.
		TrustLens_Data_Export::instance();

		// Initialize REST API.
		TrustLens_REST_API::instance();

		// Phase 3: Initialize advanced features.
		TrustLens_Linked_Accounts::instance();

		// Pro features (classes stripped from free version by Freemius deployment).
		if ( class_exists( 'TrustLens_Automation' ) && wstl_can_use_pro() ) {
			TrustLens_Automation::instance();
		}
		if ( class_exists( 'TrustLens_Webhooks' ) && wstl_can_use_pro() ) {
			TrustLens_Webhooks::instance();
		}
		if ( class_exists( 'TrustLens_Scheduled_Reports' ) && wstl_can_use_pro() ) {
			TrustLens_Scheduled_Reports::instance();
			add_action( 'init', array( 'TrustLens_Scheduled_Reports', 'schedule_reports' ) );
		}

		// Initialize checkout blocker (frontend).
		if ( ! is_admin() ) {
			TrustLens_Checkout_Blocker::instance();
		}

		// Initialize admin.
		if ( is_admin() ) {
			TrustLens_Admin::instance();

			// Test data seeder (stripped from production builds).
			if ( class_exists( 'TrustLens_Test_Data_Seeder' ) ) {
				TrustLens_Test_Data_Seeder::instance();
			}
		}
	}

	/**
	 * Initialize detection modules.
	 *
	 * @since 1.0.0
	 */
	public function init_modules(): void {
		// Register core modules (Free).
		$this->register_module( new TrustLens_Module_Returns() );
		$this->register_module( new TrustLens_Module_Orders() );
		$this->register_module( new TrustLens_Module_Coupons() );
		$this->register_module( new TrustLens_Module_Categories() );

		// Register Pro modules (stripped from free version by Freemius deployment).
		if ( class_exists( 'TrustLens_Module_Chargebacks' ) ) {
			$this->register_module( new TrustLens_Module_Chargebacks() );
		}

		// Allow extensions to register modules.
		do_action( 'trustlens/register_modules', $this );

		// Initialize registered modules.
		foreach ( $this->modules as $module ) {
			if ( $module->is_enabled() ) {
				$module->register_hooks();
			}
		}
	}

	/**
	 * Register a module.
	 *
	 * @since 1.0.0
	 * @param TrustLens_Module $module Module instance.
	 */
	public function register_module( TrustLens_Module $module ): void {
		$this->modules[ $module->get_id() ] = $module;
	}

	/**
	 * Get all registered modules.
	 *
	 * @since 1.0.0
	 * @return array Registered modules.
	 */
	public function get_modules(): array {
		return $this->modules;
	}

	/**
	 * Get active modules.
	 *
	 * @since 1.0.0
	 * @return array Active modules.
	 */
	public function get_active_modules(): array {
		return array_filter( $this->modules, function( $module ) {
			return $module->is_enabled();
		} );
	}

	/**
	 * Process score calculation (called via Action Scheduler).
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 */
	public function process_score_calculation( string $email_hash ): void {
		$calculator = new TrustLens_Score_Calculator();
		$result = $calculator->calculate( $email_hash );

		if ( $result ) {
			wstl_update_customer( $email_hash, array(
				'trust_score'      => $result['score'],
				'segment'          => $result['segment'],
				'score_updated_at' => current_time( 'mysql' ),
			) );

			// Fire action for extensions.
			do_action( 'trustlens/score_updated', $email_hash, $result['score'], $result['segment'] );
		}
	}
}
