<?php
/**
 * Trust score calculator.
 *
 * @package TrustLens
 * @since   1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Score calculator class.
 *
 * @since 1.0.0
 */
class TrustLens_Score_Calculator {

	/**
	 * Base score for new customers.
	 *
	 * @var int
	 */
	private int $base_score = 50;

	/**
	 * Collected signals.
	 *
	 * @var array
	 */
	private array $signals = array();

	/**
	 * Calculate trust score for customer.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @return array|null Result array or null on failure.
	 */
	public function calculate( string $email_hash ): ?array {
		$customer = wstl_get_customer( $email_hash );

		if ( ! $customer ) {
			return null;
		}

		// Check allowlist - always trusted.
		if ( $customer->is_allowlisted ) {
			return array(
				'score'   => 100,
				'segment' => 'vip',
				'signals' => array(),
			);
		}

		// Check minimum orders threshold.
		$min_orders = (int) get_option( 'trustlens_min_orders', 3 );
		if ( $customer->total_orders < $min_orders ) {
			return array(
				'score'   => $this->base_score,
				'segment' => 'normal',
				'signals' => array(
					array(
						'module' => 'system',
						'score'  => 0,
						'reason' => sprintf(
							/* translators: %1$d: current order count, %2$d: minimum orders required */
							__( 'Insufficient data (%1$d/%2$d orders)', 'trustlens' ),
							$customer->total_orders,
							$min_orders
						),
					),
				),
			);
		}

		// Reset signals.
		$this->signals = array();

		// Collect signals from all active modules.
		$modules = wstl_get_active_modules();

		foreach ( $modules as $module ) {
			$signal = $module->get_signal( $email_hash );

			if ( ! empty( $signal['score'] ) || ! empty( $signal['reason'] ) ) {
				$this->signals[] = array(
					'module' => $module->get_id(),
					'score'  => $signal['score'] ?? 0,
					'reason' => $signal['reason'] ?? '',
				);
			}
		}

		// Add account age bonus.
		$age_bonus = $this->get_account_age_bonus( $customer );
		if ( $age_bonus > 0 ) {
			$this->signals[] = array(
				'module' => 'account_age',
				'score'  => $age_bonus,
				'reason' => $this->get_age_bonus_reason( $customer ),
			);
		}

		/**
		 * Filter signals before final score calculation.
		 *
		 * @since 1.1.0
		 * @param array  $signals    Current signals.
		 * @param string $email_hash Customer email hash.
		 */
		$this->signals = apply_filters( 'trustlens/score_signals', $this->signals, $email_hash );

		// Calculate total adjustment.
		$total_adjustment = 0;
		foreach ( $this->signals as $signal ) {
			$total_adjustment += $signal['score'];
		}

		// Calculate final score.
		$final_score = $this->base_score + $total_adjustment;

		// Clamp to 0-100.
		$final_score = max( 0, min( 100, $final_score ) );

		// Apply filter for customization.
		$final_score = apply_filters( 'trustlens/trust_score', $final_score, $email_hash, $this->signals );

		// Determine segment.
		$segment = wstl_get_segment_from_score( $final_score );

		// Log signals for debugging.
		$this->log_signals( $email_hash, $final_score );

		return array(
			'score'   => (int) $final_score,
			'segment' => $segment,
			'signals' => $this->signals,
		);
	}

	/**
	 * Get account age bonus.
	 *
	 * @since 1.0.0
	 * @param object $customer Customer object.
	 * @return int Bonus score.
	 */
	private function get_account_age_bonus( object $customer ): int {
		if ( empty( $customer->first_order_date ) ) {
			return 0;
		}

		$days = wstl_days_between( $customer->first_order_date, current_time( 'timestamp' ) );

		if ( $days >= 365 ) {
			return 15;
		}
		if ( $days >= 180 ) {
			return 10;
		}
		if ( $days >= 90 ) {
			return 5;
		}

		return 0;
	}

	/**
	 * Get age bonus reason.
	 *
	 * @since 1.0.0
	 * @param object $customer Customer object.
	 * @return string Reason text.
	 */
	private function get_age_bonus_reason( object $customer ): string {
		$days = wstl_days_between( $customer->first_order_date, current_time( 'timestamp' ) );

		if ( $days >= 365 ) {
			return __( 'Long-term customer (1+ year)', 'trustlens' );
		}
		if ( $days >= 180 ) {
			return __( 'Established customer (6+ months)', 'trustlens' );
		}
		if ( $days >= 90 ) {
			return __( 'Regular customer (3+ months)', 'trustlens' );
		}

		return '';
	}

	/**
	 * Log signals to database.
	 *
	 * @since 1.0.0
	 * @param string $email_hash  Customer email hash.
	 * @param int    $final_score Final score.
	 */
	private function log_signals( string $email_hash, int $final_score ): void {
		global $wpdb;

		// Delete old signals for this customer.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$wpdb->delete(
			$wpdb->prefix . 'trustlens_signals',
			array( 'email_hash' => $email_hash )
		);

		// Insert new signals.
		foreach ( $this->signals as $signal ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
			$wpdb->insert(
				$wpdb->prefix . 'trustlens_signals',
				array(
					'email_hash'    => $email_hash,
					'module_id'     => $signal['module'],
					'signal_score'  => $signal['score'],
					'signal_reason' => $signal['reason'],
					'created_at'    => current_time( 'mysql' ),
				),
				array( '%s', '%s', '%d', '%s', '%s' )
			);
		}

		// Store final score as score_update so dashboard Trust Score Trends uses actual 0-100 scores.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name from $wpdb->prefix, safe.
		$wpdb->insert(
			$wpdb->prefix . 'trustlens_signals',
			array(
				'email_hash'    => $email_hash,
				'module_id'     => 'score_update',
				'signal_score'  => $final_score,
				'signal_reason' => __( 'Store snapshot', 'trustlens' ),
				'created_at'    => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%d', '%s', '%s' )
		);
	}

	/**
	 * Get signals for customer.
	 *
	 * @since 1.0.0
	 * @param string $email_hash Customer email hash.
	 * @return array Signals array.
	 */
	public static function get_signals( string $email_hash ): array {
		global $wpdb;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix, safe.
		return $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}trustlens_signals WHERE email_hash = %s ORDER BY signal_score ASC",
			$email_hash
		) );
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
	}

	/**
	 * Single instance.
	 *
	 * @var TrustLens_Score_Calculator|null
	 */
	private static ?TrustLens_Score_Calculator $instance = null;

	/**
	 * Get instance.
	 *
	 * @since 1.1.0
	 * @return TrustLens_Score_Calculator
	 */
	public static function instance(): TrustLens_Score_Calculator {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Recalculate and save score for a customer.
	 *
	 * @since 1.1.0
	 * @param string $email_hash Customer email hash.
	 * @return array|null Result array or null on failure.
	 */
	public function recalculate_score( string $email_hash ): ?array {
		$result = $this->calculate( $email_hash );

		if ( $result ) {
			wstl_update_customer( $email_hash, array(
				'trust_score'      => $result['score'],
				'segment'          => $result['segment'],
				'score_updated_at' => current_time( 'mysql' ),
			) );

			// Fire action for extensions.
			do_action( 'trustlens/score_updated', $email_hash, $result['score'], $result['segment'] );
		}

		return $result;
	}
}
